/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Sean Maher <seanptmaher@gmail.com>
 */

#if !defined(SIMDE_ARM_NEON_ZIP2_H)
#define SIMDE_ARM_NEON_ZIP2_H

#include "types.h"

HEDLEY_DIAGNOSTIC_PUSH
SIMDE_DISABLE_UNWANTED_DIAGNOSTICS
SIMDE_BEGIN_DECLS_

SIMDE_FUNCTION_ATTRIBUTES
simde_float32x2_t
simde_vzip2_f32(simde_float32x2_t a, simde_float32x2_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2_f32(a, b);
  #else
    simde_float32x2_private
      r_,
      a_ = simde_float32x2_to_private(a),
      b_ = simde_float32x2_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_float32x2_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2_f32
  #define vzip2_f32(a, b) simde_vzip2_f32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int8x8_t
simde_vzip2_s8(simde_int8x8_t a, simde_int8x8_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2_s8(a, b);
  #else
    simde_int8x8_private
      r_,
      a_ = simde_int8x8_to_private(a),
      b_ = simde_int8x8_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_int8x8_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2_s8
  #define vzip2_s8(a, b) simde_vzip2_s8((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int16x4_t
simde_vzip2_s16(simde_int16x4_t a, simde_int16x4_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2_s16(a, b);
  #else
    simde_int16x4_private
      r_,
      a_ = simde_int16x4_to_private(a),
      b_ = simde_int16x4_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_int16x4_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2_s16
  #define vzip2_s16(a, b) simde_vzip2_s16((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int32x2_t
simde_vzip2_s32(simde_int32x2_t a, simde_int32x2_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2_s32(a, b);
  #else
    simde_int32x2_private
      r_,
      a_ = simde_int32x2_to_private(a),
      b_ = simde_int32x2_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_int32x2_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2_s32
  #define vzip2_s32(a, b) simde_vzip2_s32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint8x8_t
simde_vzip2_u8(simde_uint8x8_t a, simde_uint8x8_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2_u8(a, b);
  #else
    simde_uint8x8_private
      r_,
      a_ = simde_uint8x8_to_private(a),
      b_ = simde_uint8x8_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_uint8x8_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2_u8
  #define vzip2_u8(a, b) simde_vzip2_u8((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint16x4_t
simde_vzip2_u16(simde_uint16x4_t a, simde_uint16x4_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2_u16(a, b);
  #else
    simde_uint16x4_private
      r_,
      a_ = simde_uint16x4_to_private(a),
      b_ = simde_uint16x4_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_uint16x4_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2_u16
  #define vzip2_u16(a, b) simde_vzip2_u16((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint32x2_t
simde_vzip2_u32(simde_uint32x2_t a, simde_uint32x2_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2_u32(a, b);
  #else
    simde_uint32x2_private
      r_,
      a_ = simde_uint32x2_to_private(a),
      b_ = simde_uint32x2_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_uint32x2_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2_u32
  #define vzip2_u32(a, b) simde_vzip2_u32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float32x4_t
simde_vzip2q_f32(simde_float32x4_t a, simde_float32x4_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2q_f32(a, b);
  #elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_v8x16_shuffle(a, b,
                             8, 9, 10, 11, 24, 25, 26, 27, 12, 13, 14, 15, 28, 29, 30, 31);
  #else
    simde_float32x4_private
      r_,
      a_ = simde_float32x4_to_private(a),
      b_ = simde_float32x4_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_float32x4_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2q_f32
  #define vzip2q_f32(a, b) simde_vzip2q_f32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float64x2_t
simde_vzip2q_f64(simde_float64x2_t a, simde_float64x2_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2q_f64(a, b);
  #elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_v8x16_shuffle(a, b,
                             8, 9, 10, 11, 12, 13, 14, 15, 24, 25, 26, 27, 28, 29, 30, 31);
  #else
    simde_float64x2_private
      r_,
      a_ = simde_float64x2_to_private(a),
      b_ = simde_float64x2_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_float64x2_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2q_f64
  #define vzip2q_f64(a, b) simde_vzip2q_f64((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int8x16_t
simde_vzip2q_s8(simde_int8x16_t a, simde_int8x16_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2q_s8(a, b);
  #elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_v8x16_shuffle(a, b,
                             8, 24, 9, 25, 10, 26, 11, 27, 12, 28, 13, 29, 14, 30, 15, 31);
  #else
    simde_int8x16_private
      r_,
      a_ = simde_int8x16_to_private(a),
      b_ = simde_int8x16_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_int8x16_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2q_s8
  #define vzip2q_s8(a, b) simde_vzip2q_s8((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int16x8_t
simde_vzip2q_s16(simde_int16x8_t a, simde_int16x8_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2q_s16(a, b);
  #elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_v8x16_shuffle(a, b,
                             8, 9, 24, 25, 10, 11, 26, 27, 12, 13, 28, 29, 14, 15, 30, 31);
  #else
    simde_int16x8_private
      r_,
      a_ = simde_int16x8_to_private(a),
      b_ = simde_int16x8_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_int16x8_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2q_s16
  #define vzip2q_s16(a, b) simde_vzip2q_s16((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int32x4_t
simde_vzip2q_s32(simde_int32x4_t a, simde_int32x4_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2q_s32(a, b);
  #elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_v8x16_shuffle(a, b,
                             8, 9, 10, 11, 24, 25, 26, 27, 12, 13, 14, 15, 28, 29, 30, 31);
  #else
    simde_int32x4_private
      r_,
      a_ = simde_int32x4_to_private(a),
      b_ = simde_int32x4_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_int32x4_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2q_s32
  #define vzip2q_s32(a, b) simde_vzip2q_s32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int64x2_t
simde_vzip2q_s64(simde_int64x2_t a, simde_int64x2_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2q_s64(a, b);
  #elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_v8x16_shuffle(a, b,
                             8, 9, 10, 11, 12, 13, 14, 15, 24, 25, 26, 27, 28, 29, 30, 31);
  #else
    simde_int64x2_private
      r_,
      a_ = simde_int64x2_to_private(a),
      b_ = simde_int64x2_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_int64x2_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2q_s64
  #define vzip2q_s64(a, b) simde_vzip2q_s64((a), (b))
#endif


SIMDE_FUNCTION_ATTRIBUTES
simde_uint8x16_t
simde_vzip2q_u8(simde_uint8x16_t a, simde_uint8x16_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2q_u8(a, b);
  #elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_v8x16_shuffle(a, b,
                             8, 24, 9, 25, 10, 26, 11, 27, 12, 28, 13, 29, 14, 30, 15, 31);
  #else
    simde_uint8x16_private
      r_,
      a_ = simde_uint8x16_to_private(a),
      b_ = simde_uint8x16_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_uint8x16_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2q_u8
  #define vzip2q_u8(a, b) simde_vzip2q_u8((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint16x8_t
simde_vzip2q_u16(simde_uint16x8_t a, simde_uint16x8_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2q_u16(a, b);
  #elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_v8x16_shuffle(a, b,
                             8, 9, 24, 25, 10, 11, 26, 27, 12, 13, 28, 29, 14, 15, 30, 31);
  #else
    simde_uint16x8_private
      r_,
      a_ = simde_uint16x8_to_private(a),
      b_ = simde_uint16x8_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_uint16x8_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2q_u16
  #define vzip2q_u16(a, b) simde_vzip2q_u16((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint32x4_t
simde_vzip2q_u32(simde_uint32x4_t a, simde_uint32x4_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2q_u32(a, b);
  #elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_v8x16_shuffle(a, b,
                             8, 9, 10, 11, 24, 25, 26, 27, 12, 13, 14, 15, 28, 29, 30, 31);
  #else
    simde_uint32x4_private
      r_,
      a_ = simde_uint32x4_to_private(a),
      b_ = simde_uint32x4_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_uint32x4_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2q_u32
  #define vzip2q_u32(a, b) simde_vzip2q_u32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint64x2_t
simde_vzip2q_u64(simde_uint64x2_t a, simde_uint64x2_t b) {
  #if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
    return vzip2q_u64(a, b);
  #elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_v8x16_shuffle(a, b,
                             8, 9, 10, 11, 12, 13, 14, 15, 24, 25, 26, 27, 28, 29, 30, 31);
  #else
    simde_uint64x2_private
      r_,
      a_ = simde_uint64x2_to_private(a),
      b_ = simde_uint64x2_to_private(b);

    const size_t halfway_point = sizeof(r_.values) / sizeof(r_.values[0]) / 2;
    SIMDE_VECTORIZE
    for (size_t i = 0 ; i < halfway_point ; i++) {
      r_.values[(2 * i)    ] = a_.values[halfway_point + i];
      r_.values[(2 * i) + 1] = b_.values[halfway_point + i];
    }

    return simde_uint64x2_from_private(r_);
  #endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
  #undef vzip2q_u64
  #define vzip2q_u64(a, b) simde_vzip2q_u64((a), (b))
#endif

SIMDE_END_DECLS_
HEDLEY_DIAGNOSTIC_POP

#endif /* !defined(SIMDE_ARM_NEON_ZIP2_H) */
