/***********************************************************************
* File name: synaptics.c                                               *
* Written by: Bruce Kall (extracted from tpconfig.c by C. Scott Ananian*
* Language: C                                                          *
* Version: 1.0                                                         *
* Purpose: Routines to manipulate the Synaptics touchpad.              *
*                                                                      *
*                                                                      *
***********************************************************************/

 
/* Copyright (C) 1997  C. Scott Ananian <cananian@alumni.princeton.edu>
 * Copyright (c) 1998-2000 Bruce Kalk <kall@compass.com>
 *
 * Currently maintained by: Bruce Kall, <kall@compass.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   $Id: synaptics.c,v 1.9 2002/02/22 20:31:30 bruce Exp $
 */

/*$Log: synaptics.c,v $
 *Revision 1.9  2002/02/22 20:31:30  bruce
 *Added for Mode 4 and Mode 4 options .. -M
 *
 *Revision 1.8  2001/04/23 20:37:38  bruce
 *Chanegd if(b1=AUX_ACK) to while(b1=AUX_ACK) to get
 *around kernel ack problem.
 *
 *Revision 1.7  2000/10/31 18:39:15  cph
 *Fix tab problem.
 *
 *Revision 1.6  2000/10/31 18:38:15  cph
 *Implement --sleep option to control sleep mode.
 *
 *Revision 1.5  2000/10/31 18:29:00  cph
 *Fix bug: was calling show_tap_mode in two places that should have been
 *calling show_packet_mode.
 *Fix bug: disallow -m for single_mode_byte pads.
 *Adjust usage message to show options appropriate for this pad.
 *
 *Revision 1.4  2000/10/31 18:17:40  cph
 *Fix error in usage message.  Add comments back to option switch.
 *
 *Revision 1.3  2000/10/31 18:07:49  cph
 ** Extern declarations for utilities are now in .h file.
 *
 ** Fix typo in usage.
 *
 ** Fix bug in send_cmd.
 *
 ** Add high-level debugging output.
 *
 ** Capture version information in is_Synaptics to eliminate unneeded
 *  additional identify request in set_firmware_options.
 *
 *Revision 1.2  2000/10/31 15:18:32  cph
 *Complete rewrite.  Modularized code, added symbolic definitions for
 *commands and bit layouts.
 *
 *Revision 1.1  2000/09/28 13:37:36  bruce
 *Initial revision
 **/

static char rcsid[]="$Header: /home/bruce/linux-stuff/tpconfig/tpconfig-3.1.3/RCS/synaptics.c,v 1.9 2002/02/22 20:31:30 bruce Exp $";

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <assert.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <getopt.h>
#include <termios.h>
#include "tpconfig.h"

/* Capability bits */
#define STP_CAP_EXTENDED	0x8000 /* 1 = other cap bits supported */
#define STP_CAP_SLEEP		0x0010 /* 1 = sleep mode */
#define STP_CAP_FOUR_BUTTON	0x0004 /* 1 = four buttons */
#define STP_CAP_MULTI_FINGER	0x0002 /* 1 = multi-finger detection */
#define STP_CAP_PALM_DETECT	0x0001 /* 1 = palm detection */

/* Mode bits */
#define STP_MODE_ABSOLUTE(m)	(((m) & 0x80) != 0)
#define STP_MODE_RATE(m)	(((m) & 0x40) != 0)
#define STP_MODE_SLEEP(m)	(((m) & 0x08) != 0)
#define STP_MODE_GESTURE(m)	(((m) & 0x04) == 0)
#define STP_MODE_WMODE(m)	(((m) & 0x01) != 0)
/* STP_MODE_WMODE valid only in absolute mode when STP_CAP_EXTENDED is set. */

#define STP_SET_MODE_ABSOLUTE(m, a)	(((m) &~ 0x80) | ((a) << 7))
#define STP_SET_MODE_RATE(m, a)		(((m) &~ 0x40) | ((a) << 6))
#define STP_SET_MODE_SLEEP(m, a)	(((m) &~ 0x08) | ((a) << 3))
#define STP_SET_MODE_GESTURE(m, a)	(((m) &~ 0x04) | ((!a) << 2))
#define STP_SET_MODE_WMODE(m, a)	(((m) &~ 0x01) | (a))

/* Old-style mode bits */
#define STP_OMODE_CORNER(m)	(((m) & 0x80000000) >> 31)
#define STP_OMODE_Z_THRESH(m)	(((m) & 0x70000000) >> 28)
#define STP_OMODE_TAP_MODE(m)	(((m) & 0x0C000000) >> 26)
#define STP_OMODE_EDGE_MOTN(m)	(((m) & 0x03000000) >> 24)
#define STP_OMODE_ABSOLUTE(m)	(((m) & 0x00800000) >> 23)
#define STP_OMODE_RATE(m)	(((m) & 0x00400000) >> 22)
#define STP_OMODE_BAUD(m)	(((m) & 0x00080000) >> 19)
#define STP_OMODE_3_BUTTON(m)	(((m) & 0x00040000) >> 18)
#define STP_OMODE_MIDDLE(m)	(((m) & 0x00020000) >> 17)
#define STP_OMODE_HOP(m)	(((m) & 0x00010000) >> 16)
#define STP_OMODE_RIGHT_MGN(m)	(((m) & 0x0000F000) >> 12)
#define STP_OMODE_LEFT_MGN(m)	(((m) & 0x00000F00) >> 8)
#define STP_OMODE_TOP_MGN(m)	(((m) & 0x000000F0) >> 4)
#define STP_OMODE_BOTTOM_MGN(m)	 ((m) & 0x0000000F)

#define STP_SET_OMODE_CORNER(m, a)	(((m) &~ 0x80000000) | ((a) << 31))
#define STP_SET_OMODE_Z_THRESH(m, a)	(((m) &~ 0x70000000) | ((a) << 28))
#define STP_SET_OMODE_TAP_MODE(m, a)	(((m) &~ 0x0C000000) | ((a) << 26))
#define STP_SET_OMODE_EDGE_MOTN(m, a)	(((m) &~ 0x03000000) | ((a) << 24))
#define STP_SET_OMODE_ABSOLUTE(m, a)	(((m) &~ 0x00800000) | ((a) << 23))
#define STP_SET_OMODE_RATE(m, a)	(((m) &~ 0x00400000) | ((a) << 22))
#define STP_SET_OMODE_BAUD(m, a)	(((m) &~ 0x00080000) | ((a) << 19))
#define STP_SET_OMODE_3_BUTTON(m, a)	(((m) &~ 0x00040000) | ((a) << 18))
#define STP_SET_OMODE_MIDDLE(m, a)	(((m) &~ 0x00020000) | ((a) << 17))
#define STP_SET_OMODE_HOP(m, a)		(((m) &~ 0x00010000) | ((a) << 16))
#define STP_SET_OMODE_RIGHT_MGN(m, a)	(((m) &~ 0x0000F000) | ((a) << 12))
#define STP_SET_OMODE_LEFT_MGN(m, a)	(((m) &~ 0x00000F00) | ((a) << 8))
#define STP_SET_OMODE_TOP_MGN(m, a)	(((m) &~ 0x000000F0) | ((a) << 4))
#define STP_SET_OMODE_BOTTOM_MGN(m, a)	(((m) &~ 0x0000000F) |  (a))

/* Model ID bits */
#define STP_INFO_ROT180(id)	(((id) & 0x800000) >> 23)
#define STP_INFO_PORTRAIT(id)	(((id) & 0x400000) >> 22)
#define STP_INFO_SENSOR(id)	(((id) & 0x3F0000) >> 16)
#define STP_INFO_HARDWARE(id)	(((id) & 0x00FE00) >> 9)
#define STP_INFO_NEW_ABS(id)	(((id) & 0x000080) >> 7)
#define STP_INFO_CAP_PEN(id)	(((id) & 0x000040) >> 6)
#define STP_INFO_SIMPLE_CMD(id)	(((id) & 0x000020) >> 5)
#define STP_INFO_GEOMETRY(id)	 ((id) & 0x00000F)

/* Sensor types */
#define STP_SENSOR_UNKNOWN	0
#define STP_SENSOR_STANDARD	1
#define STP_SENSOR_MINI		2
#define STP_SENSOR_SUPER	3
#define STP_SENSOR_FLEXIBLE	7
#define STP_SENSOR_ULTRA_THIN	8
#define STP_SENSOR_WIDE_MODULE	9
#define STP_SENSOR_STAMP_MODULE	11
#define STP_SENSOR_SUBMINI	12
#define STP_SENSOR_MULTISWITCH	13
#define STP_SENSOR_ADV_TECH	15
#define STP_SENSOR_ULTRA_THIN_R	16

/* Geometry types */
#define STP_GEOMETRY_UNKNOWN	0
#define STP_GEOMETRY_STANDARD	1
#define STP_GEOMETRY_OVAL	2

/* Synaptics special queries */
#define STP_QRY_IDENTIFY	0x00
#define STP_QRY_READ_MODES	0x01
#define STP_QRY_READ_CAPS	0x02
#define STP_QRY_READ_MODEL_ID	0x03
#define STP_QRY_READ_SN_PREFIX	0x06
#define STP_QRY_READ_SN_SUFFIX	0x07
#define STP_QRY_READ_RES	0x08

static unsigned int fw_version_major;
static unsigned int fw_version_minor;
static unsigned int fw_version;
static unsigned int model_id;
/* Firmware prior to version 3.2 has four mode bytes */
#define QUAD_MODE_BYTE		0x0302

void
synaptics_usage (char *progname) 
{
  copyright ();
  printf ("Usage: %s [OPTION]...\n", progname);
  printf
    ("Configure a Synaptics TouchPad.\n"
     "\n"
     "  -i, --info                    display current TouchPad configuration\n"
     "  -x, --reset                   perform a software reset on the TouchPad\n"
     "  -q, --quiet, --silent         suppress verbose output\n"
     "\n" 
     "  -a                            display absolute/relative mode\n"
     "      --absolute, --relative    set absolute/relative mode\n"
     "  -r, --rate=[0,1]              display/set reporting rate\n"
     "                                   0 = normal, 1 = high\n");
  if (single_mode_byte)
    printf
      ("  -t, --tapmode=[0-1]           display/set tapping mode:\n"
       "                                   0 = no tap gestures\n"
       "                                   1 = tap and drag\n"
       "  -s, --sleep=[0-1]             display/set sleep mode:\n"
       "                                   0 = disable, 1 = enable\n");
  else
    {
      printf
	("  -t, --tapmode=[0-3]           display/set tapping mode:\n"
	 "                                   0 = no tap gestures\n"
	 "                                   1 = tap-to-click only\n"
	 "                                   2 = tap and non-locking drag\n"
	 "                                   3 = tap and locking drag\n"
	 "  -2, --two-button              set two-button mode\n"
	 "  -3, --three-button            set three-button mode\n"
	 "  -c, --corner=[0-1]            display/enable corner taps:\n"
	 "                                   0 = disable, 1 = enable\n"
	 "  -e, --edgemode=[0,1,3]        display/set edge motion:\n"
	 "                                   0 = never\n"
	 "                                   1 = always\n"
	 "                                   3 = only during drag\n"
	 "  -m                            display which button corner taps simulate\n"
	 "      --middle-button           make corner taps simulate middle button\n"
	 "      --right-button            make corner taps simulate right button\n"
	 "  -z                            display z threshold setting\n"
	 "      --zthreshold=[0-7]        set z threshold (tap sensitivity; 0=light)\n"
	 "      --threshold=[0-7]         same as --zthreshold\n");
      if (fw_version < QUAD_MODE_BYTE)
	{
	  printf
	    ("  -M                            display edge margins\n"
	     "      --right-margin=[0-15]     set right edge margin (0=narrow)\n"
	     "      --left-margin=[0-15]      set left edge margin (0=narrow)\n"
	     "      --top-margin=[0-15]       set top edge margin (0=narrow)\n"
	     "      --bottom-margin=[0-15]    set bottom edge margin (0=narrow)\n");
	}
    }
  printf
    ("\n"
     "      --help                    display this help and exit\n"
     "      --version                 output version information\n"
     "\n"
     "Report bugs to <kall@compass.com>\n");
  exit(1);
}


/******************** Synaptics Specific Functions ****************************/

/* use the Synaptics extended ps/2 syntax to write a special command byte */
static void
send_cmd (int fd, byte cmd) 
{
  putbyte (fd, AUX_SET_SCALE11);
  putbyte (fd, AUX_SET_RES);
  putbyte (fd, (cmd & 0xC0) >> 6);
  putbyte (fd, AUX_SET_RES);
  putbyte (fd, (cmd & 0x30) >> 4);
  putbyte (fd, AUX_SET_RES);
  putbyte (fd, (cmd & 0x0C) >> 2);
  putbyte (fd, AUX_SET_RES);
  putbyte (fd,  cmd & 0x03);
}

static void
status_rqst (int fd, byte cmd, byte *byte1, byte *byte2, byte *byte3) 
{
  send_cmd (fd, cmd);
  putbyte (fd, AUX_GET_STATUS);
  *byte1 = getbyte (fd);
  *byte2 = getbyte (fd);
  *byte3 = getbyte (fd);
  if (DEBUG_LEVEL)
    fprintf (stderr, "[query %#02x => %#02x %#02x %#02x]\n",
	     cmd, *byte1, *byte2, *byte3);
}

static void
set_modes (int fd, unsigned int modes)
{
  if (DEBUG_LEVEL)
    fprintf (stderr, "[set modes: %#02x]\n", modes);
  if (single_mode_byte)
    {
      send_cmd (fd, modes);
      putbyte (fd, AUX_SET_SAMPLE);
      putbyte (fd, 20);
    }
  else
    {
      send_cmd (fd, (modes & 0xFF000000) >> 24);
      putbyte (fd, AUX_SET_SAMPLE);
      putbyte (fd, 10);
      send_cmd (fd, (modes & 0x00FF0000) >> 16);
      putbyte (fd, AUX_SET_SAMPLE);
      putbyte (fd, 20);
      if (fw_version < QUAD_MODE_BYTE)
	{
	  send_cmd (fd, (modes & 0x0000FF00) >> 8);
	  putbyte (fd, AUX_SET_SAMPLE);
	  putbyte (fd, 40);
	  send_cmd (fd, (modes & 0x000000FF));
	  putbyte (fd, AUX_SET_SAMPLE);
	  putbyte (fd, 60);
	}
    }
}

static void
query_identify (int fd,
		unsigned int *version_major,
		unsigned int *version_minor,
		unsigned int *model_code)
{
  byte b1, b2, b3;
  status_rqst (fd, STP_QRY_IDENTIFY, &b1, &b2, &b3);
  assert (b2 == 0x47);
  if (version_major) (*version_major) = (b3 & 0x0F);
  if (version_minor) (*version_minor) = b1;
  if (model_code)    (*model_code)    = ((b3 & 0xF0) >> 4);
}

static unsigned int
query_modes (int fd)
{
  byte b1, b2, b3;
  byte b4, b5, b6;
  status_rqst (fd, STP_QRY_READ_MODES, &b1, &b2, &b3);
  assert (b2 == 0x47);
  if (single_mode_byte)
    {
      assert (b1 == 0x3B);
      return b3;
    }
  else
    {
      /* Firmware prior to version 3.2 has four mode bytes */
      if (fw_version < QUAD_MODE_BYTE)
	{
	  status_rqst (fd, STP_QRY_READ_CAPS, &b4, &b5, &b6);
	  assert (b5 == 0x47);
	  return ((unsigned int) b1 << 24) | ((unsigned int) b3 << 16) |
	    ((unsigned int) b4 << 8) | (unsigned int) b6;
	}
      return ((unsigned int) b1 << 24) | ((unsigned int) b3 << 16);
    }
}

static unsigned int
query_caps (int fd)
{
  byte b1, b2, b3;
  status_rqst (fd, STP_QRY_READ_CAPS, &b1, &b2, &b3);
  assert (b2 == 0x47);
  return ((unsigned int) b1 << 8) | (unsigned int) b3;
}

static unsigned int
query_model_id (int fd)
{
  byte b1, b2, b3;
  unsigned int id;
  status_rqst (fd, STP_QRY_READ_MODEL_ID, &b1, &b2, &b3);
  id = ((b2 & 1)
	? 0x000001
	: (((unsigned int) b1 << 16)
	   | ((unsigned int) b2 << 8)
	   | (unsigned int) b3));
  single_mode_byte = STP_INFO_SIMPLE_CMD (id);
  return id;
}

static void
query_serial_number (int fd, unsigned int *sn_high, unsigned int *sn_low)
{
  byte b1, b2, b3, b4, b5, b6;
  status_rqst (fd, STP_QRY_READ_SN_PREFIX, &b1, &b2, &b3);
  status_rqst (fd, STP_QRY_READ_SN_SUFFIX, &b4, &b5, &b6);
  (*sn_high) = (b2 & 0xF0) >> 4;
  (*sn_low) = (((unsigned int) b1 << 24)
	       | ((unsigned int) b4 << 16)
	       | ((unsigned int) b5 << 8)
	       | (unsigned int) b6);
}

static void
query_resolutions (int fd, unsigned int *xres, unsigned int *yres)
{
  byte b1, b2, b3;
  status_rqst (fd, STP_QRY_READ_RES, &b1, &b2, &b3);
  assert ((b2 & 0x80) != 0);
  (*xres) = b1;
  (*yres) = b3;
}

/* initialize and determine whether this is a synaptics touchpad */
int
is_Synaptics (int fd)
{
  byte b1, b2, b3;
  int retval;
  status_rqst (fd, STP_QRY_IDENTIFY, &b1, &b2, &b3);
#if defined(__linux) && !defined(PATCHED_LINUX_KERNEL)
  while (b1 == AUX_ACK)
    {
      /* 2.2.x kernel fix an extra ack preceeded the status */
      b1 = b2;
      b2 = b3;
      b3 = getbyte (fd);
    }
#endif
  retval = (b2 == 0x47);
  if (retval)
    {
      fw_version_major = (b3 & 0x0F);
      fw_version_minor = b1;
      fw_version = (fw_version_major << 8) | fw_version_minor;
    }
  return retval;
}

void
reset_synaptics (int fd)
{
  fprintf (stderr,"Performing software reset on TouchPad.\n");
  putbyte (fd, AUX_RESET);
  getbyte_expected (fd, AUX_RESET_ACK1, "software reset ACK 1");
  getbyte_expected (fd, AUX_RESET_ACK2, "software reset ACK 2");
  fprintf (stderr, "Software reset of TouchPad complete.\n");
}

void
set_firmware_options (int fd, FILE *out)
{
  model_id = query_model_id (fd);
  firmware_rev
    = ((float) fw_version_major) + ((float) fw_version_minor / 10.0);
  fprintf (out, "Firmware: %d.%d (%s-byte mode).\n",
	   fw_version_major, fw_version_minor,
	   single_mode_byte ? "single" : "multiple");
}

static void
show_packet_mode (unsigned int modes, FILE *info)
{
  if (single_mode_byte)
    fprintf (info, "Packets: %s, %d packets per second.\n",
	     (STP_MODE_ABSOLUTE (modes)
	      ? (STP_INFO_NEW_ABS (model_id)
		 ? "absolute (new style)"
		 : "absolute (old style)")
	      : "relative"),
	     STP_MODE_RATE (modes) ? 80 : 40);
  else
    fprintf (info, "Packets: %s, %d packets per second.\n",
	     STP_OMODE_ABSOLUTE (modes) ? "absolute" : "relative",
	     STP_OMODE_RATE (modes) ? 80 : 40);
}

static void
show_sleep_mode (unsigned int modes, FILE *info)
{
  fprintf (info, "Sleep mode: %s.\n",
	   STP_MODE_SLEEP (modes) ? "on" : "off");
}

static void
show_tap_mode (unsigned int modes, FILE *info)
{
  if (single_mode_byte)
    fprintf (info, "Tap gestures: %s.\n",
	     STP_MODE_GESTURE (modes) ? "enabled" : "disabled");
  else
    {
      fprintf (info, "Corner taps %s;\t\t",
	       STP_OMODE_CORNER (modes) ? "enabled" : "disabled");
      switch (STP_OMODE_TAP_MODE (modes))
	{
	case 0:
	  fprintf (info, "no tap gestures");
	  break;
	case 1:
	  fprintf (info, "tap-to-click only");
	  break;
	case 2:
	  fprintf (info, "tap and non-locking drag");
	  break;
	case 3:
	  fprintf (info, "tap and locking drag");
	  break;
	}
      fprintf (info, ".\n");
    }
}

static void
show_edge_motion (unsigned int modes, FILE *info)
{
  fprintf (info, "Edge motion: ");
  switch (STP_OMODE_EDGE_MOTN (modes))
    {
    case 0:
      fprintf (info, "none");
      break;
    case 1:
      fprintf (info, "always");
      break;
    case 2:
      fprintf (info, "[illegal value 2]");
      break;
    case 3:
      fprintf (info, "during drag");
      break;
    }
  fprintf (info, ".\n");
}

static void
show_z_threshold (unsigned int modes, FILE *info)
{
  fprintf (info, "Z threshold: %d of 7.\n", STP_OMODE_Z_THRESH (modes));
}

static void
show_button_mode (unsigned int modes, FILE *info)
{
  fprintf (info, "%d button mode; corner tap is %s button click.\n",
	   STP_OMODE_3_BUTTON (modes) ? 3 : 2,
	   STP_OMODE_MIDDLE (modes) ? "middle" : "right");
}
static void
show_margins (unsigned int modes, FILE *info)
{
  fprintf (info, "Right margin: %ld, Left margin: %ld,\n",
	   STP_OMODE_RIGHT_MGN (modes), STP_OMODE_LEFT_MGN (modes));
  fprintf (info, "Top margin: %ld, Bottom margin: %ld.\n",
	   STP_OMODE_TOP_MGN (modes), STP_OMODE_BOTTOM_MGN (modes));
}

static int
get_argument (unsigned int limit)
{
  int delta;
  if (optarg[0] == '\0')
    return -1;
  else if (optarg[1] == '\0')
    {
      delta = optarg[0] - '0';
      return (delta >= 0 && delta <= limit ? delta : -1);
    }
  else if (optarg[2] == '\0')
    {
      delta = (optarg[0] - '0') * 10 + optarg[1] - '0';
      return (delta >= 0 && delta <= limit ? delta : -1);
    }
  else
    return -1;
}

static void
set_z_threshold (int fd)
{
  if (single_mode_byte)
    {
      fprintf (stderr, "Z threshold not supported on this TouchPad.\n");
      return;
    }
  if (optarg)
    {
      int a = get_argument (7);
      if (a < 0)
	{
	  fprintf (stderr, "Invalid Z threshold \"%s\" [use 0-7].\n", optarg);
	  return;
	}
      set_modes (fd, STP_SET_OMODE_Z_THRESH (query_modes (fd), a));
    }
  if (!silent)
    show_z_threshold (query_modes (fd), stderr);
}

static void
set_tap_mode (int fd)
{
  unsigned int limit = single_mode_byte ? 1 : 3;
  if (optarg)
    {
      int a = get_argument (limit);
      if (a < 0)
	{
	  fprintf (stderr, "Invalid tap mode \"%s\" [use 0-%d].\n",
		   optarg, limit);
	  return;
	}
      if (single_mode_byte)
	set_modes (fd, STP_SET_MODE_GESTURE (query_modes (fd), a));
      else
	set_modes (fd, STP_SET_OMODE_TAP_MODE (query_modes (fd), a));
    }
  if (!silent)
    show_tap_mode (query_modes (fd), stderr);
}

static void
set_edge_motion (int fd)
{
  if (single_mode_byte)
    {
      fprintf (stderr, "Edge motion not supported on this TouchPad.\n");
      return;
    }
  if (optarg)
    {
      int a = get_argument (3);
      if (a < 0 || a == 2)
	{
	  fprintf (stderr, "Invalid edge motion \"%s\" [use 0, 1, 3].\n",
		   optarg);
	  return;
	}
      set_modes (fd, STP_SET_OMODE_EDGE_MOTN (query_modes (fd), a));
    }
  if (!silent)
    show_edge_motion (query_modes (fd), stderr);
}

static void
set_corner_mode (int fd)
{
  if (single_mode_byte)
    {
      fprintf (stderr, "Corner mode not supported on this TouchPad.\n");
      return;
    }
  if (optarg)
    {
      int a = get_argument (1);
      if (a < 0)
	{
	  fprintf (stderr, "Invalid corner mode \"%s\" [use 0-1].\n", optarg);
	  return;
	}
      set_modes (fd, STP_SET_OMODE_CORNER (query_modes (fd), a));
    }
  if (!silent)
    show_tap_mode (query_modes (fd), stderr);
}

static void
set_packet_mode (int fd, int absolute)
{
  if (single_mode_byte)
    set_modes (fd, STP_SET_MODE_ABSOLUTE (query_modes (fd), absolute));
  else
    set_modes (fd, STP_SET_OMODE_ABSOLUTE (query_modes (fd), absolute));
  if (!silent)
    show_packet_mode (query_modes (fd), stderr);
}

static void
set_rate (int fd)
{
  if (optarg)
    {
      int a = get_argument (1);
      if (a < 0)
	{
	  fprintf (stderr, "Invalid rate \"%s\" [use 0-%c].\n",
		   optarg);
	  return;
	}
      if (single_mode_byte)
	set_modes (fd, STP_SET_MODE_RATE (query_modes (fd), a));
      else
	set_modes (fd, STP_SET_OMODE_RATE (query_modes (fd), a));
    }
  if (!silent)
    show_packet_mode (query_modes (fd), stderr);
}

static void
set_button_mode (int fd, int three)
{
  if (single_mode_byte)
    {
      fprintf (stderr, "Button mode not supported on this TouchPad.\n");
      return;
    }
  set_modes (fd, STP_SET_OMODE_3_BUTTON (query_modes (fd), three));
  if (!silent)
    show_button_mode (query_modes (fd), stderr);
}

static void
set_corner_click (int fd, int middle)
{
  if (single_mode_byte)
    {
      fprintf (stderr, "Corner click not supported on this TouchPad.\n");
      return;
    }
  set_modes (fd, STP_SET_OMODE_MIDDLE (query_modes (fd), middle));
  if (!silent)
    show_button_mode (query_modes (fd), stderr);
}

static void
set_sleep_mode (int fd)
{
  if (!single_mode_byte)
    {
      fprintf (stderr, "Sleep mode not supported on this TouchPad.\n");
      return;
    }
  if (optarg)
    {
      int a = get_argument (1);
      if (a < 0)
	{
	  fprintf (stderr, "Invalid sleep mode \"%s\" [use 0-1].\n", optarg);
	  return;
	}
      set_modes (fd, STP_SET_MODE_SLEEP (query_modes (fd), a));
    }
  if (!silent)
    show_sleep_mode (query_modes (fd), stderr);
}

static void
set_right_margin (int fd)
{
  if (single_mode_byte || (fw_version >= QUAD_MODE_BYTE))
    {
      fprintf (stderr, "Adjustable margins not supported on this TouchPad.\n");
      return;
    }
  if (optarg)
    {
      int a = get_argument (15);
      if (a < 0)
	{
	  fprintf (stderr, "Invalid right margin \"%s\" [use 0-15].\n", optarg);
	  return;
	}
      set_modes (fd, STP_SET_OMODE_RIGHT_MGN (query_modes (fd), a));
    }
  if (!silent)
    show_margins (query_modes (fd), stderr);
}

static void
set_left_margin (int fd)
{
  if (single_mode_byte || (fw_version >= QUAD_MODE_BYTE))
    {
      fprintf (stderr, "Adjustable margins not supported on this TouchPad.\n");
      return;
    }
  if (optarg)
    {
      int a = get_argument (15);
      if (a < 0)
	{
	  fprintf (stderr, "Invalid left margin \"%s\" [use 0-15].\n", optarg);
	  return;
	}
      set_modes (fd, STP_SET_OMODE_LEFT_MGN (query_modes (fd), a));
    }
  if (!silent)
    show_margins (query_modes (fd), stderr);
}

static void
set_top_margin (int fd)
{
  if (single_mode_byte || (fw_version >= QUAD_MODE_BYTE))
    {
      fprintf (stderr, "Adjustable margins not supported on this TouchPad.\n");
      return;
    }
  if (optarg)
    {
      int a = get_argument (15);
      if (a < 0)
	{
	  fprintf (stderr, "Invalid top margin \"%s\" [use 0-15].\n", optarg);
	  return;
	}
      set_modes (fd, STP_SET_OMODE_TOP_MGN (query_modes (fd), a));
    }
  if (!silent)
    show_margins (query_modes (fd), stderr);
}

static void
set_bottom_margin (int fd)
{
  if (single_mode_byte || (fw_version >= QUAD_MODE_BYTE))
    {
      fprintf (stderr, "Adjustable margins not supported on this TouchPad.\n");
      return;
    }
  if (optarg)
    {
      int a = get_argument (15);
      if (a < 0)
	{
	  fprintf (stderr, "Invalid bottom margin \"%s\" [use 0-15].\n", optarg);
	  return;
	}
      set_modes (fd, STP_SET_OMODE_BOTTOM_MGN (query_modes (fd), a));
    }
  if (!silent)
    show_margins (query_modes (fd), stderr);
}

static void
synaptics_info (int fd, FILE *info)
{
  unsigned int modes;

  fprintf (info, "Sensor type: ");
  switch (STP_INFO_SENSOR (model_id))
    {
    case STP_SENSOR_STANDARD:
      fprintf (info, "Standard TouchPad");
      break;
    case STP_SENSOR_MINI:
      fprintf (info, "Mini module");
      break;
    case STP_SENSOR_SUPER:
      fprintf (info, "Super module");
      break;
    case STP_SENSOR_FLEXIBLE:
      fprintf (info, "Flexible pad");
      break;
    case STP_SENSOR_ULTRA_THIN:
      fprintf (info, "Ultra-thin module");
      break;
    case STP_SENSOR_WIDE_MODULE:
      fprintf (info, "Wide pad module");
      break;
    case STP_SENSOR_STAMP_MODULE:
      fprintf (info, "Stamp pad module");
      break;
    case STP_SENSOR_SUBMINI:
      fprintf (info, "SubMini module");
      break;
    case STP_SENSOR_MULTISWITCH:
      fprintf (info, "MultiSwitch module");
      break;
    case STP_SENSOR_ADV_TECH:
      fprintf (info, "Advanced Technology Pad");
      break;
    case STP_SENSOR_ULTRA_THIN_R:
      fprintf (info, "Ultra-thin module, connector reversed");
      break;
    default:
      fprintf (info, "unknown (%d)", STP_INFO_SENSOR (model_id));
      break;
    }
  fprintf (info, ".\n");
  fprintf (info, "Geometry: ");
  switch (STP_INFO_GEOMETRY (model_id))
    {
    case STP_GEOMETRY_STANDARD:
      fprintf (info, "rectangular");
      break;
    case STP_GEOMETRY_OVAL:
      fprintf (info, "oval");
      break;
    default:
      fprintf (info, "unknown (%d)", STP_INFO_GEOMETRY (model_id));
      break;
    }
  fprintf (info, "/%s/%s.\n",
	   (STP_INFO_PORTRAIT (model_id)) ? "portrait" : "landscape",
	   (STP_INFO_ROT180 (model_id)) ? "down" : "up");
  modes = query_modes (fd);
  show_packet_mode (modes, info);
  show_tap_mode (modes, info);
  if (single_mode_byte)
    show_sleep_mode (modes, info);
  else
    {
      show_edge_motion (modes, info);
      show_z_threshold (modes, info);
      show_button_mode (modes, info);
      if (fw_version < QUAD_MODE_BYTE)
	show_margins (modes, info);
    }
}

void
synaptics_functions (int c, int fd, char **argv)
{
  switch (c) 
    {
    case 'h': /* --help */
      synaptics_usage (argv[0]);
      break;
    case 'q': /* --quiet */
      silent = 1;
      break;
    case 'x': /* --reset */
      reset_synaptics (fd);
      break;
    case 'v': /* --version */
      version_info ();
      break;
    case 'i': /* --info */
      if (!silent)
	synaptics_info (fd, stderr);
      break;
    case 'z': /* --zthreshold */
      set_z_threshold (fd);
      break;
    case 't': /* --tapmode */
      set_tap_mode (fd);
      break;
    case 'e': /* --edgemode */
      set_edge_motion (fd);
      break;
    case 'c': /* --corner */
      set_corner_mode (fd);
      break;
    case 'A': /* --absolute */
      set_packet_mode (fd, 1);
      break;
    case 'R': /* --relative */
      set_packet_mode (fd, 0);
      break;
    case 'a':
      if (optarg)
	{
	  int a = get_argument (1);
	  if (a < 0)
	    fprintf (stderr, "Invalid packet mode \"%s\" [use 0-1].\n",
		     optarg);
	  else
	    set_packet_mode (fd, a);
	}
      else if (!silent)
	show_tap_mode (query_modes (fd), stderr);
      break;
    case 'r': /* --rate */
      set_rate (fd);
      break;
    case '2': /* --two-button */
      set_button_mode (fd, 0);
      break;
    case '3': /* --three-button */
      set_button_mode (fd, 1);
      break;
    case '<': /* --middle-button */
      set_corner_click (fd, 1);
      break;
    case '>': /* --right-button */
      set_corner_click (fd, 0);
      break;
    case 'm':
      if (single_mode_byte)
	fprintf (stderr, "Corner click not supported on this TouchPad.\n");
      else if (optarg)
	{
	  int a = get_argument (1);
	  if (a < 0)
	    fprintf (stderr, "Invalid corner-click mode \"%s\" [use 0-1].\n",
		     optarg);
	  else
	    set_corner_click (fd, a);
	}
      else if (!silent)
	show_button_mode (query_modes (fd), stderr);
      break;
    case 's': /* --sleep */
      set_sleep_mode (fd);
      break;
    case 'M':
      if (single_mode_byte || (fw_version >= QUAD_MODE_BYTE))
	fprintf (stderr, "Adjustable margins not supported on this TouchPad.\n");
      else if (!silent)
	/* Firmware prior to version 3.2 has four mode bytes */
	show_margins (query_modes (fd), stderr);
      break;
    case '4': /* --right-margin */
      set_right_margin (fd);
      break;
    case '5': /* --left-margin */
      set_left_margin (fd);
      break;
    case '6': /* --top-margin */
      set_top_margin (fd);
      break;
    case '7': /* --bottom-margin */
      set_bottom_margin (fd);
      break;
    case '?':
      break;
    default:
      fprintf (stderr, "Unknown option %c.\n", c);
      break;
    }
}
