/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#include "ImageCache.hpp"

#include "ImageManipulation.hpp"

indii::ImageCache::ImageCache(Image* original) :
    original(original) {
  displayOriginal = new wxImage(original->getWidth(), original->getHeight(), false);
  convert(*original, *displayOriginal);
}

indii::ImageCache::~ImageCache() {
  cache_t::iterator iter1;
  for (iter1 = cache.begin(); iter1 != cache.end(); ++iter1) {
    delete iter1->second;
  }
  for (iter1 = lowCache.begin(); iter1 != lowCache.end(); ++iter1) {
    delete iter1->second;
  }

  display_cache_t::iterator iter2;
  for (iter2 = displayCache.begin(); iter2 != displayCache.end(); ++iter2) {
    delete iter2->second;
  }

  delete displayOriginal;
}

indii::Image* indii::ImageCache::get(const float scale) {
  Image* img;
  if (scale <= 1.0f) {
    img = original;
  } else {
    cache_t::const_iterator find = cache.find(scale);
    if (find != cache.end()) {
      img = find->second;
    } else {
      int width = std::floor(original->getWidth()/scale);
      int height = std::floor(original->getHeight()/scale);
      img = new Image(width, height);
      
      for (int x = 0; x < width; ++x) {
        int x1 = std::floor(scale*x);
        int x2;
        int x3 = std::floor(scale*(x + 1));
        float wx1 = 1.0f - (scale*x - x1);
        float wx2;
        float wx3 = scale*(x + 1) - x3;

        for (int y = 0; y < height; ++y) {
          int y1 = std::floor(scale*y);
          int y2;
          int y3 = std::floor(scale*(y + 1));
          float wy1 = 1.0f - (scale*y - y1);
          float wy2;
          float wy3 = scale*(y + 1) - y3;
          
          float r = 0.0f, g = 0.0f, b = 0.0f, w = 0.0f;

          for (x2 = x1; x2 <= x3; ++x2) {
            if (x2 == x1) {
              wx2 = wx1;
            } else if (x2 == x3) {
              wx2 = wx3;
            } else {
              wx2 = 1.0f;
            }

            for (y2 = y1; y2 <= y3; ++y2) {
              if (y2 == y1) {
                wy2 = wy1;
              } else if (y2 == y3) {
                wy2 = wy3;
              } else {
                wy2 = 1.0f;
              }

              float w2 = wx2*wy2;
              if (w2 > 0.0f) {
                w += w2;
                r += w2*original->r(y2, x2);
                g += w2*original->g(y2, x2);
                b += w2*original->b(y2, x2);
              }
            }
          }
          
          img->r(y, x) = r/w;
          img->g(y, x) = g/w;
          img->b(y, x) = b/w;
        }
      }
      cache.insert(std::make_pair(scale, img));
    }
  }
  return img;
}

indii::Image* indii::ImageCache::getLow(const float scale) {
  Image* img;
  if (scale <= 1.0f) {
    img = original;
  } else {
    cache_t::const_iterator find = lowCache.find(scale);
    if (find != lowCache.end()) {
      img = find->second;
    } else {
      img = new Image(*original, scale);
      lowCache.insert(std::make_pair(scale, img));
    }
  }
  return img;
}

wxImage* indii::ImageCache::getDisplay(const int width) {
  /* pre-condition */
  assert(width >= 1);

  double ratio = static_cast<double>(displayOriginal->GetWidth())/displayOriginal->GetHeight();
  wxImage* img;
  if (width == displayOriginal->GetWidth()) {
    img = displayOriginal;
  } else {
    display_cache_t::const_iterator find = displayCache.find(width);
    if (find != displayCache.end()) {
      img = find->second;
    } else {
      int height = intround(width/ratio);
      img = new wxImage(displayOriginal->Scale(width, height, wxIMAGE_QUALITY_HIGH));
      displayCache.insert(std::make_pair(width, img));
    }
  }
  return img;
}
