use ui;
use core:io;
use core:lang;
use core:sync;
use lang;
use lang:bs:macro;

/**
 * Class representing some loaded program that we want to execute and possibly single-step, etc.
 */
class Program on Render {
	// Package created by us as a temporary workspace.
	private Package pkg;

	// All view hints that are active.
	public ViewHints[] hints;

	// Pointer to the 'main' function.
	public Fn<void> main;

	// Source code, pre formatted.
	private Url->Text code;

	// Output from the program since the last check. This is not wired to stdout in any way at the
	// moment, but we could do that. Use 'addOutput' to access it, with convenience functions below.
	private StrBuf output;

	// Event called whenever the state of a thread changed.
	private Fn<void>? changeNotification;

	// Event called whenever an error is encountered from one of the threads.
	private Fn<void, Nat, Exception>? errorNotification;

	// Event called when a particular thread has been advanced (for recording).
	private Fn<void, Nat>? advanceNotification;

	// Set the change notification.
	public assign onChange(Fn<void> fn) {
		changeNotification = fn;
		fn;
	}

	// Set the error notification.
	public assign onError(Fn<void, Nat, Exception> fn) {
		errorNotification = fn;
		fn;
	}

	// Set the "advance" notification.
	public assign onAdvance(Fn<void, Nat> fn) {
		advanceNotification = fn;
		fn;
	}

	// Call the notification.
	void notifyChange() {
		if (changeNotification)
			changeNotification.call();
	}

	// Call the advance notification.
	void notifyAdvance(Nat thread) {
		if (advanceNotification)
			advanceNotification.call(thread);
	}

	package init(Package pkg, Fn<void> main, ViewHints[] hints) {
		init() {
			pkg = pkg;
			main = main;
			hints = hints;
		}

		// Add us into the global map.
		programInstances.put(this);
	}

	// Find supported file types.
	public Set<Str> supportedFiles() : static {
		findSupportedFiles();
	}

	public Program load(Url file) : static {
		load([file]);
	}

	public Program load(Url[] files) : static {
		loadProgram(files);
	}

	// Add output to the buffer. Note: We don't call 'onChange' for this, since the program will
	// soon be halted at the next step anyway, which causes a full repaint anyway.
	public void addOutput(Str data) {
		output << data;
	}

	// Get all output in the buffer. Returns null if no new output.
	public Str? getOutput() {
		if (output.empty)
			return null;

		Str result = output.toS;
		output.clear();
		result;
	}

	// Thread ID:s. We maintain our own thread ID:s that have lower numbers. They are only unique for each program.
	private Nat usedThreadId;

	// Last thread that terminated. We use this to tell the user which thread crashed. If multiple
	// threads crash at the same time, this might be inaccurate, but it is good enough for now.
	private Nat lastTerminatedId;

	// All threads in the program. The key is the ID of the UThread that executes the code.
	Word->ProgThread threads;

	// Start a new thread executing a particular function assumed to be located inside the
	// environment specified earlier. Always spawns a new thread..
	void spawn(Fn<void> fn) {
		(spawn run(fn)).detach();
	}

	// Run the function.
	void run(Fn<void> fn) {
		try {
			// Add the thread now so that it may be used to capture the result of 'main'.
			threads.put(currentUThread(), ProgThread(this, ++usedThreadId));

			fn.call();
		} catch (ExitError e) {
			// Just ignore it.
		} catch (Exception e) {
			if (errorNotification) {
				errorNotification.call(lastTerminatedId, e);
			} else {
				print("The program encountered an error:\n${e}");
			}
		}

		// If the thread remains by now, remove it.
		// This is just to be safe. It should not happen.
		Word id = currentUThread();
		threads.remove(id);
	}

	// Load source code.
	package void loadSource(Url[] files) {
		for (f in files) {
			code.put(f, highlightSource(f));
		}
	}

	// Get source for a file.
	package Text? source(Url file) {
		if (code.has(file))
			return code.get(file);
		else
			return null;
	}

	// Find and return all global variables in the program managed by this instance.
	GlobalVar[] findGlobals() {
		findGlobalsHelp(pkg);
	}

	// Terminate any threads running in the program, and wait for them to exit properly.
	void terminate() {
		for (t in threads) {
			t.terminate();
		}

		// Wait for them to terminate...
		while (threads.count > 0) {
			yield();
		}

		// Reset the threa ID:s
		usedThreadId = 0;
	}

	// This function will be called by the executed code to notify us of their current state in
	// their execution. This also lets us execute them at any location.
	package void newLocation(SrcPosWrap pos, StackVar[] vars) {
		Word id = currentUThread();
		ProgThread t = threads.get(id);

		t.onNewLocation(pos.v, vars);
	}

	// This function will be called when the function is about to return. It is basically the same
	// as 'newLocation', except that we get the return value here, and we might want to indicate
	// that the function is returning somehow.
	package void functionReturn(StackVar[] vars) {
		Word id = currentUThread();
		ProgThread t = threads.get(id);

		t.onFunctionReturn(vars);
	}

	// This function will be called by the executed code to notify that a new (traced) function was
	// called, and that we shall open a new scope.
	package MemTracker functionEntered(Str name, SrcPosWrap pos, ViewHints hints, StackVar[] vars) {
		Word id = currentUThread();
		ProgThread t = if (threads.has(id)) {
			threads.get(id);
		} else {
			ProgThread t(this, ++usedThreadId);
			threads.put(id, t);
			t;
		};

		t.onFunctionEntered(name, pos.v, hints, vars);
	}

	// This function will be called on function entry of a non-traced function. 'functionExited'
	// will not be called for this function. The main purpose of this function is to acquire the
	// MemTracker object for this thread.
	package MemTracker functionEntered() {
		Word id = currentUThread();
		ProgThread t = if (threads.has(id)) {
			threads.get(id);
		} else {
			ProgThread t(this, ++usedThreadId);
			threads.put(id, t);
			t;
		};

		t.memory;
	}

	// This function will be called by the executed code to notify that a (traced) function is about
	// to exit.
	package void functionExited() {
		Word id = currentUThread();
		if (!threads.has(id))
			return;

		ProgThread t = threads.get(id);
		var notify = t.onFunctionExited();
		if (!t.alive) {
			threads.remove(id);

			if (f = t.watch)
				spawn checkErrors(t.threadId, f);

			lastTerminatedId = t.threadId;
			if (notify)
				notifyChange();
		}
	}

	// Allocate a new thread ID for a thread that will eventually start.
	Nat allocFutureId(Word threadId) {
		// We can just preemptively add the thread.
		Nat id = ++usedThreadId;
		ProgThread t(this, id);
		threads.put(threadId, t);

		id;
	}

	// Check for errors from a future. Will block until anything is posted.
	private void checkErrors(Nat threadId, FutureBase from) {
		try {
			from.errorResult();
		} catch (ExitError e) {
			// Just ignore it.
		} catch (Exception e) {
			if (errorNotification) {
				errorNotification.call(lastTerminatedId, e);
			} else {
				print("The program encountered an error:\n${e}");
			}
		}
	}
}

/**
 * Wrapper to make a pointer of an SrcPos for easier management from ASM.
 */
class SrcPosWrap {
	SrcPos v;

	init(SrcPos pos) {
		init() { v = pos; }
	}
}


package PkgReader? progvisReader(Str ext, Url[] files, Package pkg) {
	unless (name = readerName(ext))
		return null;

	name.add(0, "progvis");
	return createReader(name, files, pkg);
}

// Find a standard Storm reader.
package PkgReader? stdReader(Str ext, Url[] files, Package pkg) {
	if (name = readerName(ext))
		return createReader(name, files, pkg);
	null;
}

private void addProgvisHints(Str ext, Hints[] to) {
	unless (name = readerName(ext))
		return;

	name.add(0, "progvis");
	name.last.name = "hints";
	name.last.params.clear();
	unless (fn = rootScope.find(name) as Function)
		return;

	unless (ptr = fn.pointer as Fn<Hints>)
		return;

	to << ptr.call();
}

private void addStdHints(Str ext, Hints[] to) {
	unless (name = readerName(ext))
		return;

	name.last.name = "hints";
	name.last.params.clear();
	unless (fn = rootScope.find(name) as Function)
		return;

	unless (ptr = fn.pointer as Fn<Hints>)
		return;

	to << ptr.call();
}

// Helper to load a program on a different thread.
private Program loadProgram(Url[] files) on Compiler {
	Package pkg("<sandbox>");
	pkg.parentLookup = rootPkg;
	pkg.noDiscard();

	Map<Str, Url[]> exts;
	for (f in files) {
		exts[f.ext] << f;
	}

	Hints[] hints;
	PkgReader[] readers;
	for (ext, files in exts) {
		// Try to find a reader ourselves first.
		if (r = progvisReader(ext, files, pkg)) {
			readers << r;
			addProgvisHints(ext, hints);
		} else if (r = stdReader(ext, files, pkg)) {
			readers << r;
			addStdHints(ext, hints);
		} else {
			throw LoadError("The file type ${ext} is not supported.");
		}
	}

	if (readers.empty()) {
		throw LoadError("The desired file type(s) are not supported.");
	}

	// Add the standard hints last, so that they have a chance to do things.
	hints << defHints();

	// Load all code.
	read(readers);

	// Produce errors now rather than later.
	pkg.compile();

	// Extract the view hints for later use.
	ViewHints[] viewHints;
	for (h in hints)
		if (view = h.view)
			viewHints << view;

	// Create the program instance and patch all code.
	Program program(pkg, findMain(hints, pkg), viewHints);
	patchFunctions(program, hints, pkg);
	program.loadSource(files);
	return program;
}

// Find the main function.
private Fn<void> findMain(Hints[] hints, Package pkg) {
	for (h in hints) {
		if (r = h.code.findMain(pkg))
			return r;
	}

	throw ProgvisError("Unable to find a 'main' function!");
}


// Find supported file types. On compiler thread.
private Set<Str> findSupportedFiles() on Compiler {
	Package pkg = named{lang};
	Set<Str> to;
	findSupportedFiles(to, named{:lang});
	findSupportedFiles(to, named{:progvis:lang});
	to;
}

private void findSupportedFiles(Set<Str> out, Package pkg) on Compiler {
	for (elem in pkg) {
		if (elem as Package)
			out.put(elem.name);
	}
}

// Find global variables in a package.
private GlobalVar[] findGlobalsHelp(Package pkg) on Compiler {
	GlobalVar[] result;

	for (elem in pkg) {
		if (elem as GlobalVar) {
			if (elem.owner is named{Render})
				result << elem;
		}
	}

	result;
}

/**
 * Exception thrown during load.
 */
class LoadError extends Exception {
	init(Str msg) { init { msg = msg; } }
	Str msg;
	void message(StrBuf to) : override {
		to << msg;
	}
}


// Global variable that keeps track of all Program instances, so that we can find the owner to a
// particular thread by its thread id.
// TODO: This can't be private...
package WeakSet<Program> programInstances on Render = WeakSet<Program>();

// Find the Program instance containing a particular thread ID.
private Program? findProgram(Word threadId) on Render {
	for (i in programInstances) {
		if (i.threads.has(threadId))
			return i;
	}
	return null;
}

public Program? findProgram() {
	findProgram(currentUThread());
}

// Find a particular thread.
public ProgThread? findThread(Word threadId) on Render {
	for (i in programInstances) {
		if (i.threads.has(threadId))
			return i.threads.get(threadId);
	}
	return null;
}

private Nat findThisThreadId(Word id) on Render {
	if (t = findThread(id)) {
		return t.threadId;
	}

	0;
}

// Find a thread id of this (existing) thread.
public Nat findThisThreadId() {
	findThisThreadId(currentUThread());
}

// Find the "simple" thread ID for a thread. This might create the thread ID in the appropriate
// location, which is why the current thread ID is needed as well.
private Nat findNewThreadId(Word currentThread, Word query) on Render {
	if (thread = findThread(query)) {
		return thread.threadId;
	}

	// We need to create it.
	if (p = findProgram(currentThread)) {
		return p.allocFutureId(query);
	}

	0;
}

// Wrapper not bound to a particular thread, making it easier to call from ASM.
public Nat findNewThreadId(Word query) {
	findNewThreadId(currentUThread(), query);
}

private void watchErrorsI(Word threadId, FutureBase future) on Render {
	if (thread = findThread(threadId)) {
		thread.watch = future;
	} else {
		future.detach();
	}
}

// Tell the system to watch for errors for the specified thread ID using the supplied future.
public void watchErrors(Word threadId, FutureBase future) {
	watchErrorsI(threadId, future);
}

// Tell the system that we are about to throw some kind of exception, and that this will likely terminate the program.
public void onFatalException() {
	if (p = findThread(currentUThread()))
		p.onFatalException();
}

// Add output to the program associated with the current thread, or output to stdout if no thread
// currently available. No newlines are added automatically.
public void output(Str text) {
	if (program = findProgram(currentUThread())) {
		program.addOutput(text);
	} else {
		stdOut.write(text);
		stdOut.flush();
	}
}
