% -----------------------------------------------------------------------------
%  (C) Altran Praxis Limited
% -----------------------------------------------------------------------------
% 
%  The SPARK toolset is free software; you can redistribute it and/or modify it
%  under terms of the GNU General Public License as published by the Free
%  Software Foundation; either version 3, or (at your option) any later
%  version. The SPARK toolset is distributed in the hope that it will be
%  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
%  Public License for more details. You should have received a copy of the GNU
%  General Public License distributed with the SPARK toolset; see file
%  COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
%  the license.
% 
% =============================================================================

%###############################################################################
% PURPOSE
%-------------------------------------------------------------------------------
% Stores, retrieves and replaces hypothesis into the database.
%###############################################################################

%###############################################################################
% MODULE
%###############################################################################

:- module(data__hyp, [add_hyp/3,
                      add_hyp_with_id/3,
                      get_hyp/3,
                      get_next_hyp_id/1,
                      replace_hyp/4,
                      prune_hyp/3,
                      prune_all_hyps/3,
                      reset_next_hyp_id/0,
                      add_forall_hyp/3,
                      get_forall_hyp/3,
                      prune_all_forall_hyp/0]).

%###############################################################################
% DEPENDENCIES
%###############################################################################

:- use_module('data__formats.pro',
              [add_state/2,
              add_type/2]).

:- use_module('data__provenance.pro',
              [path_functions/0]).

:- use_module('ioutilities.pro',
              [show_error/2]).

:- use_module('opdeclar.pro',
              [declare_operators/0,
               hide_operators/0]).

:- use_module(library(sets),
              [list_to_set/2]).

:- declare_operators.

%###############################################################################
% TYPES
%###############################################################################

:- add_type('HypothesisType',
            [x,
             ss,
             [s, 'TYPE']]).

%###############################################################################
% DATA
%###############################################################################

:- add_state(get_hyp,
             get_hyp('Hypothesis_Term', 'HypothesisType', 'Id_Int')).

:- add_state(get_next_hyp_id,
             get_hyp('Id_Int')).

:- add_state(get_forall_hyp,
	     get_forall_hyp('Id_Int', 'Hypothesis_Term', 'Condition_TermList')).

:- dynamic(get_hyp/3).

:- dynamic(get_next_hyp_id/1).

:- dynamic(get_forall_hyp/3).

%###############################################################################
% PREDICATES
%###############################################################################

%===============================================================================
% add_hyp(+Hyp_Term, +HypType, -Id_Int)
%
% The predicate is determinate and always succeeds. The identfier of the
% inserted hypothesis is bound to Id_Int;
%
% If the hypothesis is a duplicate then it is not inserted into the
% database and its identifier is bound to Id_Int;
%
% No hypothesis is added to the database if the hyp is "true" and
% it is not a path function.




%===============================================================================

add_hyp(true,_,0) :-
    \+ path_functions,
    !.

%
% Adding a duplicate hypothesis.
%

add_hyp(Hyp_Term, HypType, Id_Int) :-
    get_hyp(Hyp_Term, HypType, Id_Int),
    user:assert_log_fact(duplicate_hyp, [Id_Int, Hyp_Term]),
    !.

%
% Adding a hypothesis that is not a duplicate.
%

add_hyp(Hyp_Term, HypType, Id_Int) :-
    next_hyp_id(Id_Int),
    assertz(get_hyp(Hyp_Term, HypType, Id_Int)),
    retractall(user:could_not_infer(_)),
    !.

%===============================================================================

%===============================================================================
% add_hyp_with_id(+Hyp_Term, +HypType, +Id_Int)
%
% Adds a hypothesis into the database with a specific hypothesis identifier.
%
% There are two cases:
%   HypType = x : typically used when loading hypothesises from VCG files
%                 where the identifiers are specified. In this case,
%                 update the database to record that this identifier is
%                 used.
%
%   HypType = ss or [s, Type] : adding a hypothesis that is related (a variant)
%                               of a hypothesis with the same hypothesis id.
%===============================================================================

add_hyp_with_id(Hyp_Term, x, Id_Int) :-
    assertz(get_hyp(Hyp_Term, x, Id_Int)),
    set_next_hyp_id(Id_Int),
    !.

add_hyp_with_id(Hyp_Term, HypType, Id_Int) :-
    assertz(get_hyp(Hyp_Term, HypType, Id_Int)),
    !.

%===============================================================================

%===============================================================================
% replace_hyp(+OldHyp_Term, +HypType, +HypId_Int, +NewHyp_Term)
%
% Replace a hypothesis in the database with a new hypothesis.
%
% The predicate fails if OldHyp_Term, HypType, HypId_Ind is not
% in the database.
%===============================================================================

replace_hyp(OldHyp_Term, HypType, HypId_Int, NewHyp_Term):-
    prune_hyp(OldHyp_Term, HypType, HypId_Int),
    assertz(get_hyp(NewHyp_Term, HypType, HypId_Int)).

%===============================================================================

%===============================================================================
% prune_hyp(?Hyp_Term, ?HypType, ?HypId_Int)
%
% Predicate retracts a hypthesis from the database.
%
% Predicate fails if nothing is deleted from the database.
%===============================================================================

prune_hyp(Hyp_Term, HypType, HypId_Int):-
    retract(get_hyp(Hyp_Term, HypType, HypId_Int)).

%===============================================================================

%===============================================================================
% prune_all_hyps(?Hyp_Term, ?HypType, ?HypId_Int)
%
% Retracts all hypothesises from the database that matches the
% arguments.
%
% Predicate always succeeds independent of whether anything is deleted
% from the database.
%===============================================================================

prune_all_hyps(Hyp_Term, HypType, HypId_Int):-
    retractall(get_hyp(Hyp_Term, HypType, HypId_Int)).

%===============================================================================

%===============================================================================
% next_hyp_id(-NextId_Int)
%-------------------------------------------------------------------------------
% Return the next hypothesis identifier and record its use in the database.
%===============================================================================

next_hyp_id(HypId_Int):-
    retract(get_next_hyp_id(HypId_Int)),
    HypId1_Int is HypId_Int + 1,
    assert(get_next_hyp_id(HypId1_Int)),
    !.

%
% If no hypothesises have been inserted into the database, then the hypothesis
% identifier is 1 and the next free identifier is 2.
%

next_hyp_id(1):-
    assert(get_next_hyp_id(2)),
    !.

%===============================================================================

%===============================================================================
% set_next_hyp_id(+HypId_Int)
%-------------------------------------------------------------------------------
% Record that the hypothesis identifier has been used and increment
% get_next_hyp_id if the identifier is greater than get_next_hyp_id.
%===============================================================================

set_next_hyp_id(HypId_Int):-
    HypId1_Int is HypId_Int + 1,
    set_next_hyp_id_x(HypId1_Int).

%===============================================================================
% set_next_hyp_id_x(+HypId1_Int)
%===============================================================================

set_next_hyp_id_x(HypId1_Int):-
    retract(get_next_hyp_id(MaxHypId_Int)),
    set_next_hyp_id_y(HypId1_Int, MaxHypId_Int),
    !.

set_next_hyp_id_x(HypId1_Int):-
    set_next_hyp_id_y(HypId1_Int, 0),
    !.

%===============================================================================
% set_next_hyp_id_Y(+HypId_Int, MaxHypId_Int)
%===============================================================================

set_next_hyp_id_y(HypId_Int, MaxHypId_Int):-
    MaxHypId_Int =< HypId_Int,
    assert(get_next_hyp_id(HypId_Int)),
    !.

set_next_hyp_id_y(_HypId_Int, MaxHypId_Int):-
    assert(get_next_hyp_id(MaxHypId_Int)),
    !.

%===============================================================================

%===============================================================================
% reset_next_hyp_id
%-------------------------------------------------------------------------------
% Reset the counter for hypothesis identifiers by removing get_next_hyp_id/1.
%===============================================================================

reset_next_hyp_id:-
    retractall(get_next_hyp_id(_)),
    !.

%===============================================================================
% add_forall_hyp(+Hypothesis_Term, +Conditions_TermList, +HypId_Int)
%-------------------------------------------------------------------------------
% Insert a univerally quantified hypothesis into the database.
%===============================================================================

add_forall_hyp(Hypothesis_Term, Condition_TermList, HypId_Int) :-
    assertz(get_forall_hyp(Hypothesis_Term, Condition_TermList, HypId_Int)),
    !.

%===============================================================================

%===============================================================================
% prune_all_forall_hyp
%-------------------------------------------------------------------------------
% Delete all univerally quantified hypotheses from the database.
%===============================================================================

prune_all_forall_hyp :-
    retractall(get_forall_hyp(_Hypothesis_Term, _Condition_TermList, _HypId_Int)),
    !.

%===============================================================================

%###############################################################################
% END-OF-FILE
