///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/snapping/SnappingManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/actions/ActionManager.h>
#include <core/data/DataSetManager.h>
#include "GridSnappingProvider.h"

namespace Core {

/// The singleton instance of this class.
SnappingManager* SnappingManager::_singletonInstance = NULL;

/******************************************************************************
* Initializes the snapping manager.
******************************************************************************/
SnappingManager::SnappingManager()
{
	// Reset the snapping manager when a new scene has been loaded.
	connect(&DATASET_MANAGER, SIGNAL(dataSetReset(DataSet*)), this, SLOT(reset()));
	
	// Find all installed snapping providers.
	/*
	Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(SnappingProvider))) {
		providers.push_back(static_object_cast<SnappingProvider>(clazz->createInstance()));
	}
	*/

	// Create actions.
	objectSnappingEnabledAction = ACTION_MANAGER.addAction(new Action(ACTION_SNAPPING_OBJECT), tr("Snaps Toggle"), ":/core/snapping/snap_objects.png");
	objectSnappingEnabledAction->setCheckable(true);	
	angleSnappingEnabledAction = ACTION_MANAGER.addAction(new Action(ACTION_SNAPPING_ANGLE), tr("Angle Snap Toggle"), ":/core/snapping/snap_angle.png");
	angleSnappingEnabledAction->setCheckable(true);
	percentSnappingEnabledAction = ACTION_MANAGER.addAction(new Action(ACTION_SNAPPING_PERCENT), tr("Percent Snap Toggle"), ":/core/snapping/snap_percent.png");
	percentSnappingEnabledAction->setCheckable(true);
}

/******************************************************************************
* Resets the snap settings to their initial values.
******************************************************************************/
void SnappingManager::reset()
{
	_snapStrength = 6;
	_angleSnapStepSize = 5.0f * FLOATTYPE_PI / 180.0f;
	_percentSnapStepSize = 0.1f;
	_lastSnapPoint = SnappingRecord();
	objectSnappingEnabledAction->setChecked(false);
	angleSnappingEnabledAction->setChecked(false);
	percentSnappingEnabledAction->setChecked(false);
	
	// Deactivate all snapping providers.
	_activeProviders.clear();
	
	// Enable the grid snapping provider by default.
	_activeProviders.push_back(new GridSnappingProvider());
}

/******************************************************************************
* Rounds an angle value to the nearest multiple of the current angle snap value.
******************************************************************************/
FloatType SnappingManager::snapAngle(FloatType angle)
{
	if(!angleSnappingEnabledAction->isChecked()) return angle;
	return (floor(angle / angleSnapStepSize() + 0.5) * angleSnapStepSize()); 
}

/******************************************************************************
* Rounds a percent value to the nearest multiple of the current percent snap value.
******************************************************************************/
FloatType SnappingManager::snapPercent(FloatType p)
{
	if(!percentSnappingEnabledAction->isChecked()) return p;
	return (floor(p / percentSnapStepSize() + 0.5) * percentSnapStepSize()); 
}

/******************************************************************************
* Performs a snap in a viewport.
*    viewport - The viewport to use.
*    screenPos - The input screen coordinates in pixels.
*    result - The resulting snap point in world space.
*    snappingPlane - The construction plane used to do snapping.
*    snapPointFlags - Bitwise combination of flags.
* Returns true if a snap point has been found, false otherwise.
******************************************************************************/
bool SnappingManager::snapPoint(Viewport* viewport, const Point2I& screenPos, Point3& result, 
	const AffineTransformation& snappingPlane, SnappingFlags snapPointFlags)
{
	if(!objectSnappingEnabledAction->isChecked())
		return false;

	// Initialize context object.
	SnappingContext context(viewport, ANIM_MANAGER.time(), screenPos, snapStrength());

	// Perform snapping.
	Q_FOREACH(const SnappingProvider::SmartPtr& provider, _activeProviders) {
		CHECK_OBJECT_POINTER(provider.get());
		provider->snap(context);
	}

	_lastSnapPoint = context.bestHit();

	if(!_lastSnapPoint.isValid()) 
		return false; // No hit found.

    result = _lastSnapPoint.worldPoint;
	return true;
}


/******************************************************************************
* Clears the last hit marker and invalidates the viewports.
******************************************************************************/
void SnappingManager::clearLastSnapPoint()
{
	if(_lastSnapPoint.isValid()) {
		_lastSnapPoint = SnappingRecord();
		VIEWPORT_MANAGER.updateViewports();
	}
}


};
