/* Light Unix I/O for Lua
 * Copyright 2012 Rob Kendrick <rjek+luxio@rjek.com>
 *
 * Copyright 2012 Daniel Silverstone <dsilvers@digital-scurf.org>
 *
 * Distributed under the same terms as Lua itself (MIT).
 */

/***
Low-level bindings to POSIX functionality.

Luxio provides a very light-weight binding to many of the standard POSIX
and common Unix library calls.  Where possible, calls are very raw.  In
cases such as the `dirent` family, BSD sockets, the `getaddrinfo` family,
and some other cases, they interfaces are somewhat "cooked" either to make
them more efficient or even possible.

For the simple raw uncooked functions, all we present here is an example
of the C prototype, and possible styles for use in Lua.  You'll have to
go looking in man pages for actual information on their use.

Not all systems will provide all the functions described here.

@module luxio
*/
 
#define LUXIO_RELEASE 8 /* clients use to check for features/bug fixes */
#define LUXIO_ABI 0 /* clients use to check the ABI of calls is same */
#define LUXIO_COPYRIGHT "Copyright 2013 Rob Kendrick <rjek+luxio@rjek.com>\n" \
	"Copyright 2014 Daniel Silverstone <dsilvers@digital-scurf.org>"

#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <lua.h>
#include <lauxlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/wait.h>
#include <sys/utsname.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <net/if.h>
#include <fcntl.h>
#include <poll.h>
#include <unistd.h>
#include <dirent.h>
#include <time.h>
#include <limits.h>
#include <signal.h>
#include <syslog.h>
#include <iconv.h>

#ifdef HAVE_SENDFILE
# include <sys/sendfile.h>
#endif

#if (LUA_VERSION_NUM == 501)
# define lua_rawlen(L, idx) lua_objlen((L), (idx))
#endif

#define INVALID_MODE ((mode_t) -1)

/* External interface to Lua *************************************************/

int luaopen_luxio(lua_State *L);

/*** Process creation and execution.
Functions to do with the creation of new processes and the execution of other
programs.
@section process
*/

/*** Create a child process.

@treturn number return value
@treturn errno errno
@function fork
*/
static int
luxio_fork(lua_State *L) /* 3.1.1 */
{
	lua_pushinteger(L, fork());
	lua_pushinteger(L, errno);

	return 2;
}

static int
luxio__exec(lua_State *L, bool usep)
{
	const char *path = luaL_checkstring(L, 1);
	int params = lua_gettop(L) - 1;
	char **args;
	int c, ret;

	/* we probably at least need them to fill in arg[0] ... */
	luaL_checkstring(L, 2);

	args = calloc(params + 1, sizeof(*args));

	for (c = 0; c < params; c++) {
		/* gah!  constness */
		args[c] = (char *)luaL_checkstring(L, c + 2);
	}

	args[c] = NULL;

	if (usep) {
		ret = execvp(path, args);
	} else {
		ret = execv(path, args);
	}

	/* if we got here, there's an error. */
	free(args);
	lua_pushinteger(L, ret);
	lua_pushinteger(L, errno);

	return 2;
}

/*** Execute a file.
Calls execv().

@tparam string path absolute path to executable
@tparam string ... parameters to pass to executatable
@treturn number return value
@treturn errno errno
@function exec
*/
static int
luxio_exec(lua_State *L) /* 3.1.2 */
{
	return luxio__exec(L, false);
}

/*** Execute a file, searching PATH.
Calls execp().

@tparam string path name of executable
@tparam string ... parameters to pass to executatable
@treturn number return value
@treturn number errno
@function execp
*/
static int
luxio_execp(lua_State *L) /* 3.1.2 */
{
	return luxio__exec(L, true);
}

/* TODO: pthread_atfork() 3.1.3 */

/*** Process termination.
Functions to do with the termination of processes.
@section termination
*/

/*** Wait for a process to change state.

@tparam number pid pid to wait on
@tparam number options options for call
@treturn number return value
@treturn number|errno status code or errno if error
@function waitpid
*/
static int
luxio_waitpid(lua_State *L) /* 3.2.1 */
{
	pid_t pid = luaL_checkinteger(L, 1);
	int options = luaL_checkinteger(L, 2);
	int status;
	pid_t proc;

	proc = waitpid(pid, &status, options);
	lua_pushinteger(L, proc);
	if (proc == -1) {
		lua_pushinteger(L, errno);
	} else {
		lua_pushinteger(L, status);
	}

	return 2;
}

#define WAITPID_STATUS(x) static int luxio_##x(lua_State *L) { \
		int status = luaL_checkinteger(L, 1);	      \
		lua_pushinteger(L, x(status));		      \
		return 1;				      \
	}

/*** Check status macro for WIFEXITED.
@tparam number status code from `waitpid`()
@treturn number true or false
@function WIFEXITED
*/
WAITPID_STATUS(WIFEXITED)

/**% WEXITSTATUS
 *  retval = WEXITSTATUS(status);
 *  retval = WIFEXITED(status)
 */
/*** Obtain exit status code from child
@tparam number status code from `waitpid`()
@treturn number exit status code
@function WEXITSTATUS
*/
WAITPID_STATUS(WEXITSTATUS)

/*** Check status macro for WIFSIGNALED.
@tparam number status code from `waitpid`()
@treturn number true or false
@function WIFSIGNALLED
*/
WAITPID_STATUS(WIFSIGNALED)

/*** Obtain signal used to kill child.
@tparam number status code from `waitpid`()
@treturn number signal number
@function WTERMSIG
*/
WAITPID_STATUS(WTERMSIG)

#ifdef WCOREDUMP
/**% WCOREDUMP
 *  retval = WCOREDUMP(status);
 *  retval = WCOREDUMP(status)
 */
/*** Check status macro for WCOREDUMP.
@tparam number status code from `waitpid`()
@treturn number true or false
@function WCOREDUMP
*/
WAITPID_STATUS(WCOREDUMP)
#endif

/*** Check status macro for WIFSTOPPED.
@tparam number status code from `waitpid`()
@treturn number true or false
@function WIFSTOPPED
*/
WAITPID_STATUS(WIFSTOPPED)

/*** Obtain signal number used to stop child.
@tparam number status code from `waitpid`()
@treturn number signal number
@function WSTOPSIG
*/
WAITPID_STATUS(WSTOPSIG)

#ifdef WIFCONTINUED
/*** Check status macro for WIFCONTINUED.
Linux 2.6.10.
@tparam number status code from `waitpid`()
@treturn number true or false
@function WIFCONTINUED
*/
WAITPID_STATUS(WIFCONTINUED)
#endif

#undef WAITPID_STATUS

/*** Terminate calling process.  Does not return.
@tparam[opt=0] number exit status code.
@function _exit
*/
static int
luxio__exit(lua_State *L) /* 3.2.2 */
{
	int ret = luaL_optinteger(L, 1, 0);

	_exit(ret);

	return 0;
}

/*** Signals.  Functions related to sending signals.
@section signals
*/

/**% kill
 *  retval = kill(pid, sig);
 *  retval, errno = kill(pid, sig)
 */
/*** Send signal to a process.
@tparam number pid signal destination
@tparam number signal signal to send
@treturn number return value
@treturn errno errno
@function kill
*/
static int
luxio_kill(lua_State *L) /* 3.3.2 */
{
	pid_t pid = luaL_checkinteger(L, 1);
	int sig = luaL_checkinteger(L, 2);

	lua_pushinteger(L, kill(pid, sig));
	lua_pushinteger(L, errno);

	return 2;
}

/* Signals are going to be almost impossible to do nicely and safely. */

/* TODO: Manipulate Signal Sets 3.3.3 */
/* TODO: sigaction() 3.3.4 */
/* TODO: pthread_sigmask(), sigprocmask() 3.3.5 */
/* TODO: sigpending() 3.3.6 */
/* TODO: sigsuspend() 3.3.7 */
/* TODO: sigwait(), sigwaitinfo(), sigtimedwait() 3.3.8 */
/* TODO: sigqueue() 3.3.9 */
/* TODO: pthread_kill() 3.3.10 */

/*** Timer operations.
@section timer
*/

/*** Set an alarm clock for delivery of a signal.
@tparam number seconds How long to wait before signal delivery
@treturn number return value
@function alarm
*/
static int
luxio_alarm(lua_State *L) /* 3.4.1 */
{
	unsigned int seconds = luaL_checkinteger(L, 1);
	lua_pushinteger(L, alarm(seconds));

	return 1;
}

/*** Wait for a signal.
@treturn number return value
@function pause
*/
static int
luxio_pause(lua_State *L) /* 3.4.2 */
{
	lua_pushinteger(L, pause());
	lua_pushinteger(L, errno);

	return 2;
}

/*** Sleep for the specified number of seconds.
@tparam number seconds number of seconds to sleep
@treturn number return value
@function sleep
*/
static int
luxio_sleep(lua_State *L) /* 3.4.3 */
{
	unsigned int seconds = luaL_checkinteger(L, 1);

	lua_pushinteger(L, sleep(seconds));

	return 1;
}

/*** Process identification
@section procident
*/

/*** Get process identification.
@treturn number pid
@function getpid
*/
static int
luxio_getpid(lua_State *L) /* 4.1.1 */
{
	lua_pushinteger(L, getpid());

	return 1;
}

/*** Get process's parent's identification.
@treturn number pid
@function getppid
*/
static int
luxio_getppid(lua_State *L) /* 4.1.1 */
{
	lua_pushinteger(L, getppid());

	return 1;
}

/*** User identification.
@section userident
*/

/*** Get user identity.
@treturn number uid
@function getuid
*/
static int
luxio_getuid(lua_State *L) /* 4.2.1 */
{
	lua_pushinteger(L, getuid());

	return 1;
}

/*** Get effective user identity.
@treturn number uid
@function geteuid
*/
static int
luxio_geteuid(lua_State *L) /* 4.2.1 */
{
	lua_pushinteger(L, geteuid());

	return 1;
}

/*** Get group identity.
@treturn number gid
@function getgid
*/
static int
luxio_getgid(lua_State *L) /* 4.2.1 */
{
	lua_pushinteger(L, getgid());

	return 1;
}

/*** Get effective group identity.
@treturn number gid
@function getegid
*/
static int
luxio_getegid(lua_State *L) /* 4.2.1 */
{
	lua_pushinteger(L, getegid());

	return 1;
}

/*** Set user identity.
@tparam number uid
@treturn number return value
@treturn errno errno
@function setuid
*/
static int
luxio_setuid(lua_State *L) /* 4.2.2 */
{
	uid_t uid = luaL_checkinteger(L, 1);

	lua_pushinteger(L, setuid(uid));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Set group identity
@tparam number gid
@treturn number return value
@treturn errno errno
@function setgid
*/
static int
luxio_setgid(lua_State *L) /* 4.2.2 */
{
	gid_t gid = luaL_checkinteger(L, 1);

	lua_pushinteger(L, setgid(gid));
	lua_pushinteger(L, errno);

	return 2;
}

/* TODO: getgroups() 4.2.3 */

/*** Get username.
@treturn number return value
@treturn string|errno username or errno
@function getlogin
*/
static int
luxio_getlogin(lua_State *L) /* 4.2.4 */
{
	char buf[LOGIN_NAME_MAX];
	int r = getlogin_r(buf, sizeof(buf));

	if (r != 0) {
		lua_pushinteger(L, r);
		lua_pushinteger(L, errno);

		return 2;
	}

	lua_pushinteger(L, r);
	lua_pushstring(L, buf);

	return 2;
}

/*** Process groups.
@section procgroup
*/

/* TODO: getpgrp() 4.3.1 */
/* TODO: setsid() 4.3.2 */
/* TODO: setpgid() 4.3.3 */

/*** System identification.
@section sysident
*/

/*** uname() information table.
Returned by @{uname}.  Some fields are OS-specific.
@field sysname Operating system name, such as Linux or NetBSD.
@field nodename System's name
@field release Operating systems's release version
@field machine Hardware identifier
@field domainname NIS or YP domain name
@table uname-table
*/

/*** Get name and information about current kernel.
@treturn number return value
@treturn uname-table|errno result table, or errno.
@function uname
*/
static int
luxio_uname(lua_State *L) /* 4.4.1 */
{
	struct utsname buf;
	int r = uname(&buf);

	lua_pushinteger(L, r);

	if (r < 0) {
		lua_pushinteger(L, errno);

		return 2;
	}

	lua_createtable(L, 0, 6);

#define UNAME_FIELD(n) do { lua_pushstring(L, #n); \
	lua_pushstring(L, buf.n); \
	lua_settable(L, 2); } while (0)

	UNAME_FIELD(sysname);
	UNAME_FIELD(nodename);
	UNAME_FIELD(release);
	UNAME_FIELD(version);
	UNAME_FIELD(machine);
#ifdef _GNU_SOURCE
	UNAME_FIELD(domainname);
#endif

#undef UNAME_FIELD

	return 2;
}

/*** Time.
@section time
*/

/*** Get time in seconds.
@treturn number seconds since epoch.
@treturn number errno
@function time
*/
static int
luxio_time(lua_State *L) /* 4.5.1 */
{
	lua_pushinteger(L, time(NULL));
	lua_pushinteger(L, errno);

	return 2;
}

/*** times() information table.
Returned by @{times}.
@field utime user time
@field stime system time
@field cutime user time of children
@field cstime system time of children
@table times-table
*/

/*** Get process times.
@treturn number return value
@treturn times-table|errno Time data, or errno.
@function times
*/
static int
luxio_times(lua_State *L) /* 4.5.2 */
{
	struct tms buf;
	clock_t r = times(&buf);

	if (r == (clock_t)-1) {
		lua_pushinteger(L, r);
		lua_pushinteger(L, errno);
		return 2;
	}

	lua_pushinteger(L, r);
	lua_createtable(L, 0, 4);

#define TIMES_FIELD(n) do { lua_pushstring(L, #n); \
	lua_pushinteger(L, buf.tms_##n); \
	lua_settable(L, 2); } while (0)

	TIMES_FIELD(utime);
	TIMES_FIELD(stime);
	TIMES_FIELD(cutime);
	TIMES_FIELD(cstime);

#undef TIMES_FIELD
	return 2;
}

/*** Environment variables.
@section envvar
*/

/*** Get an environment variable.
@tparam string name
@treturn string|nil return value
@function getenv
*/
static int
luxio_getenv(lua_State *L) /* 4.6.1 */
{
	const char *envvar = luaL_checkstring(L, 1);

	char *envval = getenv(envvar);

	if (envval == NULL)
		return 0;

	lua_pushstring(L, envval);

	return 1;
}

/*** Set an environment variable.
@tparam string name
@tparam string value
@tparam[opt=0] number overwrite
@treturn number return value
@treturn errno
@function setenv
*/
static int
luxio_setenv(lua_State *L) /* POSIX.1-2001 */
{
	const char *envvar = luaL_checkstring(L, 1);
	const char *envval = luaL_checkstring(L, 2);
	int overwrite = luaL_optint(L, 3, 1);

	lua_pushinteger(L, setenv(envvar, envval, overwrite));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Unsets an environment variable.
@tparam string name
@treturn number return value
@treturn errno
@function unsetenv
*/
static int
luxio_unsetenv(lua_State *L) /* POSIX.1-2001 */
{
	const char *envvar = luaL_checkstring(L, 1);

	lua_pushinteger(L, unsetenv(envvar));
	lua_pushinteger(L, errno);

	return 2;
}

/* 4.7 Terminal identification ***********************************************/

/* TODO: ctermid() 4.7.1 */
/* TODO: ttyname(), ttyname_r(), isatty() 4.7.2 */

/* 4.8 Configurable system variables *****************************************/

/* TODO: sysconf() 4.8.1 */

/*** Directories.
readdir() is a hideous API.  As such, we need to be reasonably high-level here,
otherwise everything is just too painful.
@section dir
*/

#define LUXIO_READDIR_METATABLE "luxio.readdir"

typedef struct {
	DIR *dirp;
	struct dirent *buf, *ent;
} luxio_readdir_state;

static int
luxio_readdir_gc(lua_State *L)
{
	luxio_readdir_state *s = luaL_checkudata(L, 1, LUXIO_READDIR_METATABLE);

	closedir(s->dirp);
	free(s->buf);

	return 0;
}

static int
luxio_readdir_tostring(lua_State *L)
{
	luxio_readdir_state *s = luaL_checkudata(L, 1, LUXIO_READDIR_METATABLE);
	char buf[sizeof("dirent: 0xffffffffffffffff")];

	/* we can't use lua_pushfstring here, because our pointer might
	 * be 64 bits.
	 */
	snprintf(buf, sizeof(buf), "dirent: %p", s);
	lua_pushstring(L, buf);

	return 1;
}

static void
luxio__bless_readdir(lua_State *L)
{
	int create = luaL_newmetatable(L, LUXIO_READDIR_METATABLE);

	if (create != 0) {
		lua_pushcfunction(L, luxio_readdir_gc);
		lua_setfield(L, -2, "__gc");
		lua_pushcfunction(L, luxio_readdir_tostring);
		lua_setfield(L, -2, "__tostring");
	}

	lua_setmetatable(L, -2);
}

/*** Open a directory for enumeration.
@tparam string dir directory to enumerate
@treturn DIR|nil DIR object, or nil on error
@treturn errno errno
@function opendir
*/
static int
luxio_opendir(lua_State *L) /* 5.1.2 */
{
	const char *path = luaL_checkstring(L, 1);
	DIR *d = opendir(path);
	size_t bufz;
	luxio_readdir_state *s;

	if (d == NULL) {
		lua_pushnil(L);
		lua_pushinteger(L, errno);
		return 2;
	}

	s = lua_newuserdata(L, sizeof(*s));

	s->dirp = d;
	/* + 256 because it'd always be +1 if it weren't for the horrors
	 * of Solaris.  If we were using autoconf, we could use Ben
	 * Hutchings' function mentioned in his article "readdir_r considered
	 * harmful".
	 */
	bufz = sizeof(struct dirent) + pathconf(path, _PC_NAME_MAX) + 256;
	s->buf = malloc(bufz);

	luxio__bless_readdir(L);

	return 1;
}

/*** Open a directory for enumeration by open fd.
@tparam number fd file descriptor
@treturn DIR|nil DIR object, or nil on error
@treturn errno errno
@function fdopendir
*/
static int
luxio_fdopendir(lua_State *L) /* POSIX.1-2008 */
{
	int fd = luaL_checkinteger(L, 1);
	DIR *d = fdopendir(fd);
	size_t bufz;
	luxio_readdir_state *s;

	if (d == NULL) {
		lua_pushnil(L);
		lua_pushinteger(L, errno);
		return 2;
	}

	s = lua_newuserdata(L, sizeof(*s));

	s->dirp = d;
	/* + 256 because it'd always be +1 if it weren't for the horrors
	 * of Solaris.  If we were using autoconf, we could use Ben
	 * Hutchings' function mentioned in his article "readdir_r considered
	 * harmful".
	 */
	bufz = sizeof(struct dirent) + fpathconf(fd, _PC_NAME_MAX) + 256;
	s->buf = malloc(bufz);

	luxio__bless_readdir(L);

	return 1;
}

/*** Close a previously open directory.
@tparam DIR handle DIR object to close
@function closedir
*/
static int
luxio_closedir(lua_State *L) /* 5.1.2 */
{
	luxio_readdir_state *s = luaL_checkudata(L, 1, LUXIO_READDIR_METATABLE);

	if (s->dirp != NULL) {
		closedir(s->dirp);
		s->dirp = NULL;
	}

	free(s->buf);
	s->buf = NULL;

	return 0;
}

/*** readdir() information table.
Returned by @{readdir}.  Some fields are OS-specific.
@field d_ino inode number
@field d_name file name
@field d_type file type
@table dirent
*/

/*** Read a directory entry
@tparam DIR handle directory handle
@treturn dirent|nil directory entry table, or nil on error
@treturn errno errno
@function readdir
*/
static int
luxio_readdir(lua_State *L) /* 5.1.2 */
{
	luxio_readdir_state *s = luaL_checkudata(L, 1, LUXIO_READDIR_METATABLE);
	int err;

	err = readdir_r(s->dirp, s->buf, &s->ent);

	if (err == 0 && s->ent != NULL) {
		lua_pushinteger(L, 0);
		lua_createtable(L, 0, 3);
		lua_pushinteger(L, s->ent->d_ino);
		lua_setfield(L, -2, "d_ino");
		lua_pushstring(L, s->ent->d_name);
		lua_setfield(L, -2, "d_name");
#ifdef HAVE_D_TYPE
		lua_pushinteger(L, s->ent->d_type);
		lua_setfield(L, -2, "d_type");
#endif
		return 2;
	}

	if (s->ent == NULL) {
		/* end of directory */
		lua_pushnil(L);

		return 1;
	}

	lua_pushinteger(L, err);

	return 1;
}

/*** Reset directory stream
@tparam DIR handle
@function rewinddir
*/
static int
luxio_rewinddir(lua_State *L) /* 5.1.2 */
{
	luxio_readdir_state *s = luaL_checkudata(L, 1, LUXIO_READDIR_METATABLE);

	rewinddir(s->dirp);

	return 0;
}

/*** Working directory.
@section workdir
*/

/*** Change working directory.
@tparam string path
@treturn number return value
@treturn errno errno
@function chdir
*/
static int
luxio_chdir(lua_State *L) /* 5.2.1 */
{
	const char *path = luaL_checkstring(L, 1);

	lua_pushinteger(L, chdir(path));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Get the current working directory.
@treturn string|nil path or nil on error
@treturn errno
@function getcwd
*/
static int
luxio_getcwd(lua_State *L) /* 5.2.2 */
{
	char buf[PATH_MAX];

	if (getcwd(buf, PATH_MAX) == NULL) {
		lua_pushnil(L);
	} else {
		lua_pushstring(L, buf);
	}

	lua_pushinteger(L, errno);

	return 2;
}

/*** General file creation.
@section genfile
*/

/*** Open and possibly create a file or device.
@tparam string path
@tparam number flags
@tparam[opt] number mode, must be specified if creating.
@function open
*/
static int
luxio_open(lua_State *L) /* 5.3.1 */
{
	const char *pathname = luaL_checkstring(L, 1);
	int flags = luaL_checkint(L, 2);
	mode_t mode = luaL_optinteger(L, 3, INVALID_MODE);
	int result;

	if ((flags & O_CREAT) && mode == INVALID_MODE) {
		lua_pushstring(L, "open with O_CREAT called with no mode");
		lua_error(L);
	}

	if (mode == INVALID_MODE)
		result = open(pathname, flags);
	else
		result = open(pathname, flags, mode);

	lua_pushinteger(L, result);
	lua_pushinteger(L, errno);

	return 2;
}

/* TODO: creat() 5.3.2 */

/*** Set file mode creation mask.
@tparam number mask
@function umask
*/
static int
luxio_umask(lua_State *L) /* 5.3.3 */
{
	mode_t mask = luaL_checkinteger(L, 1);

	lua_pushinteger(L, umask(mask));

	return 1;
}

/*** Make a new name for a file.
@tparam string existing existing file name
@tparam string new new file name
@treturn number return value
@treturn errno
@function link
*/
static int
luxio_link(lua_State *L) /* 5.3.4 */
{
	const char *existing = luaL_checkstring(L, 1);
	const char *new = luaL_checkstring(L, 2);

	lua_pushinteger(L, link(existing, new));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Make a new (symbolic) name for a file.
@tparam string existing existing file name
@tparam string new new file name
@treturn number return value
@treturn errno
@function symlink
*/
static int
luxio_symlink(lua_State *L) /* POSIX.1-2001, Unknown location */
{
	const char *oldpath = luaL_checkstring(L, 1);
	const char *newpath = luaL_checkstring(L, 2);

	lua_pushinteger(L, symlink(oldpath, newpath));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Read value of symlink.
@tparam string path
@treturn number return value
@treturn string|errno value, or errno on error
@function readlink
*/
static int
luxio_readlink(lua_State *L) /* POSIX.1-2001, Unknown location */
{
	char buffer[PATH_MAX];
	ssize_t ret;
	const char *path = luaL_checkstring(L, 1);

	lua_pushinteger(L, (ret = readlink(path, buffer, PATH_MAX)));
	if (ret > 0) {
		lua_pushinteger(L, errno);
	} else {
		lua_pushstring(L, buffer);
	}

	return 2;
}

/*** Special file creation
@section specfile
*/

/*** Create a directory.
@tparam string path
@tparam number mode
@treturn number return value
@treturn errno
@function mkdir
*/
static int
luxio_mkdir(lua_State *L) /* 5.4.1 */
{
	const char *pathname = luaL_checkstring(L, 1);
	mode_t mode = luaL_checkinteger(L, 2);

	lua_pushinteger(L, mkdir(pathname, mode));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Make a FIFO special file (a named pipe)
@tparam string path
@tparam number mode
@treturn number return value
@treturn errno
@function mkfifo
*/
static int
luxio_mkfifo(lua_State *L) /* 5.4.2 */
{
	const char *pathname = luaL_checkstring(L, 1);
	mode_t mode = luaL_checkinteger(L, 2);

	lua_pushinteger(L, mkfifo(pathname, mode));
	lua_pushinteger(L, errno);

	return 2;
}

/*** File removal
@section filerem
*/

/*** Delete a name and possibly the file it points to.
@tparam string path
@treturn number return value
@treturn errno
@function unlink
*/
static int
luxio_unlink(lua_State *L) /* 5.5.1 */
{
	const char *s = luaL_checkstring(L, 1);

	lua_pushinteger(L, unlink(s));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Delete a directory.
@tparam string path
@treturn number return value
@treturn errno
@function rmdir
*/
static int
luxio_rmdir(lua_State *L) /* 5.5.2 */
{
	const char *pathname = luaL_checkstring(L, 1);

	lua_pushinteger(L, rmdir(pathname));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Change the name or location of a file.
@tparam string oldpath
@tparam string newpath
@treturn number return value
@treturn errno
@function rename
*/
static int
luxio_rename(lua_State *L) /* 5.5.3 */
{
	const char *old = luaL_checkstring(L, 1);
	const char *new = luaL_checkstring(L, 2);

	lua_pushinteger(L, rename(old, new));
	lua_pushinteger(L, errno);

	return 2;
}

/*** File characteristics
@section filechar
*/

/*** stat() information table
Returned by @{stat} and family.
@field dev id of device containing file
@field ino inode of file
@field mode protection mode
@field nlink number of links
@field uid user id of owner
@field gid group id of owner
@field rdev device id (if special file)
@field size total size, in bytes
@field blksize blocksize for file system I/O
@field blocks number of blocks allocated
@field atime time of last access
@field mtime time of last modification
@field ctime time of last status change
@table stat-table
*/

static void
luxio_push_stat_table(lua_State *L, struct stat *s)
{
	lua_createtable(L, 0, 13);

#define PUSH_ENTRY(e) do { lua_pushstring(L, #e);	\
	lua_pushinteger(L, s->st_##e);			\
	lua_settable(L, 3); } while (0)

	PUSH_ENTRY(dev);
	PUSH_ENTRY(ino);
	PUSH_ENTRY(mode);
	PUSH_ENTRY(nlink);
	PUSH_ENTRY(uid);
	PUSH_ENTRY(gid);
	PUSH_ENTRY(rdev);
	PUSH_ENTRY(size);
	PUSH_ENTRY(blksize);
	PUSH_ENTRY(blocks);
	PUSH_ENTRY(atime);
	PUSH_ENTRY(mtime);
	PUSH_ENTRY(ctime);

#undef PUSH_ENTRY
}

/*** Get file status by path.
@tparam string path
@treturn number return value
@treturn errno|stat-table
@function stat
*/
static int
luxio_stat(lua_State *L) /* 5.6.2 */
{
	const char *pathname = luaL_checkstring(L, 1);
	struct stat s;
	int r = stat(pathname, &s);

	lua_pushinteger(L, r);

	if (r < 0) {
		lua_pushinteger(L, errno);
	} else {
		luxio_push_stat_table(L, &s);
	}

	return 2;
}

/*** Get file status by fd.
@tparam number fd
@treturn number return value
@treturn errno|stat-table
@function fstat
*/
static int
luxio_fstat(lua_State *L) /* 5.6.2 */
{
	int fd = luaL_checkinteger(L, 1);
	struct stat s;
	int r = fstat(fd, &s);

	lua_pushinteger(L, r);

	if (r < 0) {
		lua_pushinteger(L, errno);
	} else {
		luxio_push_stat_table(L, &s);
	}

	return 2;
}

/*** Get symlink status by path.
@tparam string path to symlink
@treturn number return value
@treturn errno|stat-table
@function lstat
*/
static int
luxio_lstat(lua_State *L) /* POSIX.1-2001 */
{
	const char *pathname = luaL_checkstring(L, 1);
	struct stat s;
	int r = lstat(pathname, &s);

	lua_pushinteger(L, r);

	if (r < 0) {
		lua_pushinteger(L, errno);
	} else {
		luxio_push_stat_table(L, &s);
	}

	return 2;
}

#define STAT_IS(x) static int luxio_S_IS##x(lua_State *L) { \
	int mode = luaL_checkinteger(L, 1);               \
	lua_pushinteger(L, S_IS##x(mode));                 \
	return 1;                                         \
	}

/*** Check status macro S_ISREG for stat mode field.
@function S_ISREG
@tparam number mode field from a stat call
@treturn number
*/
STAT_IS(REG)

/*** Check status macro S_ISDIR for stat mode field.
@function S_ISDIR
@tparam number mode field from a stat call
@treturn number
*/
 STAT_IS(DIR)

/*** Check status macro S_ISCHR for stat mode field.
@function S_ISCHR
@tparam number mode field from a stat call
@treturn number
*/
STAT_IS(CHR)

/*** Check status macro S_ISBLK for stat mode field.
@function S_ISBLK
@tparam number mode field from a stat call
@treturn number
*/
STAT_IS(BLK)

/*** Check status macro S_ISFIFO for stat mode field.
@function S_ISFIFO
@tparam number mode field from a stat call
@treturn number
*/
STAT_IS(FIFO)

#ifdef S_ISLNK
/*** Check status macro S_ISLNK for stat mode field.
Not always available.
@function S_ISLNK
@tparam number mode field from a stat call
@treturn number
*/
STAT_IS(LNK)
#endif

#ifdef S_ISSOCK
/*** Check status macro S_ISSOCK for stat mode field.
Not always available.
@function S_ISSOCK
@tparam number mode field from a stat call
@treturn number
*/

STAT_IS(SOCK)
#endif

#undef STAT_IS

/* TODO: access() 5.6.3 */

/*** Change permissions of a file.
@tparam string path
@tparam number mode
@treturn number return value
@treturn errno
@function chmod
*/
static int
luxio_chmod(lua_State *L) /* 5.6.4 */
{
	const char *path = luaL_checkstring(L, 1);
	mode_t mode = luaL_checkinteger(L, 2);

	lua_pushinteger(L, chmod(path, mode));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Change permissions of a file by fd.
@tparam number fd
@tparam number mode
@treturn number return value
@treturn errno
@function chmod
*/
static int
luxio_fchmod(lua_State *L) /* 5.6.4 */
{
	int fd = luaL_checkinteger(L, 1);
	mode_t mode = luaL_checkinteger(L, 2);

	lua_pushinteger(L, fchmod(fd, mode));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Change ownership of a file.
@tparam string path
@tparam number owner
@tparam number group
@treturn number return value
@treturn errno
@function chmod
*/
static int
luxio_chown(lua_State *L) /* 5.6.5 */
{
	const char *path = luaL_checkstring(L, 1);
	uid_t owner = luaL_checkinteger(L, 2);
	gid_t group = luaL_checkinteger(L, 3);

	lua_pushinteger(L, chown(path, owner, group));
	lua_pushinteger(L, errno);

	return 2;
}

/* TODO: utime() 5.6.6 */

/*** Truncate a file to a specified length.
@tparam number fd
@tparam number lenght
@treturn number return value
@treturn errno
@function ftruncate
*/
static int
luxio_ftruncate(lua_State *L) /* 5.6.7 */
{
	int fildes = luaL_checkinteger(L, 1);
        off_t length = luaL_checkinteger(L, 2);

	lua_pushinteger(L, ftruncate(fildes, length));
	lua_pushinteger(L, errno);

	return 2;
}

/* 5.7 Configurable pathname variables ***************************************/

/* TODO: pathconf(), fpathconf() 5.7.1 */

/*** Pipes
@section pipes
*/

/*** Create pipe.
@tparam table pipe A table for which this function will fill in the keys 1 and 2
@treturn number return value
@treturn errno
@function pipe
*/
static int
luxio_pipe(lua_State *L) /* 6.1.1 */
{
	int res, pipefd[2];
	luaL_checktype(L, 1, LUA_TTABLE);

	res = pipe(pipefd);
	if (res == 0) {
		lua_pushinteger(L, pipefd[0]);
		lua_rawseti(L, 1, 1);
		lua_pushinteger(L, pipefd[1]);
		lua_rawseti(L, 1, 2);
	}

	lua_pushinteger(L, res);
	lua_pushinteger(L, errno);

	return 2;
}

#ifdef _GNU_SOURCE

/*** Create a pipe, with flags.
Not available on all systems.
@tparam table pipe A table for which this function will fill in the keys 1 and 2
@tparam number flags
@treturn number return value
@treturn errno
@function pipe2
*/
static int
luxio_pipe2(lua_State *L) /* GNU extension */
{
	int res, pipefd[2];
	int flags;

	luaL_checktype(L, 1, LUA_TTABLE);
	flags = luaL_checkinteger(L, 2);

	res = pipe2(pipefd, flags);
	if (res == 0) {
		lua_pushinteger(L, pipefd[0]);
		lua_rawseti(L, 1, 1);
		lua_pushinteger(L, pipefd[1]);
		lua_rawseti(L, 1, 2);
	}

	lua_pushinteger(L, res);
	lua_pushinteger(L, errno);

	return 2;
}
#endif

/*** Create a pair of connected sockets.
@tparam number domain
@tparam number type
@tparam number protocol
@tparam table fdarray The values [1] and [2] are filled in with descriptors
@treturn number return value
@treturn errno
@function socketpair
*/
static int
luxio_socketpair(lua_State *L) /* POSIX.1-2001 */
{
	int domain = luaL_checkinteger(L, 1);
	int type = luaL_checkinteger(L, 2);
	int protocol = luaL_checkinteger(L, 3);
	int sv[2];
	int res;
	luaL_checktype(L, 4, LUA_TTABLE);

	res = socketpair(domain, type, protocol, sv);
	if (res == 0) {
		lua_pushinteger(L, sv[0]);
		lua_rawseti(L, 4, 1);
		lua_pushinteger(L, sv[1]);
		lua_rawseti(L, 4, 2);
	}

	lua_pushinteger(L, res);
	lua_pushinteger(L, errno);

	return 2;
}

/*** File descriptor manipulation.
@section fdmanip
*/

/*** Duplicate a file descriptor.
@tparam number oldfd
@treturn number return value
@treturn errno
@function dup
*/
static int
luxio_dup(lua_State *L) /* 6.2.1 */
{
	int oldfd = luaL_checkint(L, 1);

	lua_pushinteger(L, dup(oldfd));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Duplicate a file descriptor to a specific one
@tparam number oldfd
@tparam number newfd
@treturn number return value
@treturn errno
@function dup2
*/
static int
luxio_dup2(lua_State *L) /* 6.2.1 */
{
	int oldfd = luaL_checkint(L, 1);
	int newfd = luaL_checkint(L, 2);

	lua_pushinteger(L, dup2(oldfd, newfd));
	lua_pushinteger(L, errno);

	return 2;
}

#ifdef _GNU_SOURCE
/** Duplicate a file descriptor to a specific one, with flags.
Not available on all platforms.
@tparam number oldfd
@tparam number newfd
@tparam number flags
@treturn number return value
@treturn errno
@function dup3
*/
static int
luxio_dup3(lua_State *L) /* GNU extension */
{
	int oldfd = luaL_checkint(L, 1);
	int newfd = luaL_checkint(L, 2);
	int flags = luaL_checkint(L, 3);

	lua_pushinteger(L, dup3(oldfd, newfd, flags));
	lua_pushinteger(L, errno);

	return 2;
}
#endif

/*** File descriptor deassignment.
@section fileclose
*/

/*** Close a file descriptor.
@tparam number fd
@treturn number return value
@treturn errno
@function close
*/
static int
luxio_close(lua_State *L) /* 6.3.1 */
{
	lua_pushinteger(L, close(luaL_checkint(L, 1)));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Input and output
@section io
*/

/*** Read from a file descriptor.
@tparam number fd
@tparam number count
@treturn number|string return value or read data
@treturn errno
@function read
*/
static int
luxio_read(lua_State *L) /* 6.4.1 */
{
    int fd = luaL_checkint(L, 1);
    int count = luaL_checkint(L, 2);
    ssize_t result;
    char *buf = malloc(count);

    if (buf == NULL) {
        lua_pushstring(L, "unable to allocate read buffer: memory exhausted");
        lua_error(L);
    }

    result = read(fd, buf, count);

    if (result == -1) {
        lua_pushinteger(L, result);
        lua_pushinteger(L, errno);
    } else {
        /* sadly there appears to be no way to avoid this copy.
         * luaL_Buffer actually builds things on the C stack bytes at a time,
         * and there is no way to pre-allocate a Lua type other than a
         * userdatum.  Additionally, should Lua call its panic function because
         * it can't allocate memory to copy this into, our buf will be leaked.
         * We could perhaps fix this with a lot of faff, involving creating
         * a userdatum for our buffer, and setting a __gc metamethod.
         */
        lua_pushlstring(L, buf, result);
        lua_pushinteger(L, errno);
    }

    free(buf);

    return 2;
}

/*** Write to a file descriptor.
@tparam number fd
@tparam string data
@tparam[opt=0] number start_offset
@treturn number return value
@treturn errno
@function write
*/
static int
luxio_write(lua_State *L) /* 6.4.2 */
{
	int fd = luaL_checkint(L, 1);
	size_t count;
	const char *buf = luaL_checklstring(L, 2, &count);
	size_t offset = luaL_optinteger(L, 3, 0);

	if (offset > count) offset = count;

	lua_pushinteger(L, write(fd, buf + offset, count - offset));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Write data from multiple buffers.
@tparam number fd
@tparam string ...
@treturn number return value
@treturn errno
@function writev
*/
static int
luxio_writev(lua_State *L) /* POSIX.1-2001 */
{
	int fd = luaL_checkint(L, 1);
	int blks = lua_gettop(L) - 1;
	int c;
	struct iovec *iov;

	/* check there is at least one string to write */
	luaL_checkstring(L, 2);

	iov = malloc(blks * sizeof(*iov));

	for (c = 0; c < blks; c++) {
		iov[c].iov_base = (void *)luaL_checkstring(L, c + 2);
		iov[c].iov_len = lua_rawlen(L, c + 2);
	}

	lua_pushinteger(L, writev(fd, iov, blks));
	lua_pushinteger(L, errno);

	free(iov);

	return 2;
}

#ifdef HAVE_SENDFILE
/*** Transfer data between descriptors.
Not available on all systems.
@tparam number out_fd
@tparam number in_fd
@tparam[opt=nil] number offset
@tparam number count
@treturn number return value
@treturn errno
@function sendfile
*/
static int
luxio_sendfile(lua_State *L) /* Linux-specific */
{
	int out_fd = luaL_checkint(L, 1);
	int in_fd = luaL_checkint(L, 2);
	off_t offset;
	size_t count = luaL_checkint(L, 4);
	ssize_t r;

	if (lua_isnil(L, 3)) {
		r = sendfile(out_fd, in_fd, NULL, count);
		lua_pushinteger(L, r);
		lua_pushinteger(L, errno);
		return 2;
	}

	offset = luaL_checkint(L, 3);
	r = sendfile(out_fd, in_fd, &offset, count);
	lua_pushinteger(L, r);
	lua_pushinteger(L, errno);
	lua_pushinteger(L, offset);

	return 3;
}
#endif /* HAVE_SENDFILE */

#ifdef HAVE_SPLICE
/*** Splice data to or from a pipe.
Not available on all systems.
@tparam number fd_in
@tparam number off_in
@tparam number fd_out
@tparam number off_out
@tparam number len
@tparam number flags
@treturn number return value
@treturn errno
@function splice
*/
static int
luxio_splice(lua_State *L) /* Linux-specific */
{
	int fd_in = luaL_checkinteger(L, 1);
	loff_t off_in = luaL_optlong(L, 2, -1);
	int fd_out = luaL_checkinteger(L, 3);
	loff_t off_out = luaL_optlong(L, 4, -1);
	size_t len = luaL_checkinteger(L, 5);
	unsigned int flags = luaL_checkinteger(L, 6);

	loff_t *poff_in = &off_in;
	loff_t *poff_out = &off_out;

	if (off_in == -1) poff_in = NULL;
	if (off_out == -1) poff_out = NULL;

	lua_pushinteger(L, splice(fd_in, poff_in, fd_out, poff_out,
				  len, flags));
	lua_pushinteger(L, errno);

	return 2;
}
#endif

/*** Control operations on files
@section fcntl
*/

/*** Manipulate file descriptor.
Supported commands: F_GETFD/F_SETFD, F_GETFL/F_SETFL,
F_GETPIPE_SZ/F_SETPIPE_SZ, F_DUPFD, F_DUPFD_CLOEXEC,
F_SETLK, F_SETLKW, F_GETLK.

Commands that take a struct, such as F_SETLK, accept
a table as the argument, with keys named as the struct's.

@tparam number fd
@tparam number command
@tparam ... ...
@treturn number return value
@treturn errno
@function fcntl
*/
static int
luxio_fcntl(lua_State *L) /* 6.5.2 */
{
	int fd = luaL_checkint(L, 1);
	int cmd = luaL_checkint(L, 2);
	long arg_long;
	struct flock flock;

	switch (cmd) {
	/* commands that take no argument */
	case F_GETFD:
	case F_GETFL:
#ifdef F_GETPIPE_SZ
	case F_GETPIPE_SZ:
#endif
		lua_pushinteger(L, fcntl(fd, cmd));
		lua_pushinteger(L, errno);
		return 2;

	/* commands that take a long */
	case F_DUPFD:
#ifdef F_DUPFD_CLOEXEC
	case F_DUPFD_CLOEXEC:
#endif
	case F_SETFD:
	case F_SETFL:
#ifdef F_SETPIPE_SZ
	case F_SETPIPE_SZ:
#endif
		arg_long = luaL_checkinteger(L, 3);
		lua_pushinteger(L, fcntl(fd, cmd, arg_long));
		lua_pushinteger(L, errno);

		return 2;

	/* commands that take exciting things */
	case F_SETLK:
	case F_SETLKW:
	case F_GETLK:
		luaL_checktype(L, 3, LUA_TTABLE);

		lua_getfield(L, 3, "l_type");
		lua_getfield(L, 3, "l_whence");
		lua_getfield(L, 3, "l_start");
		lua_getfield(L, 3, "l_len");
		flock.l_type = lua_tonumber(L, -4);
		flock.l_whence = lua_tonumber(L, -3);
		flock.l_start = lua_tonumber(L, -2);
		flock.l_len = lua_tonumber(L, -1);
		flock.l_pid = 0;

		lua_pushinteger(L, fcntl(fd, cmd, &flock));
		lua_pushinteger(L, errno);

		if (cmd == F_GETLK) {
			lua_pushnumber(L, flock.l_type);
			lua_pushnumber(L, flock.l_whence);
			lua_pushnumber(L, flock.l_start);
			lua_pushnumber(L, flock.l_len);
			lua_pushnumber(L, flock.l_pid);
			lua_setfield(L, 3, "l_pid");
			lua_setfield(L, 3, "l_len");
			lua_setfield(L, 3, "l_start");
			lua_setfield(L, 3, "l_whence");
			lua_setfield(L, 3, "l_type");
		}

		return 2;

	default:
		lua_pushstring(L, "unhandled fcntl() command");
		lua_error(L);
	}

	return 0; /* never get here, but keep compiler happy */
}

#ifdef _LARGEFILE64_SOURCE
static int
luxio_lseek(lua_State *L) /* 6.5.3 */
{
	int fd = luaL_checkint(L, 1);
	off64_t offset = (off64_t)luaL_checknumber(L, 2); /* 56b is enough! */
	int whence = luaL_checkint(L, 3);

	lua_pushinteger(L, (lua_Number)lseek64(fd, offset, whence));
	lua_pushinteger(L, errno);

	return 2;
}
#else
/*** Reposition read/write file offset.
@tparam number fd
@tparam number offset
@tparam number whence
@treturn number return value
@treturn errno
@function lseek
*/
static int
luxio_lseek(lua_State *L) /* 6.5.3 */
{
	int fd = luaL_checkint(L, 1);
	off_t offset = luaL_checkinteger(L, 2);
	int whence = luaL_checkint(L, 3);

	lua_pushinteger(L, lseek(fd, offset, whence));
	lua_pushinteger(L, errno);

	return 2;
}
#endif

/*** File synchronisation.
@section filesync
*/

/*** Synchronise a file's in-core state with storage device.
@tparam number fd
@treturn number return value
@treturn errno
@function fsync
*/
static int
luxio_fsync(lua_State *L) /* 6.6.1 */
{
	int fildes = luaL_checkinteger(L, 1);

	lua_pushinteger(L, fsync(fildes));
	lua_pushinteger(L, errno);

	return 2;
}

#ifdef HAVE_FDATASYNC
/*** Synchronise only a file's data and not unnessercery metadata.
Not available on all systems.
@tparam number fd
@treturn number return value
@treturn errno
@function fdatasync
*/
static int
luxio_fdatasync(lua_State *L) /* 6.6.2 */
{
	int fildes = luaL_checkinteger(L, 1);

	lua_pushinteger(L, fdatasync(fildes));
	lua_pushinteger(L, errno);

	return 2;
}
#endif

/* 6.7 Asynchronous input and output */

/* TODO: aio_read() 6.7.2 */
/* TODO: aio_write() 6.7.3 */
/* TODO: lio_listio() 6.7.4 */
/* TODO: aio_error() 6.7.5 */
/* TODO: aio_return() 6.7.6 */
/* TODO: aio_cancel() 6.7.7 */
/* TODO: aio_suspend() 6.7.8 */
/* TODO: aio_fsync() 6.7.9 */

/*** General Terminal Interface.
@section genterm
*/

/* TODO: cfgetispeed(), cfgetospeed(), cfsetispeed(), cfsetospeed() 7.1.3 */
/* TODO: tcgetattr(), tcsetattr() 7.2.1 */
/* TODO: tcsendbreak(), tcdrain(), tcflush(), tcflow() 7.2.2 */

/*** Get terminal forground process group.
@tparam number fd
@treturn number return value
@treturn errno
@function tcgetpgrp
*/
static int
luxio_tcgetpgrp(lua_State *L) /* 7.2.3 */
{
	lua_pushinteger(L, tcgetpgrp(luaL_checkinteger(L, 1)));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Set terminal forground process group.
@tparam number fd
@tparam number pgrp_id
@treturn number return value
@treturn errno
@function tcsetpgrp
*/
static int
luxio_tcsetpgrp(lua_State *L) /* 7.2.4 */
{
	int fildes = luaL_checkinteger(L, 1);
	pid_t pgrp_id = luaL_checkinteger(L, 2);

	lua_pushinteger(L, tcsetpgrp(fildes, pgrp_id));
	lua_pushinteger(L, errno);

	return 2;
}

/* 8.1 Referenced C Language Routines ****************************************/

/* These are ANSI C functions POSIX imports.  TODO any Lua doesn't already */

/* 9.1 Database access *******************************************************/

/* TODO: getgrgid(), getgrgid_r(), getgrnam(), getgrnam_r() 9.2.1 */
/* TODO: getpwuid(), getpwuid_r(), getpwnam(), getpwnam_r() 9.2.2 */

/* 10 Data interchange format ************************************************/

/* This is just related to data structures and file formats, not functions */

/* 11 Synchronisation ********************************************************/

/* Semaphores, mutexes, etc should be handled by a dedicated threading lib */

/* 12 Memory management ******************************************************/

/* While it might be interesting to bind mmap and mlock etc, it's difficult to
 * see how this might work in Lua.  Perhaps emulate a string?
 */

/* 13 Execution scheduling ***************************************************/

/* TODO: all of this. */

/*** Clock and timer functions.
@section clocktimer
*/

/* TODO: clock_settime(), clock_gettime(), clock_getres() 14.2.1 */
/* Timer functions excluded, based on signals */

/*** High-resolution sleep.
@tparam number seconds
@tparam number nanoseconds
@treturn number return value
@treturn errno
@treturn number remaining seconds
@treturn number remaining nanosections
@function nanosleep
*/
static int
luxio_nanosleep(lua_State *L) /* 14.2.5 */
{
	struct timespec req, rem = { 0, 0 };

	req.tv_sec = luaL_checkinteger(L, 1);
	req.tv_nsec = luaL_checkinteger(L, 2);

	lua_pushinteger(L, nanosleep(&req, &rem));
	lua_pushinteger(L, errno);
	lua_pushinteger(L, rem.tv_sec);
	lua_pushinteger(L, rem.tv_nsec);

	return 4;
}

/*** Message passing.
POSIX message passing is not available on all platforms.
@section msgpass
*/
#if defined(_POSIX_MESSAGE_PASSING)

#include <mqueue.h>

#define LUXIO_MQ_METATABLE_NAME "luxio.mq"

struct luxio_mq_data {
	mqd_t mq;
	char name[NAME_MAX];
};

static int
luxio__mq_tostring(lua_State *L)
{
	char buf[NAME_MAX + 64];

	struct luxio_mq_data *m =
		luaL_checkudata(L, 1, LUXIO_MQ_METATABLE_NAME);

	sprintf(buf, "<mq %p %s>", (void *)(intptr_t)m->mq, m->name);

	lua_pushstring(L, buf);
	return 1;
}

/*** Open a message queue.
@tparam string name
@tparam number oflag
@tparam[opt] number mode
@treturn mq|number message queue, or return value
@treturn errno
@function mq_open
*/
static int
luxio_mq_open(lua_State *L) /* 15.2.1 */
{
	const char *name = luaL_checkstring(L, 1);
	int oflag = luaL_checkinteger(L, 2);
	mode_t mode = luaL_optinteger(L, 3, INVALID_MODE);
	mqd_t mq;
	struct luxio_mq_data *m;

	if ((oflag & O_CREAT) && mode == INVALID_MODE) {
		lua_pushstring(L, "mq_open with O_CREATE called with no mode");
		lua_error(L);
	}

	if (oflag & O_CREAT) {
		mq = mq_open(name, oflag, mode, NULL);
	} else {
		mq = mq_open(name, oflag);
	}

	if (mq == (mqd_t)-1) {
		lua_pushnumber(L, -1);
		lua_pushinteger(L, errno);

		return 2;
	}

	m = lua_newuserdata(L, sizeof(*m));
	m->mq = mq;
	strncpy(m->name, name, NAME_MAX);

	if (luaL_newmetatable(L, LUXIO_MQ_METATABLE_NAME) != 0) {
		lua_pushcfunction(L, luxio__mq_tostring);
		lua_setfield(L, -2, "__tostring");
	}

	lua_setmetatable(L, -2);
	lua_pushinteger(L, errno);

	return 2;
}

/*** Close a message queue descriptor.
@tparam mq mqdes
@treturn number return value
@treturn errno
@function mq_close
*/
static int
luxio_mq_close(lua_State *L) /* 15.2.2 */
{
	struct luxio_mq_data *m =
		luaL_checkudata(L, 1, LUXIO_MQ_METATABLE_NAME);

	lua_pushinteger(L, mq_close(m->mq));
	lua_pushinteger(L, errno);
	return 2;
}

/*** Remove a message queue.
@tparam string name
@treturn number return value
@treturn errno
@function mq_unlink
*/
static int
luxio_mq_unlink(lua_State *L) /* 15.2.3 */
{
	const char *name = luaL_checkstring(L, 1);

	lua_pushinteger(L, mq_unlink(name));
	lua_pushinteger(L, errno);
	return 2;
}

/*** Send a message to a message queue.
@tparam mq queue
@tparam string message
@tparam number priority
@treturn number return value
@treturn errno
@function mq_send
*/
static int
luxio_mq_send(lua_State *L) /* 15.2.4 */
{
	struct luxio_mq_data *m =
		luaL_checkudata(L, 1, LUXIO_MQ_METATABLE_NAME);
	size_t msg_len;
	const char *msg_ptr = luaL_checklstring(L, 2, &msg_len);
	unsigned int msg_prio = luaL_checkinteger(L, 3);

	lua_pushinteger(L, mq_send(m->mq, msg_ptr, msg_len, msg_prio));
	lua_pushinteger(L, errno);
	return 2;
}


/*** Receive a message from a message queue.
@tparam mq queue
@treturn number return value
@treturn errno
@treturn string|nil message data, or nil in case of error
@treturn number|nil message priority, or nil in case of error
@function mq_receive
*/
static int
luxio_mq_receive(lua_State *L) /* 15.2.5 */
{
	struct luxio_mq_data *m =
		luaL_checkudata(L, 1, LUXIO_MQ_METATABLE_NAME);
	unsigned int msg_prio;
	struct mq_attr attr;

	/* Find out the maximum size of a message */
	if (mq_getattr(m->mq, &attr) == -1) {
		lua_pushinteger(L, -1);
		lua_pushinteger(L, errno);
		return 2;
	} else {
		char msg_ptr[attr.mq_msgsize];
		int r = mq_receive(m->mq, msg_ptr, sizeof(msg_ptr), &msg_prio);
		lua_pushinteger(L, r);
		lua_pushinteger(L, errno);
		if (r == -1) {
			return 2;
		}
		lua_pushlstring(L, msg_ptr, r);
		lua_pushinteger(L, msg_prio);
		return 4;
	}
}

/* TODO: mq_notify() 15.2.6 */

/*** Message queue attributes table.
@field mq_flags 0 or O_NONBLOCK
@field mq_maxmsg Maximum number of messages on queue
@field mq_msgsize Maximum size of message (in bytes)
@field mq_curmsgs Number of messages currently on queue
@table mqattr-table
*/
static int
luxio_make_attr_table(lua_State *L, struct mq_attr *attr)
{
	int top = lua_gettop(L) + 1;

	lua_createtable(L, 0, 4);

#define PUSH_ENTRY(e) do { lua_pushstring(L, "mq_" #e); \
	lua_pushinteger(L, attr->mq_##e); \
	lua_settable(L, top); } while (0)

	PUSH_ENTRY(flags);
	PUSH_ENTRY(maxmsg);
	PUSH_ENTRY(msgsize);
	PUSH_ENTRY(curmsgs);

#undef PUSH_ENTRY

	return 1;
}

/*** Set message queue attributes.
As only the flags can be changed, this does not take a table.
@tparam mq mqdes
@tparam number flags
@treturn number return value
@treturn errno
@treturn mqattr-table new attribute table
@function mq_setattr
*/
static int
luxio_mq_setattr(lua_State *L) /* 15.2.7 */
{
	struct luxio_mq_data *m =
		luaL_checkudata(L, 1, LUXIO_MQ_METATABLE_NAME);
	struct mq_attr mqstat = { luaL_checkinteger(L, 2), 0, 0, 0 };
	struct mq_attr omqstat = { 0, 0, 0, 0 };

	lua_pushinteger(L, mq_setattr(m->mq, &mqstat, &omqstat));
	lua_pushinteger(L, errno);
	luxio_make_attr_table(L, &omqstat);

	return 3;
}

/*** Get message queue attributes.
@tparam mq mqdes
@treturn number return value
@treturn errno
@treturn mqattr-table current queue attributes
@function mq_getattr
*/
static int
luxio_mq_getattr(lua_State *L) /* 15.2.8 */
{
	struct luxio_mq_data *m =
		luaL_checkudata(L, 1, LUXIO_MQ_METATABLE_NAME);
	struct mq_attr mqstat;

	lua_pushinteger(L, mq_getattr(m->mq, &mqstat));
	lua_pushinteger(L, errno);
	luxio_make_attr_table(L, &mqstat);

	return 3;
}

#if _XOPEN_SOURCE >= 600 || _POSIX_C_SOURCE >= 200112L

/*** Send a message to a queue, with a timeout.
Not available on some systems.
@tparam mq queue
@tparam string message
@tparam number priority
@tparam number tv_sec
@tparam number tv_nsec
@treturn number return value
@treturn errno
@function mq_timedsend
*/
/**% mq_timedsend
 * retval = mq_timedsend(mq, msg, len, prio, timeout)
 * retval, errno = mq_timedsend(mq, msg, prio, tv_secs, tv_nsec)
 */
static int
luxio_mq_timedsend(lua_State *L) /* POSIX.1-2001 */
{
	struct luxio_mq_data *m =
		luaL_checkudata(L, 1, LUXIO_MQ_METATABLE_NAME);
	size_t msg_len;
	const char *msg_ptr = luaL_checklstring(L, 2, &msg_len);
	unsigned int msg_prio = luaL_checkinteger(L, 3);
	time_t tv_secs = luaL_checkinteger(L, 4);
	long tv_nsec = luaL_checkinteger(L, 5);
	struct timespec abs_timeout = { tv_secs, tv_nsec };

	lua_pushinteger(L, mq_timedsend(m->mq, msg_ptr, msg_len, msg_prio,
				&abs_timeout));
	lua_pushinteger(L, errno);
	return 2;
}

/*** Receive a message from a message queue, with a timeout
@tparam mq queue
@treturn number return value
@treturn errno
@tparam number tv_sec
@tparam number tv_nsec
@treturn string|nil message data, or nil in case of error
@treturn number|nil message priority, or nil in case of error
@function mq_timedreceive
*/
/**% mq_timedreceive
 * retval = mq_timedsend(mq, bug, len, prio, timeout)
 * retval, errno, msg, prio = mq_timedreceive(mq, tv_secs, tv_nsec)
 */
static int
luxio_mq_timedreceive(lua_State *L) /* POSIX.1-2001 */
{
	struct luxio_mq_data *m =
		luaL_checkudata(L, 1, LUXIO_MQ_METATABLE_NAME);
	unsigned int msg_prio;
	struct mq_attr attr;
	time_t tv_secs = luaL_checkinteger(L, 2);
	long tv_nsec = luaL_checkinteger(L, 3);
	struct timespec abs_timeout = { tv_secs, tv_nsec };

	/* Find out the maximum size of a message */
	if (mq_getattr(m->mq, &attr) == -1) {
		lua_pushinteger(L, -1);
		lua_pushinteger(L, errno);
		return 2;
	} else {
		char msg_ptr[attr.mq_msgsize];
		int r = mq_timedreceive(m->mq, msg_ptr, sizeof(msg_ptr),
				&msg_prio, &abs_timeout);
		lua_pushinteger(L, r);
		lua_pushinteger(L, errno);
		if (r == -1) {
			return 2;
		}
		lua_pushlstring(L, msg_ptr, r);
		lua_pushinteger(L, msg_prio);
		return 4;
	}
}
#endif /* POSIX.1-2001 check */
#endif /* _POSIX_MESSAGE_PASSING */

/* 16 Thread management ******************************************************/

/* Nope: use a threading library. */

/*** Socket handling.
This interface is slightly cooked.  We provide userdata encapsulations for
sockaddr and addrinfo types.  Use `getaddrinfo()` to obtain an addrinfo
object, then use `ipairs()` over it to get each entry to try.

  r, addrinfo = getaddrinfo("www.rjek.com", "80", 0, l.AF_UNSPEC, l.SOCK_STREAM)
  for _, ai in ipairs(addrinfo) do
    sock = socket(ai.ai_family, ai.ai_socktype, ai.ai_protocol)
    if connect(sock, ai.ai_addr) >= 0 then break end
  end
@section sock
*/

#define LUXIO_SOCKADDR_METATABLE_NAME "luxio.sockaddr"
#ifndef UNIX_PATH_MAX
/* From man 7 unix */
#define UNIX_PATH_MAX   108
#endif

static int
luxio__sockaddr_index(lua_State *L)
{
	struct sockaddr *sa = luaL_checkudata(L, 1, LUXIO_SOCKADDR_METATABLE_NAME);
	const char *var = luaL_checkstring(L, 2);
	if (strcmp(var, "family") == 0) {
		lua_pushnumber(L, sa->sa_family);
		return 1;
	}
	if (sa->sa_family == AF_INET) {
		struct sockaddr_in *sa_in = (struct sockaddr_in *)sa;
		if (strcmp(var, "port") == 0) {
			lua_pushnumber(L, ntohs(sa_in->sin_port));
			return 1;
		} else if (strcmp(var, "address") == 0) {
			char tmp_buf[INET_ADDRSTRLEN];
			lua_pushstring(L, inet_ntop(AF_INET, &sa_in->sin_addr, tmp_buf, INET_ADDRSTRLEN));
			return 1;
		}
	}
	if (sa->sa_family == AF_INET6) {
		struct sockaddr_in6 *sa_in = (struct sockaddr_in6 *)sa;
		if (strcmp(var, "port") == 0) {
			lua_pushnumber(L, ntohs(sa_in->sin6_port));
			return 1;
		} else if (strcmp(var, "flowinfo") == 0) {
			lua_pushnumber(L, sa_in->sin6_flowinfo);
			return 1;
		} else if (strcmp(var, "scope_id") == 0) {
			lua_pushnumber(L, sa_in->sin6_scope_id);
			return 1;
		} else if (strcmp(var, "address") == 0) {
			char tmp_buf[INET6_ADDRSTRLEN];
			lua_pushstring(L, inet_ntop(AF_INET6, &sa_in->sin6_addr, tmp_buf, INET6_ADDRSTRLEN));
			return 1;
		}
	}
	if (sa->sa_family == AF_UNIX) {
		struct sockaddr_un *sa_un = (struct sockaddr_un *)sa;
		if (strcmp(var, "path") == 0) {
			lua_pushstring(L, sa_un->sun_path);
			return 1;
		}
	}
	return luaL_error(L, "unknown field %s in sockaddr", var);
}

static int
luxio__sockaddr_tostring(lua_State *L)
{
	struct sockaddr *sa = luaL_checkudata(L, 1, LUXIO_SOCKADDR_METATABLE_NAME);
	if (sa->sa_family == AF_INET) {
		struct sockaddr_in *sa_in = (struct sockaddr_in *)sa;
		char tmp_buf[INET_ADDRSTRLEN];
		inet_ntop(AF_INET, &sa_in->sin_addr, tmp_buf, INET_ADDRSTRLEN);
		lua_pushfstring(L, "sockaddr: AF_INET %d %s", ntohs(sa_in->sin_port), tmp_buf);
	} else if (sa->sa_family == AF_INET6) {
		struct sockaddr_in6 *sa_in = (struct sockaddr_in6 *)sa;
		char tmp_buf[INET6_ADDRSTRLEN];
		inet_ntop(AF_INET6, &sa_in->sin6_addr, tmp_buf, INET6_ADDRSTRLEN);
		lua_pushfstring(L, "sockaddr: AF_INET6 %d %s", ntohs(sa_in->sin6_port), tmp_buf);
	} else if (sa->sa_family == AF_UNIX) {
		struct sockaddr_un *sa_un = (struct sockaddr_un *)sa;
		lua_pushfstring(L, "sockaddr: AF_UNIX %s", sa_un->sun_path);
	} else {
		lua_pushfstring(L, "sockaddr: unknown family %d", sa->sa_family);
	}
	return 1;
}

/* Label the userdata on the top of the stack as a sockaddr */
static int
luxio___makesockaddr(lua_State *L)
{
	int create_mt = luaL_newmetatable(L, LUXIO_SOCKADDR_METATABLE_NAME);

	if (create_mt) {
		lua_pushcfunction(L, luxio__sockaddr_index);
		lua_setfield(L, -2, "__index");
		lua_pushcfunction(L, luxio__sockaddr_tostring);
		lua_setfield(L, -2, "__tostring");
	}

	lua_setmetatable(L, -2);

	return 1;
}

/* Copy the given sockaddr and push a userdata onto the stack labelled as a sockaddr */
static int
luxio__makesockaddr(lua_State *L, const struct sockaddr *sa, socklen_t len)
{
	struct sockaddr *sa_copy = lua_newuserdata(L, len);

	memcpy(sa_copy, sa, len);

	return luxio___makesockaddr(L);
}

/* Utility for sizing supported socket addresses */
static socklen_t
luxio___sockaddr_len(lua_State *L, const struct sockaddr *sa)
{
	switch (sa->sa_family) {
	case AF_INET:
		return sizeof(struct sockaddr_in);
	case AF_INET6:
		return sizeof(struct sockaddr_in6);
	case AF_UNIX:
		return SUN_LEN((struct sockaddr_un *)sa);
	}

	return luaL_error(L, "unknown address family %d", sa->sa_family);
}

static int
luxio_makesockaddr(lua_State *L)
{
	int family = luaL_checkinteger(L, 1);
	if (family == AF_INET) {
		struct sockaddr_in *sa_in;
		int port = luaL_checkinteger(L, 2);
		const char *address = luaL_checkstring(L, 3);
		sa_in = lua_newuserdata(L, sizeof(*sa_in));
		if (inet_pton(AF_INET, address, &sa_in->sin_addr) != 1) {
			return luaL_error(L, "unable to parse address: %s", address);
		}
		if (port < 0 || port > 65535) {
			return luaL_error(L, "port %d out of range", port);
		}
		sa_in->sin_family = AF_INET;
		sa_in->sin_port = htons(port);
		return luxio___makesockaddr(L);
	}
	if (family == AF_INET6) {
		struct sockaddr_in6 *sa_in;
		int port = luaL_checkinteger(L, 2);
		const char *address = luaL_checkstring(L, 3);
		sa_in = lua_newuserdata(L, sizeof(*sa_in));
		if (inet_pton(AF_INET6, address, &sa_in->sin6_addr) != 1) {
			return luaL_error(L, "unable to parse address: %s", address);
		}
		if (port < 0 || port > 65535) {
			return luaL_error(L, "port %d out of range", port);
		}
		/* TODO: Support flow and scope */
		sa_in->sin6_family = AF_INET6;
		sa_in->sin6_port = htons(port);
		sa_in->sin6_flowinfo = 0;
		sa_in->sin6_scope_id = 0;
		return luxio___makesockaddr(L);
	}
	if (family == AF_UNIX) {
		struct sockaddr_un *sa_un;
		size_t pathlen;
		const char *path = luaL_checklstring(L, 2, &pathlen);
		if (pathlen >= UNIX_PATH_MAX) {
			return luaL_error(L, "unable to create local socket, path too long (%d chars)",
					  pathlen);
		}
		sa_un = lua_newuserdata(L, sizeof(sa_un->sun_family) + pathlen + 1);
		sa_un->sun_family = AF_UNIX;
		strcpy(&sa_un->sun_path[0], path);
		return luxio___makesockaddr(L);
	}
	return luaL_error(L, "unknown socket family %d", family);
}

/**% socket
 * retval = socket(domain, type, protocol);
 * retval, errno = socket(domain, type, protocol)
 */
/*** Create an endpoint for communication.
@tparam number domain
@tparam number type
@tparam number protocol
@treturn number return value
@treturn errno
@function socket
*/
static int
luxio_socket(lua_State *L)
{
	int domain = luaL_checkint(L, 1);
	int type = luaL_checkint(L, 2);
	int protocol = luaL_checkint(L, 3);

	lua_pushinteger(L, socket(domain, type, protocol));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Listen for connections on a socket.
@tparam number sockfd
@tparam number backlog
@treturn number return value
@treturn errno
@function listen
*/
static int
luxio_listen(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	int backlog = luaL_checkint(L, 2);

	lua_pushinteger(L, listen(sockfd, backlog));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Shut down part of a full-duplex connection.
@tparam number sockfd
@tparam number how
@treturn number return value
@treturn errno
@function shutdown
*/
static int
luxio_shutdown(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	int how = luaL_checkint(L, 2);

	lua_pushinteger(L, shutdown(sockfd, how));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Initiate a connection on a socket.
@tparam number fd
@tparam sockaddr sockaddr
@treturn number return value
@treturn errno
@function connect
*/
static int
luxio_connect(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	struct sockaddr *addr = luaL_checkudata(L, 2, LUXIO_SOCKADDR_METATABLE_NAME);
	socklen_t l = luxio___sockaddr_len(L, addr);

	lua_pushinteger(L, connect(sockfd, addr, l));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Bind a name to a socket.
@tparam number fd
@tparam sockaddr sockaddr
@treturn number return value
@treturn errno
@function bind
*/
static int
luxio_bind(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	struct sockaddr *addr = luaL_checkudata(L, 2, LUXIO_SOCKADDR_METATABLE_NAME);
	socklen_t l = luxio___sockaddr_len(L, addr);

	lua_pushinteger(L, bind(sockfd, addr, l));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Accept a connection on a socket.
@tparam number fd
@treturn number return value
@treturn errno|sockaddr
@function accept
*/
static int
luxio_accept(lua_State *L)
{
	int listenfd = luaL_checkint(L, 1);
	int clientfd;
	struct sockaddr_storage addr;
	socklen_t l = sizeof(addr);

	clientfd = accept(listenfd, (struct sockaddr *)&addr, &l);

	lua_pushinteger(L, clientfd);

	if (clientfd > -1) {
		luxio__makesockaddr(L, (const struct sockaddr *)&addr, l);
	} else {
		lua_pushinteger(L, errno);
	}

	return 2;
}

/*** Get options on a socket.
@tparam number fd
@tparam number level
@tparam number optname
@treturn number return value
@treturn errno|number|string
@function getsockopt
*/
static int
luxio_getsockopt(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	int level = luaL_checkint(L, 2);
	int optname = luaL_checkint(L, 3);
	int res, r_int;
	char r_ifname[IFNAMSIZ];
	socklen_t l;

	switch (level) {
	case SOL_SOCKET:
		switch (optname) {
		/* options that involve strings */
#ifdef SO_BINDTODEVICE
		case SO_BINDTODEVICE:
#endif
			l = IFNAMSIZ;
			res = getsockopt(sockfd, level, optname, r_ifname, &l);
			lua_pushinteger(L, res);
			if (res == -1) {
				lua_pushinteger(L, errno);
			} else {
				lua_pushstring(L, r_ifname);
			}

			return 2;

                /* other options probably expect integers */
		default:
			l = sizeof(r_int);
			res = getsockopt(sockfd, level, optname, &r_int, &l);
			lua_pushinteger(L, res);
			if (res == -1) {
				lua_pushinteger(L, errno);
			} else {
				lua_pushinteger(L, r_int);
			}

			return 2;
		}
	case IPPROTO_IP:
		switch (optname) {
#ifdef TCP_CORK
		case TCP_CORK:
#endif
#ifdef TCP_DEFER_ACCEPT
		case TCP_DEFER_ACCEPT:
#endif
#ifdef TCP_NODELAY
		case TCP_NODELAY:
#endif
			l = sizeof(r_int);
			res = getsockopt(sockfd, level, optname, &r_int, &l);
			lua_pushinteger(L, res);
			if (res == -1) {
				lua_pushinteger(L, errno);
			} else {
				lua_pushinteger(L, r_int);
			}

			return 2;

		default:
			return luaL_error(L, "unhandled IPPROTO_IP option %d",
					  optname);
		}
	}

	return luaL_error(L, "unhandled socket level %d", level);
}

/*** Set an option on a socket.
@tparam number fd
@tparam number level
@tparam number optname
@param optvalue
@treturn number return value
@treturn errno
@function setsockopt
*/
static int
luxio_setsockopt(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	int level = luaL_checkint(L, 2);
	int optname = luaL_checkint(L, 3);
	int res, r_int;
	const char *r_ifname;
	socklen_t l;
	size_t sz;

	switch (level) {
	case SOL_SOCKET:
		switch (optname) {
		/* options that involve strings */
#ifdef SO_BINDTODEVICE
		case SO_BINDTODEVICE:
#endif
			r_ifname = luaL_checklstring(L, 4, &sz);
			res = setsockopt(sockfd, level, optname,
					 (void *)r_ifname, (socklen_t)sz);
			lua_pushinteger(L, res);
			lua_pushinteger(L, errno);

			return 2;
		/* other options are probably integers */
		default:
			l = sizeof(r_int);
			r_int = luaL_checkint(L, 4);
			res = setsockopt(sockfd, level, optname, &r_int, l);
			lua_pushinteger(L, res);
			lua_pushinteger(L, errno);

			return 2;
		}
	case IPPROTO_IP:
		switch (optname) {
#ifdef TCP_CORK
		case TCP_CORK:
#endif
#ifdef TCP_DEFER_ACCEPT
		case TCP_DEFER_ACCEPT:
#endif
#ifdef TCP_NODELAY
		case TCP_NODELAY:
#endif
			l = sizeof(r_int);
			r_int = luaL_checkint(L, 4);
			res = setsockopt(sockfd, level, optname, &r_int, l);
			lua_pushinteger(L, res);
			lua_pushinteger(L, errno);

			return 2;

		default:
			return luaL_error(L, "unhandled IPPROTO_IP option %d",
					  optname);
		}
	}

	return luaL_error(L, "unhandled socket level %d", level);
}

/*** Convert getaddrinfo-specific errors to strings.
@tparam errno errno
@treturn string error string
@function gai_strerror
*/
static int
luxio_gai_strerror(lua_State *L)
{
	lua_pushstring(L, gai_strerror(luaL_checkint(L, 1)));

	return 1;
}

/*** Result table from `getaddrinfo`.
@field ai_flags number
@field ai_family number
@field ai_socktype number
@field ai_protocol number
@field ai_canonname string
@field ai_addr sockaddr type containing address information.
@table addrinfo
*/

/*** Network address and service translation.
@tparam string node
@tparam string service
@tparam[opt=0] number ai_flags
@tparam[opt=AF_UNSPEC] number ai_family
@tparam[opt=0] number ai_socktype
@tparam[opt=0] number ai_protocol
@treturn errno|table table of result `addrinfo` entries
@function getaddrinfo
*/
static int
luxio_getaddrinfo(lua_State *L)
{
	const char *node = luaL_checkstring(L, 1);
	const char *serv = luaL_checkstring(L, 2);
	struct addrinfo hints, *results, *rp;
	int r, c;

	memset(&hints, '\0', sizeof(hints));
	hints.ai_flags = luaL_optint(L, 3, 0);
	hints.ai_family = luaL_optint(L, 4, AF_UNSPEC);
	hints.ai_socktype = luaL_optint(L, 5, 0);
	hints.ai_protocol = luaL_optint(L, 6, 0);

	if (node[0] == '\0')
		node = NULL;

	if (serv[0] == '\0')
		serv = NULL;

	r = getaddrinfo(node, serv, &hints, &results);

	lua_pushinteger(L, r);

	if (r < 0)
		return 1;

	lua_newtable(L); /* table we return with entries */

	for (rp = results, c = 1; rp != NULL; rp = rp->ai_next, c++) {
		lua_createtable(L, 0, 6); /* entry table */

		lua_pushliteral(L, "ai_flags");
		lua_pushinteger(L, rp->ai_flags);
		lua_rawset(L, -3);

		lua_pushliteral(L, "ai_family");
		lua_pushinteger(L, rp->ai_family);
		lua_rawset(L, -3);

		lua_pushliteral(L, "ai_socktype");
		lua_pushinteger(L, rp->ai_socktype);
		lua_rawset(L, -3);

		lua_pushliteral(L, "ai_protocol");
		lua_pushinteger(L, rp->ai_protocol);
		lua_rawset(L, -3);

		lua_pushliteral(L, "ai_canonname");
		lua_pushstring(L, rp->ai_canonname);
		lua_rawset(L, -3);

		lua_pushliteral(L, "ai_addr");
		luxio__makesockaddr(L, rp->ai_addr, rp->ai_addrlen);
		lua_rawset(L, -3);

		lua_rawseti(L, -2, c);
	}

	freeaddrinfo(results);

	return 2;
}

/*** Socket-related send and receive functions.
@section socksendrecv
*/

/*** Send a message on a socket.
@tparam number fd
@tparam string data
@tparam[opt=0] number flags
@treturn number return value
@treturn errno
@function send
*/
static int
luxio_send(lua_State *L)
{
    int fd = luaL_checkint(L, 1);
    size_t count;
    const char *buf = luaL_checklstring(L, 2, &count);
    int flags = luaL_optint(L, 3, 0);
    
    lua_pushinteger(L, send(fd, buf, count, flags));
    lua_pushinteger(L, errno);
    
    return 2;
}

/*** Send a message on a socket to a specific destination.
@tparam number fd
@tparam string data
@tparam[opt=0] number flags
@tparam sockaddr sockaddr
@treturn number return value
@treturn errno
@function sendto
*/
static int
luxio_sendto(lua_State *L)
{
    int fd = luaL_checkint(L, 1);
    size_t count;
    const char *buf = luaL_checklstring(L, 2, &count);
    int flags = luaL_optint(L, 3, 0);
    struct sockaddr *addr = luaL_checkudata(L, 4, LUXIO_SOCKADDR_METATABLE_NAME);
    socklen_t l = luxio___sockaddr_len(L, addr);

    lua_pushinteger(L, sendto(fd, buf, count, flags, addr, l));
    lua_pushinteger(L, errno);
    
    return 2;
}

/*** Receive a message from a socket.
@tparam number fd
@tparam number count
@tparam[opt=0] number flags
@treturn number|string return value if error, otherwise string
@treturn errno
@function recv
*/
static int
luxio_recv(lua_State *L)
{
    int fd = luaL_checkint(L, 1);
    int count = luaL_checkint(L, 2);
    int flags = luaL_optint(L, 3, 0);
    ssize_t result;
    char *buf = malloc(count);
    
    if (buf == NULL) {
        lua_pushstring(L, "unable to allocate read buffer: memory exhausted");
        lua_error(L);
    }
    
    result = recv(fd, buf, count, flags);
    
    if (result == -1) {
        lua_pushinteger(L, result);
        lua_pushinteger(L, errno);
    } else {
        /* sadly there appears to be no way to avoid this copy.
         * luaL_Buffer actually builds things on the C stack bytes at a time,
         * and there is no way to pre-allocate a Lua type other than a
         * userdatum.  Additionally, should Lua call its panic function because
         * it can't allocate memory to copy this into, our buf will be leaked.
         * We could perhaps fix this with a lot of faff, involving creating
         * a userdatum for our buffer, and setting a __gc metamethod.
         */
        lua_pushlstring(L, buf, result);
        lua_pushinteger(L, errno);
    }

    free(buf);

    return 2;
}

/*** Receive a message from a socket, also returning sender information.
@tparam number fd
@tparam number count
@tparam[opt=0] number flags
@treturn number|string return value if error, otherwise string
@treturn errno
@treturn sockaddr|nil
@function recvfrom
*/
static int
luxio_recvfrom(lua_State *L)
{
    int fd = luaL_checkint(L, 1);
    int count = luaL_checkint(L, 2);
    int flags = luaL_optint(L, 3, 0);
    struct sockaddr_storage addr;
    socklen_t l = sizeof(addr);
    ssize_t result;
    char *buf = malloc(count);
    
    if (buf == NULL) {
        lua_pushstring(L, "unable to allocate read buffer: memory exhausted");
        lua_error(L);
    }
    
    result = recvfrom(fd, buf, count, flags, (struct sockaddr *)&addr, &l);
    
    if (result == -1) {
        free(buf);
        lua_pushinteger(L, result);
        lua_pushinteger(L, errno);
        return 2;
    }
    
    /* sadly there appears to be no way to avoid this copy.
     * luaL_Buffer actually builds things on the C stack bytes at a time,
     * and there is no way to pre-allocate a Lua type other than a
     * userdatum.  Additionally, should Lua call its panic function because
     * it can't allocate memory to copy this into, our buf will be leaked.
     * We could perhaps fix this with a lot of faff, involving creating
     * a userdatum for our buffer, and setting a __gc metamethod.
     */
    lua_pushlstring(L, buf, result);
    free(buf);
    lua_pushinteger(L, errno);
    luxio__makesockaddr(L, (const struct sockaddr *)&addr, l);
    
    return 3;
}

/*** Poll-binding functions.
@section poll
*/

#define LUXIO_POLLFD_METATABLE "luxio.pollfdarray"

typedef struct {
	struct pollfd *pollfds;
	int allocated;
} luxio_pollfds;

static int
luxio__pollfds_gc(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	free(pfds->pollfds);
	pfds->pollfds = NULL;
	pfds->allocated = 0;
	return 0;
}

static int
luxio__pollfds_tostring(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	lua_pushfstring(L, "pollfds: %d slot%s", pfds->allocated, pfds->allocated == 1 ? "" : "s");
	return 1;
}

static int
luxio__pollfds_len(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	lua_pushnumber(L, pfds->allocated);
	return 1;
}

static int
luxio_pollfds_new(lua_State *L)
{
	luxio_pollfds *pfds = lua_newuserdata(L, sizeof(*pfds));
	int create_table = luaL_newmetatable(L, LUXIO_POLLFD_METATABLE);
	pfds->pollfds = NULL;
	pfds->allocated = 0;

	if (create_table) {
		lua_pushcfunction(L, luxio__pollfds_gc);
		lua_setfield(L, -2, "__gc");
		lua_pushcfunction(L, luxio__pollfds_tostring);
		lua_setfield(L, -2, "__tostring");
		lua_pushcfunction(L, luxio__pollfds_len);
		lua_setfield(L, -2, "__len");
	}

	lua_setmetatable(L, -2);

	return 1;
}

static int
luxio_pollfds_resize(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	int desired_size = luaL_checkint(L, 2);
	int idx;
	struct pollfd *newfds = realloc(pfds->pollfds, sizeof(struct pollfd) * desired_size);
	if (newfds != NULL) {
		if (desired_size > pfds->allocated) {
			for (idx = pfds->allocated; idx < desired_size; ++idx) {
				newfds[idx].fd = -1;
				newfds[idx].events = newfds[idx].revents = 0;
			}
		}
		pfds->pollfds = newfds;
		pfds->allocated = desired_size;
	} else {
		return luaL_error(L, "Unable to resize pollfds array");
	}
	/* Return the pollfds array for neatness */
	return 1;
}

static int
luxio_pollfds_set_slot(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	int slot = luaL_checkint(L, 2);
	int fd = luaL_checkint(L, 3);
	short events = luaL_checkint(L, 4);
	short revents;
	if (slot == 0 || slot > pfds->allocated || slot < -pfds->allocated) {
		return luaL_error(L, "slot out of range 1 .. %d", pfds->allocated);
	}

	revents = luaL_optint(L, 4, pfds->pollfds[slot - 1].revents);

	pfds->pollfds[slot - 1].fd = fd;
	pfds->pollfds[slot - 1].events = events;
	pfds->pollfds[slot - 1].revents = revents;

	return 0;
}

static int
luxio_pollfds_get_slot(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	int slot = luaL_checkint(L, 2);

	if (slot == 0 || slot > pfds->allocated || slot < -pfds->allocated) {
		return luaL_error(L, "slot out of range 1 .. %d", pfds->allocated);
	}

	lua_pushnumber(L, pfds->pollfds[slot - 1].fd);
	lua_pushnumber(L, pfds->pollfds[slot - 1].events);
	lua_pushnumber(L, pfds->pollfds[slot - 1].revents);
	return 3;
}

static int
luxio_poll(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	int timeout = luaL_checkint(L, 2);

	lua_pushinteger(L, poll(pfds->pollfds, pfds->allocated, timeout));
	lua_pushinteger(L, errno);

	return 2;
}

/*** Bit and flag operation functions.
@section bit
*/

static int
luxio_bitop_or(lua_State *L)
{
	int value = luaL_checkint(L, 1);
	int n = lua_gettop(L);

	while (n > 1) {
		value |= luaL_checkint(L, n--);
	}

	lua_pushnumber(L, value);

	return 1;
}

static int
luxio_bitop_and(lua_State *L)
{
	int value = luaL_checkint(L, 1);
	int n = lua_gettop(L);

	while (n > 1) {
		value &= luaL_checkint(L, n--);
	}

	lua_pushnumber(L, value);

	return 1;
}

static int
luxio_bitop_clear(lua_State *L)
{
	int value = luaL_checkint(L, 1);
	int n = lua_gettop(L);

	while (n > 1) {
		value &= ~luaL_checkint(L, n--);
	}

	lua_pushnumber(L, value);

	return 1;
}

static int
luxio_bitop_invert(lua_State *L)
{
	int value = luaL_checkint(L, 1);
	int n = lua_gettop(L);

	/* Special case, passed 1 value, we invert that rather than
	 * inverting the other bits supplied
	 */
	if (n == 1) {
		value = ~value;
	} else {
		while (n > 1) {
		     value ^= luaL_checkint(L, n--);
		}
	}

	lua_pushnumber(L, value);

	return 1;
}

static int
luxio_bitop_test(lua_State *L)
{
	int value = luaL_checkint(L, 1);
	int goal = 0;
	int n = lua_gettop(L);

	while (n > 1) {
		goal |= luaL_checkint(L, n--);
	}

	lua_pushboolean(L, (value & goal) == goal);

	return 1;
}

/*** Time-related functions.
The time-related functions in Luxio are medium-rare.  A timeval type is exposed
as a userdata type, complete with comparison, addition/subtraction, and tostring
metamethods.  You can set the fields tv_sec, tv_usec, seconds, and useconds.
@section time
*/
#define LUXIO_TIMEVAL_METATABLE "luxio.timeval"

static int
luxio_timeval_lt(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	struct timeval *b = luaL_checkudata(L, 2, LUXIO_TIMEVAL_METATABLE);

	lua_pushboolean(L, timercmp(a, b, <));

	return 1;
}

static int
luxio_timeval_le(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	struct timeval *b = luaL_checkudata(L, 2, LUXIO_TIMEVAL_METATABLE);

	/* <= is not portable, so use ! > */
	lua_pushboolean(L, !timercmp(a, b, >));

	return 1;
}

static int
luxio_timeval_eq(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	struct timeval *b = luaL_checkudata(L, 2, LUXIO_TIMEVAL_METATABLE);

	/* == is not portable, so use ! != */
	lua_pushboolean(L, !timercmp(a, b, !=));

	return 1;
}

#define LUXIO_TIME_BUFLEN (1024)
static int
luxio_timeval_tostring(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	char buffer[LUXIO_TIME_BUFLEN];

	snprintf(buffer, LUXIO_TIME_BUFLEN, "timeval: %ld.%06ld",
		 (long)a->tv_sec, (long)a->tv_usec);

	lua_pushstring(L, buffer);

	return 1;
}

static int
luxio_timeval_index(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	const char *s = luaL_checkstring(L, 2);

	if (strcmp(s, "tv_sec") == 0)
		lua_pushinteger(L, a->tv_sec);
	else if (strcmp(s, "tv_usec") == 0)
		lua_pushinteger(L, a->tv_usec);
	else if (strcmp(s, "seconds") == 0)
		lua_pushnumber(L, (lua_Number)(a->tv_sec) +
			       ((lua_Number)(a->tv_usec) / 1000000));
	else if (strcmp(s, "useconds") == 0)
		lua_pushinteger(L, a->tv_usec + (a->tv_sec * 1000000));
	else
		luaL_error(L, "Unknown field %s in timeval", s);

	return 1;
}

static int
luxio_timeval_newindex(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	const char *s = luaL_checkstring(L, 2);

	if (strcmp(s, "tv_sec") == 0)
		a->tv_sec = luaL_checkinteger(L, 3);
	else if (strcmp(s, "tv_usec") == 0)
		a->tv_usec = luaL_checkinteger(L, 3);
	else if (strcmp(s, "seconds") == 0) {
		lua_Number v = luaL_checknumber(L, 3);
		a->tv_sec = (time_t)v;
		a->tv_usec = (suseconds_t)((v - a->tv_sec) * 1000000);
	} else if (strcmp(s, "useconds") == 0) {
		lua_Number v = luaL_checknumber(L, 3);
		a->tv_sec = (time_t)(v / 1000000);
		a->tv_usec = (suseconds_t)v % 1000000;
	} else
		luaL_error(L, "Unknown field %s in timeval", s);

	return 0;
}

static void luxio__bless_timeval(lua_State *L);

static int
luxio_timeval_add(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	struct timeval *b = luaL_checkudata(L, 2, LUXIO_TIMEVAL_METATABLE);
	struct timeval *ret = lua_newuserdata(L, sizeof(*ret));

	timeradd(a, b, ret);

	luxio__bless_timeval(L);

	return 1;
}

static int
luxio_timeval_sub(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	struct timeval *b = luaL_checkudata(L, 2, LUXIO_TIMEVAL_METATABLE);
	struct timeval *ret = lua_newuserdata(L, sizeof(*ret));

	timersub(a, b, ret);

	luxio__bless_timeval(L);

	return 1;
}

static void
luxio__bless_timeval(lua_State *L)
{
	int create = luaL_newmetatable(L, LUXIO_TIMEVAL_METATABLE);

	if (create) {
		lua_pushcfunction(L, luxio_timeval_le);
		lua_setfield(L, -2, "__le");
		lua_pushcfunction(L, luxio_timeval_lt);
		lua_setfield(L, -2, "__lt");
		lua_pushcfunction(L, luxio_timeval_eq);
		lua_setfield(L, -2, "__eq");
		lua_pushcfunction(L, luxio_timeval_tostring);
		lua_setfield(L, -2, "__tostring");
		lua_pushcfunction(L, luxio_timeval_index);
		lua_setfield(L, -2, "__index");
		lua_pushcfunction(L, luxio_timeval_newindex);
		lua_setfield(L, -2, "__newindex");
		lua_pushcfunction(L, luxio_timeval_add);
		lua_setfield(L, -2, "__add");
		lua_pushcfunction(L, luxio_timeval_sub);
		lua_setfield(L, -2, "__sub");
	}

	lua_setmetatable(L, -2);
}

/*** Create a new timeval, set to the epoch.
@treturn timeval
@function zero_timeval
*/
static int
luxio_timeval_zero(lua_State *L)
{
	struct timeval *r = lua_newuserdata(L, sizeof(*r));

	timerclear(r);

	luxio__bless_timeval(L);

	return 1;
}

/*** Get the time of day.
@treturn timeval|number return value
@treturn errno
@function gettimeofday
*/
static int
luxio_gettimeofday(lua_State *L)
{
	struct timeval *r = lua_newuserdata(L, sizeof(*r));

	int ret = gettimeofday(r, NULL);

	if (ret == -1) {
		lua_pushinteger(L, -1);
		lua_pushinteger(L, errno);
		return 2;
	}

	luxio__bless_timeval(L);

	return 1;
}

/*** Misc utility functions.
@section misc
*/

/*** Return a string describing an error number.
@tparam errno errno
@treturn string
@function strerror
*/
static int
luxio_strerror(lua_State *L)
{
	lua_pushstring(L, strerror(luaL_checkint(L, 1)));

	return 1;
}

static char *luxio_openlog_ident = NULL;

/*** Open a log file.
@tparam string ident
@tparam number option
@tparam number facility
@function openlog
*/
static int
luxio_openlog(lua_State *L)
{
	size_t len;
	const char *ident = luaL_checklstring(L, 1, &len);
	int option = luaL_checkint(L, 2);
	int facility = luaL_checkint(L, 3);

	/* openlog doesn't make its own copy of ident
	 * and lua could garbage collect ident
	 * so take a copy of ident
	 */
	free(luxio_openlog_ident);
	luxio_openlog_ident = malloc(len);
	strncpy(luxio_openlog_ident, ident, len);

	openlog(ident, option, facility);

	return 0;
}

/*** Write a message to the open log.
@tparam number priority
@tparam string log message
@function syslog
*/
static int
luxio_syslog(lua_State *L)
{
	int priority = luaL_checkint(L, 1);
	const char *msg = luaL_checkstring(L, 2);

	syslog(priority, "%s", msg);

	return 0;
}

/*** Close the open log.
@function closelog
*/
static int
luxio_closelog(lua_State *L)
{
	free(luxio_openlog_ident);
	luxio_openlog_ident = NULL;
	closelog();

	return 0;
}

/*** Set the log priority mask.
@tparam number newmask
@tparam number old mask
@function setlogmask
*/
static int
luxio_setlogmask(lua_State *L)
{
	int mask = luaL_checkint(L, 1);

	int oldmask = setlogmask(mask);
	lua_pushinteger(L, oldmask);

	return 1;
}

/*** Discover the bit used for a specific priority.
@tparam number priority
@treturn number mask
@funciton LOG_MASK
*/
static int
luxio_LOG_MASK(lua_State *L)
{
	int priority = luaL_checkint(L, 1);

	int mask = LOG_MASK(priority);
	lua_pushinteger(L, mask);

	return 1;
}

/*** Allocate descriptor for character set conversion.
@tparam string tocode
@tparam string fromcode
@treturn iconv|number return value
@treturn errno
@function icon_open
*/
static int
luxio_iconv_open(lua_State *L)
{
	const char *tocode = luaL_checkstring(L, 1);
	const char *fromcode = luaL_checkstring(L, 2);
	iconv_t r = iconv_open(tocode, fromcode);

	if (r == (iconv_t)-1) {
		lua_pushnumber(L, -1);
		lua_pushnumber(L, errno);
		return 2;
	} else {
		lua_pushlightuserdata(L, (void *)r);
		return 1;
	}
}

/*** Close a previously-allocated iconv descriptor.
@tparam iconv handle
@treturn number return vlaue
@treturn errno
@function iconv_close
*/
static int
luxio_iconv_close(lua_State *L)
{
	iconv_t c = (iconv_t)lua_touserdata(L, 1);

	if (iconv_close(c) == -1) {
		lua_pushnumber(L, -1);
	} else {
		lua_pushnumber(L, 0);
	}

	lua_pushnumber(L, errno);
	return 2;
}

#define ICONV_BUF_SIZE 256

/*** Perform character set conversion.
@tparam iconv handle
@tparam string buf
@treturn string|number resulting string or return value in case of error
@treturn errno
@treturn string|nil partial result when error.
@function iconv
*/
static int
luxio_iconv(lua_State *L)
{
	/* based on Alexandre Erwin Ittner <alexandre@ittner.com.br>'s code */
	/* either returns completed string, or -1, errno, partial result */
    iconv_t cd = (iconv_t)lua_touserdata(L, 1);
    size_t ibleft;
    char *inbuf = (char* )luaL_checklstring(L, 2, &ibleft);
    char outbufs[ICONV_BUF_SIZE];
    char *outbuf = outbufs;
    size_t obleft = ICONV_BUF_SIZE;
    size_t ret = -1;

    luaL_Buffer b;
    luaL_buffinit(L, &b);

    do {
        ret = iconv(cd, (ICONV_IN_TYPE)&inbuf, &ibleft, &outbuf, &obleft);
        if (ret == (size_t)(-1)) {
            luaL_addlstring(&b, outbufs, ICONV_BUF_SIZE - obleft);

            if (errno == E2BIG) {
              obleft = ICONV_BUF_SIZE;
              outbuf = outbufs;
            } else {
              lua_pushnumber(L, -1);
              lua_pushnumber(L, errno);
              luaL_pushresult(&b);
              return 3;
            }
        }
    } while (ret == (size_t)-1);

    luaL_addlstring(&b, outbufs, ICONV_BUF_SIZE - obleft);
    luaL_pushresult(&b);
    return 1;
}

#undef ICONV_BUF_SIZE

static const struct luaL_Reg
luxio_functions[] = {
	{ "open", luxio_open },
	{ "close", luxio_close },
	{ "read", luxio_read },
	{ "write", luxio_write },
	{ "writev", luxio_writev },
	{ "lseek", luxio_lseek },
	{ "ftruncate", luxio_ftruncate },
	{ "fsync", luxio_fsync },
#ifdef HAVE_FDATASYNC
	{ "fdatasync", luxio_fdatasync },
#endif
	{ "rename", luxio_rename },
	{ "link", luxio_link },
	{ "unlink", luxio_unlink },
	{ "symlink", luxio_symlink },
	{ "readlink", luxio_readlink },
#ifdef HAVE_SENDFILE
	{ "sendfile", luxio_sendfile },
#endif
#ifdef HAVE_SPLICE
	{ "splice", luxio_splice },
#endif
	{ "dup", luxio_dup },
	{ "dup2", luxio_dup2 },
#ifdef _GNU_SOURCE
	{ "dup3", luxio_dup3 },
#endif
        { "pipe", luxio_pipe },
#ifdef _GNU_SOURCE
        { "pipe2", luxio_pipe2 },
#endif
	{ "socketpair", luxio_socketpair },
	{ "fcntl", luxio_fcntl },
	{ "umask", luxio_umask },
	{ "chmod", luxio_chmod },
	{ "fchmod", luxio_fchmod },
	{ "chown", luxio_chown },
	{ "mkfifo", luxio_mkfifo },
	{ "mkdir", luxio_mkdir },
	{ "rmdir", luxio_rmdir },

#define STAT_IS_ENTRY(x) { "S_IS" #x, luxio_S_IS##x }
	STAT_IS_ENTRY(REG),
	STAT_IS_ENTRY(DIR),
	STAT_IS_ENTRY(CHR),
	STAT_IS_ENTRY(BLK),
	STAT_IS_ENTRY(FIFO),
#ifdef S_ISLNK
	STAT_IS_ENTRY(LNK),
#endif
#ifdef S_ISSOCK
	STAT_IS_ENTRY(SOCK),
#endif
#undef STAT_IS_ENTRY

	{ "stat", luxio_stat },
	{ "lstat", luxio_lstat },
	{ "fstat", luxio_fstat },

	{ "socket", luxio_socket },
	{ "listen", luxio_listen },
	{ "shutdown", luxio_shutdown },
	{ "connect", luxio_connect },
	{ "bind", luxio_bind },
	{ "accept", luxio_accept },
	{ "getsockopt", luxio_getsockopt },
	{ "setsockopt", luxio_setsockopt },

	{ "getaddrinfo", luxio_getaddrinfo },
	{ "gai_strerror", luxio_gai_strerror },

	{ "make_sockaddr", luxio_makesockaddr },

    { "send", luxio_send },
    { "sendto", luxio_sendto },
    { "recv", luxio_recv },
    { "recvfrom", luxio_recvfrom },
    
	{ "pollfds_new", luxio_pollfds_new },
	{ "pollfds_resize", luxio_pollfds_resize },
	{ "pollfds_setslot", luxio_pollfds_set_slot },
	{ "pollfds_getslot", luxio_pollfds_get_slot },
	{ "poll", luxio_poll },

	{ "zero_timeval", luxio_timeval_zero },
	{ "gettimeofday", luxio_gettimeofday },

	{ "fork", luxio_fork },
	{ "exec", luxio_exec },
	{ "execp", luxio_execp },
	{ "waitpid", luxio_waitpid },

#define WAITPID_STATUS_ENTRY(x) { #x, luxio_##x }
	WAITPID_STATUS_ENTRY(WIFEXITED),
	WAITPID_STATUS_ENTRY(WEXITSTATUS),
	WAITPID_STATUS_ENTRY(WIFSIGNALED),
	WAITPID_STATUS_ENTRY(WTERMSIG),
#ifdef WCOREDUMP
	WAITPID_STATUS_ENTRY(WCOREDUMP),
#endif
	WAITPID_STATUS_ENTRY(WIFSTOPPED),
	WAITPID_STATUS_ENTRY(WSTOPSIG),
#ifdef WIFCONTINUED
	WAITPID_STATUS_ENTRY(WIFCONTINUED),
#endif
#undef WAITPID_STATUS_ENTRY

	{ "kill", luxio_kill },

	{ "strerror",  luxio_strerror },
	{ "_exit", luxio__exit },

	{ "setenv", luxio_setenv },
	{ "unsetenv", luxio_unsetenv },
	{ "getenv", luxio_getenv },

	{ "opendir", luxio_opendir },
	{ "fdopendir", luxio_fdopendir },
	{ "closedir", luxio_closedir },
	{ "readdir", luxio_readdir },
	{ "rewinddir", luxio_rewinddir },

	{ "chdir", luxio_chdir },
	{ "getcwd", luxio_getcwd },

	{ "alarm", luxio_alarm },
	{ "pause", luxio_pause },
	{ "sleep", luxio_sleep },

	{ "getpid", luxio_getpid },
	{ "getppid", luxio_getppid },

	{ "getuid", luxio_getuid },
	{ "geteuid", luxio_geteuid },
	{ "getgid", luxio_getgid },
	{ "getegid", luxio_getegid },
	{ "setuid", luxio_setuid },
	{ "setgid", luxio_setgid },
	{ "getlogin", luxio_getlogin },

	{ "uname", luxio_uname },

	{ "time", luxio_time },
	{ "times", luxio_times },

	{ "tcgetpgrp", luxio_tcgetpgrp},
	{ "tcsetpgrp", luxio_tcsetpgrp},

	{ "nanosleep", luxio_nanosleep },

#if defined(_POSIX_MESSAGE_PASSING)

#if _XOPEN_SOURCE >= 600 || _POSIX_C_SOURCE >= 200112L
	{ "mq_timedsend", luxio_mq_timedsend },
	{ "mq_timedreceive", luxio_mq_timedreceive },
#endif
	{ "mq_open", luxio_mq_open },
	{ "mq_close", luxio_mq_close },
	{ "mq_unlink", luxio_mq_unlink },
	{ "mq_send", luxio_mq_send },
	{ "mq_receive", luxio_mq_receive },
	{ "mq_setattr", luxio_mq_setattr },
	{ "mq_getattr", luxio_mq_getattr },
#endif
	{ "openlog", luxio_openlog },
	{ "syslog", luxio_syslog },
	{ "closelog", luxio_closelog },
	{ "setlogmask", luxio_setlogmask },
	{ "LOG_MASK", luxio_LOG_MASK },

	{ "iconv_open", luxio_iconv_open },
	{ "iconv_close", luxio_iconv_close },
	{ "iconv", luxio_iconv },

	{ NULL, NULL }
};

static const struct luaL_Reg
luxio_bitop_functions[] = {
	{ "bor", luxio_bitop_or },
	{ "band", luxio_bitop_and },
	{ "binvert", luxio_bitop_invert },
	{ "btest", luxio_bitop_test },
	{ "bclear", luxio_bitop_clear },

	{ NULL, NULL }
};

#include "luxio_constants.h"

#define NUMERIC_CONSTANT(x) do { lua_pushstring(L, #x);  \
				 lua_pushinteger(L, x); \
				 lua_settable(L, -3); } while (0)

int
luaopen_luxio(lua_State *L)
{
	int e;
	const char *n;
	lua_Number v;

#if (LUA_VERSION_NUM > 501)
	luaL_newlib(L, luxio_functions);
	luaL_newlib(L, luxio_bitop_functions);
#else
	luaL_register(L, "luxio", luxio_functions);
	lua_createtable(L, 0, (sizeof(luxio_bitop_functions) /
			       sizeof(struct luaL_Reg)) - 1);
	luaL_register(L, NULL, luxio_bitop_functions);
#endif
	lua_setfield(L, -2, "bit");

	for (e = 0;; e++) {
		n = luxio_numeric_constants[e].name;
		v = luxio_numeric_constants[e].number;

		if (n == NULL) break;
		lua_pushstring(L, n);
		lua_pushnumber(L, v);
		lua_settable(L, -3);
	}

	lua_pushstring(L, "_VERSION");
	lua_pushfstring(L, "Luxio %d", LUXIO_RELEASE);
	lua_settable(L, -3);

	lua_pushstring(L, "_COPYRIGHT");
	lua_pushstring(L, LUXIO_COPYRIGHT);
	lua_settable(L, -3);

	lua_pushstring(L, "_RELEASE");
	lua_pushnumber(L, LUXIO_RELEASE);
	lua_settable(L, -3);

	lua_pushstring(L, "_ABI");
	lua_pushnumber(L, LUXIO_ABI);
	lua_settable(L, -3);

	/* push values that are not compile-time known */
#ifdef SIGRTMIN
	NUMERIC_CONSTANT(SIGRTMIN);
	NUMERIC_CONSTANT(SIGRTMAX);
#endif
#ifdef HAVE_D_TYPE
	NUMERIC_CONSTANT(DT_UNKNOWN);
	NUMERIC_CONSTANT(DT_FIFO);
	NUMERIC_CONSTANT(DT_CHR);
	NUMERIC_CONSTANT(DT_DIR);
	NUMERIC_CONSTANT(DT_BLK);
	NUMERIC_CONSTANT(DT_REG);
	NUMERIC_CONSTANT(DT_LNK);
	NUMERIC_CONSTANT(DT_SOCK);
#endif
	return 1;
}

#undef NUMERIC_CONSTANT
