#include "nsec3_rr.h"

namespace ADNS {

	NSEC3_RR::NSEC3_RR()
	{
		rr_type = RR_TYPE::NSEC3;
		rr_class = RR_CLASS::IN;
	}

	array<Byte>^ NSEC3_RR::GetNextHashedOwner()
	{
		array<Byte>^ output = gcnew array<Byte>(NextHashedOwner->Length);
		NextHashedOwner->CopyTo(output,0);
		return output;
	}

	Void NSEC3_RR::SetNextHashedOwner(array<Byte>^ newhash)
	{
		if (NextHashedOwner == nullptr)
			NextHashedOwner = gcnew array<Byte>(newhash->Length);
		else
			NextHashedOwner->Resize(NextHashedOwner,newhash->Length);

		newhash->CopyTo(NextHashedOwner,0);

		UpdateRdata();	
		return;
	}

	HASH_ALGORITHM NSEC3_RR::GetHashAlgorithm()
	{
		return algo;
	}

	Void NSEC3_RR::SetHashAlgorithm(HASH_ALGORITHM n)
	{
		algo = n;
		UpdateRdata();
	}

	bool NSEC3_RR::GetOptOutFlag()
	{
		if (Flags & 0x01)
			return true;

		return false;
	}

	Void NSEC3_RR::SetOptOutFlag(bool val)
	{
		if (val)
			Flags = Flags | 0x01;
		else
			Flags = Flags & 0xFE;

		UpdateRdata();
		return;
	}

	UInt16 NSEC3_RR::GetIterations()
	{
		return Iterations;
	}

	Void NSEC3_RR::SetIterations(UInt16 val)
	{
		Iterations = val;
		UpdateRdata();
		return;
	}

	array<Byte>^ NSEC3_RR::GetSalt()
	{
		array<Byte>^ output = gcnew array<Byte>(Salt->Length);
		Salt->CopyTo(output,0);
		return output;
	}

	Void NSEC3_RR::SetSalt(array<Byte>^ newsalt)
	{
		if (Salt == nullptr)
			Salt = gcnew array<Byte>(newsalt->Length);
		else
			Salt->Resize(Salt,newsalt->Length);

		newsalt->CopyTo(Salt,0);

		UpdateRdata();	
		return;
	}

	Void NSEC3_RR::ToCanonical()
	{
		owner->MakeCanonical();
		UpdateRdata();	
		return;
	}

	array<Byte>^ NSEC3_RR::GetBitmap()
	{
		array<Byte>^ bm;
		if (!Bitmap)
			return nullptr;

		bm = gcnew array<Byte>(Bitmap->Length);
		Bitmap->CopyTo(bm,0);

		return bm;
	}


	Void NSEC3_RR::SetBitmap(List<RR_TYPE>^ typelist)
	{
		size_t i = 0;
		unsigned short int bm_len = 0;
		unsigned short int i_type;
		unsigned short int max_windows = 0;
		Byte window_num = 0;
		Byte winlen = 0;
		int datapos = 0;
		array<Byte>^ data = nullptr;
		array<Byte>^ cur_data = gcnew array<Byte>(32);
		Byte cur_window = 0;
		Byte cur_window_max = 0;
		unsigned short int cur_data_size = 0;
		
		if (!typelist)
			return;

		i_type = 0;

		for (i = 0; i < typelist->Count; i++) {
			if ( i_type < (unsigned short int) typelist[i])
				i_type = (unsigned short int) typelist[i];
		}

		if (i_type < (unsigned short int) RR_TYPE::NSEC) {
			i_type = (unsigned short int) RR_TYPE::NSEC;
		}

		bm_len = i_type / 8 + 2;
		if (!data)
			data = gcnew array<Byte>(bm_len);
		else
			data->Resize(data,bm_len);

		data->Clear(Bitmap,0,data->Length);

		for (i = 0; i < typelist->Count; i++) {
			SetBit(data,(unsigned short int) typelist[i],true);
		}
		
		Bitmap = gcnew array<Byte>(1);
		datapos = 0;

		i = 0;
		window_num = 0;

		/* Fold it into windows */
		while (i < data->Length)
		{
			winlen = 32;
			if (i + winlen >= data->Length)
			{
				winlen = data->Length - i;
				break;
			}
			//Are there any set bits in the window?
			winlen = find_last_non_zero_byte(Bitmap,i,winlen) + 1; //+1 since this is a length, not a zero based position
			if (winlen > 0)
			{
				Bitmap->Resize(Bitmap,Bitmap->Length + winlen + 2); // +2 for windows number and windows length bytes
				Bitmap[datapos++] = window_num;
				Bitmap[datapos++] = winlen;
				data->Copy(data,i,Bitmap,datapos,winlen);
			}
			window_num++;
			i += 32;
		}

		UpdateRdata();
	}
	
	List<RR_TYPE>^ NSEC3_RR::GetTypesFromBitmap()
	{
		List<RR_TYPE>^ output = gcnew List<RR_TYPE>(0);
		int pos = 0;
		Byte window_num = 0;
		Byte window_len = 0;
		int i = 0;
		int j = 0;

		while (pos < Bitmap->Length)
		{
			window_num = Bitmap[pos++];
			window_len = Bitmap[pos++];

			for (j = 0; j < window_len; j++)
			{
				if (Bitmap[pos + j] != 0)
				{
					//Need to find the set bits, and reconstruct them into values
					for (i = 0; i < 8; ++i)
					{
						if (Bitmap[pos + j] & (0x1 <<(7 - i)))
						{
							output->Add((RR_TYPE) ((window_num * 256) * (j * 8) + i));
						}
					}
				}
			}

			pos += window_len;
		}

		return output;
	}

	
	Void NSEC3_RR::SetBitmap(array<Byte>^ bm)
	{
		if (!Bitmap)
			Bitmap = gcnew array<Byte>(bm->Length);
		else
			Bitmap->Resize(Bitmap,bm->Length);

		bm->CopyTo(Bitmap,0);
		UpdateRdata();
	}

	String^ NSEC3_RR::Print()
	{
		String^ output;
		List<RR_TYPE>^ l;
		int i;

		output = PrintHeader();
		output += " ";
		output += Convert::ToString((Byte) algo);
		output += " ";
		output += Convert::ToString(Flags);
		output += " ";
		output += Convert::ToString(Iterations);
		output += " ";
		output += ArrayToHexString(Salt);
		output += " ";
		output += Base32Encode(NextHashedOwner);
		l = GetTypesFromBitmap();
		for (i = 0; i < l->Count; ++i)
		{
			output += Enum::GetName(RR_TYPE::typeid,l[i]);
			output += " ";
		}
		
		return output;
	}

	bool NSEC3_RR::BitmapCoversType(RR_TYPE t)
	{
		List<RR_TYPE>^ l;

		l = GetTypesFromBitmap();
		if (l->Contains(t))
			return true;

		return false;
	}

	Void NSEC3_RR::UpdateRdata()
	{
		int len = 6;
		int pos = 0;

		if (Bitmap)
			len += Bitmap->Length;
		if (Salt)
			len += Salt->Length;
		if (NextHashedOwner)
			len += NextHashedOwner->Length;

		if (rdata)
			rdata->Resize(rdata,len);
		else
			rdata = gcnew array<Byte>(len);

		rdata[pos++] = (Byte)algo;
		rdata[pos++] = Flags;
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder(Iterations))->CopyTo(rdata,pos);
		pos +=2;
		if (Salt)
		{
			rdata[pos] = Salt->Length;
			Salt->CopyTo(rdata,pos);
			pos += Salt->Length;
		}
		else
		{
			rdata[pos++] = 0;
		}
		if (NextHashedOwner)
		{
			rdata[pos++] = NextHashedOwner->Length;
			NextHashedOwner->CopyTo(rdata,pos);
			pos += NextHashedOwner->Length;
		}
		else
		{
			rdata[pos++] = 0;
		}
		if (Bitmap)
		{
			Bitmap->CopyTo(rdata,pos);
		}

		return;
	}

	NSEC3_RR^ NSEC3_RR::Clone()
	{
		NSEC3_RR^ newrr = gcnew NSEC3_RR();
		newrr->rr_type = rr_type;
		newrr->owner = owner->Clone();
		newrr->ttl = ttl;
		newrr->rr_class = rr_class;
		newrr->algo = algo;
		newrr->Flags = Flags;
		newrr->Iterations = Iterations;
		newrr->Salt = gcnew array<Byte>(Salt->Length);
		Salt->CopyTo(newrr->Salt,0);
		newrr->NextHashedOwner = gcnew array<Byte>(NextHashedOwner->Length);
		NextHashedOwner->CopyTo(newrr->NextHashedOwner,0);
		newrr->Bitmap = gcnew array<Byte>(Bitmap->Length);
		Bitmap->CopyTo(newrr->Bitmap,0);
		newrr->UpdateRdata();
		return newrr;
	}

	ResourceRecord^ NSEC3_RR::ParseResourceRecord(array<Byte>^ domainname, UInt16 rr_type, UInt16 rr_class, UInt32 ttl, UInt16 rdata_len, array<Byte>^ packet, int rdata_start)
	{
		NSEC3_RR^ nsec3out;
		array<Byte>^ tmparray;
		int len;
		int pos;

		nsec3out = gcnew NSEC3_RR();
		nsec3out->owner = gcnew DOMAIN_NAME(domainname);
		nsec3out->rr_class = (RR_CLASS) rr_class;
		nsec3out->ttl = ttl;
		pos = rdata_start;
		nsec3out->SetHashAlgorithm((HASH_ALGORITHM) packet[pos++]);
		if (packet[pos++] & 0x01)
			nsec3out->SetOptOutFlag(true);
		nsec3out->SetIterations(IPAddress::NetworkToHostOrder((short int)BitConverter::ToUInt16(packet,pos)));
		pos += 2;
		if (packet[pos] > 0)
		{
			tmparray = gcnew array<Byte>(packet[pos]);
			packet->Copy(packet,pos+1,tmparray,0,packet[pos]);
			nsec3out->SetSalt(tmparray);
		}
		pos += packet[pos] + 1;
		if (packet[pos] > 0)
		{
			tmparray = gcnew array<Byte>(packet[pos]);
			packet->Copy(packet,pos+1,tmparray,0,packet[pos]);
			nsec3out->SetNextHashedOwner(tmparray);
		}
		pos += packet[pos] + 1;
		len = rdata_start + rdata_len - pos;
		tmparray->Resize(tmparray,len);
		packet->Copy(packet,pos,tmparray,0,len);
		nsec3out->SetBitmap(tmparray);
		return nsec3out;
	}

	String^ NSEC3_RR::PrintRR(ResourceRecord^ rec)
	{
		return safe_cast<NSEC3_RR^>(rec)->Print();
	}

	ResourceRecord^ NSEC3_RR::CloneRR(ResourceRecord^ rec)
	{
		return safe_cast<NSEC3_RR^>(rec)->Clone();
	}
}