/*
  Copyright © 2010 Harald Sitter <apachelogger@ubuntu.com>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 2 of
  the License or (at your option) version 3 or any later version
  accepted by the membership of KDE e.V. (or its successor approved
  by the membership of KDE e.V.), which shall act as a proxy
  defined in Section 14 of version 3 of the license.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "Module.h"
#include "ui_Module.h"

#include <QtCore/QFile>
#include <QtGui/QPaintEngine>

#include <KAboutData>
#include <KPluginFactory>
#include <KStandardDirs>

#include "EngineProbe.h"
#include "Version.h"

K_PLUGIN_FACTORY_DECLARATION(KcmQtGraphicsSystemFactory);

Module::Module(QWidget *parent, const QVariantList &args) :
    KCModule(KcmQtGraphicsSystemFactory::componentData(), parent, args),
    ui(new Ui::Module),
    probedButton(0)
{
    KAboutData *about = new KAboutData("qt-graphicssystem", 0,
                                       ki18n("Qt Graphics System"),
                                       global_s_versionStringFull,
                                       ki18n("Configure the Qt Graphics System"),
                                       KAboutData::License_GPL_V3,
                                       ki18n("Copyright 2010 Harald Sitter"),
                                       KLocalizedString(), QByteArray(),
                                       "apachelogger@ubuntu.com");

    about->addAuthor(ki18n("Harald Sitter"), ki18n("Mad Man"), "apachelogger@ubuntu.com");
    setAboutData(about);

    ui->setupUi(this);
    scriptFile = new QFile(componentData().dirs()->localkdedir() + "env/" + "qt-graphicssystem.sh");

    connect(ui->nativeButton, SIGNAL(clicked()), SLOT(changed()));
    connect(ui->rasterButton, SIGNAL(clicked()), SLOT(changed()));
    connect(ui->openglButton, SIGNAL(clicked()), SLOT(changed()));

    ui->infoIconLabel->hide();
    ui->infoLabel->hide();

    // We have no help so remove the button from the buttons.
    setButtons(buttons() ^ KCModule::Help);
}

Module::~Module()
{
    delete scriptFile;
    delete ui;
}

void Module::load()
{
    if (!scriptFile->exists()) {
        // Trying to probe engine in use.
        EngineProbe *p = new EngineProbe(this);
        connect(p, SIGNAL(engineType(int)), SLOT(setType(int)));
        p->show();
        return;
    }

    if (!scriptFile->open(QIODevice::ReadOnly | QIODevice::Text)) {
        // Could not open file for whatever reason.
        return;
    }
    if (scriptFile->atEnd()) {
        // File is empty.
        return;
    }

    QString value(scriptFile->readLine().split('=').value(1));
    if (value == QLatin1String("raster")) {
        ui->rasterButton->setChecked(true);
    } else if (value == QLatin1String("opengl")) {
        ui->openglButton->setChecked(true);
    }
    scriptFile->close();
}

void Module::save()
{
    scriptFile->flush();
    scriptFile->open(QIODevice::WriteOnly | QIODevice::Text);
    if (ui->nativeButton->isChecked() && ui->nativeButton != probedButton) {
        scriptFile->write("export QT_GRAPHICSSYSTEM=native");
    } else if (ui->rasterButton->isChecked() && ui->rasterButton != probedButton) {
        scriptFile->write("export QT_GRAPHICSSYSTEM=raster");
    } else if (ui->openglButton->isChecked() && ui->openglButton != probedButton) {
        scriptFile->write("export QT_GRAPHICSSYSTEM=opengl");
    } else if (!ui->nativeButton->isChecked() &&
               !ui->rasterButton->isChecked() &&
               !ui->openglButton->isChecked()) {
        // User switched to defaults
        scriptFile->remove();
        return;
    }
    scriptFile->close();
}

void Module::defaults()
{
    ui->nativeButton->setChecked(false);
    ui->rasterButton->setChecked(false);
    ui->openglButton->setChecked(false);
}

void Module::changed()
{
    ui->infoLabel->show();
    ui->infoIconLabel->setPixmap(KIcon("dialog-information").pixmap(ui->infoLabel->height()));
    ui->infoIconLabel->show();
    KCModule::changed();
}

void Module::setType(int type)
{
    QPaintEngine::Type type_ = static_cast<QPaintEngine::Type>(type);
    switch (type_) {
    case QPaintEngine::X11 :
        ui->nativeButton->setChecked(true);
        probedButton = ui->nativeButton;
        break;
    case QPaintEngine::Raster :
        ui->rasterButton->setChecked(true);
        probedButton = ui->rasterButton;
        break;
    case QPaintEngine::OpenGL2 :
        ui->openglButton->setChecked(true);
        probedButton = ui->openglButton;
        break;
    default:
        // Unhandled type.
        // TODO: throw warning?
        break;
    }
}
