# bzr-dbus: dbus support for bzr/bzrlib.
# Copyright (C) 2007 Canonical Limited.
#   Author: Robert Collins.
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 

"""Tests for the url mapping logic.."""

from bzrlib.tests import TestCase

from bzrlib.plugins.dbus import mapper


class TestMapper(TestCase):

    def test_init(self):
        a_mapper = mapper.URLMapper()
        # an empty map should have no maps.
        self.assertEqual({}, a_mapper.maps)

    def test_single_map(self):
        a_mapper = mapper.URLMapper()
        a_mapper.maps['foo/'] = ['bar/']
        self.assertEqual([], a_mapper.map('foo'))
        self.assertEqual(['bar/'], a_mapper.map('foo/'))
        self.assertEqual(['bar/thomolew'], a_mapper.map('foo/thomolew'))

    def test_multi_target_map(self):
        a_mapper = mapper.URLMapper()
        a_mapper.maps['foo/'] = ['bar/', 'gam/']
        self.assertEqual([], a_mapper.map('foo'))
        self.assertEqual(['bar/', 'gam/'], a_mapper.map('foo/'))
        self.assertEqual(['bar/thomolew', 'gam/thomolew'],
            a_mapper.map('foo/thomolew'))

    def test_multi_source_matches(self):
        a_mapper = mapper.URLMapper()
        a_mapper.maps['file:///'] = ['http://host/']
        # longer == more specific, give it both shorter and longer outputs.
        a_mapper.maps['file:///tmp/'] = ['bzr://host/', 'bzr+ssh://host/tmp/']
        self.assertEqual([], a_mapper.map('memory:///'))
        self.assertEqual(
            ['bzr://host/',
             'bzr+ssh://host/tmp/',
             'http://host/tmp/'
            ],
            a_mapper.map('file:///tmp/'))
        self.assertEqual(
            ['bzr://host/suffix',
             'bzr+ssh://host/tmp/suffix',
             'http://host/tmp/suffix'
            ],
            a_mapper.map('file:///tmp/suffix'))

    def test_irrelevant_maps_ignored(self):
        a_mapper = mapper.URLMapper()
        a_mapper.maps['a/'] = ['b/']
        a_mapper.maps['b/'] = ['c/']
        self.assertEqual([], a_mapper.map(''))
        self.assertEqual(['b/'], a_mapper.map('a/'))
        self.assertEqual(['c/'], a_mapper.map('b/'))

    def test_add_map_strips_readonly_prefix(self):
        """add_map should strip the readonly+URL to work with bzr serve."""
        a_mapper = mapper.URLMapper()
        a_mapper.add_map('readonly+foo/', 'bar/')
        self.assertEqual({'foo/':['bar/']}, a_mapper.maps)

    def test_remove_map_strips_readonly_prefix(self):
        """remove_map should strip the readonly+URL to work with bzr serve."""
        a_mapper = mapper.URLMapper()
        a_mapper.add_map('readonly+foo/', 'bar/')
        a_mapper.remove_map('readonly+foo/', 'bar/')
        self.assertEqual({}, a_mapper.maps)
