//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleDesigner/SampleEditorController.h
//! @brief     Defines class SampleEditorController
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_SAMPLEDESIGNER_SAMPLEEDITORCONTROLLER_H
#define BORNAGAIN_GUI_VIEW_SAMPLEDESIGNER_SAMPLEEDITORCONTROLLER_H

#include "GUI/Model/Sample/FormFactorItemCatalog.h"
#include "GUI/Model/Sample/ItemWithParticlesCatalog.h"
#include <QUndoStack>

class ISelectionContainerForm;
class ISelectionProperty;
class CompoundItem;
class CoreAndShellForm;
class DoubleProperty;
class InterferenceForm;
class InterferenceItem;
class Item3D;
class ItemWithMaterial;
class ItemWithParticles;
class LatticeTypeSelectionForm;
class LayerForm;
class LayerItem;
class MaterialModel;
class MesocrystalForm;
class SampleForm;
class SampleItem;
class ParticleLayoutItem;
class ProjectDocument;
class SelectionContainerForm;

//! Class to modify a sample from the layer oriented sample editor.
//!
//! Use this class to modify the sample model. It takes care of notifications and creating undo/redo
//! commands.
//! It operates on one SampleItem.
class SampleEditorController : public QObject {
    Q_OBJECT
public:
    SampleEditorController(ProjectDocument* document, SampleItem* multi);

    //! Set the current form.
    //!
    //! The form can change e.g. when a different sample gets the current one
    //! in the layer editor. Also nullptr is allowed.
    void setSampleForm(SampleForm* view);

    //! The current form.
    SampleForm* sampleForm() const;

    //! The item on which this controller operates.
    SampleItem* sampleItem() const;

    //! The contained undo stack.
    QUndoStack* undoStack();

    //! The materials of the current document
    MaterialModel* materialModel() const;

    //! The current document
    ProjectDocument* projectDocument() const;

    void addLayerItem(LayerItem* before);
    QColor findColor(int atIndex);
    void onLayerAdded(LayerItem* layer);
    void addLayerItemFromUndo(int atIndex);
    void duplicateLayerItem(const LayerItem* layer);
    void removeLayerItem(LayerItem* layer);
    void removeLayerItemFromUndo(int atIndex);

    void onLayoutAdded(LayerForm* layerForm, ParticleLayoutItem* layout);
    void addLayoutItem(LayerForm* layerForm);
    void duplicateLayoutItem(LayerForm* layerForm, ParticleLayoutItem* layout);
    void removeLayoutItem(LayerForm* layerForm, ParticleLayoutItem* layout);

    void onParticleLayoutAdded(ParticleLayoutItem* layout, ItemWithParticles* newItem);
    void addParticleLayoutItem(ParticleLayoutItem* layout, ItemWithParticlesCatalog::Type type);
    void addParticleLayoutItem(ParticleLayoutItem* layout, FormFactorItemCatalog::Type t);
    ParticleLayoutItem* parentLayoutItem(ItemWithParticles* item);

    void onParticleCompoundAdded(CompoundItem* composition, ItemWithParticles* newItem);
    void addCompoundItem(CompoundItem* composition, ItemWithParticlesCatalog::Type type);
    void addCompoundItem(CompoundItem* composition, FormFactorItemCatalog::Type type);
    CompoundItem* parentCompoundItem(ItemWithParticles* item);

    void duplicateItemWithParticles(ItemWithParticles* item);
    void removeParticle(ItemWithParticles* item);

    void setCoreFormFactor(CoreAndShellForm* widget, FormFactorItemCatalog::Type type);
    void setShellFormFactor(CoreAndShellForm* widget, FormFactorItemCatalog::Type type);

    void setMesocrystalBasis(MesocrystalForm* widget, ItemWithParticlesCatalog::Type type);
    void setMesocrystalBasis(MesocrystalForm* widget, FormFactorItemCatalog::Type type);
    void selectInterference(InterferenceForm* widget, int newIndex);
    void setIntegrateOverXi(LatticeTypeSelectionForm* widget, bool newValue);

    void setDouble(double newValue, DoubleProperty& d);
    void setDoubleFromUndo(double newValue, const QString& path);

    void setCurrentIndex(ISelectionContainerForm* widget, int index, ISelectionProperty& d);

    void selectMaterial(ItemWithMaterial* item, const QString& newMaterialIdentifier);
    void setMaterialValue(ItemWithMaterial* item, double newValue, DoubleProperty& d);

    //! Set an interference function's value which affects the total particle density of the
    //! containing particle layout.
    //!
    //! Some values in interference settings affect the total density in the particle layout which
    //! contains this interference function. Call this method to provide all the related updating
    //! (data & UI).
    void setDensityRelatedValue(InterferenceItem* interferenceItem, double newValue,
                                DoubleProperty& d);

    void onStartingToMoveLayer();
    void onStoppedToMoveLayer(QWidget* widgetToMove, QWidget* moveAboveThisWidget);

    void setSampleName(const QString& name);
    void setSampleDescription(const QString& description);

signals:
    void requestViewInRealspace(Item3D* item);
    void aboutToRemoveItem(Item3D* item);
    void modified();

private:
    ItemWithParticles* createAndInitItem(FormFactorItemCatalog::Type formFactorType) const;
    ItemWithParticles* createAndInitItem(ItemWithParticlesCatalog::Type itemType) const;

private:
    QUndoStack m_undoStack;
    SampleItem* m_sampleItem;
    SampleForm* m_sampleForm;
    ProjectDocument* m_document;
};

#endif // BORNAGAIN_GUI_VIEW_SAMPLEDESIGNER_SAMPLEEDITORCONTROLLER_H
