//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Support/Util/WidgetMoverButton.h
//! @brief     Defines class WidgetMoverButton
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_SUPPORT_UTIL_WIDGETMOVERBUTTON_H
#define BORNAGAIN_GUI_SUPPORT_UTIL_WIDGETMOVERBUTTON_H

#include <QMap>
#include <QTimer>
#include <QToolButton>

class QScrollArea;
class QPropertyAnimation;

//! Button to move a widget vertically in a layout.
//!
//! With this button, the "move layer" feature in the layer oriented sample editor is realized
//! (it is the button on each LayerForm).
//!
//! When pressing the button, the parent layout of the widget is deactivated, and the widget is
//! moved to the position where the pressed mouse cursor is moved.
//! When the mouse is released, the layout is again activated and a signal finishedMoving() is
//! emitted. Any reordering of the widgets in the layout then has to be handled outside - no
//! reordering is done in here.
class WidgetMoverButton : public QToolButton {
    Q_OBJECT
public:
    //! Create a widget mover button.
    //!
    //! If the widget shall not be able to be dragged on the top position, this can be defined by
    //! ignoreOnTop (the number of widgets at the top of the layout which shall not be affected by
    //! the reordering). In the case of the layer moving, the topmost Form (the sample
    //! properties) shall not be part of reordering.
    WidgetMoverButton(QWidget* parent, QWidget* widgetToMove, int ignoreOnTop = 0);

protected:
    void mousePressEvent(QMouseEvent* event) override;
    void mouseReleaseEvent(QMouseEvent* event) override;
    void mouseMoveEvent(QMouseEvent* event) override;

signals:
    void startingToMove();
    void finishedMoving(QWidget* widgetToMove, QWidget* moveAboveThisWidget);

private:
    void scrollParent(bool up);

private:
    bool m_pressed = false;
    bool m_started = false;
    int m_globalMouseDownY;
    QLayout* m_layoutToDeactivate = nullptr;
    QWidget* m_widgetToMove;
    QWidget* m_dropAboveThisWidget;
    int m_originalWidgetY;
    int m_yOfFirstRelevantWidget;
    int m_ignoreOnTop;
    QTimer m_dragScrollTimer;
    QScrollArea* m_scrollArea;
    QPoint m_hotSpot; //!< The mouse-down coordinates in the widget to move
    QMap<QWidget*, QPropertyAnimation*> m_animations;
};


#endif // BORNAGAIN_GUI_SUPPORT_UTIL_WIDGETMOVERBUTTON_H
