if (typeof Test === 'undefined') require('../tester');

Test('decimalPlaces', function () {
    var MAX = 1e9;

    Test.areEqual(BigNumber.prototype.decimalPlaces, BigNumber.prototype.dp);

    BigNumber.config({
        DECIMAL_PLACES: 20,
        ROUNDING_MODE: 7,
        RANGE: 1E9,
        EXPONENTIAL_AT: 1E9
    });

    // Return decimal places count

    t = function (value, count){
        Test.areEqual(new BigNumber(value).decimalPlaces(), count);
    }

    t(0, 0);
    t(1, 0);
    t(1.2, 1);
    t(0.1, 1);
    t(0.25, 2);
    t(0.0625, 4);
    t(99, 0);
    t(9900, 0);
    t(1000.001, 3);
    t('1000.001', 3);
    t(-0, 0);
    t(-1, 0);
    t(-1.2, 1);
    t(-0.1, 1);
    t(-0.25, 2);
    t(-0.0625, 4);
    t(-99, 0);
    t(-9900, 0);
    t(-1000.001, 3);
    t('-1000.001', 3);

    t(NaN, null);
    t('NaN', null);
    t(Infinity, null);
    t(-Infinity, null);
    t('Infinity', null);
    t('-Infinity', null);

    t('0.00000000000000000002', 20);
    t('100.0000000000000000000000000000000000032948', 40);
    t('1.3e-11', 12);
    t('-3.52e-7', 9);
    t('6.9525e-12', 16);
    t('2.1e-8', 9);
    t('3.7015e-7', 11);
    t('-50.1839', 4);
    t('0.014836', 6);
    t('-16688', 0);
    t('-506006218906684823229.56892808849303', 14);
    t('10000000000000.00000000000000000000000000000000000001000000000000000000000000000000000000000001', 80);
    t('057643758687043658032465987410000000000000.0000000000000', 0);
    t('999999999999999999999999999999999999999999999999999999999999999999999999999.99999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999999', 113);

    // Return a new BigNumber with a maximum number of dp decimal places

    t = function (expected, value, dp, rm) {
        Test.areEqual(String(expected), new BigNumber(value).decimalPlaces(dp, rm).valueOf());
    }

    t('0', '0.000084888060736883027314038572334303632', 0);

    t('-79998', '-79998', 0);
    t('30845717889906383053', '30845717889906383052.56472015469740823', 0);
    t('-446807', '-446806.9227761147047517', 0);
    t('76', '76.0719', 0);
    t('-0', '-0.00686876124450101884528', 0);
    t('71710176', '71710176', 0);
    t('79', '79', 0);
    t('-20187079148819697784', '-20187079148819697784', 0);
    t('6309793381', '6309793381', 0);
    t('76162410487880977', '76162410487880976.81307140573688335764284', 0);
    t('7491318926', '7491318926.312122759153805942088431', 0);
    t('74909422733607112719', '74909422733607112719.13179009875', 0);
    t('559', '559.2110851431205', 0);
    t('-20306363016185', '-20306363016184.760368254019194925', 0);
    t('-6717914940586242526', '-6717914940586242525.574537480672174624149', 0);
    t('-5', '-5', 0);
    t('8', '8.0606265529', 0);
    t('24701408591129838', '24701408591129838.22163', 0);
    t('3806471925924246437202', '3806471925924246437201.71339576511057957', 0);
    t('-85883753924431578007564277', '-85883753924431578007564277.399776', 0);
    t('826068555927524695', '826068555927524695', 0);
    t('0', '0.000058441452091833136219167587256', 0);
    t('47353089', '47353089.2019161006899898', 0);
    t('-372', '-372.03039745', 0);
    t('3103', '3103', 0);
    t('5620', '5620.48101861977', 0);
    t('9105547112', '9105547112.1319443692', 0);
    t('7472595749413646784408602', '7472595749413646784408602.1095168949', 0);
    t('335816732795', '335816732794.51601276961965281205689254', 0);
    t('-7911332887831436027', '-7911332887831436027.40569259650318', 0);
    t('-52182567851715', '-52182567851715', 0);
    t('439059239410', '439059239409.55703238112278', 0);
    t('43465109466157505', '43465109466157505', 0);
    t('5390894521', '5390894520.8738091063016', 0);
    t('-583532700278695860335147', '-583532700278695860335146.6674153779', 0);
    t('485669', '485669', 0);
    t('3', '3.284095344472285718254591781467536534546', 0);
    t('-0', '-0.4210910538061556801483189', 0);
    t('3134', '3133.654494231705614', 0);
    t('8136642604255255554627376', '8136642604255255554627375.87998', 0);
    t('-54301', '-54301.2807534165371279131810401278', 0);
    t('27796900', '27796900.0685101', 0);
    t('-36', '-36', 0);
    t('-7930085746684', '-7930085746684.2599205', 0);
    t('736988809325', '736988809325', 0);
    t('-181606874378737533', '-181606874378737532.738571718716743', 0);
    t('577064478018279', '577064478018279', 0);
    t('-862419646909', '-862419646909', 0);
    t('3269018763090', '3269018763089.5711045989917554', 0);

    t('Infinity', Infinity, 0);
    t('-Infinity', -Infinity, 0);
    t('NaN', NaN, 0);

    t('1', 0.5, 0);
    t('1', 0.7, 0);
    t('1', 1, 0);
    t('1', 1.1, 0);
    t('1', 1.49999, 0);
    t('-0', -0.5, 0);
    t('-1', '-0.5000000000000001', 0);
    t('-1', -0.7, 0);
    t('-1', -1, 0);
    t('-1', -1.1, 0);
    t('-1', -1.49999, 0);
    t('-1', -1.5, 0);

    t('9007199254740990', '9007199254740990', 0);
    t('9007199254740991', '9007199254740991', 0);
    t('-9007199254740990', '-9007199254740990', 0);
    t('-9007199254740991', '-9007199254740991', 0);

    BigNumber.config({EXPONENTIAL_AT: 100});

    t('536870911', 536870910.5, 0);
    t('536870911', 536870911, 0);
    t('536870911', 536870911.4, 0);
    t('536870912', 536870911.5, 0);
    t('536870912', 536870912, 0);
    t('536870912', 536870912.4, 0);
    t('536870913', 536870912.5, 0);
    t('536870913', 536870913, 0);
    t('536870913', 536870913.4, 0);
    t('1073741823', 1073741822.5, 0);
    t('1073741823', 1073741823, 0);
    t('1073741823', 1073741823.4, 0);
    t('1073741824', 1073741823.5, 0);
    t('1073741824', 1073741824, 0);
    t('1073741824', 1073741824.4, 0);
    t('1073741825', 1073741824.5, 0);
    t('2147483647', 2147483646.5, 0);
    t('2147483647', 2147483647, 0);
    t('2147483647', 2147483647.4, 0);
    t('2147483648', 2147483647.5, 0);
    t('2147483648', 2147483648, 0);
    t('2147483648', 2147483648.4, 0);
    t('2147483649', 2147483648.5, 0);

    t('0', 0.4, 0);
    t('-0', -0.4, 0);
    t('-0', -0.5, 0);
    t('1', 0.6, 0);
    t('-1', -0.6, 0);
    t('2', 1.5, 0);
    t('2', 1.6, 0);
    t('-2', -1.6, 0);

    t('8640000000000000', '8640000000000000', 0);
    t('8640000000000001', '8640000000000001', 0);
    t('8640000000000002', '8640000000000002', 0);
    t('9007199254740990', '9007199254740990', 0);
    t('9007199254740991', '9007199254740991', 0);
    t('1.7976931348623157e+308', '1.7976931348623157e+308', 0);
    t('-8640000000000000', '-8640000000000000', 0);
    t('-8640000000000001', '-8640000000000001', 0);
    t('-8640000000000002', '-8640000000000002', 0);
    t('-9007199254740990', '-9007199254740990', 0);
    t('-9007199254740991', '-9007199254740991', 0);
    t('-1.7976931348623157e+308', '-1.7976931348623157e+308', 0);

    BigNumber.config({EXPONENTIAL_AT: 1E9});

    t('0', 0, 0);
    t('-0', -0, 0);
    t('0', '0', 0);
    t('-0', '-0', 1);
    t('-0', '-0', 1, 0);
    t('-0', '-0', 1, 3);
    t('-0', '-0', 1, 6);
    t('0', '0', 10);
    t('-0', '-0', 20);

    // Tests generated using java BigDecimal and random values

    t('0.0039', '0.0039', 4, 3);
    t('-41620.65299', '-41620.65299', 6, 5);
    t('0.0011117', '0.0011117', 7, 0);
    t('938116', '938116', 6, 5);
    t('-7187', '-7187', 4, 4);
    t('-2.4263', '-2.426346', 4, 6);
    t('-0.537', '-0.5374743', 3, 5);
    t('652506', '652506', 3, 6);
    t('-178587', '-178587', 3, 2);
    t('4093', '4093', 3, 0);
    t('44', '44', 1, 2);
    t('-0.0496', '-0.049635', 4, 4);
    t('-1', '-0.05', 0, 0);
    t('520216039', '520216039', 7, 1);
    t('-2932289', '-2932289', 8, 3);
    t('53937.3', '53937.399', 1, 1);
    t('63101619', '63101619', 5, 0);
    t('-0.00207', '-0.00207', 6, 0);
    t('-0.032', '-0.03169086', 3, 0);
    t('-755', '-755', 5, 3);
    t('-3583', '-3583', 5, 1);
    t('-6.616', '-6.615375', 3, 0);
    t('7528739', '7528739', 8, 0);
    t('-0.9028', '-0.9028782', 4, 2);
    t('-8.78762908', '-8.78762908', 8, 2);
    t('-0.000262', '-0.0002613', 6, 0);
    t('-888980.3', '-888980.3', 1, 6);
    t('0.00009075', '0.00009075', 8, 0);
    t('-4.42', '-4.4195', 2, 0);
    t('-0.336889', '-0.336888682', 6, 6);
    t('43759', '43759', 6, 4);
    t('0.0000804', '0.0000804', 8, 6);
    t('-4', '-4', 0, 2);
    t('-3', '-3.614204', 0, 2);
    t('336379823', '336379823', 4, 1);
    t('310614', '310614', 7, 0);
    t('-375811635', '-375811635', 8, 3);
    t('-5446775', '-5446775', 7, 4);
    t('59.7954405', '59.7954405', 7, 0);
    t('47086', '47085.84', 0, 4);
    t('-2564', '-2564', 3, 4);
    t('1.4069997', '1.40699967', 7, 2);
    t('-0.4', '-0.42528874', 1, 6);
    t('-0.00005573', '-0.000055732', 8, 1);
    t('-0.00003014', '-0.000030137', 8, 6);
    t('57', '57', 3, 6);
    t('365474.9164', '365474.9164', 4, 5);
    t('-7', '-7', 0, 4);
    t('0.843206', '0.84320562', 6, 5);
    t('-62734', '-62734', 5, 3);
    t('-51246', '-51246', 7, 2);
    t('0.0003', '0.000332', 4, 5);
    t('-609', '-609', 2, 0);
    t('-0.00004595', '-0.00004595', 8, 1);
    t('-22243', '-22243', 0, 1);
    t('54693', '54693', 5, 0);
    t('5', '5', 1, 6);
    t('2', '2', 2, 6);
    t('3808', '3808', 0, 4);
    t('2616', '2616', 0, 5);
    t('-2477', '-2477', 3, 2);
    t('0.00009', '0.00008892', 5, 4);
    t('8.8953', '8.8953', 5, 5);
    t('0.07', '0.06922', 2, 4);
    t('0.08', '0.08', 3, 2);
    t('-773.0965536', '-773.0965536', 8, 2);
    t('-326', '-326', 2, 1);
    t('-0.356', '-0.3564473', 3, 6);
    t('-312283993', '-312283992.7', 0, 3);
    t('-2.3995861', '-2.39958606', 7, 3);
    t('-0.0449', '-0.0449', 4, 3);
    t('-0', '-0.003', 0, 4);
    t('85942', '85942', 2, 5);
    t('6738', '6738.1977', 0, 3);
    t('-0.003', '-0.002995', 5, 3);
    t('-572936', '-572936', 3, 6);
    t('684.2426407', '684.2426407', 8, 6);
    t('-0', '-0.0007851', 1, 6);
    t('-0', '-0.002', 0, 5);
    t('85.05', '85.05', 3, 3);
    t('1.66', '1.66', 3, 3);
    t('-17.14', '-17.14', 2, 0);
    t('80182', '80182', 6, 5);
    t('-0.00064', '-0.00063953', 6, 6);
    t('-3.76703298', '-3.767032983', 8, 4);
    t('-0.000378', '-0.00037877', 6, 2);
    t('675', '675.76294', 0, 3);
    t('0.0003', '0.00023235', 4, 0);
    t('-0.0155175', '-0.0155175', 7, 0);
    t('-645406477.5', '-645406477.5', 2, 1);
    t('-0.89603', '-0.8960278', 5, 6);
    t('866034', '866034', 7, 3);
    t('484.002', '484.002', 4, 6);
    t('420', '419.83', 0, 6);
    t('4611867124', '4611867124', 4, 4);
    t('0.0080252', '0.0080252', 8, 0);
    t('-0', '-0.02829833', 1, 4);
    t('-8', '-8', 0, 4);
    t('-0.0073554', '-0.00735541', 7, 4);
    t('28', '28', 3, 5);
    t('-3903', '-3903', 6, 0);
    t('228', '228', 3, 4);
    t('0.88', '0.883', 2, 5);
    t('-0.01', '-0.009', 2, 3);
    t('7796.6', '7796.6', 2, 2);
    t('0', '0.005', 0, 6);
    t('805.25', '805.2467', 2, 0);
    t('-85023', '-85023', 4, 3);
    t('-0', '-0.008097491', 0, 2);
    t('-0.026', '-0.0259', 3, 4);
    t('-0.00801', '-0.008007359', 5, 4);
    t('30745', '30745', 6, 5);
    t('-81', '-81', 4, 6);
    t('-386', '-386', 4, 2);
    t('-2876', '-2876', 5, 6);
    t('0', '0.00002329', 4, 5);
    t('-62453828', '-62453828', 7, 6);
    t('0.317', '0.317', 3, 3);
    t('-706', '-706', 0, 1);
    t('36.911', '36.9109527', 4, 4);
    t('5040908', '5040908', 1, 0);
    t('-0.08', '-0.08', 2, 4);
    t('-0.352847', '-0.352847085', 6, 6);
    t('0.001', '0.00006', 3, 2);
    t('-0.4389', '-0.438905', 4, 1);
    t('0.00390964', '0.00390964', 8, 2);
    t('-35105.1', '-35105.1788', 1, 2);
    t('-8858843.829', '-8858843.829', 3, 1);
    t('-5946.90816', '-5946.90816', 5, 3);
    t('608475', '608475', 6, 6);
    t('3.5873016', '3.58730156', 7, 5);
    t('-7988864', '-7988864', 8, 0);
    t('-6056.29', '-6056.29', 3, 0);
    t('-4481788', '-4481788', 3, 5);
    t('-1.6', '-1.5678', 1, 6);
    t('-351561391', '-351561391', 5, 5);
    t('3104642', '3104642', 8, 2);
    t('-7.30071897', '-7.30071897', 8, 2);
    t('-0.0256033', '-0.02560321', 7, 3);
    t('-0.5', '-0.5', 1, 0);
    t('-7.173', '-7.173', 4, 6);
    t('0.1', '0.05', 1, 2);
    t('0.0006736', '0.0006736', 7, 3);
    t('-54.37621076', '-54.37621076', 8, 4);
    t('4823.725785', '4823.725785', 7, 4);
    t('-0', '-0.000084', 1, 4);
    t('-7491', '-7491', 4, 6);
    t('-226', '-226.80088', 0, 2);
    t('-0.8', '-0.8109', 1, 6);
    t('-0.0548897', '-0.054889654', 7, 0);
    t('-91.1579356', '-91.15793558', 7, 6);
    t('47988267', '47988267', 1, 3);
    t('91', '91', 3, 5);
    t('-5.0091', '-5.009088', 4, 0);
    t('-1382.949', '-1382.949', 4, 0);
    t('-697', '-697', 2, 5);
    t('-5376196', '-5376196', 1, 6);
    t('-0.6', '-0.6', 1, 6);
    t('-887591', '-887591', 8, 2);
    t('0.02000666', '0.020006669', 8, 3);
    t('-39983.5195', '-39983.5195', 5, 6);
    t('-0.723', '-0.722771', 3, 4);
    t('-0.0007', '-0.0007', 5, 5);
    t('-8637388', '-8637388', 4, 1);
    t('-775918', '-775918', 8, 0);
    t('3281682.74', '3281682.74', 2, 3);
    t('193', '193', 4, 6);
    t('-0', '-0.369577856', 0, 5);
    t('-0.34504', '-0.34504336', 5, 2);
    t('-22', '-22', 0, 5);
    t('-548324', '-548324', 4, 2);
    t('-28', '-28', 3, 5);
    t('-6.70773', '-6.707727', 5, 0);
    t('11.062', '11.06169236', 3, 5);
    t('0.00052', '0.00051869', 5, 6);
    t('-0.0421539', '-0.0421539', 7, 3);
    t('-0.248552', '-0.248552', 6, 4);
    t('-864296', '-864296', 8, 0);
    t('0.8599583', '0.8599583', 7, 6);
    t('-49755.36744', '-49755.36744', 5, 5);
    t('-605.095664', '-605.095664', 6, 3);
    t('-103637', '-103637', 5, 2);
    t('868619', '868619', 3, 4);
    t('-0.0003926', '-0.0003926', 7, 4);
    t('1', '0.9', 0, 0);
    t('4.3567', '4.35670557', 4, 4);
    t('-238', '-238', 3, 1);
    t('1271.594054', '1271.594054', 7, 4);
    t('-4.1787893', '-4.178789328', 7, 1);
    t('372', '372', 3, 6);
    t('-9.1645', '-9.1645354', 4, 6);
    t('83.241462', '83.241462', 6, 6);
    t('-0.00001262', '-0.00001262', 8, 6);
    t('63.4147', '63.414735', 4, 3);
    t('-2', '-2', 3, 5);
    t('538601216', '538601216', 2, 3);
    t('2138522', '2138522', 6, 1);
    t('301527.309', '301527.309', 3, 5);
    t('-7.568', '-7.5683', 3, 5);
    t('-1969563', '-1969563', 8, 1);

    t('-249815496353142605063083339706423770452529255711064841247978.1649', '-249815496353142605063083339706423770452529255711064841247978.16482575714251625720191747855256', 4, 3);
    t('-8757939868931375459275247990933877309', '-8757939868931375459275247990933877309', 37, 5);
    t('556228688742576287254121716253667920283844645854057755303259369', '556228688742576287254121716253667920283844645854057755303259369', 25, 3);
    t('0', '0.00000000000000000000000000000000000000000000000029171651270270187897342247995644546232788426', 35, 3);
    t('-0.0000058979376152458865088785307705', '-0.000005897937615245886508878530770431196412050562641578155968', 34, 0);
    t('-0.00000000000000847145658', '-0.00000000000000847145657888758222743159237', 23, 3);
    t('-6049825281564367887763596943301191584240212075976455', '-6049825281564367887763596943301191584240212075976455', 53, 0);
    t('-64680661822322715719008107701612.741312367131318202976964422', '-64680661822322715719008107701612.74131236713131820297696442216284615573809', 27, 4);
    t('-0', '-0.0000000000000000000000000000000000048810387700772580768508931440796984745771392', 29, 2);
    t('232730083859746141163.7653402888578932348762190816502', '232730083859746141163.76534028885789323487621908165018267559238917447610653431235463768839108', 31, 5);
    t('-499311204526771278437755030091270601.471802335', '-499311204526771278437755030091270601.47180233568698638607829334564', 9, 2);
    t('-4793558721695298336734196955569628362250.1', '-4793558721695298336734196955569628362250.107513868618556605590137359591047843592593', 1, 6);
    t('0.0000000000000000000000000000000000000622', '0.000000000000000000000000000000000000062176566496', 40, 5);
    t('-393323198834362857317921391989917230.1575124832953752419421908305674188204', '-393323198834362857317921391989917230.15751248329537524194219083056741882044261538788798', 37, 5);
    t('-0', '-0.000000000000000169092170076859363', 3, 5);
    t('0.00085774283834918184933525579624189945402764', '0.0008577428383491818493352557962418994540276420616048890965876345513', 44, 4);
    t('-127706837731025454069338274697755478243.226555768723254468591', '-127706837731025454069338274697755478243.226555768723254468591', 21, 4);
    t('0.0000000000000000000000000000002', '0.00000000000000000000000000000024', 31, 6);
    t('0', '0.0000000000000000006', 0, 4);
    t('0.000000000000000000000007810237984104', '0.00000000000000000000000781023798410401506609033130277607174', 36, 6);
    t('-0.0000000000000000000000000000000000000000000003', '-0.00000000000000000000000000000000000000000000030032464295099044566372323', 46, 1);
    t('-685966223751248103958158215191994961086468451812432', '-685966223751248103958158215191994961086468451812432', 15, 6);
    t('-342139733926317660019.326340161146875487297', '-342139733926317660019.326340161146875487297346019153178', 21, 6);
    t('0', '0.000000000000000000000000000000000000000000901836202', 15, 4);
    t('54703159147681578.1514852895273075959730711237955491690133829927977209580124', '54703159147681578.1514852895273075959730711237955491690133829927977209580124', 59, 0);
    t('-460718430436988680547605508933197195738591668941440213118174196', '-460718430436988680547605508933197195738591668941440213118174196', 16, 6);
    t('70597943458486911871858911963104830835210711517857117431432573585719054629372', '70597943458486911871858911963104830835210711517857117431432573585719054629372', 17, 3);
    t('0', '0.0000000000000000000000005', 17, 1);
    t('612277653819449429818059108521923195679275819340316389136081740197236376194', '612277653819449429818059108521923195679275819340316389136081740197236376194', 24, 6);
    t('0', '0.000004077002116450599035175933786605407147118579202271612670066923117668943811493201471493894', 0, 3);
    t('0', '0.0000000000000000454362126792715044166595091', 4, 5);
    t('-0.00000000000000000000000247815', '-0.00000000000000000000000247815', 29, 2);
    t('-0.000000000000000000000000000000039578053693', '-0.000000000000000000000000000000039578053693375996216932325600263217353654', 42, 4);
    t('-232816283401672701377791127757043050155711815150567236808118951', '-232816283401672701377791127757043050155711815150567236808118951', 5, 2);
    t('-29071348593675752783607764808066026', '-29071348593675752783607764808066026', 0, 3);
    t('-0', '-0.0000000000000000000000000000001261801374206123908708248209346534624420321535319795165638569', 3, 5);
    t('-400979013779505784551704647545324555644743917317817725', '-400979013779505784551704647545324555644743917317817725', 51, 0);
    t('-707984518604562445981560648691697348187535181109991331112833004096934116229906958', '-707984518604562445981560648691697348187535181109991331112833004096934116229906958', 33, 3);
    t('-0.00000000000000000000054943542078511748325889425', '-0.00000000000000000000054943542078511748325889425029425118', 48, 2);
    t('0.000000000000000000000000000001', '0.0000000000000000000000000000000004', 30, 2);
    t('8361324754302.257', '8361324754302.25608823898722021345099998761427662952277448263515002938433228636937277798222125', 3, 2);
    t('47845474280015515238795364952786480611238289992139357357828.2393536', '47845474280015515238795364952786480611238289992139357357828.23935359', 7, 2);
    t('2549907485257905040787802731022172814.032473410309278713663931353982863019132463', '2549907485257905040787802731022172814.03247341030927871366393135398286301913246263649610180999011', 42, 4);
    t('7245563391265598645357861460253613932139592382610560614764364520097782949512752649', '7245563391265598645357861460253613932139592382610560614764364520097782949512752649', 40, 1);
    t('80140230840302995211398777354070268652682793260175436489292', '80140230840302995211398777354070268652682793260175436489292', 35, 5);
    t('-6237457908203764779831412120415113057397781437576879126960196359736451276', '-6237457908203764779831412120415113057397781437576879126960196359736451276', 46, 3);
    t('-64022435787355811014521281511793708435812347405139910972682589', '-64022435787355811014521281511793708435812347405139910972682589', 59, 4);
    t('23773726397.29160222442891928943538179152516815326677387048397317891529', '23773726397.291602224428919289435381791525168153266773870483973178915287792', 59, 2);
    t('-0.0000000000000000009191736362201691567821986127083610325635', '-0.00000000000000000091917363622016915678219861270836103256344707473557037101545788367584278310776658', 58, 3);
    t('576493832674677623548573635478304255.82263895978273', '576493832674677623548573635478304255.82263895978273', 14, 5);
    t('0.00000000001', '0.00000000000000000000000000000000000000000000000007822968340296738406', 11, 2);
    t('-0', '-0.00000000000000000000000000000000000000000000000803017739', 16, 2);
    t('-76782672919180281245123823777032511965124724736456274885479622075418722', '-76782672919180281245123823777032511965124724736456274885479622075418722', 48, 4);
    t('4455851923984827076883518360355220330356', '4455851923984827076883518360355220330356', 36, 6);
    t('6861599706553109843427643365400432566541688718985576240352998201999416183743', '6861599706553109843427643365400432566541688718985576240352998201999416183743', 2, 2);
    t('0', '0.0000083135366316053183734543904737952651532784316140061929170739473518406297062533554026617147464', 0, 1);
    t('72361127388902796012114362477322144735533135802605580980837229637633107.0925818254603', '72361127388902796012114362477322144735533135802605580980837229637633107.09258182546029609579105130429', 13, 5);
    t('-273219046129778472266.058485499109338596573075', '-273219046129778472266.0584854991093385965730749815531346353045049027336176088162559', 24, 4);
    t('-0.00000000000007280978660680040221854732703892674800762227646', '-0.00000000000007280978660680040221854732703892674800762227646226836240552990665555', 59, 5);
    t('6276464097096605785329824864148.527049815380996985914', '6276464097096605785329824864148.52704981538099698591393138250952524233217779', 21, 0);
    t('-597198', '-597197.628834953506966767991553710700934413500204012426446876175175114500037146677042239668', 0, 0);
    t('-433359038877962603713455049783', '-433359038877962603713455049783', 30, 0);
    t('0.0000000000000000000000000000000006381735336173415547900207', '0.0000000000000000000000000000000006381735336173415547900206847223271181528556195', 58, 0);
    t('807050820.676699481791051456', '807050820.676699481791051455096629111775329549493444751897213743866128850061622892', 18, 2);
    t('0.0000000000000000000000000000489752745584111598026871408099', '0.000000000000000000000000000048975274558411159802687140809907', 58, 3);
    t('0', '0.0000000000000000000000000000000000000000000226067497099004331941912856', 36, 6);
    t('39094507220356523575213839820749561678857032020620050894656716293.67495126324', '39094507220356523575213839820749561678857032020620050894656716293.67495126323710280607325675762434', 11, 2);
    t('-22678769.817248435493696742588538', '-22678769.817248435493696742588538331241538369550386799148219117165563326051964281', 24, 4);
    t('5767307789536064608781837241295188919', '5767307789536064608781837241295188919', 30, 0);
    t('168139421336703751454723348941581.0989620502676347173183006762310303463068390391675336175', '168139421336703751454723348941581.0989620502676347173183006762310303463068390391675336175', 55, 6);
    t('-88504154823150878334701258558002569539793415193610842759120001088201133334307983', '-88504154823150878334701258558002569539793415193610842759120001088201133334307983', 44, 4);
    t('-329655464734888739743767364510089523323', '-329655464734888739743767364510089523323', 25, 1);
    t('0.00000000000000000000084501920385200277918', '0.000000000000000000000845019203852002779189', 41, 1);
    t('-5840123488663376216608124416421102623453621285401805153644515702191', '-5840123488663376216608124416421102623453621285401805153644515702191', 59, 3);
    t('-0.00000062136', '-0.00000062135994705765807424168688040116009865960794', 13, 3);
    t('-0.000000000000000000001', '-0.00000000000000000000000000000000000000001462340018509', 21, 0);
    t('0.006', '0.0059936079684973012978514720463815026319867172', 4, 2);
    t('526400924922583880463185416850358172941032433541198785132036360013561525242705246671523133061757', '526400924922583880463185416850358172941032433541198785132036360013561525242705246671523133061757', 38, 6);
    t('-882873192309626516849737955750920016208815071464396145314', '-882873192309626516849737955750920016208815071464396145314', 46, 6);
    t('-6758735221049379519.2572393', '-6758735221049379519.25723934716782621573177906', 7, 2);
    t('-0.000000790786371520517683432954223230156393', '-0.00000079078637152051768343295422323015639290504929', 42, 4);
    t('-0.00000000000000000000000000000000000091966', '-0.0000000000000000000000000000000000009196610338039484256720548095', 41, 6);
    t('-0.00000000000541933780252974657673566396725809671507', '-0.0000000000054193378025297465767356639672580967150744942399', 50, 4);
    t('-0.000000000000000000000583479626', '-0.000000000000000000000583479626800396501245049473177007318461587644875162633137088', 30, 2);
    t('11604528953141243475', '11604528953141243475', 13, 3);
    t('566807769866890.02801522720838416179840190775794257854696482', '566807769866890.02801522720838416179840190775794257854696482590456963092114076898', 44, 3);
    t('-35620697798492911.0669248410687861643251268323797667576839159303711022', '-35620697798492911.066924841068786164325126832379766757683915930371102255700535220012496346147093317', 52, 1);
    t('-0.0000000000000000000050597138398214172387021051', '-0.000000000000000000005059713839821417238702105087169671933387005', 46, 0);
    t('-0.8174627036499993060703761337876311', '-0.81746270364999930607037613378763105641195817852303184573911882', 34, 4);
    t('0.0000000000000000000016801618027332596946', '0.000000000000000000001680161802733259694523102897059933026814910108114561982575916856344241', 40, 2);
    t('-0.0005550919624693963627417354876271038367574907', '-0.000555091962469396362741735487627103836757490736793688', 46, 6);
    t('0.00000000000001', '0.0000000000000000005989657036', 14, 2);
    t('26923162467831521466200388799932149017792464401239965995848900909703513553682', '26923162467831521466200388799932149017792464401239965995848900909703513553682', 59, 0);
    t('0.000000000000034716604224301619113605162061332208', '0.0000000000000347166042243016191136051620613322078875442915881933', 48, 5);
    t('4554587644116353728395891927482', '4554587644116353728395891927482', 4, 4);
    t('83', '83', 1, 4);
    t('-0.0000000000000000679512312597136188846870750884127001656543', '-0.0000000000000000679512312597136188846870750884127001656543162955358', 58, 5);
    t('-0.000000000000000005843127017405787209162333', '-0.0000000000000000058431270174057872091623337743081992327769648', 42, 2);
    t('698142412071442689875148052035405913655358215178078673989', '698142412071442689875148052035405913655358215178078673989', 31, 2);
    t('-0.000000000000000000000000001', '-0.0000000000000000000000000000000000000009', 27, 3);
    t('0', '0.000000000000000000000000000000000000000000000000086322062431379058693358', 36, 6);
    t('-7097679626212584135194693334505819500.76271239784243114877303953752095973790591', '-7097679626212584135194693334505819500.7627123978424311487730395375209597379059174819443305631091738', 41, 1);
    t('720941616590530465684319461159925340787620861616050215112729354513077297889437424470222725372.4341', '720941616590530465684319461159925340787620861616050215112729354513077297889437424470222725372.43418', 4, 1);
    t('-8565045.0741813', '-8565045.07418120224', 7, 3);
    t('7179182230007440380654240229988748528461622212340003478705', '7179182230007440380654240229988748528461622212340003478705', 16, 1);
    t('128138852434106311723518159896099183377408757231649238006509175039', '128138852434106311723518159896099183377408757231649238006509175039', 51, 0);
    t('0.0000000000000000000000000000000000071444116240971906475', '0.000000000000000000000000000000000007144411624097190647497044019271703768941206', 56, 5);
    t('0', '0.00000000000000000000000000000000000000000000021714481944', 43, 6);
    t('-0.0000000000000001', '-0.0000000000000000000000000034534834', 16, 0);
    t('-0.0000000000000000000008286492885892336366506247591678062602', '-0.00000000000000000000082864928858923363665062475916780626021532507656936043414109352811732', 58, 1);
    t('31161421229261423772494491055755169247634601491928', '31161421229261423772494491055755169247634601491928', 51, 3);
    t('1748715317929813133410156549170209422179478560908330825848622104018934659066', '1748715317929813133410156549170209422179478560908330825848622104018934659066', 57, 4);
    t('-554303811557294466.269761483473739646624314242607077', '-554303811557294466.269761483473739646624314242607077815435340758612837177421989342652', 33, 1);
    t('-0.19004485', '-0.19004485473016995992614957080209680408919713640428488619', 8, 1);
    t('0.000252525605711892', '0.000252525605711891731866212005527221984514215823140269482997380211714402276122070798053', 18, 6);
    t('86245969707619', '86245969707619', 14, 3);
    t('-31435925010862195008998682761', '-31435925010862195008998682761', 9, 3);
    t('-7376.12482208', '-7376.124822087908447175101', 8, 2);
    t('0.000080221434577125714125624728141606555230430450054', '0.0000802214345771257141256247281416065552304304500535613033078792598113626175', 51, 0);
    t('-6050582615205191601389958119203059837835097590785064613410822037914417495686026661', '-6050582615205191601389958119203059837835097590785064613410822037914417495686026661', 31, 1);
    t('0.00000000000000003455493043277403696896050418463562482', '0.0000000000000000345549304327740369689605041846356248273019036250593274409786087865531980510674', 53, 3);
    t('-20967.65067808575', '-20967.6506780857520813447701235001803657407937', 11, 6);
    t('568254966593770.553753276551449605949', '568254966593770.553753276551449605948238816764309803642928261672349658172008375162162314878680613', 21, 0);
    t('-451.149170475956331479335', '-451.1491704759563314793340102771090380475440750267088321862964226523091325726805634622304958123379', 21, 3);
    t('-0.00000000000000000000000089587927281631480799', '-0.00000000000000000000000089587927281631480798176250533957436898566513857011780218162097370714526', 44, 0);
    t('0.00006', '0.00005989898942742447311513559', 6, 2);
    t('0.00000000000000000000000000000000000084666165975', '0.000000000000000000000000000000000000846661659751645', 47, 6);
    t('-274912.8896024699', '-274912.8896024699118787839924993246206752520896053416203239133353705', 10, 1);
    t('2884607226068313798651836226110538860137007975185098848', '2884607226068313798651836226110538860137007975185098848', 55, 6);
    t('-0.00000000000000000002773912168', '-0.000000000000000000027739121678637743521714908420713523555500095414277237781740952561559672', 29, 3);
    t('-2651358523359639', '-2651358523359639', 16, 4);
    t('-4813276596516131663456402092.38944176', '-4813276596516131663456402092.3894417517916213734867', 8, 3);
    t('-0.000000000000000000000000000000000000112895859', '-0.000000000000000000000000000000000000112895858119340820153717620708673416', 45, 0);
    t('-0.0000000000000000021', '-0.0000000000000000020224974226159732231298241899365279396237510862449464734463589', 19, 3);
    t('-7589382547973376572325815568183010749314649708', '-7589382547973376572325815568183010749314649708', 33, 5);
    t('842229243093860852173.05443961588175098377444082861489172139756969332835', '842229243093860852173.0544396158817509837744408286148917213975696933283408713841831638764', 50, 0);
    t('-699708233495.712278374225898965891885496', '-699708233495.71227837422589896589188549642075052667001859282382939797996992686357419809583', 27, 4);
    t('-0.00000000071616892037859054', '-0.000000000716168920378590537721800581109521242491374877', 26, 0);
    t('3951438403.277181863849692687044689923289027507211228628773736', '3951438403.2771818638496926870446899232890275072112286287737359542096', 51, 2);
    t('109559.01297171197941271991562118124015322', '109559.012971711979412719915621181240153223610175595763848035079898263530366', 35, 3);
    t('-0.00000000000000000000000000000000000000000000489005761683947', '-0.00000000000000000000000000000000000000000000489005761683946256390055271579820502403', 59, 3);
    t('-0.00158', '-0.00158276409341184629828973465939702170102740363504722102864514069935682480434491', 5, 2);
    t('-2625562538887919963240549817430379735187837775384', '-2625562538887919963240549817430379735187837775384', 44, 0);
    t('0', '0.00000000000000000000002', 9, 1);
    t('6744808.28630410992583043657159413157574982400492', '6744808.28630410992583043657159413157574982400491999838773235902794700459508120619939892', 42, 6);
    t('0.000000000000000009', '0.000000000000000009', 18, 3);
    t('6067881766683695479556751950119377724336039886809300136812181462', '6067881766683695479556751950119377724336039886809300136812181462', 31, 1);
    t('0.000000000000000000000000000000000000000077241022793', '0.0000000000000000000000000000000000000000772410227935606591033087412064412570098277', 51, 1);
    t('743268323094543466714.5656060177009296968524561226948074051004264', '743268323094543466714.565606017700929696852456122694807405100426400624651576046667459841457817767865', 43, 3);
    t('-596135', '-596135', 3, 2);
    t('0.07662404229928167815820747802086592943344134084819', '0.0766240422992816781582074780208659294334413408481864862625859275536716954542357278357044523255', 50, 4);
    t('-0.000000000000000000000000000885256', '-0.000000000000000000000000000885255222637729340070545710310579917592457286140653', 33, 0);
    t('55893020145100569857309952693924435456669213.356281068124371401302229274073839082240544043386519', '55893020145100569857309952693924435456669213.356281068124371401302229274073839082240544043386519', 51, 0);
    t('0.000000000044398802543439239843437872656117', '0.000000000044398802543439239843437872656117345711426288269229', 42, 1);
    t('0.000000000000000000000000000026311845042475573561274888459', '0.00000000000000000000000000002631184504247557356127488845876295156781061349824020867304', 57, 6);
    t('-127546673151161350153045221729711', '-127546673151161350153045221729711', 2, 5);
    t('0.0000000000000000000000846081', '0.00000000000000000000008460817346743347251678412215465432867047065857', 28, 3);
    t('-51296363216658187515760473402291', '-51296363216658187515760473402291', 27, 4);
    t('-8623967090632480689370959269411792112227821530531202541003992985', '-8623967090632480689370959269411792112227821530531202541003992985', 54, 5);
    t('-966322003490784535731979767036707880939452169502093813823674817588484', '-966322003490784535731979767036707880939452169502093813823674817588484', 56, 5);
    t('-272472379898107893040761485379027824396136886208', '-272472379898107893040761485379027824396136886208', 40, 1);
    t('0.000000000000000000007578294653267877535740172858883482', '0.00000000000000000000757829465326787753574017285888348202441720573692830214583542806201', 55, 6);
    t('25958781094763030494523126276550672249453141421083859868561356127876297714106602172585', '25958781094763030494523126276550672249453141421083859868561356127876297714106602172585', 29, 5);
    t('0.00000000000000000000000000000000000002', '0.000000000000000000000000000000000000015806075051053843417566111132263641553843698646164', 38, 5);
    t('-73099329416955017257739991488405608089647281115099586', '-73099329416955017257739991488405608089647281115099586', 21, 6);
    t('-88307320.90319141093816288808957670297463', '-88307320.903191410938162888089576702974638474154077235587259266824247410860533478169707304792818', 32, 2);
    t('-35236814557361445958969411965214520.30399877405421', '-35236814557361445958969411965214520.303998774054210302477326477', 14, 2);
    t('-2908087206403224796868158372851057.39782067611', '-2908087206403224796868158372851057.3978206761117082535006786132666114643644421715709235528929678', 11, 4);
    t('-19581413994383784948718328954653', '-19581413994383784948718328954653', 28, 1);
    t('0.00000000000000000000000000004', '0.000000000000000000000000000040608230569366379665209245709785291404206510745612419401142658', 29, 3);
    t('-0.906238977426875501362745', '-0.9062389774268755013627459831223515271076867650054627805581400941104277662892230888597201', 24, 2);
    t('0', '0.000000000000000000000025517761823978244291048210421988594612225022695964910425529', 21, 1);
    t('663336.45734367931832199866', '663336.4573436793183219986595282312647796998714487327022132545955984591825466144183', 20, 4);
    t('-461221276.52047063868261544207237644195170184', '-461221276.5204706386826154420723764419517018397461911607', 36, 4);
    t('92664692270788697481952993240101', '92664692270788697481952993240101', 9, 1);
    t('0.00000000000001', '0.0000000000000000417218487798321067688965201563233239322412080713783058725771499175637', 14, 0);
    t('-0.00000000000000291161', '-0.0000000000000029116100944057582762735577555', 21, 6);
    t('3593092835933826522602826541522473977510499517734180584911059518', '3593092835933826522602826541522473977510499517734180584911059518', 39, 2);
    t('0.00000000000240671872382434259', '0.00000000000240671872382434258731759266816542514881094906467645247797685306920193421', 29, 6);
    t('98445608185462908936594271820438358882244286228', '98445608185462908936594271820438358882244286228', 43, 4);
    t('-38275693035627727048820160757435988950513161', '-38275693035627727048820160757435988950513161', 28, 2);
    t('0.000000000000000007838540274247171557849126', '0.000000000000000007838540274247171557849125417291807341828802631329086161031903106930206518314803525', 42, 0);
    t('0', '0.0000000000000000000000000000000000000008760161376', 38, 6);
    t('-274462946120897177140732986024361620867165740004629583369799434.752624497847', '-274462946120897177140732986024361620867165740004629583369799434.7526244978472107528071824755321251', 12, 1);
    t('805407519521180265118391229', '805407519521180265118391229', 27, 0);
    t('4837.60752412303502514', '4837.60752412303502513085517977565448616961234363618524491896285197928079211821305450283483280681236', 17, 0);
    t('19446.966116400627008657626752077217581388920002', '19446.966116400627008657626752077217581388920001650492489914443643730475', 42, 6);
    t('259711114990641', '259711114990641', 1, 3);
    t('-0.0000000001', '-0.0000000000000000002', 10, 0);
    t('80565580.149080509196072203611545842035798415', '80565580.1490805091960722036115458420357984148292066084650845696757793247885015079193577022206528568', 36, 5);
    t('0.000000005126609951384222369644011060208031', '0.00000000512660995138422236964401106020803193692417523731220078957765716642', 42, 3);
    t('1758006742538130240388703498480688686072955030356.7524172233646685', '1758006742538130240388703498480688686072955030356.75241722336466855', 16, 1);
    t('-128086989690.52297791537155595882257011356310571016034760745375196584546', '-128086989690.5229779153715559588225701135631057101603476074537519658454607884909539951', 59, 2);
    t('-361166416734639117275346911473316661252428418964862.401', '-361166416734639117275346911473316661252428418964862.400974469693342990746', 4, 5);
    t('71633752430127836728495483808.07484204148983764432608854793881189', '71633752430127836728495483808.074842041489837644326088547938811892934755621628332271860178432369', 35, 4);
    t('5744453717566208360238616812981884481035389801', '5744453717566208360238616812981884481035389801', 46, 1);
    t('2988347090930431122495200201632971168964831173901728', '2988347090930431122495200201632971168964831173901728', 31, 0);
    t('2320482568655945754105081468655336561175857', '2320482568655945754105081468655336561175857', 13, 6);
    t('85972608415515722814484953328108168327861432', '85972608415515722814484953328108168327861432', 27, 3);
    t('735203353605367618463183405487305335425', '735203353605367618463183405487305335425', 3, 4);
    t('-0.000000000000038685598587815953783879233', '-0.00000000000003868559858781595378387923293510596996438543568027764671', 39, 4);
    t('39417777876786859925775897228.414515103190999545938444157261853177310365', '39417777876786859925775897228.4145151031909995459384441572618531773103646804525566998911941365927', 42, 6);
    t('-0.00042729698662369', '-0.0004272969866236854339301384420491548821671090593596154031221535925979534328791', 17, 3);
    t('-0.001', '-0.000000000000000000000000000000000005405329234046634574698589533972367873335830589026185244', 3, 3);
    t('780253449349101270418037346037809619092763848109347741511253928090145408496147938502521', '780253449349101270418037346037809619092763848109347741511253928090145408496147938502521', 54, 4);
    t('-0.00000000000000000000000000000000067451325390149', '-0.0000000000000000000000000000000006745132539014862711234', 47, 0);
    t('31.0602835210115', '31.06028352101159116694707249229172391419071049341139', 13, 1);
    t('0.0000000000000000000000000000000000000000000000000495', '0.000000000000000000000000000000000000000000000000049499482322322842956811022466314', 54, 0);
    t('0.00000000000000000000000000036747356681216113823925277207', '0.0000000000000000000000000003674735668121611382392527720704138622451221314733501', 57, 1);
    t('320701762086356051254296033221502209341446339599208346957463888343997136', '320701762086356051254296033221502209341446339599208346957463888343997136', 32, 0);
    t('0.0000000000000000006229756350932003529439584563700569061023', '0.0000000000000000006229756350932003529439584563700569061023', 59, 0);
    t('-0.00000000000000000000000000000001', '-0.000000000000000000000000000000000002779554968030959772451850537321855864675620017588372276903500536', 32, 3);
    t('19504345494555478351182477703565457647566876858685627027379505944698694624726169090861984', '19504345494555478351182477703565457647566876858685627027379505944698694624726169090861984', 0, 4);
    t('-884684852128576325857920870487060224224863003891902747660876195738072', '-884684852128576325857920870487060224224863003891902747660876195738072', 29, 2);
    t('-0.00000000000000001927120482', '-0.000000000000000019271204816104208235432769229247337783831896121690184707573452963345555747139521261', 26, 3);
    t('-17476187434.5449326082784577197411497434939321355651616981523713348786', '-17476187434.544932608278457719741149743493932135565161698152371334878519825208420735438717588271', 58, 3);
    t('0.000000000000000000000000000000000000000000000000671942359', '0.000000000000000000000000000000000000000000000000671942359005812', 59, 1);
    t('0.0000000000000000000000000000000004917118136701612', '0.000000000000000000000000000000000491711813670161116275010479557699441078855323071908', 49, 0);
    t('634166172293272998158876.8413', '634166172293272998158876.84128062603520572376508413295607032369731651393103539143496669193918096843', 4, 0);
    t('-28119944334730255826834221613922194267786170327968560369991839067', '-28119944334730255826834221613922194267786170327968560369991839067', 32, 6);
    t('0.00154819734046993', '0.00154819734046992352816640604555843660775840335436322950452318455817', 17, 0);
    t('-105913645497507508669973802578229519978763548843414114661.54980455412849338374', '-105913645497507508669973802578229519978763548843414114661.549804554128493383737493', 20, 4);
    t('7289695139546080448297917061342289649234847530317672856838500148', '7289695139546080448297917061342289649234847530317672856838500148', 20, 1);
    t('0', '0.000000042979420754347314988434544000447696776811769466455886485920655', 0, 4);
    t('0', '0.000000000000000000000000000000000924077178027327', 24, 6);
    t('282942634910281488152070543.81977405730745731368050238', '282942634910281488152070543.81977405730745731368050238222825081313', 26, 5);
    t('9829583855774518751527887659888828937143309416650.01061691363', '9829583855774518751527887659888828937143309416650.010616913621011666291387767132900103741582580722946', 11, 2);
    t('2843.5579809036184', '2843.5579809036184434137308509550969034869063413306363790685675', 13, 3);
    t('0.0000000000000000000000000000006935112798', '0.000000000000000000000000000000693511279803301217125480155013139742885382224949324725789805812', 40, 5);
    t('38402002761517.0373', '38402002761517.03726339201008642458566869258242930397372981070695839443682553444750792077030923352', 4, 2);
    t('-0.00000000000000001', '-0.00000000000000000000000000000000000284', 17, 0);
    t('-0.00000000000000000000000000000000000000000328559961', '-0.0000000000000000000000000000000000000000032855996112178825812918129', 50, 2);
    t('-10857376233280.74203', '-10857376233280.74203521516266128862336259662162270285388060400166956185959712257522107', 5, 2);
    t('-0.00000000000000000000000000000001', '-0.000000000000000000000000000000011428328539927202527982255539515744018395155', 32, 1);
    t('0', '0.0000000000000000006362859957256', 11, 4);
    t('4325993183488742208455442659071054631148', '4325993183488742208455442659071054631148', 11, 3);
    t('-1163899294989774891167097029980639467528238979769081614588667904', '-1163899294989774891167097029980639467528238979769081614588667904', 27, 3);
    t('-0.000008771176272398358315944222330846161437656443089', '-0.000008771176272398358315944222330846161437656443089372426117023897251811651901343373735604735', 51, 2);
    t('-0.000000000000000000000003961015971842', '-0.000000000000000000000003961015971841840897153173232191143604645950090459018394', 36, 3);
    t('-0.000000000000044516333', '-0.0000000000000445163325234697173332966370812630194498631820837056547152354402952422626816765', 21, 0);
    t('-0.001', '-0.000000000000000000000000560288645294823225485858180788451420057727431', 3, 3);
    t('70431396474321596927755.0616724357770297517853368489945852116010015382199953', '70431396474321596927755.061672435777029751785336848994585211601001538219995373286052198980239404', 52, 1);
    t('-290563347.4018618590644496547932201955716045339506671235383081837373', '-290563347.40186185906444965479322019557160453395066712353830818373739840892194032111993775', 58, 1);
    t('-49838122704426512042478', '-49838122704426512042478', 12, 4);
    t('65731122722942661959967857762.18', '65731122722942661959967857762.18226768355365269085129302733326389972791821455019761189700189', 2, 4);
    t('684132827465730082618383537053250641255400578134', '684132827465730082618383537053250641255400578134', 39, 1);
    t('710427365331082486088196567.85693834564371170948582954685019', '710427365331082486088196567.8569383456437117094858295468501870885566575655950543328963118398231759', 32, 4);
    t('36646445.4', '36646445.42824516523285194093354092096642673855547686910006', 1, 4);
    t('0.43424917263886958982405140577413037118', '0.43424917263886958982405140577413037118797467795184', 38, 1);
    t('-0.00000000000000000000000036588414680611016888031201521581', '-0.000000000000000000000000365884146806110168880312015215814', 56, 5);
    t('-0.00000000000000000000051018357371038710352736277553359016842', '-0.0000000000000000000005101835737103871035273627755335901684125664187418843827676645536966728787', 59, 3);
    t('-35159732781823.97716511119720206064558667', '-35159732781823.977165111197202060645586672946470926242812549089110584121041', 26, 4);
    t('0.000000000000000000000000000000000003364306363591', '0.000000000000000000000000000000000003364306363590521522380481558802243742946674208931', 48, 4);
    t('-4306747914757314388890273349986190769173944.39291819807120899471231887129063365500089582609', '-4306747914757314388890273349986190769173944.39291819807120899471231887129063365500089582608837', 47, 3);
    t('-0.0000206161005604222009966692502', '-0.00002061610056042220099666925021597014337038232489136426449787194485033490838383', 31, 5);
    t('29588529819920739172563477650035803968437251317221514791', '29588529819920739172563477650035803968437251317221514791', 38, 4);
    t('0', '0.00000000000000000000000000000076512226709342597183392545931193073264251232258187043154966155', 10, 4);
    t('642634746714122.488629063706747265885497766263668674846', '642634746714122.4886290637067472658854977662636686748455506370584335783149463152080002791966876190272', 39, 5);
    t('-0.000000000001', '-0.0000000000000000000000068031607833728788218500078443799438145', 12, 0);
    t('4478793059557.919366066899688681937166624937', '4478793059557.919366066899688681937166624937037651207942775207986303596524642642692579', 31, 1);
    t('719126016681991174787244179437135662886106', '719126016681991174787244179437135662886106', 21, 6);
    t('612370379.042280015162966981091543536169122', '612370379.04228001516296698109154353616912118620911001007', 33, 0);
    t('-0.000000000000000000381148646471375542749', '-0.00000000000000000038114864647137554274933758', 39, 1);
    t('-0.00000336921984240686812663817912975898', '-0.00000336921984240686812663817912975898003855606655763928424753588827973199516045', 38, 1);
    t('6614908420764830574613829691575811991052782880888895087915', '6614908420764830574613829691575811991052782880888895087915', 1, 5);
    t('-75576843314854333307451548', '-75576843314854333307451548', 8, 2);
    t('-7094775936348915.769263075129454561542743852557970402', '-7094775936348915.7692630751294545615427438525579704015758356635999962', 36, 4);
    t('0.000000000000000647255404244969058746125607', '0.0000000000000006472554042449690587461256070155216132767410199433477488145722851174', 43, 4);
    t('212191747967678055259704020533185819.7153877789442928970247278', '212191747967678055259704020533185819.7153877789442928970247278005685961467993140759589633126', 25, 5);
    t('0.000000000000000000000000000000086866886236741', '0.000000000000000000000000000000086866886236741491884373982593431074047696748', 45, 5);
    t('-21896707599295196444127', '-21896707599295196444127', 7, 6);
    t('67963466958486868433411681965', '67963466958486868433411681965', 8, 1);
    t('-0.0004618114320999748548', '-0.0004618114320999748548227742586597847789768814695996979397676648662277615278571112727802', 22, 1);
    t('0', '0.000000000000000000000000000000000000000072014201946018446425103586624', 34, 3);
    t('0.000000000000000000000000040626999', '0.000000000000000000000000040626998084581879309101940084150929007748602990641311915569084218', 33, 0);
    t('-0.00000000000000000000000000000000000002364670058488838781', '-0.000000000000000000000000000000000000023646700584888387803963852870078295754301293417634319', 56, 0);
    t('-61312195050009621627250721752682.05458', '-61312195050009621627250721752682.05458776416651498653588646631355122', 5, 2);
    t('88558823477934754368770485901911168548905661612129649349536492903025217066', '88558823477934754368770485901911168548905661612129649349536492903025217066', 27, 3);
    t('-5761274593909054578402071821461472994565942226296450524728000900348368713263854424411989', '-5761274593909054578402071821461472994565942226296450524728000900348368713263854424411989', 54, 4);
    t('4397525934482', '4397525934482', 10, 1);
    t('0', '0.000000000000008', 3, 3);
    t('3832505074710.8772241868692955359876737605513385536', '3832505074710.8772241868692955359876737605513385535657156382235008771952598281484', 37, 5);
    t('0.0000000000000000000000000000000000000000000064039785838957', '0.00000000000000000000000000000000000000000000640397858389567536112989058804434974', 58, 6);
    t('6899986941802317801470081277005392880851864737732161265779438709', '6899986941802317801470081277005392880851864737732161265779438709', 36, 2);
    t('0.048906322648425185632628713058196452313509', '0.0489063226484251856326287130581964523135085310426446391', 42, 5);
    t('-29068667316141265922020323299656223961691022983876401059969277924346', '-29068667316141265922020323299656223961691022983876401059969277924346', 52, 1);
    t('57896564283529490722728198636247192473162.955437706', '57896564283529490722728198636247192473162.955437706585876490347082', 9, 1);
    t('-1377052660765266598224922578884764678982220316470617198042071.84', '-1377052660765266598224922578884764678982220316470617198042071.842763168339803148921', 2, 6);
    t('-0.0000008472294237476439209693038490556088683', '-0.00000084722942374764392096930384905560886836719554554921990211', 43, 2);
    t('-0.00000000000006211378313284557438319018', '-0.0000000000000621137831328455743831901849526312006248559047579838', 38, 6);
    t('-64132448927708626844067839788020.56104426721067307770150847932963341750262', '-64132448927708626844067839788020.56104426721067307770150847932963341750262451491448713527', 41, 4);
    t('-0.00080510769276', '-0.0008051076927558161863905844170590891349375387966097254255501665792278513428990177180729', 14, 4);
    t('-0.000000000523495070412194527222', '-0.000000000523495070412194527222245530916445', 30, 6);
    t('-776981501784122653016725778489064077038674642417508137241453166338683', '-776981501784122653016725778489064077038674642417508137241453166338683', 25, 6);
    t('-85.754361554275952592904696122168206579', '-85.754361554275952592904696122168206579375505261', 36, 6);
    t('41971686031617689086324682083921394293937303185646952166039268116965193583', '41971686031617689086324682083921394293937303185646952166039268116965193583', 7, 5);
    t('0.00039420287057734077878369443635344083421129327830247522826', '0.0003942028705773407787836944363534408342112932783024752282629774898259222473719', 59, 3);
    t('0.00000000000000000000000000000000000000000000000675866720473', '0.0000000000000000000000000000000000000000000000067586672047328993602503820470338', 59, 5);
    t('-0', '-0.000000000000000000000000000000000000000000000088705230880406784828084160640801289390145869866', 36, 1);
    t('-304191535324135351444668650070326671', '-304191535324135351444668650070326671', 28, 0);
    t('-0.000000001', '-0.000000000000000000000000000000000000645202999', 9, 3);
    t('-1475932977772754862911365548920235009', '-1475932977772754862911365548920235009', 1, 1);
    t('0.00000000000000748482924758', '0.00000000000000748482924758907665350364240578489492856151184272179973', 26, 1);
    t('-1', '-0.0000800368117241283577308465283698286330423189888237592540844483', 0, 3);
    t('0.00000000000000000005436973621557718727915791605', '0.0000000000000000000543697362155771872791579160464793433', 47, 4);
    t('-0', '-0.000000000000000000000000008105812474775967121736315163', 10, 2);
    t('-162061415319361123389400237365697567694754077666970.95367098983987825787039', '-162061415319361123389400237365697567694754077666970.95367098983987825787039768461541526', 23, 1);
    t('-0.00000000000000000000000078141830260660528317962609581', '-0.0000000000000000000000007814183026066052831796260958153167081281', 53, 2);
    t('-760450372966873218121690338416694511378045594432280482201.6', '-760450372966873218121690338416694511378045594432280482201.5743953451455', 1, 6);
    t('-8969924824887750.078167576440491913960555993', '-8969924824887750.0781675764404919139605559934327012046423719029', 27, 6);
    t('-0.00000000000000000000000000000000000000000008850594039644162', '-0.00000000000000000000000000000000000000000008850594039644162037424244532346653', 59, 2);
    t('-260213417057955846142647463609725738084865895592932493537647341857', '-260213417057955846142647463609725738084865895592932493537647341857', 51, 2);
    t('565141621099385566886468464466', '565141621099385566886468464466', 10, 1);
    t('83664.29974759139108443286033487', '83664.2997475913910844328603348762233403', 26, 1);
    t('159684867991933938188684086.84829503153331708926629620984773984546340485586957818589', '159684867991933938188684086.84829503153331708926629620984773984546340485586957818589283205', 56, 4);
    t('4783482240722.889565506051950025731026748125580032', '4783482240722.88956550605195002573102674812558003270323257226548107196735812', 36, 1);
    t('0', '0.0000000000000000003', 9, 1);
    t('-73191538351494680542101.097679556493531497949879615674615869073278632189', '-73191538351494680542101.09767955649353149794987961567461586907327863218941724713424903198', 48, 6);
    t('1377196679485199902841228391473638877532', '1377196679485199902841228391473638877532', 20, 0);
    t('-17772153798727918242886392361333023042650831518530553', '-17772153798727918242886392361333023042650831518530553', 46, 0);
    t('0.08328506397736199', '0.08328506397736199281189505020297381756462266529288847578', 17, 6);
    t('0.000000000000000000000073181265651336435623978981763', '0.000000000000000000000073181265651336435623978981762922547674290225409720410306838731933043556', 52, 0);
    t('2290053898142311723842871344607971300300.4', '2290053898142311723842871344607971300300.434151478993367452964917933417740817968087394784172', 1, 4);
    t('-4526903994128138106185254976553.417454138027126670650882487065659163773887419248789373', '-4526903994128138106185254976553.4174541380271266706508824870656591637738874192487893725731607711', 54, 4);
    t('0.000000000000000000000000000004529650848633752087137797', '0.00000000000000000000000000000452965084863375208713779692', 54, 2);
    t('41790148346489387876273166411501052202', '41790148346489387876273166411501052202', 8, 0);
    t('-0.000000000069328660259658436177', '-0.0000000000693286602596584361762589565091715124753019116431269', 30, 0);
    t('934822253075684362315304330796079406852179', '934822253075684362315304330796079406852179', 20, 0);
    t('6138685955473706895.27191846437634217823067', '6138685955473706895.271918464376342178230674192673963926883446477458738606505318684196487099828273', 23, 1);
    t('-4995207495760013820781008', '-4995207495760013820781008', 9, 5);
    t('-315575628071433482711097581911695580094607581401113654130956785', '-315575628071433482711097581911695580094607581401113654130956785', 53, 2);
    t('14237742301009518268544482664657681590264.3699281317800082179089', '14237742301009518268544482664657681590264.36992813178000821790889652431207007693905617376813211534954', 22, 6);
    t('6933324085798448906217582066829684616544680559688340085427', '6933324085798448906217582066829684616544680559688340085427', 59, 5);
    t('0.00000000000000076326740786404408348160812965284434866778', '0.00000000000000076326740786404408348160812965284434866778747625162316325969396896612585666', 56, 3);
    t('0.000000000000086813905963104592136996896057455896843', '0.000000000000086813905963104592136996896057455896843175022732573388730069970550757', 51, 5);
    t('51801898867306672572496403839561629956832174080646016677953156', '51801898867306672572496403839561629956832174080646016677953156', 8, 6);
    t('-0.000000007315101480415581536176467876648223789983200337', '-0.00000000731510148041558153617646787664822378998320033685682677794506948090883516907520915603842', 54, 6);
    t('0', '0.0000000000000000002560972936441568309868552845452461441', 1, 4);
    t('-4148384973258837308705900867312127926039628611927712693805299864565985491', '-4148384973258837308705900867312127926039628611927712693805299864565985491', 57, 2);
    t('0', '0.00000000000008081392179', 11, 3);
    t('-0', '-0.00000000000000000000000000000000000000000000000045718218551581160547165', 32, 5);
    t('126371519643424683947919776390544767992024526388', '126371519643424683947919776390544767992024526388', 33, 3);
    t('-148196', '-148195.22330505940168207920977604763714302872718112527157', 0, 3);
    t('0.000000002384518595846426', '0.00000000238451859584642619680666569487976797096982144404778137134898742245621718069', 24, 1);
    t('0.0000000000000000000001', '0.000000000000000000000000000000004625', 22, 0);
    t('-16911.7419832374869', '-16911.7419832374868955960680613558461083659651878098', 13, 5);
    t('-0.000012', '-0.0000122184223277020419635939917289023624698188785457519029895428324113502565328671558', 6, 2);
    t('0.0000000000000000020288542156920825061442115612294420237', '0.000000000000000002028854215692082506144211561229442023610384333574312922', 55, 0);
    t('-926348619407629319522119393447968619', '-926348619407629319522119393447968619', 9, 6);
    t('0.000000000000000011172996664320313188853949533840233', '0.00000000000000001117299666432031318885394953384023300040666548394933514811631768406408144820507299', 51, 6);
    t('8676085578448129636.46125142815995954615633699159686307427564866280997975', '8676085578448129636.461251428159959546156336991596863074275648662809979754896', 53, 6);
    t('0.000000000000000000006911673453476612828647988180511', '0.00000000000000000000691167345347661282864798818051074447846', 51, 5);
    t('-0', '-0.000000000000000000000000000000000000000037274661429910134438335779699', 20, 6);
    t('-0', '-0.00000000000000000000000000000022912429406409774126564962929569092713860155341547', 5, 2);
    t('0', '0.00000000000000000000007', 9, 4);
    t('-0.000000000000000000000000000647', '-0.00000000000000000000000000064679122351281910373', 30, 6);
    t('628253552286979258688877224011928439269818309', '628253552286979258688877224011928439269818309', 39, 2);
    t('-7473225474290375395406057107593674409', '-7473225474290375395406057107593674409', 9, 4);
    t('-0', '-0.000000000000000000000000000000000000000000044980341254278389339188217614184616236700512663', 42, 1);
    t('-0', '-0.0000000000000000000000008169851496807201249112045355202923', 10, 4);
    t('44979151873456538589159481816187996063744819950498068666276244930081', '44979151873456538589159481816187996063744819950498068666276244930081', 44, 2);
    t('-5656502680950453.307129232', '-5656502680950453.3071292317099969560559506107', 9, 6);
    t('0.000012916983213677608946870037474', '0.00001291698321367760894687003747317197534115902429539309092', 33, 0);
    t('186614411842436994154261287798158888894696101481267805116', '186614411842436994154261287798158888894696101481267805116', 48, 4);
    t('707.380045900803464776244462046183', '707.380045900803464776244462046182987', 31, 6);
    t('632194297368023228055660745485057870911107495734770014', '632194297368023228055660745485057870911107495734770014', 46, 6);
    t('35280495669924811181641930079387327223136396121465358386312882495656970347', '35280495669924811181641930079387327223136396121465358386312882495656970347', 1, 4);
    t('-6325867003724294.663698111988027895808356539794935974343081043', '-6325867003724294.66369811198802789580835653979493597434308104291301033018168875746098749871551255682', 45, 0);
    t('-799353555477135814839251032527782560596504969267009081682848404947106492', '-799353555477135814839251032527782560596504969267009081682848404947106492', 39, 1);
    t('66595791053662779211189537167463453717540557304309464', '66595791053662779211189537167463453717540557304309464', 27, 0);
    t('86863819.0795207560566246205038942404911', '86863819.07952075605662462050389424049106413333666160054769765', 31, 4);
    t('-0.000000000000000000000000000000001', '-0.00000000000000000000000000000000000000000000008', 33, 3);
    t('0.0000000000000000275', '0.0000000000000000275440961091735938216613771820776939226812444657855328490901237788725', 19, 1);
    t('-0', '-0.00000000000000000000000000000000000000000000000225083583687450018290901147795588587262327956', 21, 4);
    t('106844609528473355525467772516478055461312403965348968485756541728796', '106844609528473355525467772516478055461312403965348968485756541728796', 16, 4);
    t('-0.000000000000000814523751192634198253241238297485319828', '-0.0000000000000008145237511926341982532412382974853198275271819634843873247036611', 54, 0);
    t('2606400896622366889946488350737590139429931326875963144243247149520.3226059185533490213723413754', '2606400896622366889946488350737590139429931326875963144243247149520.3226059185533490213723413754', 28, 6);
    t('-183452660943051448746754671284613', '-183452660943051448746754671284613', 10, 2);
    t('839807437987291944449694079569638219', '839807437987291944449694079569638219', 36, 0);
    t('-8132217764461224416834766794322810048760447551363872095', '-8132217764461224416834766794322810048760447551363872095', 48, 0);
    t('-0.0000000000000000000000000000001', '-0.00000000000000000000000000000000084013749996116337258106', 31, 3);
    t('0.00000000000000000000000000000000056286653735314698', '0.000000000000000000000000000000000562866537353146980064941975206481554805', 52, 3);
    t('8826215431283658953381909149090157677663835064029813383632728867019103467383915169736679', '8826215431283658953381909149090157677663835064029813383632728867019103467383915169736679', 35, 1);
    t('65532958561624805787499.79879743257742173928432822969962227122981569885', '65532958561624805787499.79879743257742173928432822969962227122981569885', 48, 1);
    t('-2776707353947863935313425463872380342255574023352080451291774137989569263228187316199185', '-2776707353947863935313425463872380342255574023352080451291774137989569263228187316199185', 45, 3);
    t('84870733020575140260092508553754661695.368114319072', '84870733020575140260092508553754661695.36811431907190335719', 12, 4);
    t('-83503440279939.973261442707162100212414583470064391568958643329', '-83503440279939.9732614427071621002124145834700643915689586433289964', 49, 0);
    t('-0', '-0.00000000000000000000000000000000000000000000000593635309628535746', 10, 2);
    t('0.00001', '0.0000000000000000000000000000000000000000000005286316917406446218605705578095483616474', 5, 0);
    t('0', '0.000000000000000000000000000000000000000008', 21, 1);
    t('-0.000000000000000000000000000000896870792', '-0.000000000000000000000000000000896870791266', 39, 0);
    t('6740911664689729', '6740911664689729', 17, 0);
    t('-0.00000690323521226292545282', '-0.0000069032352122629254528255683555238', 26, 1);
    t('-0.00000000000000000000913384438905427651481080851110444839', '-0.000000000000000000009133844389054276514810808511104448390238589693895571906', 56, 4);
    t('-37939815047550295525500362964223610089693995885662719445639226194590163762795367', '-37939815047550295525500362964223610089693995885662719445639226194590163762795367', 40, 2);
    t('-0', '-0.000000000000000003', 3, 6);
    t('0.000000000000000000000000000000005425', '0.000000000000000000000000000000005424109435043055109323846477896259', 36, 2);
    t('-0.00864924723907831760464760343418947', '-0.00864924723907831760464760343418946520325901534', 35, 6);
    t('0', '0.0000000000000000005374923', 14, 5);
    t('11960649002', '11960649002', 11, 1);
    t('6205972.6375129878468500799969491683169898210504634996318', '6205972.6375129878468500799969491683169898210504634996317075719512025074532662', 49, 0);
    t('-0.0000327696652066428583356559', '-0.000032769665206642858335655876746559995630835559846294876125981984', 28, 0);
    t('-7106283670343099889326720878025602271833620303', '-7106283670343099889326720878025602271833620303', 13, 1);
    t('-0.000000000000000000000001', '-0.000000000000000000000000000000009', 24, 3);
    t('-0.00001', '-0.0000000000000008', 5, 3);
    t('-6133916442630955317842810214.86821373617414489008214917421152', '-6133916442630955317842810214.868213736174144890082149174211518886339289911249129630570965984027', 32, 5);
    t('-23881.773402249', '-23881.7734022481', 9, 3);
    t('-7943072313454728932408805268809876911677807547292409069069320924', '-7943072313454728932408805268809876911677807547292409069069320924', 53, 3);
    t('-0.0000000000000000000000005184238277412160705490261079943', '-0.000000000000000000000000518423827741216070549026107994302914', 55, 6);
    t('-6502553383150595', '-6502553383150595', 18, 6);
    t('-0.00000000000000000000000000000000690227527', '-0.00000000000000000000000000000000690227526594856364875794508690356161782941768', 41, 6);
    t('-0.056303708323832222458882734633919', '-0.0563037083238322224588827346339194', 33, 5);
    t('146170653842047607933686838505110978214296', '146170653842047607933686838505110978214296', 43, 4);
    t('-6440173.72698277131181168466528058795104036685676747477', '-6440173.726982771311811684665280587951040366856767474768344', 47, 3);
    t('435822485701161806484754025865621399846162883184293991478109154521248459436', '435822485701161806484754025865621399846162883184293991478109154521248459436', 4, 3);
    t('-0.000000000000000037815186790363', '-0.00000000000000003781518679036284917443497339141496107319487738088629444641566971505', 30, 4);
    t('565930492095544460683321420785084911874956.4375793957430882229833768389375836010018', '565930492095544460683321420785084911874956.43757939574308822298337683893758360100179', 40, 2);
    t('0.00000000000000000000004152', '0.000000000000000000000041522594866881469408628302850804768779275384292937914047860237548127164352', 26, 6);
    t('-17034881662549539727165518222983064951722670573249', '-17034881662549539727165518222983064951722670573248.5125902187644045205097633', 0, 6);
    t('276253842420715594878582609882869589958601896297494973', '276253842420715594878582609882869589958601896297494973', 36, 3);
    t('-2486857760541953491833813995388.9805692056860091745889', '-2486857760541953491833813995388.9805692056860091745889748664644781', 22, 1);
    t('-0.00000000001', '-0.00000000000000000000000000000000001343364842101594718687600286348068', 11, 3);
    t('0.00000000000000000000000000000000000000000000007224', '0.0000000000000000000000000000000000000000000000722473881816060373641', 50, 1);
    t('142224026751646621535876997526167808747124874491.8218160208781636', '142224026751646621535876997526167808747124874491.82181602087816353785918805465', 16, 2);
    t('2054722951827331212352617437865565678128465963', '2054722951827331212352617437865565678128465963', 46, 0);
    t('-0.00000000000000000259639195619', '-0.000000000000000002596391956187822298458567687533150431', 29, 0);
    t('-0', '-0.0000000000000000000000000000000000000000000000004783091262527', 35, 1);
    t('-0', '-0.00000000000000000000503086613010142201530941795090898678065118486089549316623317411283716858', 6, 1);
    t('-679.96730494488662265292122816939430658266', '-679.96730494488662265292122816939430658266', 38, 6);
    t('-4377850752966056', '-4377850752966056', 6, 5);
    t('445003949563154132133680872348175496003398462600208906574865645004', '445003949563154132133680872348175496003398462600208906574865645004', 39, 2);
    t('-0', '-0.00000000000000000000000000000000000000001178758870852233489202054', 33, 6);
    t('91936091236094033738149099398253993909296246976317855675164622956828', '91936091236094033738149099398253993909296246976317855675164622956828', 40, 4);
    t('0', '0.00000000000000000000000000000000000003', 31, 1);
    t('-898629194744631919581491799266418977651679996825530874', '-898629194744631919581491799266418977651679996825530874', 34, 3);
    t('0.000003467700718336729714643811584256', '0.0000034677007183367297146438115842561720677348016451716438690590079270204', 36, 3);
    t('-22546644480772822606187389406474962988389256779810505350888', '-22546644480772822606187389406474962988389256779810505350888', 52, 4);
    t('-55082020290223830995673899832810918291437742407509337067868', '-55082020290223830995673899832810918291437742407509337067868', 2, 0);
    t('1', '0.00000000000000000000000000000000000000007958953405109879', 0, 0);
    t('-6621241335299355957642029172.6', '-6621241335299355957642029172.6111786502352678896678324397010972489567', 1, 4);
    t('-0.0033627', '-0.00336269742972398707755429219378413396', 8, 0);
    t('22743478716848471813054723697837.9399566865524', '22743478716848471813054723697837.9399566865524005957435185765916723113785', 13, 3);
    t('0.0000000005992973504613566123673628822458384541', '0.0000000005992973504613566123673628822458384541', 46, 3);
    t('1042630777875163117448109887506935525', '1042630777875163117448109887506935525', 15, 4);
    t('221830139371014042990375785475202619672388755376281035547439322905698500192065908918841004849327', '221830139371014042990375785475202619672388755376281035547439322905698500192065908918841004849327', 1, 4);
    t('-77589.796358', '-77589.7963584873098058997214465516938135824363', 6, 1);
    t('0.000000000001819253160967', '0.0000000000018192531609674821174014441217356', 24, 5);
    t('0', '0.00000000000000000000000000000023716920156897944671996308882014675', 13, 5);
    t('-44856.778787975729305979715', '-44856.77878797572930597971463489707665657021962477424374876719956949220865646499239709551116', 21, 5);
    t('-824225425379483553475089927776690875947364334175898675', '-824225425379483553475089927776690875947364334175898675', 2, 0);
    t('87122038423032466553825622664295676697747152041634317', '87122038423032466553825622664295676697747152041634317', 31, 3);
    t('-0.000000000000000070582005393075667', '-0.00000000000000007058200539307566721824589217372033295032', 33, 6);
    t('489167543699462369.29206154', '489167543699462369.2920615397068117518054274410906616895686747969646065180181122832961', 8, 4);
    t('0.000000000000000000000000000047024525824', '0.00000000000000000000000000004702452582390298338191679929233670770515206599960650313812431598721', 40, 2);
    t('0.000000000000000000000000015966925641239639', '0.000000000000000000000000015966925641239638324777906483140603500193', 42, 0);
    t('0.000000000000000000000000000000000001', '0.000000000000000000000000000000000000000072802411951783460267416348', 36, 0);
    t('-7140703693356903422350486769685753717988557066347312584825633769978886.3629', '-7140703693356903422350486769685753717988557066347312584825633769978886.362931', 4, 2);
    t('1736062301921704.3544399700210783', '1736062301921704.354439970021078301187861306249924535576', 17, 4);
    t('2', '2', 2, 6);
    t('0', '0.00000000000000000000000000000000000000000000000024894143738651545275937572115200791168651', 34, 6);
    t('-68729847089572652805731356629429656670148403900488325598543', '-68729847089572652805731356629429656670148403900488325598543', 59, 6);
    t('-346930759824.8932559917769593999', '-346930759824.893255991776959399952245588688', 19, 1);
    t('-4340547831472081803462962486388614438943892232', '-4340547831472081803462962486388614438943892232', 48, 1);
    t('61616001329905068913964', '61616001329905068913964', 16, 0);
    t('0', '0.000000000000000006', 5, 5);
    t('-49307863225330198431734', '-49307863225330198431734', 13, 4);
    t('298024701456973658362011859788351474', '298024701456973658362011859788351474', 30, 5);
    t('-425424728525125303177668913491238360367678741', '-425424728525125303177668913491238360367678741', 36, 2);
    t('24056413149115583034277863478145', '24056413149115583034277863478145', 33, 5);
    t('-68662564068039829470775924161168787917897062354279632533471702872755.986663', '-68662564068039829470775924161168787917897062354279632533471702872755.9866639308147823', 6, 2);
    t('-27101699509736837178436357.31', '-27101699509736837178436357.3100268139', 4, 1);
    t('-0', '-0.00000000000000000000419594981553879981941810661', 1, 4);
    t('273877393288860727439964596728486053482512130635.909', '273877393288860727439964596728486053482512130635.90913019', 3, 1);
    t('102473879333862989052456171039164098', '102473879333862989052456171039164098', 15, 4);
    t('-39381337887967273.903118568586532771031839637750510080526', '-39381337887967273.90311856858653277103183963775051008052576239248866475699378', 39, 3);
    t('-3388998591124688079', '-3388998591124688079', 13, 5);
    t('-150567903893972682463213974956', '-150567903893972682463213974956', 16, 4);
    t('-0.0000000000000000000000000000000000000000493095312761862', '-0.0000000000000000000000000000000000000000493095312761861866451073373325095600742298256276', 55, 0);
    t('-7953667604', '-7953667604', 5, 1);
    t('4604171467545769381273154343552810484270535161', '4604171467545769381273154343552810484270535161', 40, 3);
    t('590588071250370946139445102570765117263428130058924147992458327492312529138', '590588071250370946139445102570765117263428130058924147992458327492312529138', 23, 5);
    t('-480177823684830812634801592253452743963534397293491478160746754380654483824166', '-480177823684830812634801592253452743963534397293491478160746754380654483824166', 50, 2);
    t('-0', '-0.0000000000000000000000038859594862', 21, 5);
    t('0', '0.00000000000000000000748620339565797', 8, 5);
    t('0.00002499835647695345918605661945571070775336188', '0.00002499835647695345918605661945571070775336187978592874439346863', 47, 0);
    t('4159973699834074325565638', '4159973699834074325565638', 4, 5);
    t('0.000000000000000000000055715176658', '0.00000000000000000000005571517665842097957139681288363708572', 33, 6);
    t('0.00000001', '0.0000000000000000000000000000000000041593906324710134345258956249480933017685', 8, 0);
    t('278216706345217684734', '278216706345217684734', 4, 0);
    t('-515039538585', '-515039538585', 12, 5);
    t('-5156282216060787005389201808449011961455755', '-5156282216060787005389201808449011961455755', 32, 1);
    t('1771197246191236979629384473461.83606571991', '1771197246191236979629384473461.8360657199109008737', 11, 1);
    t('0.0011131469353', '0.0011131469352378249447935166749689277812743785507290928314830661852468909137787363822177', 13, 0);
    t('92115423409743', '92115423409743', 12, 5);
    t('858.2661281409167634547302646312611320956987', '858.26612814091676345473026463126113209569860657925193123562', 40, 2);
    t('-1', '-0.0719443', 0, 3);
    t('-0.0000000000000009', '-0.00000000000000081331379483643709', 16, 0);
    t('0.00002697436643419847452779353409412459199', '0.0000269743664341984745277935340941245919918623531756557204282278703617', 41, 5);
    t('0.00000000000000000000001', '0.000000000000000000000000000000000000000000023801773695682720261287808392663674484132464627', 23, 0);
    t('-0.00068344382770949159674695980739380635445', '-0.0006834438277094915967469598073938063544463198925827927897963802449983538', 41, 4);
    t('0', '0.0000000000000000000000000000000000000000000000050279214787679199305', 11, 4);
    t('-588947409514785639282485632722383004287882037069573499096951.3145927013550028697730488', '-588947409514785639282485632722383004287882037069573499096951.31459270135500286977304875961', 25, 6);
    t('-0', '-0.00000000000000000000000000000000186436816132981351522904214050221580878', 3, 4);
    t('-2372.6417339151826508053284670230939641484', '-2372.6417339151826508053284670230939641484398', 37, 1);
    t('62178674327740811042508553727964', '62178674327740811042508553727964', 27, 5);
    t('0', '0.0000000000000000000000000000000547293246671362566489950300676458220362775126945519', 7, 4);
    t('49251652967453539726484585223974655821592759', '49251652967453539726484585223974655821592759', 31, 5);
    t('-296495898199.58115237407948666112', '-296495898199.5811523740794866611232178652873298488744263', 20, 6);
    t('-95652710114780760171030049795936499140918623272399896688565724875008683384093', '-95652710114780760171030049795936499140918623272399896688565724875008683384093', 57, 1);
    t('0.00000000000058426002118757843091739264518599980607', '0.00000000000058426002118757843091739264518599980606639684373685326021916693294', 50, 2);
    t('2772843216253911884822930634835873229720748.35205347294264485414722818229074309092764957', '2772843216253911884822930634835873229720748.35205347294264485414722818229074309092764957', 45, 1);
    t('0', '0.0002', 0, 1);
    t('0.0000000000000000000009', '0.0000000000000000000009', 22, 4);
    t('-0.00000063377460212762032425176962047901364151137812855454938', '-0.000000633774602127620324251769620479013641511378128554549388907545372024969906304253032', 59, 1);
    t('-0.00000006646203569453454444708452552277428018333642748745699', '-0.0000000664620356945345444470845255227742801833364274874569813785409098392133873982373960550117596', 59, 0);
    t('0', '0.000000000000000000000000000000000000000000018538496533514574221730165720097', 31, 5);
    t('128484998383401376820534767951931012435370666908970446', '128484998383401376820534767951931012435370666908970446', 44, 1);
    t('4387738573794731066916336352026833902940795.8852740466656380585076098286734723', '4387738573794731066916336352026833902940795.8852740466656380585076098286734723472438018787975', 34, 5);
    t('-0.00000000000000000003405365175859259155568376', '-0.0000000000000000000340536517585925915556837608377203983174081729250328', 44, 6);
    t('-187112408058.7234570771077', '-187112408058.723457077107737466758', 13, 5);
    t('531063677635775810446921905944792595119327655554346497', '531063677635775810446921905944792595119327655554346497', 21, 1);
    t('-0.0000000000000000000000006791908058045691073817', '-0.00000000000000000000000067919080580456910738169344941664639835178052777348953964845369', 46, 0);
    t('307899704537986547769348847994164364550840818783793849762539', '307899704537986547769348847994164364550840818783793849762539', 14, 4);
    t('-435801778695359687591423580.427694338967317762706300283493834282886549420729852', '-435801778695359687591423580.427694338967317762706300283493834282886549420729852017', 52, 5);
    t('-0.00000000003917777044145525747611380850933123472924171119', '-0.000000000039177770441455257476113808509331234729241711186132211007540710149907', 56, 6);
    t('62928039693272623682271.27541008736912709308116480951633335017303574647668632724622', '62928039693272623682271.27541008736912709308116480951633335017303574647668632724621535388', 59, 5);
    t('-0.0000000000000000000000000000000000000000001', '-0.000000000000000000000000000000000000000000000007000182632515075', 43, 0);
    t('907824547333163182771486955205512585202844466110951892146887907879200198061176', '907824547333163182771486955205512585202844466110951892146887907879200198061176', 24, 1);
    t('-1544386137499412381886438484', '-1544386137499412381886438484', 25, 6);
    t('-0.0000000000000000000000000076', '-0.00000000000000000000000000768056506477829996048633181519', 28, 2);
    t('4890192706601958001267805328506308744627862445358671612', '4890192706601958001267805328506308744627862445358671612', 43, 4);
    t('-0.001', '-0.000000000000000000000000000000000000000003024424622329171771637603264679763397312441', 3, 0);
    t('-0.000000232115524236009432926', '-0.0000002321155242360094329258', 27, 0);
    t('49465993391406248092231222.0801712487751425033643', '49465993391406248092231222.08017124877514250336430361542116940315722', 23, 3);
    t('4180158455264386144163419573028038177659222536720952339274298247333816505172', '4180158455264386144163419573028038177659222536720952339274298247333816505172', 45, 3);
    t('-0.000000000000000000000000000060254658062', '-0.0000000000000000000000000000602546580626641397565454810564671579837553', 39, 1);
    t('0.0000019257899344941444964', '0.00000192578993449414449637565345019538269354146347', 25, 0);
    t('0', '0.0000000000000007', 13, 4);
    t('-0.000000000000000000000000000000000000007', '-0.0000000000000000000000000000000000000077279149653677340608947700558217181', 39, 2);
    t('-448376506822431348847333799886', '-448376506822431348847333799886', 0, 1);
    t('69867122143002481442816258286999167805918307181607743074904979008771841710463824003', '69867122143002481442816258286999167805918307181607743074904979008771841710463824003', 27, 3);
    t('2178943445964550611', '2178943445964550611', 17, 0);
    t('588387425474973011615137458350384132121857043644620739740517932.762', '588387425474973011615137458350384132121857043644620739740517932.76207074', 4, 3);
    t('0.000000000000000891030290232918573851', '0.0000000000000008910302902329185738510097719', 36, 1);
    t('3082156529442568521513326353208207517976381714116608.9434238802561825638640926017', '3082156529442568521513326353208207517976381714116608.943423880256182563864092601674468', 28, 6);
    t('-269047218334585112717070569715', '-269047218334585112717070569715', 20, 3);
    t('794322372403220770530191708', '794322372403220770530191708', 17, 4);
    t('-0.0000088706942874980615965377284', '-0.0000088706942874980615965377283623276238900639610011564988041972', 31, 3);
    t('-18762392383129800103435772599858039146861024540145071643', '-18762392383129800103435772599858039146861024540145071643', 39, 0);
    t('0.00000000000000000005418504797326945324468962601702987844065', '0.000000000000000000054185047973269453244689626017029878440651506342739333005455074', 59, 6);
    t('0', '0.00000000000000000000000000000001455552106862800727428835727449445351956', 2, 6);
    t('-845184120395843729177378264132032391293931221704496190873', '-845184120395843729177378264132032391293931221704496190873', 3, 2);
    t('-22928704982133259844820264310150045267.8329497378477851667', '-22928704982133259844820264310150045267.8329497378477851666787234740604182218327262831065786', 19, 0);
    t('-753282787421295813612681585527569219114746603', '-753282787421295813612681585527569219114746603', 6, 3);
    t('-0', '-0.000000000000000000000000000001114691322399392322421', 0, 5);
    t('-4470337.856339789341346743328195402056', '-4470337.8563397893413467433281954020555766465023528610798962516623771308', 30, 4);
    t('-30259936572085000771571577042343302258005437254494921143265833920775250590681264568', '-30259936572085000771571577042343302258005437254494921143265833920775250590681264568', 37, 5);
    t('0.00000000046019029818715', '0.000000000460190298187149944541124538982908213632543612794526400527723071339103700250052883', 24, 2);
    t('488789246040736024354688778283025974423012332029061586465391.05657634342713960705278', '488789246040736024354688778283025974423012332029061586465391.0565763434271396070527794705650531927', 23, 0);
    t('-6754886810347257311369815292802990704436078888019328596021676996446581588371838421302183', '-6754886810347257311369815292802990704436078888019328596021676996446581588371838421302183', 17, 5);
    t('-2219143949146.543192979755734', '-2219143949146.5431929797557336049181837818262147309272936749984443073357094259', 15, 3);
    t('25642596132200847246923459489123979011650156', '25642596132200847246923459489123979011650156', 34, 4);
    t('-869559474794910580.4702422545', '-869559474794910580.470242254580425701508344203389450340088616045503344062196', 10, 1);
    t('873581632317363936846913522953208947257070629891494175.1987423890811633389352647', '873581632317363936846913522953208947257070629891494175.19874238908116333893526471237141433112991286', 25, 4);
    t('-149595188298843646987316097444673881', '-149595188298843646987316097444673881', 36, 5);
    t('1314405778222365740681306509390140309', '1314405778222365740681306509390140309', 23, 5);
    t('-43669276969497246217249931136079143593668679948650792993674045.547575505339753429132018021618569924', '-43669276969497246217249931136079143593668679948650792993674045.547575505339753429132018021618569924', 36, 2);
    t('778437789780670221576981092962324328921406844694078635347625850330135239051380.593313827', '778437789780670221576981092962324328921406844694078635347625850330135239051380.5933138270296', 9, 3);
    t('-0', '-0.000000000000000000000670532931448014402320805028695397125747680104246730672291510758362070539', 0, 4);
    t('-269860226.555168110686802', '-269860226.555168110686802406496811651255242690733933523277397815449890266344932488292555', 15, 6);
    t('-0.00000000000000000000000000000000000001', '-0.00000000000000000000000000000000000000000000001', 38, 3);
    t('8255334830426368087757667915495316773111372', '8255334830426368087757667915495316773111372', 20, 4);
    t('-498.288', '-498.2877551595647284860735236925190739383420573843285964682707105606057458671350918750993', 3, 6);
    t('90275906866311700418329778663.453021601770244972374', '90275906866311700418329778663.4530216017702449723738726140283616676480577425596798642530312', 21, 4);
    t('-58249061380532235.683643040770210682572744295', '-58249061380532235.68364304077021068257274429468', 27, 4);
    t('-0.00000000000000000005874732963', '-0.00000000000000000005874732963903011696675103873912247327240795349388835921', 29, 1);
    t('-0.00000000000000000000000891849846684592', '-0.00000000000000000000000891849846684592', 39, 0);
    t('-317305030488680572949072951462138920358812787719661772458316652068525445', '-317305030488680572949072951462138920358812787719661772458316652068525445', 58, 0);
    t('154344118085858988045736881532862450933453810805000737193200.8311028461495237875501818767647353', '154344118085858988045736881532862450933453810805000737193200.8311028461495237875501818767647353', 34, 0);
    t('-0.00000000000000000006151217562036449', '-0.00000000000000000006151217562036449913306415978013249544', 35, 2);
    t('0.00000000000000000000000000000000000000635481634', '0.000000000000000000000000000000000000006354816333226622007293085007542785238974690455300658', 47, 0);
    t('-72104134714968004316834628241028266449839170680955147176037571084261', '-72104134714968004316834628241028266449839170680955147176037571084261', 54, 6);
    t('449011810931249785286308861907624', '449011810931249785286308861907624', 8, 0);
    t('-5344557217.952340951684183652094', '-5344557217.9523409516841836520936077337920245188603000165', 21, 0);
    t('26763304918.6071132280297527249552897014', '26763304918.607113228029752724955289701414261410830249608167321604531533418376564217', 28, 5);
    t('-5.9354', '-5.935401329955359917419849571921209501551', 4, 2);
    t('0.0000000000000002842770147295499709278', '0.000000000000000284277014729549970927748689066908625707440474139245449815727102349121121703585184', 37, 2);
    t('-0.00000006536334906818', '-0.00000006536334906818007245367103563161239', 21, 1);
    t('3987294744.961449803344679269558748056110724042180299042772', '3987294744.961449803344679269558748056110724042180299042771934156686853', 48, 0);
    t('3670550979301523677411452367600923241928927', '3670550979301523677411452367600923241928927', 16, 6);
    t('8186460244315446238612369289', '8186460244315446238612369289', 14, 5);
    t('0.0000000000001205038039008310782739771735999409', '0.0000000000001205038039008310782739771735999408450010198815947147909379151151097718078660678', 46, 2);
    t('17790773861566.9127491793774523928', '17790773861566.9127491793774523927742086885', 19, 6);
    t('-45973691512980904832772447657128646184729588347720906399317356124291', '-45973691512980904832772447657128646184729588347720906399317356124291', 6, 0);
    t('-0.000052578846594552360365003261412', '-0.00005257884659455236036500326141292751072228323274374276273184032573733734930586182920538869496014', 33, 1);
    t('-8451030.38187527151851343992271353120147534273330863994132400507052', '-8451030.38187527151851343992271353120147534273330863994132400507052', 59, 5);
    t('-6913729808236204745703016735366143142854267317875', '-6913729808236204745703016735366143142854267317875', 48, 6);
    t('-0', '-0.000000000000000000515816546619846222740519650659244354', 14, 1);
    t('-0.0000000000000000000000000059770244313969237968717', '-0.0000000000000000000000000059770244313969237968717414549698422124221759010405670994', 49, 2);
    t('727784599805850866261167964.5891386990587296184', '727784599805850866261167964.5891386990587296183810717', 19, 0);
    t('60455336921761571297330062708834209593753553971798623628973754356', '60455336921761571297330062708834209593753553971798623628973754356', 34, 5);
    t('-0', '-0.00000000000000000000000000000000000000000000002', 14, 2);
    t('-0.1', '-0.0000000000000000000000000000000000000000268246264078205', 1, 3);
    t('7867937945277457921518274222617202749835981396620016683948282927454552756', '7867937945277457921518274222617202749835981396620016683948282927454552756', 45, 5);
    t('10798101545970440531103848', '10798101545970440531103848', 17, 5);
    t('-662862030859665015252209082020387338342609', '-662862030859665015252209082020387338342609', 41, 6);
    t('-44509783950769310032819987921393220034667288606302636', '-44509783950769310032819987921393220034667288606302636', 10, 2);
    t('0.000721', '0.000721015686405377087216290192146931451368869346018108792313779', 7, 3);
    t('-429429431898.92798147529486', '-429429431898.9279814752948602', 15, 5);
    t('0.000000000001', '0.00000000000000000000004107', 12, 0);
    t('-1523647444816999427298977167480522782325629813153844', '-1523647444816999427298977167480522782325629813153844', 27, 0);
    t('159233142565064172480281097992202157794955635367088722816913405230709', '159233142565064172480281097992202157794955635367088722816913405230709', 52, 3);
    t('17734356336750552208117881', '17734356336750552208117881', 13, 3);
    t('0.00000000000000000010516440931', '0.000000000000000000105164409309278409833867961542984330406756444964971586826869142', 29, 0);
    t('-0.000000005607865232286053212511', '-0.0000000056078652322860532125106865502282497985819996577284499049194079530138833114274376005211', 30, 5);
    t('-1848824671208162.036118636322531620559621761039257152', '-1848824671208162.0361186363225316205596217610392571519859349877420578173659325511361795', 36, 6);
    t('474121748.258771413359324788', '474121748.258771413359324788219351236297274357376611118786304644016953198682051204228387618', 18, 5);
    t('0', '0.000000000000000000000000000000000000002', 9, 6);
    t('-4082518153742954140701880216101112933681579', '-4082518153742954140701880216101112933681579', 2, 6);
    t('485561797431709913', '485561797431709913', 8, 5);
    t('-0.00000005169583157473170767787', '-0.000000051695831574731707677872752277227315759207157035374591647416039798341232', 29, 5);
    t('-31444130984617191013282867777267050079417810443829', '-31444130984617191013282867777267050079417810443829', 43, 3);
    t('8800583786371595440869107', '8800583786371595440869107', 4, 6);
    t('-8815.8895192223252405825465547234263054813143616472980391', '-8815.889519222325240582546554723426305481314361647298039143773292329917612441361288012832403085661674', 52, 4);
    t('24193532123207943347431137579390758056362200617916927652557', '24193532123207943347431137579390758056362200617916927652557', 8, 6);
    t('-1.337926538823846889672640236908474426590700086', '-1.3379265388238468896726402369084744265907000856185156132950540139555844532982184035787153238', 45, 4);
    t('1558843239.299544313206143220207974', '1558843239.299544313206143220207974153124813171090852680122860660088788288026393496', 24, 3);
    t('-59589972311323891941872406112978568670975162132927304.08231126490356', '-59589972311323891941872406112978568670975162132927304.08231126490356049195534195298897388522', 15, 1);
    t('-743294747766629171048480641662764189036069326862056657538375964541286934', '-743294747766629171048480641662764189036069326862056657538375964541286934', 5, 4);
    t('22265', '22265', 3, 4);
    t('960209483951475422891115334433638881085096.26588580775357747606657536401221311373248396571', '960209483951475422891115334433638881085096.265885807753577476066575364012213113732483965707', 47, 0);
    t('44550253.862798116411937890173846464380254722540077', '44550253.862798116411937890173846464380254722540076612180961686217459396341111804577388882586249791', 42, 6);
    t('43402590248081760572362131719223.753209477296474800070796246196620361', '43402590248081760572362131719223.753209477296474800070796246196620360891', 36, 2);
    t('23302193755473927447547.0400871612708276023203865329170219302115999767999', '23302193755473927447547.0400871612708276023203865329170219302115999767999489', 49, 4);
    t('0.00000000000000000000000809577123038853204379', '0.0000000000000000000000080957712303885320437905131515608754123689963443056574402556634345', 44, 4);
    t('5757764236108545477867413983', '5757764236108545477867413983', 19, 3);
    t('0.0000000000000000000000000000000000000009', '0.0000000000000000000000000000000000000009', 40, 0);
    t('3559280912885112533280423534223324103634509100163633301839596057', '3559280912885112533280423534223324103634509100163633301839596057', 59, 1);
    t('-417080158865542582854655944868550771791.046398310335547707041055645769929951581637541553712', '-417080158865542582854655944868550771791.04639831033554770704105564576992995158163754155371200816', 52, 6);
    t('89983719097904808464', '89983719097904808464', 1, 4);
    t('-735429992378182205630619451123590.3947291140982175810651340488182275175', '-735429992378182205630619451123590.394729114098217581065134048818227517476', 37, 6);
    t('419077969568256244823720087754694848763408887642073576695421791', '419077969568256244823720087754694848763408887642073576695421791', 36, 1);
    t('191461.889031503960878427762073940304873639', '191461.88903150396087842776207394030487363817696768685154419315097935845754622181742599271', 36, 2);
    t('-0.00000000000000000000062342', '-0.000000000000000000000623414', 26, 3);
    t('127194045893446729075351171047', '127194045893446729075351171047', 21, 0);
    t('-229500965923850.70051429223116003250968598721393599828363661825696333', '-229500965923850.700514292231160032509685987213935998283636618256963339786490679495749782904303115', 53, 1);
    t('-0.0000000000000000001', '-0.0000000000000000000056041231775239402844541555853241836954476161592639531971958216635667906218', 19, 3);
    t('-0', '-0.00000000000000000000000000000000000000000000000037582095633137036754003160897330055440194309661', 35, 5);
    t('3374255136237325085491942038388147404432627458905669250744778', '3374255136237325085491942038388147404432627458905669250744778', 45, 0);
    t('7690224951153160774526032743679621', '7690224951153160774526032743679621', 33, 6);
    t('0.00000000000000000000000000000000017074840686264748628337821', '0.000000000000000000000000000000000170748406862647486283378213576419477915575', 59, 5);
    t('-0.00000000004637857649826847069795159309827675', '-0.000000000046378576498268470697951593098276745844317758', 44, 5);
    t('0.00017652709951498133638383919005461463529305123', '0.000176527099514981336383839190054614635293051230467643495742837724131', 47, 1);
    t('-0', '-0.000000000000000000000000000000000000000000000112634125418', 33, 4);
    t('723368046640065130294742958718721773826602583088486440698866940503225379031577817', '723368046640065130294742958718721773826602583088486440698866940503225379031577817', 51, 4);
    t('-50933665584184852038671079863303718636397390288718740018317103418.102', '-50933665584184852038671079863303718636397390288718740018317103418.102293292287286568351413551802', 3, 5);
    t('-2410200702497', '-2410200702497', 11, 1);
    t('0', '0.0000000000000000000000000000000000000000000042239344371306878587914753063390073941', 34, 4);
    t('-0.00000000000000000000000000000000000000000000001', '-0.0000000000000000000000000000000000000000000000000575180898854911', 47, 0);
    t('0.000000000000000004344458301', '0.000000000000000004344458300844696324', 27, 5);
    t('36415144993270477751194428581026117927500927.44375086964836279651713', '36415144993270477751194428581026117927500927.443750869648362796517124098901167909583083796664632', 23, 0);
    t('5334215483112576666595260269852476258881', '5334215483112576666595260269852476258881', 11, 2);
    t('3572677626411797212153520390351.131380872979421133958812687842550117369', '3572677626411797212153520390351.131380872979421133958812687842550117369055722382', 40, 1);
    t('-0', '-0.000000000317959455183002008629386273613319488458668668911799807504500387889375132126574196359', 2, 1);
    t('0', '0.00000000000000000000000000508716881203357016907853303274028327363526105723619349316', 9, 5);
    t('0', '0.00000000000000000000000000000000000000000000007059146600634404411', 2, 6);
    t('0.00000000000000000008044178973467803785', '0.000000000000000000080441789734678037845949859474890695888', 38, 5);
    t('0.00055142638660941844622938085879706786353818237626618', '0.0005514263866094184462293808587970678635381823762661734377953648597433', 53, 0);
    t('-17599444586797881418318563414502812139457', '-17599444586797881418318563414502812139457', 29, 1);
    t('1134429695712029417607873744122149717054918253194205124127', '1134429695712029417607873744122149717054918253194205124127', 56, 5);
    t('-0', '-0.0000000000000000000000000000032833397936417991775542735318772256', 18, 4);
    t('-0.00000000000082748682928206867048022608063733', '-0.00000000000082748682928206867048022608063733', 44, 0);
    t('-892190137424331421784517600329772157761277945477908134895153411935646898622686045781365887203805522', '-892190137424331421784517600329772157761277945477908134895153411935646898622686045781365887203805522', 21, 2);
    t('-57225378653330495581042999105043387285455715448557023450921912.731', '-57225378653330495581042999105043387285455715448557023450921912.730815794217451434674350759', 3, 3);
    t('88944978837600241675413451540.41662447490467530719484465416514', '88944978837600241675413451540.41662447490467530719484465416513364729285517299279519760671836409', 32, 0);
    t('-0.00566575519125872879638699056271355696758097', '-0.00566575519125872879638699056271355696758097875409951260368210651006', 44, 1);
    t('52101710101744937632626526544984055.2507495489669041508372470444630234511783474876012526', '52101710101744937632626526544984055.25074954896690415083724704446302345117834748760125260024', 54, 6);
    t('25552542841746347884157299301699166856285838610487.074735318379', '25552542841746347884157299301699166856285838610487.074735318379', 13, 4);
    t('-0', '-0.00000000000000000000000008442227169', 0, 5);
    t('-0', '-0.000000000000000000000000000000000000000000000024981374014605917033767473869314720387745536', 7, 2);
    t('-480402927391768064318296499265102782166804733100958844081491000.164525046348783775063556224', '-480402927391768064318296499265102782166804733100958844081491000.16452504634878377506355622448905', 27, 6);
    t('-83963278511727947.84244986971165729356', '-83963278511727947.8424498697116572935579418275332796753821475270524', 20, 3);
    t('905516486335617027566122447589928241315669260318', '905516486335617027566122447589928241315669260318', 35, 4);
    t('6552963687555211427868764317.90444524', '6552963687555211427868764317.90444524617', 8, 1);
    t('-0.0000000000000000000000000000000000000296558323', '-0.0000000000000000000000000000000000000296558322254714710784144978368282230551424514', 46, 0);
    t('71233459565941813768643527698052038451814432324674330534.182221666', '71233459565941813768643527698052038451814432324674330534.182221665956151724061357615822583899564696', 9, 0);
    t('-0.000000000000000000000000000000000000000007692392', '-0.000000000000000000000000000000000000000007692392232033940021796786860993673', 48, 2);
    t('0.0000004142741301072051705173415526', '0.000000414274130107205170517341552625080742', 34, 4);
    t('0.00000000000000000000000000000049', '0.0000000000000000000000000000004844225578184253767706840696504802592', 32, 0);
    t('0', '0.00000000000000000000000000000000000000000000586004362384730418793774932852260308', 42, 5);
    t('0.000000000000000000000000000000000057244859768', '0.000000000000000000000000000000000057244859768035439766499099449028148846', 45, 5);
    t('-827141688560282316577471043529435271414984', '-827141688560282316577471043529435271414984', 27, 2);
    t('-368402916220866.36823893036045234503764', '-368402916220866.36823893036045234503764172264459272073949130478550198453350142034438121269721972', 23, 5);
    t('9103003038081670721316405', '9103003038081670721316405', 18, 6);
    t('-7852165677474979.5938719858816080949974229232217347241', '-7852165677474979.59387198588160809499742292322173472401655014999447181095216397', 37, 0);
    t('-6715185377564.74139538353174075359773259963669721907215595224', '-6715185377564.7413953835317407535977325996366972190721559522323030678428100131570087002950444421', 47, 0);
    t('1706349694.76350749557800040706127419362257045545428491995', '1706349694.763507495578000407061274193622570455454284919951774181697932220714269', 47, 4);
    t('78588826016299437743619705479876210302676979259827058565445', '78588826016299437743619705479876210302676979259827058565445', 30, 4);
    t('0.000000000001', '0.000000000000000000001', 12, 2);
    t('-2589582973458706776296834369656361634439983', '-2589582973458706776296834369656361634439983', 39, 0);
    t('1743441368337597162867911.2951052396023395109004737465489', '1743441368337597162867911.29510523960233951090047374654895392832378911231920027143594963', 31, 3);
    t('-73199678471', '-73199678471', 3, 0);
    t('0', '0.00000000000000000000000000000000000000000003', 40, 1);
    t('0.0000000000000000000000000000000004667447051117350698', '0.000000000000000000000000000000000466744705111735069864674296158720400075859650953', 52, 3);
    t('-0.000000000007594399656167355193566547921', '-0.0000000000075943996561673551935665479215549769984862601569756133629823648103539458', 39, 2);
    t('3.1652680597256250889013415555546409571597090799265', '3.1652680597256250889013415555546409571597090799264826416835791052587978940383354082674008433382', 49, 5);
    t('50502759777095272980458734330295487385.5034260436783855', '50502759777095272980458734330295487385.5034260436783854754464126887951593337', 16, 2);
    t('-349476928883536309320539866916636955806193888951300354548141473494386', '-349476928883536309320539866916636955806193888951300354548141473494386', 55, 1);
    t('841906075761357751506861189609878618918101934357106', '841906075761357751506861189609878618918101934357106', 25, 3);
    t('-59181751544129813715475076424907785097207348427962139893929324499', '-59181751544129813715475076424907785097207348427962139893929324499', 35, 2);
    t('-1521196888137155531754333457458022000831356615', '-1521196888137155531754333457458022000831356615', 41, 3);
    t('-0', '-0.000000000000000000000000000000000000009', 8, 4);
    t('-8289795629.1717218118944', '-8289795629.171721811894356069463885446122', 13, 0);
    t('-0.000000000000223889609679829899532021178692653007372269973', '-0.0000000000002238896096798298995320211786926530073722699726206380829817413519982043525', 57, 6);
    t('-7258143388738798481910452159781024994720650112469214442658774256937667959953614727386289', '-7258143388738798481910452159781024994720650112469214442658774256937667959953614727386289', 22, 2);
    t('-359558549295372353924483405244841350664470.360298499922130388371586275901555831355657233897572726855', '-359558549295372353924483405244841350664470.360298499922130388371586275901555831355657233897572726855', 58, 1);
    t('-0.000000000000000000000000000000000000000000237870554', '-0.0000000000000000000000000000000000000000002378705539415969705231398235316441614230302', 52, 0);
    t('0.2393241059673690394127252863371489989987293405009221276', '0.2393241059673690394127252863371489989987293405009221275481644601266835423187167949827392', 55, 0);
    t('-689590028428083520425141553232150041665855900447213740339177308861649878', '-689590028428083520425141553232150041665855900447213740339177308861649878', 18, 0);
    t('0.000000078110011766889375023084610838180269', '0.0000000781100117668893750230846108381802696891356733714996828302', 42, 1);
    t('-0.0000000739636485110881770771812970480784085432490937791404', '-0.000000073963648511088177077181297048078408543249093779140448174989076410926', 58, 6);
    t('839622336342426602772908498745047009924846092679316564429', '839622336342426602772908498745047009924846092679316564429', 57, 6);
    t('-0.0000000000000000000004456538488579841701862456826545336942', '-0.00000000000000000000044565384885798417018624568265453369416231186707532116823535364936', 58, 5);
    t('5457344694848495013.70758306793373443062303840408774687096091471', '5457344694848495013.70758306793373443062303840408774687096091470997296538132975451', 45, 6);
    t('278810146198025452859993052541559705412200658428450141592249279860660994', '278810146198025452859993052541559705412200658428450141592249279860660994', 53, 6);
    t('-0.0000000000000000000000000000001', '-0.0000000000000000000000000000000008', 31, 0);
    t('0.00000003870782543245692370508', '0.000000038707825432456923705079513311568', 29, 2);
    t('0.00000000000000028905839185125858911474408672313456633629', '0.00000000000000028905839185125858911474408672313456633629272919468390905836955324418317503957', 56, 4);
    t('-51357247691860499903562435980838.792767', '-51357247691860499903562435980838.79276727924860634529292348327', 6, 4);
    t('36984058670396140.944950463552746216537895105829140616432610621131124', '36984058670396140.9449504635527462165378951058291406164326106211311240265980364777353', 52, 3);
    t('4324247714235544003727336020912959171914181632133', '4324247714235544003727336020912959171914181632133', 26, 1);
    t('-0.0000000000000000000000000000000000000000000000008937166', '-0.0000000000000000000000000000000000000000000000008937166', 55, 1);
    t('-2442300682026632859154542947848984516683026198562824', '-2442300682026632859154542947848984516683026198562824', 31, 6);
    t('0.0000000000000000000040559', '0.000000000000000000004055940890427693898528647800483601', 25, 4);
    t('-0.0000000000000000000000000000000000000000000000005', '-0.000000000000000000000000000000000000000000000000533883534094325559019527373422252174198', 49, 1);
    t('-601022544723155145451207670010213497437.14164798', '-601022544723155145451207670010213497437.1416479757860807842495953660628318061595534', 8, 6);
    t('70433022521741511411711454.3981934256304734737098098', '70433022521741511411711454.398193425630473473709809802903058576643802930136306958583', 26, 6);
    t('10908799783188529757366993500335', '10908799783188529757366993500335', 21, 3);
    t('87532.4007713298943414092792987798224322373504067', '87532.400771329894341409279298779822432237350406676898594464428537143159348548268047779886855749206', 43, 0);
    t('-0.519371276718530514624846800754582116141529266132757', '-0.5193712767185305146248468007545821161415292661327570320784267894852821486214528653', 51, 1);
    t('9843596527442895375096439245255836073.70931062242142483683526808', '9843596527442895375096439245255836073.70931062242142483683526807533268520820713372314', 26, 0);
    t('0.001', '0.0000000000000058501267284656327224026025031332724569171393523546241494751676984130276', 3, 2);
    t('88204244847517941308668679833512765096788828649032102210342660386515548133512234443', '88204244847517941308668679833512765096788828649032102210342660386515548133512234443', 9, 2);
    t('-72474655403865728422919', '-72474655403865728422919', 25, 2);
    t('-64978799280092373.2357486192905225486909493417592884977935811274', '-64978799280092373.23574861929052254869094934175928849779358112736784', 46, 3);
    t('-3900549976428555791621079898632755518751734773176044166588963', '-3900549976428555791621079898632755518751734773176044166588963', 45, 5);
    t('-0.000000000000000000000000000082479', '-0.00000000000000000000000000008247901689204830195155306799530867952682774163311949867', 34, 6);
    t('0', '0.0000000000000000000000000338254327291878442621187444947666248001402914296565659898518000736', 11, 3);
    t('-0.00000000223134', '-0.000000002231341114998828837864161402405342720671858261823364908867', 14, 6);

    // rounding mode 7
    t('-4227925468727766007369594003870427.86133603669917314695806481136062429', '-4227925468727766007369594003870427.86133603669917314695806481136062429149585050333617751', 35, 7);
    t('0.00000000000000002542497815049223865530599542', '0.000000000000000025424978150492238655305995417803361817364414794153572487661433133', 44, 7);
    t('-52231667.228724632', '-52231667.22872463243810129366255', 9, 7);
    t('34077267364474763.91234860081333406989016992096282251', '34077267364474763.9123486008133340698901699209628225111620410327006274894046060867566994460704', 35, 7);
    t('5856401617744968221752230029308.1679768661242918708685642495558141063641674034', '5856401617744968221752230029308.167976866124291870868564249555814106364167403351653', 46, 7);
    t('23796511434278450866104444583677797487678290864898492636634411901159527', '23796511434278450866104444583677797487678290864898492636634411901159527', 49, 7);
    t('-0', '-0.00000000000000000001622749974913381', 8, 7);
    t('63101238791571926526656526888', '63101238791571926526656526888', 26, 7);
    t('119975362395419219788856351528506147606875253314643205183185766', '119975362395419219788856351528506147606875253314643205183185766', 37, 7);
    t('0.00000000000000028638823981166558781214435', '0.0000000000000002863882398116655878121443495241084176650123383124435451', 41, 7);
    t('-0', '-0.0000000000000646493344101476775', 4, 7);
    t('3727665898156798714702225969249372097434686068956916500942749774019588426536', '3727665898156798714702225969249372097434686068956916500942749774019588426536', 55, 7);
    t('409593411642458946972739997054469022719146.491026', '409593411642458946972739997054469022719146.491025521359231357', 6, 7);
    t('13404825779782337.686240571775407373089634963', '13404825779782337.686240571775407373089634962567081893468942784631003987766301470481820407103286888', 27, 7);
    t('661581721939770480643', '661581721939770480643', 0, 7);
    t('82201282103354391377948704606454107732823853693730778633', '82201282103354391377948704606454107732823853693730778632.679770307020989736730343202980605582', 0, 7);
    t('188489334274192797369124659824920031586552510451', '188489334274192797369124659824920031586552510451', 20, 7);
    t('-350373000061738168422367720730846856693416103657170981784.47171168192426', '-350373000061738168422367720730846856693416103657170981784.47171168192426', 14, 7);
    t('-371822329041221965657417973472237384192854529035779701345758861333.911', '-371822329041221965657417973472237384192854529035779701345758861333.91125259284165', 3, 7);
    t('11390.30218059', '11390.3021805893183128284582351157607918846791695198255651493155092893025770778456684', 8, 7);
    t('31051565651391', '31051565651391', 10, 7);
    t('3381673772647503796291366133103', '3381673772647503796291366133103', 24, 7);
    t('543390727353666255169063924082693597248278069190643870027', '543390727353666255169063924082693597248278069190643870027', 55, 7);
    t('799421038004579749157298710029180393', '799421038004579749157298710029180393', 9, 7);
    t('-0.0362677922831722959112878701030537328697742', '-0.036267792283172295911287870103053732869774154760781501898753153437796985370382553866', 43, 7);
    t('-0.00000000000000048840974104900508184409730782', '-0.00000000000000048840974104900508184409730782', 44, 7);
    t('90721186387234427125851848111061056765611723546', '90721186387234427125851848111061056765611723546', 39, 7);
    t('0.00000000000478987425349763447387328875752791010733067', '0.0000000000047898742534976344738732887575279101073306690331160727348729434112548998768887148899', 53, 7);
    t('15590509654436423421185143897507920201916474372180041409856165500105', '15590509654436423421185143897507920201916474372180041409856165500105', 30, 7);
    t('20.749566685219600998729078427300664316775', '20.7495666852196009987290784273006643167750133692770467181607553093280822', 39, 7);
    t('-0.000053459060231372', '-0.00005345906023137227399335260530503765672935', 18, 7);
    t('-0.0000000003951749339568017915631', '-0.000000000395174933956801791563099248827925205492855063788550039874086544657303197463564591312', 32, 7);
    t('8679339310554212663278369614448693.5454104727306164118703439386963341', '8679339310554212663278369614448693.545410472730616411870343938696334073', 34, 7);
    t('33955284728592200348127638127333126188369126163765426829286263844316', '33955284728592200348127638127333126188369126163765426829286263844316', 58, 7);
    t('-594503494522354581672092284773632917525938683919588064506930195373570676846343387.84', '-594503494522354581672092284773632917525938683919588064506930195373570676846343387.8422460203946', 2, 7);
    t('-0.000000000039360661299713', '-0.000000000039360661299713264121639691847249043522176561382860930019972250155868911494243265848', 24, 7);
    t('772954079526310860071529507548873740833690851501083576478349475089452727853278468874693', '772954079526310860071529507548873740833690851501083576478349475089452727853278468874693', 43, 7);
    t('0.0000000000000087453797413879451313', '0.000000000000008745379741387945131324297953447', 34, 7);
    t('5367560449328479105254540606212332013053671843660775766151', '5367560449328479105254540606212332013053671843660775766151', 48, 7);
    t('323195274385200326456762447968768', '323195274385200326456762447968768', 32, 7);
    t('-0.00666873745951635736846493280695487641727423448', '-0.00666873745951635736846493280695487641727423447512328693008180656151418503780844889026', 47, 7);
    t('-7926588167337833445785869654809557874339194445079754658952051495268464651', '-7926588167337833445785869654809557874339194445079754658952051495268464651', 52, 7);
    t('773169576814793176793181528896362236450108', '773169576814793176793181528896362236450108', 30, 7);
    t('-8.1575381767765005493134495350395', '-8.15753817677650054931344953503948694358980563360390907469324255712308578813630240549552277566', 31, 7);
    t('-21.0687436857970456', '-21.068743685797045585476881980189937156684', 16, 7);
    t('170893057097642104712842895333915433672.71614098604436028244004249890377265867813183634', '170893057097642104712842895333915433672.71614098604436028244004249890377265867813183634', 48, 7);
    t('1419314989888991829914000521', '1419314989888991829914000521', 27, 7);
    t('60.82455319459720635738568680187', '60.82455319459720635738568680186527959160858773466033409815105410886097854659665276583589', 29, 7);
    t('-326121410879335855857666649461844584751.890652900789444294337811561299374410085', '-326121410879335855857666649461844584751.8906529007894442943378115612993744100850340106799465', 39, 7);
    t('-0', '-0.00000000047316992670949883926600553413571239606520662308124419529296046257305486386553', 9, 7);
    t('8727742090301337651614326326111889945767014507', '8727742090301337651614326326111889945767014507', 7, 7);
    t('-53786060748605241244291153266999830828105314220961448804386112379126610916027104677231615675462', '-53786060748605241244291153266999830828105314220961448804386112379126610916027104677231615675462', 41, 7);
    t('-7291471902884803069229597050854712688481', '-7291471902884803069229597050854712688481', 38, 7);
    t('847593479', '847593479', 4, 7);
    t('1373363663030811404110606699087734488483568208061113004937904474604557886571038', '1373363663030811404110606699087734488483568208061113004937904474604557886571038', 51, 7);
    t('-40747376548681307346900781782230386637811935441.3112009014674226146734340597796810731578811', '-40747376548681307346900781782230386637811935441.31120090146742261467343405977968107315788114674885', 43, 7);
    t('-9195689298211729379676090128793323261891000554', '-9195689298211729379676090128793323261891000554', 35, 7);
    t('-0.0034576620308478723296001292094731200618886', '-0.0034576620308478723296001292094731200618885915140945281', 43, 7);
    t('-834830377500140623938088', '-834830377500140623938088', 13, 7);
    t('-2591671475562784513632.7426072061600211', '-2591671475562784513632.7426072061600211449224236386965047917900134074059687266614653350608538539233', 16, 7);
    t('-0.000000071830212114487276777707572429272192915812377', '-0.00000007183021211448727677770757242927219291581237659932896988676370766529', 51, 7);
    t('-640413896696784439726742018298253474508645956359', '-640413896696784439726742018298253474508645956359', 17, 7);
    t('-0.0000000007984874929607838512528277745379438139977', '-0.0000000007984874929607838512528277745379438139976936507024', 49, 7);
    t('-191366559030669593366830590135909135496110802292432471355483754482684086882649187.03963614483', '-191366559030669593366830590135909135496110802292432471355483754482684086882649187.039636144827732224', 11, 7);
    t('8142815084790919737688042342962162664546160121642099901931307232971548145567222', '8142815084790919737688042342962162664546160121642099901931307232971548145567222', 39, 7);
    t('-332782287617659420418475853661', '-332782287617659420418475853661', 8, 7);
    t('0.000000000000000090055750282893', '0.000000000000000090055750282893334403507', 30, 7);
    t('-21214000845.333611178419231112230625322747176660692', '-21214000845.3336111784192311122306253227471766606919954698312144738447328491967907213627789511', 41, 7);
    t('-3308503261988645767218602554122821282075149204084296957091595953533605', '-3308503261988645767218602554122821282075149204084296957091595953533605', 50, 7);
    t('0.0000000000148313954826', '0.000000000014831395482560045753365765821761099871603901006599292947641903057294622382', 22, 7);
    t('-2540286161517932530549634211256175558471.6859862264011445254812864800971', '-2540286161517932530549634211256175558471.685986226401144525481286480097095411420704429841593275576', 32, 7);
    t('53072382348521019562214270117475451722847127807.9201176137241464512', '53072382348521019562214270117475451722847127807.920117613724146451247168428976198', 19, 7);
    t('254629197539322919418061.82042110551640609842973441', '254629197539322919418061.8204211055164060984297344124295121369080688739190169652019849566879168623229', 26, 7);
    t('-0.000000000000000078419972351846537383122302261056', '-0.0000000000000000784199723518465373831223022610558577146950907022375450619', 48, 7);
    t('11595408011.00123572', '11595408011.001235716255147528059507314235781168368767750180889928736025154811960130245171531056', 8, 7);
    t('-88737685279585592831151984911995996', '-88737685279585592831151984911995996', 34, 7);
    t('-37552253678559041341461701227356.158965', '-37552253678559041341461701227356.15896541954380463879908913349547402024', 6, 7);
    t('9003148188594382524113049819941839640533061131950519887235499386135779479186433401692', '9003148188594382524113049819941839640533061131950519887235499386135779479186433401692', 28, 7);
    t('5361439166674927508874087692620586593056959', '5361439166674927508874087692620586593056959', 27, 7);
    t('0.000000000091', '0.00000000009088521688422520393209050649016376675636278197884083029252', 12, 7);
    t('-85657245717942136252098076569728416271774563165859', '-85657245717942136252098076569728416271774563165859', 43, 7);
    t('-3646212560971909.35373247444', '-3646212560971909.3537324744426713329988288580585598165569188254140994718', 11, 7);
    t('370765.6513425131872580849719776983168312523', '370765.651342513187258084971977698316831252349262543626', 37, 7);
    t('-0.000000000000000000079137759978805203671743111106', '-0.00000000000000000007913775997880520367174311110627', 48, 7);
    t('-1611465593139807523207984983928164779476993618071493833576191', '-1611465593139807523207984983928164779476993618071493833576191', 54, 7);
    t('-0.00000000033726114', '-0.000000000337261144121803302303977879693673326581217799', 17, 7);
    t('-6903218163', '-6903218163', 0, 7);
    t('-7394201623023154', '-7394201623023154', 6, 7);
    t('-89820218178368462032325999901068153419219023209648734123533475637571406377255460273403483', '-89820218178368462032325999901068153419219023209648734123533475637571406377255460273403483', 56, 7);
    t('6856221590550350247215669709857430594517289.154687344282048067223977700265786705429299', '6856221590550350247215669709857430594517289.1546873442820480672239777002657867054292985437290849', 42, 7);
    t('1713217233116965517209765946255483359142721898296301440743911736', '1713217233116965517209765946255483359142721898296301440743911736', 49, 7);
    t('-61567856267375488815199698038892', '-61567856267375488815199698038892', 31, 7);
    t('34856837444274112573644196165166306557489218761519232806183215580575570476', '34856837444274112573644196165166306557489218761519232806183215580575570476', 19, 7);
    t('-605776974882763519', '-605776974882763519', 13, 7);
    t('-47418341187456097944575634049467431694845063678', '-47418341187456097944575634049467431694845063678', 18, 7);
    t('1585.955173007361693854365', '1585.95517300736169385436496480295446937143592132714724734106024847451264683205302947963649', 21, 7);
    t('-0.000000000645133213415476611350444263216682276825945168', '-0.0000000006451332134154766113504442632166822768259451679249815370738204997714034783482', 54, 7);
    t('8302757044275405219378536899241030582612132769226851500125963466033484914477.69666886019', '8302757044275405219378536899241030582612132769226851500125963466033484914477.6966688601893948', 11, 7);
    t('45270612179603070143114444578079928392447807.32', '45270612179603070143114444578079928392447807.31953', 3, 7);
    t('7301116759454965528151147816987787832077100563626120303568921', '7301116759454965528151147816987787832077100563626120303568921', 44, 7);
    t('1347.918925815662323176162011734192369953734988907893741421192', '1347.9189258156623231761620117341923699537349889078937414211917111615784661239064', 57, 7);
    t('2745551364781081262572434756', '2745551364781081262572434756', 25, 7);
    t('-0.0000000000000000007702847033377973269515', '-0.0000000000000000007702847033377973269514520366', 40, 7);
    t('-779091641487404784136567.4569597590049318948543232527454505450307', '-779091641487404784136567.4569597590049318948543232527454505450306884403', 40, 7);
    t('0.0000000000000000454', '0.000000000000000045425771078016534055033094636873778207765101376805036792267244437277308175', 19, 7);
    t('56037886202598118256341767914293907027232', '56037886202598118256341767914293907027232', 38, 7);
    t('456832180283905827046262334942695632683335636856760196', '456832180283905827046262334942695632683335636856760196', 7, 7);
    t('835408846081276629656740640207846703234335963484085602007567975275144', '835408846081276629656740640207846703234335963484085602007567975275144', 10, 7);
    t('5892208545916340693141737610813531078984299948', '5892208545916340693141737610813531078984299948', 43, 7);
    t('0.0000000000000000648486346789469454376605', '0.0000000000000000648486346789469454376604957149144419097073694775265833471038380610746092239', 41, 7);
    t('262034975067781367618641670843831216012635510399715', '262034975067781367618641670843831216012635510399715', 48, 7);
    t('0.0000000000001753084053965545624370536153934309', '0.0000000000001753084053965545624370536153934309', 46, 7);
    t('322685018236508000449322148450559922708614699674162314336435015392855585932088', '322685018236508000449322148450559922708614699674162314336435015392855585932088', 6, 7);
    t('-0.0000751245672641759875031068805293967164447087335563', '-0.00007512456726417598750310688052939671644470873355631128944237042822656', 52, 7);
    t('119777479.69030463302136982513942340039304882524620509', '119777479.69030463302136982513942340039304882524620508857791460878231096405985356133971', 44, 7);
    t('35553204237558471570629583797360426827090262465551670477505712708774604078722600420011', '35553204237558471570629583797360426827090262465551670477505712708774604078722600420011', 19, 7);
    t('-291943404908366202882510361495002297845826668767795046772085261154931', '-291943404908366202882510361495002297845826668767795046772085261154931', 35, 7);
    t('-4534915638090657787513612506781133713200439475.0867572558553302530492308505380074065078701', '-4534915638090657787513612506781133713200439475.086757255855330253049230850538007406507870065', 43, 7);
    t('-4628007825261462798342072.911906258290433814410271477744026909511632016076', '-4628007825261462798342072.91190625829043381441027147774402690951163201607620251591', 48, 7);
    t('4.4783487885384154042674977520386158763493723', '4.478348788538415404267497752038615876349372276082911556304299004285541926525363120311938399325', 43, 7);
    t('994636826214072644153969326141', '994636826214072644153969326141', 15, 7);
    t('-5505235444849638686149329852308536500811963484639683908157841253614298429625', '-5505235444849638686149329852308536500811963484639683908157841253614298429625', 4, 7);
    t('614148606644556790188235.0255904986548346451861728211971653475326466503852137269737', '614148606644556790188235.0255904986548346451861728211971653475326466503852137269737', 59, 7);
    t('2193724261100974203871998310680424123017389161396709538479', '2193724261100974203871998310680424123017389161396709538479', 57, 7);
    t('-1956878070881025122270372.337521825043239627169969252259537214', '-1956878070881025122270372.337521825043239627169969252259537214113849107454904456961545052', 36, 7);
    t('-0.000000085351', '-0.00000008535110476842370597384', 12, 7);
    t('641105.998789180105078291507747673', '641105.9987891801050782915077476729519884908863536991235498773209022192197109739345514224', 27, 7);
    t('-21.197585767523889474671349269775310955379095', '-21.197585767523889474671349269775310955379095140158', 42, 7);
    t('116146.494005975894598315121282665834747215927459854167', '116146.49400597589459831512128266583474721592745985416695', 48, 7);
    t('7485928.847855308413604360975395065166506118993796', '7485928.8478553084136043609753950651665061189937957371746846213016715624539634852', 42, 7);
    t('6076054393834960260724553991126463655232706.998159020156971123267673392190851428162156', '6076054393834960260724553991126463655232706.9981590201569711232676733921908514281621564918', 42, 7);
    t('-826178119212535027', '-826178119212535027', 15, 7);
    t('-208338831092277473', '-208338831092277473', 6, 7);
    t('69243933206144181076562868181368044771717426832955483549265453790367.885941195178679', '69243933206144181076562868181368044771717426832955483549265453790367.8859411951786792008', 15, 7);
    t('-2967', '-2967', 4, 7);
    t('-41501138725996819959124846804770912118526003045.185796167930212824055822', '-41501138725996819959124846804770912118526003045.185796167930212824055821995991', 25, 7);
    t('-18057852043794248196575675918109156108791090767', '-18057852043794248196575675918109156108791090767', 20, 7);
    t('281718409203896229312414254327976677356070865725930122957218157', '281718409203896229312414254327976677356070865725930122957218157', 58, 7);
    t('-57791881810902060710765', '-57791881810902060710765', 6, 7);
    t('897687437107677390019954648718257777584106700283464841488202716343791452515', '897687437107677390019954648718257777584106700283464841488202716343791452515', 15, 7);
    t('-2074411348618460421873509155369584886131842509398520763333892604629272098166', '-2074411348618460421873509155369584886131842509398520763333892604629272098166', 56, 7);
    t('6412163494823707737671130186405578927125993163390203934159490599840161448487', '6412163494823707737671130186405578927125993163390203934159490599840161448487', 38, 7);
    t('2570935346175677196878', '2570935346175677196878', 9, 7);
    t('2922237829561344881354422214117488.09791032403417844482040226257725168516472', '2922237829561344881354422214117488.0979103240341784448204022625772516851647155', 41, 7);
    t('-360113314811150792166325832443', '-360113314811150792166325832443', 27, 7);
    t('60014328059462865087436739179650691113359513', '60014328059462865087436739179650691113359513', 19, 7);
    t('-0.000000000006304400685027434212478867369219111834780823862', '-0.000000000006304400685027434212478867369219111834780823862330243813719233306', 57, 7);
    t('-7205740883575345558.3035609312', '-7205740883575345558.303560931199598', 12, 7);
    t('53085840415323322403425172031495', '53085840415323322403425172031495', 13, 7);
    t('501486.7129443291838593479218597548273888277689050437', '501486.71294432918385934792185975482738882776890504370426', 46, 7);
    t('0.000000000000000908932850021752246249043079995293384916', '0.00000000000000090893285002175224624904307999529338491636808', 54, 7);
    t('-0.00000000043256345933043079995692824366141', '-0.0000000004325634593304307999569282436614050910377417779512451575', 41, 7);
    t('-57356950707173599793269435282846849386194446656814419867723389602143804102082390505', '-57356950707173599793269435282846849386194446656814419867723389602143804102082390505', 54, 7);
    t('708838339642304601261089699220144910531439729426156936490112289', '708838339642304601261089699220144910531439729426156936490112289', 34, 7);
    t('-0.0046078939476187', '-0.0046078939476187097406433991788939813592464396622249737415711345731', 16, 7);
    t('-0.12323576577', '-0.123235765770465267088639292854754', 11, 7);
    t('0.00000000000000000095580555623555550814768224905099', '0.00000000000000000095580555623555550814768224905099420276782483589081456723662215426756599134555', 50, 7);
    t('-2570339352145966269634677761640260169032119', '-2570339352145966269634677761640260169032119', 28, 7);
    t('38236133483427843764612112447', '38236133483427843764612112447', 15, 7);
    t('-336931025688483976222819902541535', '-336931025688483976222819902541535', 4, 7);
    t('-9131418940.626118397155251', '-9131418940.6261183971552511339709100767636897756353245436849948601735074456013276755274', 15, 7);
    t('-0.5390546818595212848113530704801476758617232550071935366093', '-0.539054681859521284811353070480147675861723255007193536609276138828242044691640655011489223249255041', 58, 7);
    t('-71416755738083832515056950347140525452730269982573.3282170765632710695719312', '-71416755738083832515056950347140525452730269982573.32821707656327106957193116691737786397252', 25, 7);
    t('0.00000000000000000255991442876714392607715869138', '0.000000000000000002559914428767143926077158691380360601109197', 48, 7);
    t('0.0008012312849268656243687029355101489200440567', '0.0008012312849268656243687029355101489200440566607892247', 46, 7);
    t('244.1299004975136113934851571373', '244.129900497513611393485157137342367594', 28, 7);
    t('8873145.130771366', '8873145.13077136613329539431464116435056918385440213780105794207728379404299861', 9, 7);
    t('90421830817071276704486549442201016086558530905157803796367055646.90135601654869', '90421830817071276704486549442201016086558530905157803796367055646.90135601654868751923854163', 14, 7);
    t('-36.526533442608873306192392', '-36.5265334426088733061923917926640502233384568457377017150304742119162081602297', 24, 7);
    t('7756944055327836674900630471616647295.05850876072750123732246', '7756944055327836674900630471616647295.0585087607275012373224596', 23, 7);
    t('0.000000000000045081619273154020727317734277869142539661', '0.0000000000000450816192731540207273177342778691425396607541', 54, 7);
    t('-6657777084968602905612118294092900774663152', '-6657777084968602905612118294092900774663152', 43, 7);
    t('-0.000000000000000005188433862917015864715902', '-0.0000000000000000051884338629170158647159019984830235182423', 42, 7);
    t('0', '0.00000000000000000317', 15, 7);
    t('-0.4149264099071581632611228387619456297727645178621012', '-0.414926409907158163261122838761945629772764517862101198886202115978635194007190868314555521133717701', 52, 7);
    t('-6864484.8819017398097', '-6864484.88190173980971584504271598165', 13, 7);
    t('-0.000000000000000001679161583143115920667389957545331', '-0.000000000000000001679161583143115920667389957545331037811592028938944353273214379643791129', 51, 7);
    t('2757136.4258217552704766493458823230944375263657904360372', '2757136.4258217552704766493458823230944375263657904360372469', 49, 7);
    t('73592128777516437498892', '73592128777516437498892', 16, 7);
    t('0.07257197707618528010135723118752', '0.07257197707618528010135723118751971071518538894191478337895115762517662900499752291', 33, 7);
    t('-0.000004438131', '-0.0000044381314450238844267302840448432030161991750345428173', 12, 7);
    t('-595708298116615541357228353744113762188591036583833859568474759300695041038994506', '-595708298116615541357228353744113762188591036583833859568474759300695041038994506', 26, 7);
    t('-199080228621514339171672812459767795971.316371238367118391139875956069', '-199080228621514339171672812459767795971.316371238367118391139875956069353409632336022093730932235', 30, 7);
    t('373244977600891908151269122085847490535520152535249519596826077513', '373244977600891908151269122085847490535520152535249519596826077513', 29, 7);
    t('-5639594880101200527887384823782946761462906320.140387160404385845875304739793220257', '-5639594880101200527887384823782946761462906320.140387160404385845875304739793220257', 36, 7);
    t('-0.0000000000087947389416162510202344444769', '-0.000000000008794738941616251020234444476890906037567993523566767799162529587006731256066852047615', 40, 7);
    t('-2811163289244542298.314552033404146359617103886433288499526162438312', '-2811163289244542298.314552033404146359617103886433288499526162438312107482463', 48, 7);
    t('71312169853960180892335621188140367568370992911790488', '71312169853960180892335621188140367568370992911790488', 6, 7);
    t('-62901173098.92897012666077517055378671774520634', '-62901173098.92897012666077517055378671774520634147', 35, 7);
    t('-225945189426218018978712161605939563187347', '-225945189426218018978712161605939563187347', 43, 7);
    t('0.00000000000007', '0.0000000000000670546442643248060364526003698781540676945356538481', 14, 7);
    t('-80283941091068212735380828801414422106056308023', '-80283941091068212735380828801414422106056308023', 49, 7);
    t('-582347756606942164157330806711292105702409851721143752567190054087620149322902471723806842', '-582347756606942164157330806711292105702409851721143752567190054087620149322902471723806842', 1, 7);
    t('-789926098528886434848959110622484454631207369243255613772447578642734', '-789926098528886434848959110622484454631207369243255613772447578642734', 47, 7);
    t('-635798687539890156072639669428587757239', '-635798687539890156072639669428587757239', 8, 7);
    t('-772569903906438796789911092772836.040164878760556480065413195516497', '-772569903906438796789911092772836.0401648787605564800654131955164969017', 33, 7);
    t('27480851396278530398639821633448999360391209592648305694853077032079.542019003323', '27480851396278530398639821633448999360391209592648305694853077032079.542019003323244917259654092', 12, 7);
    t('-37187774977797692205929824486204948247323', '-37187774977797692205929824486204948247323', 8, 7);
    t('-3066994074128464238846827107369679938251453783778612958797836976139626988170384146746054', '-3066994074128464238846827107369679938251453783778612958797836976139626988170384146746054', 25, 7);
    t('1080460621508171861173510076169499142582659362076277316', '1080460621508171861173510076169499142582659362076277316', 26, 7);

    t('-815303438910393610000000000000000000000000000', '-815303438910393610000000000000000000000000000.5', 0, 7);
    t('48079804879135055337410501154945563557501499846191760613365317572636043210233866737200001', '48079804879135055337410501154945563557501499846191760613365317572636043210233866737200000.5', 0, 7);
    t('775141321458031580151353003447938403271713528287267233801607476638641728594600000000001', '775141321458031580151353003447938403271713528287267233801607476638641728594600000000000.5', 0, 7);
    t('-3246099003639468003580507000000000000000000000000000000000000000000000000000000000000000000000000000', '-3246099003639468003580507000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('616052744557652711', '616052744557652710.5', 0, 7);
    t('-47298707260000000000000000000000000000000', '-47298707260000000000000000000000000000000.5', 0, 7);
    t('431370739475777039213491701837719630211035950221951900000000000000000001', '431370739475777039213491701837719630211035950221951900000000000000000000.5', 0, 7);
    t('54683100000000000000000000000000000000000000000000000000000000000000000000000000000000000000001', '54683100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('672783487870821678831521822264427999625400000000000000000000000000000000000000001', '672783487870821678831521822264427999625400000000000000000000000000000000000000000.5', 0, 7);
    t('85750000000000000000000000000000000000000001', '85750000000000000000000000000000000000000000.5', 0, 7);
    t('-357280726879877087440504485199000000000000000000000', '-357280726879877087440504485199000000000000000000000.5', 0, 7);
    t('-4341299477000000000000000000000', '-4341299477000000000000000000000.5', 0, 7);
    t('31902000001', '31902000000.5', 0, 7);
    t('333458122140379389691720067000000000000000000000000000000000000000000001', '333458122140379389691720067000000000000000000000000000000000000000000000.5', 0, 7);
    t('-459839025400000000000000000000', '-459839025400000000000000000000.5', 0, 7);
    t('-82390000000000', '-82390000000000.5', 0, 7);
    t('76701', '76700.5', 0, 7);
    t('190430490057932227000000000000000000000000000000000000001', '190430490057932227000000000000000000000000000000000000000.5', 0, 7);
    t('60159411740605592996051862591038032930184607684088677320000000000000000000000001', '60159411740605592996051862591038032930184607684088677320000000000000000000000000.5', 0, 7);
    t('-2234900000', '-2234900000.5', 0, 7);
    t('56189665877093832601689082910000000000000000000000000000000000000001', '56189665877093832601689082910000000000000000000000000000000000000000.5', 0, 7);
    t('2320000000000000000000000000001', '2320000000000000000000000000000.5', 0, 7);
    t('-6305981611547961162835048322675738149574948489012480000000000000', '-6305981611547961162835048322675738149574948489012480000000000000.5', 0, 7);
    t('7135722984978446900000000000000000000000000000000000001', '7135722984978446900000000000000000000000000000000000000.5', 0, 7);
    t('7739620000000000000000000000000000000000000000000000001', '7739620000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-7310071091565097458378697755962864041334940666867342517847595229165394387202247229680000000', '-7310071091565097458378697755962864041334940666867342517847595229165394387202247229680000000.5', 0, 7);
    t('5138337172269533148524367880141880518489245800000000000001', '5138337172269533148524367880141880518489245800000000000000.5', 0, 7);
    t('-6000', '-6000.5', 0, 7);
    t('463879090754017750000000000000001', '463879090754017750000000000000000.5', 0, 7);
    t('-629850000000000000000', '-629850000000000000000.5', 0, 7);
    t('-205429044500000000000000000000', '-205429044500000000000000000000.5', 0, 7);
    t('-5416257015618874559905147873000000000000000000000000000000000000000000000000', '-5416257015618874559905147873000000000000000000000000000000000000000000000000.5', 0, 7);
    t('623364421289248201317000000000000001', '623364421289248201317000000000000000.5', 0, 7);
    t('911141136528601', '911141136528600.5', 0, 7);
    t('-276055672501612195718314543393745315761189100000', '-276055672501612195718314543393745315761189100000.5', 0, 7);
    t('-51740866419136700582070000', '-51740866419136700582070000.5', 0, 7);
    t('56999862161113789759865220000000000000000000000000000000000000000001', '56999862161113789759865220000000000000000000000000000000000000000000.5', 0, 7);
    t('-4111633071145139947224365900000', '-4111633071145139947224365900000.5', 0, 7);
    t('697451872400000000000000000000000000000000000000000000000000000000000000000000000000000001', '697451872400000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('5000000000000000000000001', '5000000000000000000000000.5', 0, 7);
    t('5350000000000000000000000000000000000000000000000000000001', '5350000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('10529587347000000000000000000000000000000000000000001', '10529587347000000000000000000000000000000000000000000.5', 0, 7);
    t('-2219716000000000000000000000000000', '-2219716000000000000000000000000000.5', 0, 7);
    t('2211360517426407426481005085784959830631111751668394294258799150000000000001', '2211360517426407426481005085784959830631111751668394294258799150000000000000.5', 0, 7);
    t('91550904625902200434312440338781865630142094511605342285084477183074410000000000000000000000001', '91550904625902200434312440338781865630142094511605342285084477183074410000000000000000000000000.5', 0, 7);
    t('658054801757836432123543171404589184297504018642564438772090020000000000000000000001', '658054801757836432123543171404589184297504018642564438772090020000000000000000000000.5', 0, 7);
    t('79944791864921246971621269400000000000000000000001', '79944791864921246971621269400000000000000000000000.5', 0, 7);
    t('360829991900000000000000000000000000000000000000000000000000000000000000000000000000000001', '360829991900000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-1480000000000000000000000000', '-1480000000000000000000000000.5', 0, 7);
    t('-64280200000000', '-64280200000000.5', 0, 7);
    t('-75061307733616254618964203889755925762439000000', '-75061307733616254618964203889755925762439000000.5', 0, 7);
    t('44130000000000000000000001', '44130000000000000000000000.5', 0, 7);
    t('44702568841946912396571635908221096745886412397401280570000000000000000000000000001', '44702568841946912396571635908221096745886412397401280570000000000000000000000000000.5', 0, 7);
    t('604039501485544598558831356363400000000000000001', '604039501485544598558831356363400000000000000000.5', 0, 7);
    t('-19917984485520', '-19917984485520.5', 0, 7);
    t('-311996565804449030000000000000000000000000000000000000000000000000000000000000000', '-311996565804449030000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-173984786383490515959556081806759606423524092895768000000', '-173984786383490515959556081806759606423524092895768000000.5', 0, 7);
    t('-53469189427189740866342899750820440774650016000000000000000000000000000000000000000000000000', '-53469189427189740866342899750820440774650016000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-75', '-75.5', 0, 7);
    t('27983400000000000000000001', '27983400000000000000000000.5', 0, 7);
    t('-55866209', '-55866209.5', 0, 7);
    t('-4236500000000', '-4236500000000.5', 0, 7);
    t('2698000000000000000000001', '2698000000000000000000000.5', 0, 7);
    t('176755986000000000000000000000000000000000000000000000001', '176755986000000000000000000000000000000000000000000000000.5', 0, 7);
    t('10061938000001', '10061938000000.5', 0, 7);
    t('-6910178749421869055443921431000000000', '-6910178749421869055443921431000000000.5', 0, 7);
    t('3577828651898578960680400000000000000000000000000000000000000000000000001', '3577828651898578960680400000000000000000000000000000000000000000000000000.5', 0, 7);
    t('68020066499785661939171192017417847730323210000000000000000000000000000000000001', '68020066499785661939171192017417847730323210000000000000000000000000000000000000.5', 0, 7);
    t('-670000000000000000000000000000', '-670000000000000000000000000000.5', 0, 7);
    t('-40412630000000000000', '-40412630000000000000.5', 0, 7);
    t('-61065894754899019888087081382802643548863976547845002280076808300000000000000', '-61065894754899019888087081382802643548863976547845002280076808300000000000000.5', 0, 7);
    t('100000001', '100000000.5', 0, 7);
    t('-14822039000000000000000000000000', '-14822039000000000000000000000000.5', 0, 7);
    t('-63700', '-63700.5', 0, 7);
    t('-15144885754320000000000000000', '-15144885754320000000000000000.5', 0, 7);
    t('-5272200000000000000000', '-5272200000000000000000.5', 0, 7);
    t('-3247040000000000000', '-3247040000000000000.5', 0, 7);
    t('-2681260809546501000', '-2681260809546501000.5', 0, 7);
    t('87599201647856572613610170000000000000000000000000000000000000001', '87599201647856572613610170000000000000000000000000000000000000000.5', 0, 7);
    t('250000000000000000000000000000000000000000001', '250000000000000000000000000000000000000000000.5', 0, 7);
    t('14130422050230001', '14130422050230000.5', 0, 7);
    t('-215000000000000000000000000000000000000000000000000000000000000000000000000', '-215000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-43726827420000000', '-43726827420000000.5', 0, 7);
    t('-4800000000000000000000000000000000000000000000000000000000000000000', '-4800000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-7300000000000000000000000000', '-7300000000000000000000000000.5', 0, 7);
    t('491352128092860358920632352122032679985453000001', '491352128092860358920632352122032679985453000000.5', 0, 7);
    t('-882188546069284263665240261471556461467006545965824525162469092995249760000', '-882188546069284263665240261471556461467006545965824525162469092995249760000.5', 0, 7);
    t('-85741667839249074', '-85741667839249074.5', 0, 7);
    t('-527599500272614000000000000000000000000000000000000000000000000000000000000000000000000000000000', '-527599500272614000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-832820000000000', '-832820000000000.5', 0, 7);
    t('-198601973381063265872709695924943530652918000000000000000000', '-198601973381063265872709695924943530652918000000000000000000.5', 0, 7);
    t('-10000', '-10000.5', 0, 7);
    t('-673944669918113000000000000000000000000000000', '-673944669918113000000000000000000000000000000.5', 0, 7);
    t('-84188225', '-84188225.5', 0, 7);
    t('771154817579711197000000000000000000000000000000000000000000000000001', '771154817579711197000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-87389583669272781200000000000000000000000000000000000', '-87389583669272781200000000000000000000000000000000000.5', 0, 7);
    t('2626469664216538014114045286014123593449500000001', '2626469664216538014114045286014123593449500000000.5', 0, 7);
    t('-696131077975000000000000', '-696131077975000000000000.5', 0, 7);
    t('-78690000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '-78690000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-442122610891418980859993032200000', '-442122610891418980859993032200000.5', 0, 7);
    t('5661090000000000000000000000000000000000000000000000000000000000000000000000000000001', '5661090000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-407000000000000000000', '-407000000000000000000.5', 0, 7);
    t('729350422974116892063715545463978850831800000000001', '729350422974116892063715545463978850831800000000000.5', 0, 7);
    t('637954990213103000000001', '637954990213103000000000.5', 0, 7);
    t('-1138365762514721521483834034584523726374412779304481429041180049963600000', '-1138365762514721521483834034584523726374412779304481429041180049963600000.5', 0, 7);
    t('2475696728487804739525182918139219026008027679054000000000000000000000000000000000001', '2475696728487804739525182918139219026008027679054000000000000000000000000000000000000.5', 0, 7);
    t('66193237237200326100000000000000000001', '66193237237200326100000000000000000000.5', 0, 7);
    t('663179971604113808385191042094969369727970974320403970845752700000000001', '663179971604113808385191042094969369727970974320403970845752700000000000.5', 0, 7);
    t('596157270311242', '596157270311241.5', 0, 7);
    t('-430859454062137256578640000000000000000000000000000000000000000', '-430859454062137256578640000000000000000000000000000000000000000.5', 0, 7);
    t('7032813000000000001', '7032813000000000000.5', 0, 7);
    t('767314152225968313318624830508438163176729385050170000000000000000001', '767314152225968313318624830508438163176729385050170000000000000000000.5', 0, 7);
    t('-321648005175685895332610984400000000000', '-321648005175685895332610984400000000000.5', 0, 7);
    t('-5914848242008475427391535555998338828544179518249385611394025548931079438027758333803040000000000', '-5914848242008475427391535555998338828544179518249385611394025548931079438027758333803040000000000.5', 0, 7);
    t('73480016864560536978879028863212252705530701', '73480016864560536978879028863212252705530700.5', 0, 7);
    t('8976064818931304755883643401472118930622270000000000000001', '8976064818931304755883643401472118930622270000000000000000.5', 0, 7);
    t('-5079754732761006567963078886040778070', '-5079754732761006567963078886040778070.5', 0, 7);
    t('-51002521076071416214838521445923939000000000', '-51002521076071416214838521445923939000000000.5', 0, 7);
    t('8684049955981816069703979854267267479779343491340480759376275884260000000000000000000000000001', '8684049955981816069703979854267267479779343491340480759376275884260000000000000000000000000000.5', 0, 7);
    t('37507194800178617805104432695868625191287264624719442564304569000000000000000000000000000000001', '37507194800178617805104432695868625191287264624719442564304569000000000000000000000000000000000.5', 0, 7);
    t('-4281839065682230128346600112666094892582742848537366052855118901071834009400000000000000', '-4281839065682230128346600112666094892582742848537366052855118901071834009400000000000000.5', 0, 7);
    t('-333000', '-333000.5', 0, 7);
    t('-502583575785454685986915469822211140422238465305000000000000000000000000000000000000', '-502583575785454685986915469822211140422238465305000000000000000000000000000000000000.5', 0, 7);
    t('-32030575969596778652213137431653825274000000000000000000', '-32030575969596778652213137431653825274000000000000000000.5', 0, 7);
    t('-4506117101680962000000000000000000000000', '-4506117101680962000000000000000000000000.5', 0, 7);
    t('-64730463575631254944664602047280130000000000000000000000000000000000000', '-64730463575631254944664602047280130000000000000000000000000000000000000.5', 0, 7);
    t('27570663944600021244014689614081019825668139000000000000000000000000001', '27570663944600021244014689614081019825668139000000000000000000000000000.5', 0, 7);
    t('-350302102100000000000000000', '-350302102100000000000000000.5', 0, 7);
    t('-3918501626000000000', '-3918501626000000000.5', 0, 7);
    t('51812774862668048505100000000000000000000000000000000000000000000000000000000001', '51812774862668048505100000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-61696491490996741116958188157141279933341663428216950000000000000000000000', '-61696491490996741116958188157141279933341663428216950000000000000000000000.5', 0, 7);
    t('4656808510000000000000000000000000000000000000000000000000000000000000000001', '4656808510000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('23272197523886798346038846952829185419450839280000000000000000000000000000000000000001', '23272197523886798346038846952829185419450839280000000000000000000000000000000000000000.5', 0, 7);
    t('4065544662345173976170000000000000000000000000000000001', '4065544662345173976170000000000000000000000000000000000.5', 0, 7);
    t('-422635258000000000000000000000000000000000000000000000000000', '-422635258000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-3582000', '-3582000.5', 0, 7);
    t('7740826542065848739126877465966868569935787226324870000000000000000001', '7740826542065848739126877465966868569935787226324870000000000000000000.5', 0, 7);
    t('1614965644012243386199869835574326000000001', '1614965644012243386199869835574326000000000.5', 0, 7);
    t('-20000', '-20000.5', 0, 7);
    t('-2291933618897112688229402464046600000000000000000000000000000000000000000000000000000000000000', '-2291933618897112688229402464046600000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-663605069281623297326832485568849000000000000000000000000000000000000000000000000000000000000000000', '-663605069281623297326832485568849000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('1939060273775317149434400000000000000000000000000000000000000000000000000000000000000000000001', '1939060273775317149434400000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-726114440793216017690000000000000000000000', '-726114440793216017690000000000000000000000.5', 0, 7);
    t('-8529164154875627047383959874881458179234088946000000000000000000000000000000000000000000000000000', '-8529164154875627047383959874881458179234088946000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('359551952970277110582000000000000000000000000000000000001', '359551952970277110582000000000000000000000000000000000000.5', 0, 7);
    t('-6822800000000000000000000000000000000000000000000000', '-6822800000000000000000000000000000000000000000000000.5', 0, 7);
    t('28024094231179167233916852000000000000000000000000000000001', '28024094231179167233916852000000000000000000000000000000000.5', 0, 7);
    t('-5768619642107599734543714041179604138353000000000000', '-5768619642107599734543714041179604138353000000000000.5', 0, 7);
    t('-4336937232847962038320625210133958133684710802758818489207914168472208196240530780800000', '-4336937232847962038320625210133958133684710802758818489207914168472208196240530780800000.5', 0, 7);
    t('294276367161162465235912532501878152936548550758709465137169762240500722165978566', '294276367161162465235912532501878152936548550758709465137169762240500722165978565.5', 0, 7);
    t('-5853377924485346629722284422247261298818024606786760212628942481993981005105147485025005541304000', '-5853377924485346629722284422247261298818024606786760212628942481993981005105147485025005541304000.5', 0, 7);
    t('-57376528507859392114198303459327987043203713812970277355663102705761801186615364048872085338570', '-57376528507859392114198303459327987043203713812970277355663102705761801186615364048872085338570.5', 0, 7);
    t('8981899774674151341273723045246779558611526093370000000000000000000000000000001', '8981899774674151341273723045246779558611526093370000000000000000000000000000000.5', 0, 7);
    t('6880184505510000000000000000000000000000000000000000000000000000000000000000000000001', '6880184505510000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-9144765542574759986504560099796367000000000000000000000000000000000000000000000000000000000000000000', '-9144765542574759986504560099796367000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('-33550940230832658386583641510', '-33550940230832658386583641510.5', 0, 7);
    t('570500000001', '570500000000.5', 0, 7);
    t('21171552652016197991647323966415490225947817225636864710000000000000000000000000001', '21171552652016197991647323966415490225947817225636864710000000000000000000000000000.5', 0, 7);
    t('-219451479377926100000000000000000000000', '-219451479377926100000000000000000000000.5', 0, 7);
    t('-46023648500000000000000000000000000000000000000000000000', '-46023648500000000000000000000000000000000000000000000000.5', 0, 7);
    t('-5788386698147635042359128255204287799666980072971472360314579333990755726292753882057820000', '-5788386698147635042359128255204287799666980072971472360314579333990755726292753882057820000.5', 0, 7);
    t('48425486699992611674833028561543095971459492320411160921', '48425486699992611674833028561543095971459492320411160920.5', 0, 7);
    t('-59344633366981652200000000000', '-59344633366981652200000000000.5', 0, 7);
    t('-65482014253687833342600000000000000000000000', '-65482014253687833342600000000000000000000000.5', 0, 7);
    t('-30128014553929563000000000000000000000000000000000000000000000000000000000000', '-30128014553929563000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('794768001', '794768000.5', 0, 7);
    t('-150865959000000000000000000000000000000000000000000000000000000000000000000000000000000', '-150865959000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('873381282566874884976602503384155621488078946785261238', '873381282566874884976602503384155621488078946785261237.5', 0, 7);
    t('-76364822618102400000000000000000000000000000000000000000000000000', '-76364822618102400000000000000000000000000000000000000000000000000.5', 0, 7);
    t('1590000000000000000000000000000000000000000000000000000000001', '1590000000000000000000000000000000000000000000000000000000000.5', 0, 7);
    t('15883352501', '15883352500.5', 0, 7);
    t('-1678796930161119374892', '-1678796930161119374892.5', 0, 7);
    t('7519225332612480000000000000001', '7519225332612480000000000000000.5', 0, 7);
    t('26782458868551013018628206420511402980374696236993201', '26782458868551013018628206420511402980374696236993200.5', 0, 7);
    t('84182962247257426278629190001', '84182962247257426278629190000.5', 0, 7);
    t('-674633568509895706371390517304490463246553534599391649648000000', '-674633568509895706371390517304490463246553534599391649648000000.5', 0, 7);
    t('2036599383754410655598351317310763047828000000000000000000000000000001', '2036599383754410655598351317310763047828000000000000000000000000000000.5', 0, 7);
    t('-1747000000', '-1747000000.5', 0, 7);
    t('3960685307896400000000000001', '3960685307896400000000000000.5', 0, 7);
    t('-1', '-1.5', 0, 7);
    t('5', '4.5', 0, 7);
    t('-8542007911699875753350816000000000', '-8542007911699875753350816000000000.5', 0, 7);
    t('229448019839399741', '229448019839399740.5', 0, 7);
    t('4340972004000000000000000000000000000000000001', '4340972004000000000000000000000000000000000000.5', 0, 7);
    t('-773300000000', '-773300000000.5', 0, 7);
    t('7155533863912051817808706333952649054169243324075500000000000000000000000000000000001', '7155533863912051817808706333952649054169243324075500000000000000000000000000000000000.5', 0, 7);
    t('994879015661912752196001000000000000000001', '994879015661912752196001000000000000000000.5', 0, 7);
    t('9735231726716596452594218428908047051010710000000000000000000000000000000001', '9735231726716596452594218428908047051010710000000000000000000000000000000000.5', 0, 7);
    t('830000001', '830000000.5', 0, 7);
    t('-621154', '-621154.5', 0, 7);
    t('-7026505943196190557292140834274961200000000', '-7026505943196190557292140834274961200000000.5', 0, 7);
    t('-5698210301557000000000000000000000000000', '-5698210301557000000000000000000000000000.5', 0, 7);
    t('633325185790000001', '633325185790000000.5', 0, 7);
    t('-64165596081004870', '-64165596081004870.5', 0, 7);
    t('-41646941216693066756777307741427764177371639363417043000000000000000000000000000000000000000', '-41646941216693066756777307741427764177371639363417043000000000000000000000000000000000000000.5', 0, 7);
    t('-30276599950453', '-30276599950453.5', 0, 7);
    t('844934410001', '844934410000.5', 0, 7);
    t('193536068479533070654088534028865405146725922969081990000000000000000000000000001', '193536068479533070654088534028865405146725922969081990000000000000000000000000000.5', 0, 7);
    t('3571390090807278635466910226000000000001', '3571390090807278635466910226000000000000.5', 0, 7);
    t('33758731615573853269038256552240314238547949393669248932000000000000000000000000000000000000000001', '33758731615573853269038256552240314238547949393669248932000000000000000000000000000000000000000000.5', 0, 7);

    // rounding mode 8
    t('66201703687996196493948073895661219741.79732939050299983848272445', '66201703687996196493948073895661219741.79732939050299983848272444531659415942809827190951', 26, 8);
    t('-432784391834045276975262545157922', '-432784391834045276975262545157922', 28, 8);
    t('1149345737914160007440457748332143986414.9571', '1149345737914160007440457748332143986414.957136799115537152433594986979', 4, 8);
    t('-8955086490844848921210649185442547', '-8955086490844848921210649185442547', 35, 8);
    t('425088038661656538743278817856878424164093767.85670041722429249440202331857043', '425088038661656538743278817856878424164093767.85670041722429249440202331857043013807', 33, 8);
    t('-309644694334054373034963302293381012433965117503313464512444644', '-309644694334054373034963302293381012433965117503313464512444644', 15, 8);
    t('-6113258580590430007589739873373442564851315606055053637944', '-6113258580590430007589739873373442564851315606055053637944', 20, 8);
    t('-79578809228411034521778967729244470830365009197702327546767865195836773388369068481948969.900741', '-79578809228411034521778967729244470830365009197702327546767865195836773388369068481948969.9007405268', 6, 8);
    t('-98635952437161220.96618', '-98635952437161220.9661782723695091452045447927505345126085', 5, 8);
    t('5306244228965561737.62447596491901907069113187489713', '5306244228965561737.62447596491901907069113187489712907837802453330192181391370636667897993', 32, 8);
    t('881494696226260293271264.36459456081728760273917638738994049272140009', '881494696226260293271264.36459456081728760273917638738994049272140008956721', 45, 8);
    t('0', '0.0000000000000000398975254', 1, 8);
    t('-2966907742563204474300609285105763571411556718257633358168823625207636310562902929129639632', '-2966907742563204474300609285105763571411556718257633358168823625207636310562902929129639632', 59, 8);
    t('4815226942078558754202763870371111530259178454324372013168459692655444158537', '4815226942078558754202763870371111530259178454324372013168459692655444158537', 58, 8);
    t('89735736519038061687.396102956878866121243318078603705593756032576774631567136', '89735736519038061687.396102956878866121243318078603705593756032576774631567136', 57, 8);
    t('-5139206863826452924558709317549576110217185', '-5139206863826452924558709317549576110217185', 10, 8);
    t('410799450059.233711597011245238081176283367945619000645947401973880735', '410799450059.23371159701124523808117628336794561900064594740197388073499637579092523992206', 59, 8);
    t('-1861173217597843873200339700375118033056165470698896905908777053211294087744', '-1861173217597843873200339700375118033056165470698896905908777053211294087744', 38, 8);
    t('-0', '-0.00000000000000006595', 11, 8);
    t('-5407783849362963393803124142731051275240080145012', '-5407783849362963393803124142731051275240080145012', 20, 8);
    t('-34168517091385789116719661427034183841563988149', '-34168517091385789116719661427034183841563988149', 17, 8);
    t('5795445846', '5795445846', 2, 8);
    t('-208487757.4947118655668802941616876105973734970142836273035', '-208487757.49471186556688029416168761059737349701428362730350350449276293511255885603228665646', 49, 8);
    t('796420627119046.3257623490013937727996276166953155756033788473505312', '796420627119046.325762349001393772799627616695315575603378847350531229664668561098545265510186232639', 52, 8);
    t('-764919.2520201660863571182709007', '-764919.252020166086357118270900678004721225081351152', 25, 8);
    t('1711778256244239.6432803280511675051780680294577052774533541577657', '1711778256244239.643280328051167505178068029457705277453354157765665691237184194081804579550628', 49, 8);
    t('-5303437679211478868344273161947940791429451823884596427841.337465439843', '-5303437679211478868344273161947940791429451823884596427841.337465439843017473765793878253395', 12, 8);
    t('7894537442022400587189513519.07614612464633', '7894537442022400587189513519.0761461246463343271909512896919921757738101749442', 14, 8);
    t('-852162476174009362657224788407373904292.060834168512303', '-852162476174009362657224788407373904292.0608341685123032205848264284500246825574591705678252075211', 15, 8);
    t('12263206466360474155347209378267179331790532796950452218705592936654936382476475166224', '12263206466360474155347209378267179331790532796950452218705592936654936382476475166224', 50, 8);
    t('286739.751858335177', '286739.75185833517663282910779633883075386374977590348991259883429540928694021465959067075843', 12, 8);
    t('-1762282392342322391451423.95138646013724750227039851', '-1762282392342322391451423.9513864601372475022703985059087755006296016821468', 26, 8);
    t('-87248674085961920378373182112140016225793103119799077836448861613155523873993867027371', '-87248674085961920378373182112140016225793103119799077836448861613155523873993867027371', 18, 8);
    t('-74092308.573751689088057271221141335221234751439463510099422792', '-74092308.5737516890880572712211413352212347514394635100994227920017954257506493659703', 55, 8);
    t('-20760047489279', '-20760047489279', 11, 8);
    t('-0.0000000000000000000513', '-0.00000000000000000005125373071584388602667', 22, 8);
    t('553838249451575699354166087894479427887538748821541', '553838249451575699354166087894479427887538748821541', 49, 8);
    t('1423248954610786032951054974531156745902169', '1423248954610786032951054974531156745902169', 18, 8);
    t('-455513127477032432052165898762670896803882354533268751050143', '-455513127477032432052165898762670896803882354533268751050143', 59, 8);
    t('-0.112326948668716031138310473397726709406273', '-0.1123269486687160311383104733977267094062725419900260888852399393388797266733492952', 42, 8);
    t('220496431007708742488102987834369132032989458669949404568519039366030060413', '220496431007708742488102987834369132032989458669949404568519039366030060413', 50, 8);
    t('-0.00000000000325811433456091828013955203891346578756293444537', '-0.000000000003258114334560918280139552038913465787562934445372834409816', 59, 8);
    t('-457504832634108650876363681507414265221', '-457504832634108650876363681507414265221', 40, 8);
    t('-424186796605368850991004250916', '-424186796605368850991004250916', 31, 8);
    t('656926.697474746923763802615025035828093602941156686984327', '656926.6974747469237638026150250358280936029411566869843273144638083499809', 51, 8);
    t('0.000000000000000005397242366818203996', '0.00000000000000000539724236681820399616017543171572414502390014285527039717119783434813739074418', 36, 8);
    t('1090684201.48935564', '1090684201.4893556381714726135555970595559735352509535058941537207185560561831', 8, 8);
    t('69374039942373.3254580716722156975426911', '69374039942373.32545807167221569754269111', 25, 8);
    t('9113500111924420134168224777108947212846008765793966704145', '9113500111924420134168224777108947212846008765793966704145.431817424332', 0, 8);
    t('30823.68236248962442', '30823.682362489624415303797164930829692243120825797614790110298558566891', 14, 8);
    t('69510854030784689448691205235110864752020705135715463000353620416781464635565813359264377', '69510854030784689448691205235110864752020705135715463000353620416781464635565813359264377', 31, 8);
    t('0.000000000000058862529560768', '0.0000000000000588625295607683198127617212575718008706568858558601170114', 27, 8);
    t('-17848760834059263091993243721898202503622421343253601508', '-17848760834059263091993243721898202503622421343253601508', 23, 8);
    t('-236803670466123595831349021922016939098010857534796192', '-236803670466123595831349021922016939098010857534796192', 21, 8);
    t('-102.198091649633311839521389772950955038', '-102.198091649633311839521389772950955038425200879430185191719111255573192340640709444075063237279', 36, 8);
    t('326989728850104213753.8335807775873', '326989728850104213753.833580777587346167082725672681674988508273423', 13, 8);
    t('-122.0291061318001501339747396650645162229074064349133', '-122.029106131800150133974739665064516222907406434913336614242606358', 49, 8);
    t('-831410519711671441355', '-831410519711671441355', 16, 8);
    t('-253620339160158444333927384234542070578', '-253620339160158444333927384234542070578', 9, 8);
    t('-7348280719663124699843895522030312381252008175705772142979880256543246476384943464878409', '-7348280719663124699843895522030312381252008175705772142979880256543246476384943464878409', 50, 8);
    t('6664146236447212814277572.0999056094992', '6664146236447212814277572.099905609499231171', 13, 8);
    t('-45131966684154664823729921239622741743388388032363350525350351', '-45131966684154664823729921239622741743388388032363350525350351', 54, 8);
    t('-40833932047032323008.1477121334939888606975457955366723579560220387882733564215', '-40833932047032323008.14771213349398886069754579553667235795602203878827335642149629', 59, 8);
    t('-610147668317151494655953188414347.513062160034037994471478116919185602379422684392', '-610147668317151494655953188414347.5130621600340379944714781169191856023794226843916275138433253', 48, 8);
    t('68480060832241105622.0309399631877587294933', '68480060832241105622.0309399631877587294932744437074077578726838602811862463', 22, 8);
    t('194232568635561193478658679661926001447122084303161530447723323350451591165847396942433467', '194232568635561193478658679661926001447122084303161530447723323350451591165847396942433467', 53, 8);
    t('301488112835117984681216562501634945527931425479867644918437173989635003275', '301488112835117984681216562501634945527931425479867644918437173989635003275', 45, 8);
    t('-6851957382967116770156790585389162381351471623360536810366727929091181605285629993', '-6851957382967116770156790585389162381351471623360536810366727929091181605285629993', 58, 8);
    t('-47712893537299565607733312598395281973889458872107933076', '-47712893537299565607733312598395281973889458872107933076', 22, 8);
    t('0', '0.0000000000000000859670879420002853724584828504688225809552905002764241975507797099562754888', 9, 8);
    t('0.004432633947152', '0.00443263394715151891254682580745644237652322307', 15, 8);
    t('69262889569389462605539771733767837082438188', '69262889569389462605539771733767837082438188', 44, 8);
    t('-3891963561578294887852441074022523556630020645085806037968401206141034018227239650504315', '-3891963561578294887852441074022523556630020645085806037968401206141034018227239650504315', 29, 8);
    t('517455700713583735581011224355747758835072', '517455700713583735581011224355747758835072', 33, 8);
    t('3286559014183405381681284382257494050726761990059', '3286559014183405381681284382257494050726761990059', 8, 8);
    t('-63655023123174740325953043492297676892760601721778538384', '-63655023123174740325953043492297676892760601721778538384', 38, 8);
    t('-3616486456360129835721411345', '-3616486456360129835721411345', 25, 8);
    t('5302046753446852414769871718814521746.07821211750152648928158273941305583283', '5302046753446852414769871718814521746.07821211750152648928158273941305583283', 39, 8);
    t('7748631583295908.9', '7748631583295908.93', 1, 8);
    t('698840974194462996670670779801788898006147', '698840974194462996670670779801788898006147', 39, 8);
    t('5115298308143358514121930351981670511745255186478118611705507709680743067049815465', '5115298308143358514121930351981670511745255186478118611705507709680743067049815465', 3, 8);
    t('-612675133207.913892413366818256954190842994', '-612675133207.91389241336681825695419084299362927950456589721655286975937658897251713491', 30, 8);
    t('-23760765404011175917916548731905430881375361039', '-23760765404011175917916548731905430881375361039', 37, 8);
    t('1.537325640830455556577284161295194462534093392', '1.53732564083045555657728416129519446253409339155685994767622443301510890518218163627512409985028874', 45, 8);
    t('85488099619561530625377.4192191219015077695264716986497372124636', '85488099619561530625377.41921912190150776952647169864973721246357123472247253263193079654814', 40, 8);
    t('-0.000000000000000000050862650388973728733959', '-0.00000000000000000005086265038897372873395911474890963231183754251032093750396822', 42, 8);
    t('5526230849632961555873105581682031232032076652', '5526230849632961555873105581682031232032076652', 33, 8);
    t('5.35603726623675478881750325031077857085', '5.3560372662367547888175032503107785708523437510821741384889170003345091454641716', 38, 8);
    t('-269', '-268.5690822533536300367', 0, 8);
    t('5757355935873247215361', '5757355935873247215361', 7, 8);
    t('246850624649836235549257820435684838216256457058297475944623153518238931267', '246850624649836235549257820435684838216256457058297475944623153518238931267', 29, 8);
    t('-692278289514395596822246030611', '-692278289514395596822246030611', 29, 8);
    t('5011884712679298647620944560807194982963074154957144073408073782.012902676141595080232825022', '5011884712679298647620944560807194982963074154957144073408073782.01290267614159508023282502208799', 27, 8);
    t('-1601088319576953191651975449908738521284', '-1601088319576953191651975449908738521284', 13, 8);
    t('-4414644735536629652380210044243136361770887495321', '-4414644735536629652380210044243136361770887495321', 1, 8);
    t('0.0000000266191991852311627685858864306795', '0.0000000266191991852311627685858864306795247265183872439736762861', 40, 8);
    t('0.000428744212691549043417829548631538073876', '0.0004287442126915490434178295486315380738764488424270530614427525474179623980503815', 42, 8);
    t('-0.000000000047', '-0.000000000046905000977649660171', 12, 8);
    t('-0.0000000000000000012796854596101', '-0.0000000000000000012796854596101112', 31, 8);
    t('0', '0.00000000000000007196452', 10, 8);
    t('-86284478039635020.88154442791139797502881588209234679333543929358069793135', '-86284478039635020.8815444279113979750288158820923467933354392935806979313474421195014946654603', 56, 8);
    t('345.7906584145207214494773389942349751491', '345.79065841452072144947733899423497514905046668326852378724478389764822926854507387015', 37, 8);
    t('-30568645130233018977700516286545868158267620169281990416832482986193', '-30568645130233018977700516286545868158267620169281990416832482986193', 43, 8);
    t('37665450910098298765355672911622917', '37665450910098298765355672911622917', 25, 8);
    t('-47992849', '-47992849', 1, 8);
    t('-2289649689772323.458246269770512751882357714601759751619429', '-2289649689772323.458246269770512751882357714601759751619428559799708215001792781583', 42, 8);
    t('205469749210966781349300242506558230047710955467288617413646386581699317243', '205469749210966781349300242506558230047710955467288617413646386581699317243', 25, 8);
    t('-32056221565262365438607056649704856185526855802381047619081869', '-32056221565262365438607056649704856185526855802381047619081869', 58, 8);
    t('-69387756776107867448507.5597896113493378218866042115232132608', '-69387756776107867448507.5597896113493378218866042115232132608308527098602632063024', 37, 8);
    t('0.000000820570605855677101583661555709', '0.00000082057060585567710158366155570875342023482713606', 36, 8);
    t('69207211379716780147073990298041848170635366340631501124460602562', '69207211379716780147073990298041848170635366340631501124460602562', 14, 8);
    t('74433853443.711309574611828991681415199898999', '74433853443.711309574611828991681415199898998743211590771456662685697848481543071016889', 33, 8);
    t('560136636765721.54952783053237', '560136636765721.5495278305323717513438', 14, 8);
    t('66999767298684323378394675772410013.28969441028366360358403105373652837191570809785171273', '66999767298684323378394675772410013.28969441028366360358403105373652837191570809785171273', 54, 8);
    t('68347634798085780973515205391681470162522109665463538490773666914168258894444', '68347634798085780973515205391681470162522109665463538490773666914168258894444', 24, 8);
    t('7697922274864830543190030076775438', '7697922274864830543190030076775438', 20, 8);
    t('278983937451071130016483992277822580223855875348190818.113846802586205314111233', '278983937451071130016483992277822580223855875348190818.11384680258620531411123264767447310686', 24, 8);
    t('-18302541929262108151338961040808216477301', '-18302541929262108151338961040808216477301', 13, 8);
    t('-917241354938259400580805453251192909557123968816077324554688.2632819065881148284', '-917241354938259400580805453251192909557123968816077324554688.263281906588114828403637', 20, 8);
    t('45819385064609897668758693853072194843813609788794421314', '45819385064609897668758693853072194843813609788794421314', 24, 8);
    t('-523986068289618485266030887200551590040084842900599214834165319686', '-523986068289618485266030887200551590040084842900599214834165319686', 18, 8);
    t('7852063910359971787869622352527374328444882187005886753422152922497829760479356719', '7852063910359971787869622352527374328444882187005886753422152922497829760479356719', 38, 8);
    t('0.0000000000247091', '0.00000000002470914629639547358185782547191865524156377299551121577023612761119607882', 16, 8);
    t('440821431387771614013770385588.03139517196419817521106343238223731692566664358823035313142', '440821431387771614013770385588.03139517196419817521106343238223731692566664358823035313142186102', 59, 8);
    t('-7.391761264366265837615347413187815498740935', '-7.391761264366265837615347413187815498740934589988129058805961426', 42, 8);
    t('41794466666300650385807822364400569925816612741244088085922306361105114', '41794466666300650385807822364400569925816612741244088085922306361105114', 48, 8);
    t('0', '0.0000000000000000038802542700538271039582060794252268615576361549601345439908', 15, 8);
    t('35343256389819549943056605473715229852', '35343256389819549943056605473715229852', 9, 8);
    t('-818984458678111129545223433258309212674764516267725284171918954770229.24954611425', '-818984458678111129545223433258309212674764516267725284171918954770229.249546114248715', 11, 8);
    t('-7230691209.6753920167936103432715638', '-7230691209.67539201679361034327156375593719410288588', 25, 8);
    t('-37368231656156642628245727340400846248755238210404670712676232100034223145733', '-37368231656156642628245727340400846248755238210404670712676232100034223145733', 9, 8);
    t('-21531849930403836723.549953750896490206543923575188', '-21531849930403836723.549953750896490206543923575188115922', 30, 8);
    t('0.00255705840261906787264907412348588448784776243920353187679', '0.0025570584026190678726490741234858844878477624392035318767939168622896408327924273682986', 59, 8);
    t('-72553243708', '-72553243708', 12, 8);
    t('-0.0000000000008856317899193899991568647156407142735462649', '-0.0000000000008856317899193899991568647156407142735462649439073', 55, 8);
    t('0.00000000000000002567536449680036305412256336138415403', '0.0000000000000000256753644968003630541225633613841540257438901', 53, 8);
    t('-2861414563473219891469834741889616', '-2861414563473219891469834741889616', 23, 8);
    t('76315153632082828597570739187599689956646538104750912445681366544986223082108062643693', '76315153632082828597570739187599689956646538104750912445681366544986223082108062643693', 58, 8);
    t('-36017699886964287552542855178455727562394623', '-36017699886964287552542855178455727562394623', 10, 8);
    t('-8039050083841916177617168262552140531689510609781882', '-8039050083841916177617168262552140531689510609781882', 7, 8);
    t('-75564522183815281041495497607512529468573045762495628', '-75564522183815281041495497607512529468573045762495628', 47, 8);
    t('-0.0000000000000028959465568008886578', '-0.000000000000002895946556800888657768819798024109048410779074303487219485197865657', 34, 8);
    t('284523781858209826579679727341459643256753883111577735224', '284523781858209826579679727341459643256753883111577735224', 17, 8);
    t('855998775255794984240155453520859416491', '855998775255794984240155453520859416491', 2, 8);
    t('-138150439221363970723340802061921303881866895', '-138150439221363970723340802061921303881866895', 10, 8);
    t('-4733097249983425263123260326467561884629992599437098006772507345.39029977752721930256', '-4733097249983425263123260326467561884629992599437098006772507345.39029977752721930256036990911', 21, 8);
    t('-0.000000000000007891361298828364332998396426', '-0.0000000000000078913612988283643329983964258020880117022087729997106646341', 42, 8);
    t('-0', '-0.00000000000000000005232135595017205522', 14, 8);
    t('6004269558735103787710387970518297095870881046535997656345481925243412966194', '6004269558735103787710387970518297095870881046535997656345481925243412966194', 47, 8);
    t('-79790582292297723097183916661305043162178089556664159', '-79790582292297723097183916661305043162178089556664159', 50, 8);
    t('868633879339750610248.8719201777840534154224473677127177', '868633879339750610248.8719201777840534154224473677127177456222197014169473933', 34, 8);
    t('-35193645352688585041808187638769417251065373576.318111185472303777399956841', '-35193645352688585041808187638769417251065373576.3181111854723037773999568411745', 27, 8);
    t('-914456499778647874582727776237', '-914456499778647874582727776237', 32, 8);
    t('-0.0000000082872578546868214209908047326678355', '-0.0000000082872578546868214209908047326678354507848678952930148008797982168429352770394', 43, 8);
    t('43709030067329728774375267529433280243651742049220974', '43709030067329728774375267529433280243651742049220974', 33, 8);
    t('-118010.225141375666233966986827467153774', '-118010.2251413756662339669868274671537736196650024082171238583391369', 33, 8);
    t('23068625486429.2799608174027102', '23068625486429.279960817402710215', 16, 8);
    t('-704545071947407222321207465462220615566', '-704545071947407222321207465462220615566', 11, 8);
    t('14375372697678701853.68596153637061158763127684500719629374218600432501980003', '14375372697678701853.68596153637061158763127684500719629374218600432501980003395932938491', 56, 8);
    t('-634203070629093.01725952920860266543339878670159', '-634203070629093.01725952920860266543339878670158808711459045809411073373681', 32, 8);
    t('-3341330166644850447365712680571164098505980754357561658133', '-3341330166644850447365712680571164098505980754357561658133', 40, 8);
    t('910.78407278867854608277033163963661161209449973499620413', '910.784072788678546082770331639636611612094499734996204129162575412', 53, 8);
    t('-74657560210242704554422776032639005330.452019595638291987749374750110848265985', '-74657560210242704554422776032639005330.45201959563829198774937475011084826598462291919991235622365', 39, 8);
    t('-470224275317729788265083630786402723652606489581196807283808757461349811868180256', '-470224275317729788265083630786402723652606489581196807283808757461349811868180256', 30, 8);
    t('1548861035159077785270281667791270424763253422330193371408', '1548861035159077785270281667791270424763253422330193371408', 39, 8);
    t('572490113577029960833837229080147476156510343599468021', '572490113577029960833837229080147476156510343599468021', 53, 8);
    t('643806729774291640618659309341323785812549216455045455115986466479734', '643806729774291640618659309341323785812549216455045455115986466479734', 59, 8);
    t('-0.0000000000000028832370380139686337031164', '-0.00000000000000288323703801396863370311643469222712049186100527525476992', 40, 8);
    t('-7189938709688507888323269632860908682422507867792553339856907644331197', '-7189938709688507888323269632860908682422507867792553339856907644331197', 30, 8);
    t('-146458270834000351215786805348778503451274270331400574216733292', '-146458270834000351215786805348778503451274270331400574216733292', 53, 8);
    t('-0.00000354623728604661459577622785392304170907', '-0.00000354623728604661459577622785392304170906690465556766249117223748961549701782239', 44, 8);
    t('2952633202166896462404492499220990667245459193', '2952633202166896462404492499220990667245459193', 18, 8);
    t('-40658243711269788486614574882823442616218332307814675126120483345884287', '-40658243711269788486614574882823442616218332307814675126120483345884287', 35, 8);
    t('-834746.130874791201613251929057183287554326432137609', '-834746.13087479120161325192905718328755432643213760902671', 45, 8);
    t('-2740259405495848461193558179879029219965128521581766328834436598769356651654532', '-2740259405495848461193558179879029219965128521581766328834436598769356651654532', 21, 8);
    t('-8698435709878691464261', '-8698435709878691464261', 24, 8);
    t('-4854.3960864787394404064207224627696634919', '-4854.39608647873944040642072246276966349186696614541346739', 37, 8);
    t('-9046342079405156048260593050430368806', '-9046342079405156048260593050430368806', 13, 8);
    t('-0.0004000833397214913009450572039483114734165469785', '-0.0004000833397214913009450572039483114734165469785253268388', 49, 8);
    t('-23588224945151001125418674123739250937217610361057096547', '-23588224945151001125418674123739250937217610361057096547', 50, 8);
    t('-7464158850661582447.135098739', '-7464158850661582447.135098739246', 9, 8);
    t('-1120821804664354.438155', '-1120821804664354.438154570308756776805085557342686456892402105759461492146293434', 6, 8);
    t('0', '0.00000000000000702791991239841893249958321', 6, 8);
    t('-0.0000000000000000000919', '-0.00000000000000000009190566893831833839554', 22, 8);
    t('64989966866.84348665817102620621477817048264530787181', '64989966866.8434866581710262062147781704826453078718131', 41, 8);
    t('-264353242609519937462651086142583415212476095512802343', '-264353242609519937462651086142583415212476095512802343', 45, 8);
    t('-319660042.9060556155850335364548437', '-319660042.90605561558503353645484368385673087962874009065861', 25, 8);
    t('782876222924236185663713124322579837808875447927.750634733', '782876222924236185663713124322579837808875447927.750634732835826802912809356260197', 9, 8);
    t('-4820197230775172031472341884498005296391387635256546231303736492', '-4820197230775172031472341884498005296391387635256546231303736492', 59, 8);
    t('-45476814068689923138802955966005002117064', '-45476814068689923138802955966005002117064', 40, 8);
    t('86962370844284513181.323731711208686245837285176301766411', '86962370844284513181.323731711208686245837285176301766411023449553787804266696487', 36, 8);
    t('404346421681633141340078973828115038719203880378027932732656742417', '404346421681633141340078973828115038719203880378027932732656742417', 20, 8);
    t('-53694967597040342895345907112.207745463281021175645491335763', '-53694967597040342895345907112.207745463281021175645491335763004711756325', 30, 8);
    t('-6559.90160086515960335469008322764225267411067302815351', '-6559.901600865159603354690083227642252674110673028153509381229252097346854', 50, 8);
    t('-0.00000000061377507317560820930017891219268306', '-0.00000000061377507317560820930017891219268306242240809651039096286454694883355904014170354', 44, 8);
    t('8466762839492012821866.4647006439743751275401538868809', '8466762839492012821866.464700643974375127540153886880890946782195395', 31, 8);
    t('8475.6568', '8475.656838687006572512673250060913774957342621772685866238642718847562956066296778675046839230187', 4, 8);
    t('3994107783920982224771924508121993204610124753099359206657026721569631', '3994107783920982224771924508121993204610124753099359206657026721569631', 29, 8);
    t('-32449538518104.832133372196883764549982574662552753596186838341', '-32449538518104.832133372196883764549982574662552753596186838341255300262261822662991490178499', 48, 8);
    t('0.00000000000000005', '0.0000000000000000488345616040752827', 17, 8);

    t('8225834668202811700000', '8225834668202811700000.5', 0, 8);
    t('-67268029621398311238261431331851556661886545942854113916050734581000000000001', '-67268029621398311238261431331851556661886545942854113916050734581000000000000.5', 0, 8);
    t('4100000', '4100000.5', 0, 8);
    t('694900000000000', '694900000000000.5', 0, 8);
    t('-2668752017792840000000000000000000000000000000000000000000000000000000000001', '-2668752017792840000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('5138303265393258323808903836559900000000000000000', '5138303265393258323808903836559900000000000000000.5', 0, 8);
    t('-61599472977109463564534878506157000000000000000000000000000000001', '-61599472977109463564534878506157000000000000000000000000000000000.5', 0, 8);
    t('-706579187032566119663804136174620000000000000000001', '-706579187032566119663804136174620000000000000000000.5', 0, 8);
    t('10', '10.5', 0, 8);
    t('903942469098308100136613765218227935820271584538277900367517300000000000000000', '903942469098308100136613765218227935820271584538277900367517300000000000000000.5', 0, 8);
    t('7259090494485579718152684060350290000000000000000000000000000000000000000000000000000000000', '7259090494485579718152684060350290000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('50', '50.5', 0, 8);
    t('-336051', '-336050.5', 0, 8);
    t('-359851700000000000000000000000000000000000000000000000000000000000000000000000000000001', '-359851700000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('8300', '8300.5', 0, 8);
    t('-5230000000000000000000000000000000000000000000000001', '-5230000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-832637338257576743857980873934631876338729504830820328750000000000000000000000000000000001', '-832637338257576743857980873934631876338729504830820328750000000000000000000000000000000000.5', 0, 8);
    t('2170', '2170.5', 0, 8);
    t('7530960067300000', '7530960067300000.5', 0, 8);
    t('-69761077704964968462722176065747076185154464193791511210000000000000000000000000000000000000001', '-69761077704964968462722176065747076185154464193791511210000000000000000000000000000000000000000.5', 0, 8);
    t('8951', '8951.5', 0, 8);
    t('-62816928986109317245318155705980721116645949369247436234018847714093151000000000001', '-62816928986109317245318155705980721116645949369247436234018847714093151000000000000.5', 0, 8);
    t('-273450712038160000001', '-273450712038160000000.5', 0, 8);
    t('870651868531859742020859624838429666098184966539196792393834098130000000000000000000', '870651868531859742020859624838429666098184966539196792393834098130000000000000000000.5', 0, 8);
    t('7467352850531742535166203140000', '7467352850531742535166203140000.5', 0, 8);
    t('-27739701695659117725205758576200000000000001', '-27739701695659117725205758576200000000000000.5', 0, 8);
    t('-21575913877090003604970871615441236644670000000000001', '-21575913877090003604970871615441236644670000000000000.5', 0, 8);
    t('-84684097772847106401567201003729899910000000000001', '-84684097772847106401567201003729899910000000000000.5', 0, 8);
    t('5639766204672194231584787000000000000000000000000000000000000000000000000000000000000000000000', '5639766204672194231584787000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('15504935934287862095572900000000000000000000000', '15504935934287862095572900000000000000000000000.5', 0, 8);
    t('-1400577619554361844128289470441748896959600001', '-1400577619554361844128289470441748896959600000.5', 0, 8);
    t('-562508502033790781280182919180003097225841259584126808028780000001', '-562508502033790781280182919180003097225841259584126808028780000000.5', 0, 8);
    t('462676260328650000000000000000000', '462676260328650000000000000000000.5', 0, 8);
    t('80610550887265206737088350828621745655000000000000000000000000000000000000000000000000000000', '80610550887265206737088350828621745655000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-802119959634183327467393167608755000000000000000000000000000000001', '-802119959634183327467393167608755000000000000000000000000000000000.5', 0, 8);
    t('78620000000000000000000000000000000000000', '78620000000000000000000000000000000000000.5', 0, 8);
    t('810000000000000000000000000000000000', '810000000000000000000000000000000000.5', 0, 8);
    t('-3676779349655275521581849548079800000000000000000000000000000000000000000000000000000000001', '-3676779349655275521581849548079800000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-81572917297627759505118569656972784453878787892500000000000001', '-81572917297627759505118569656972784453878787892500000000000000.5', 0, 8);
    t('-198003769523621', '-198003769523620.5', 0, 8);
    t('5306', '5306.5', 0, 8);
    t('-4000000000000000000000000000000000000000001', '-4000000000000000000000000000000000000000000.5', 0, 8);
    t('-6964300000000000000000000000000000001', '-6964300000000000000000000000000000000.5', 0, 8);
    t('5993000000000000000000', '5993000000000000000000.5', 0, 8);
    t('-58', '-57.5', 0, 8);
    t('-314705009617969030000000000000000000000000000000000000000001', '-314705009617969030000000000000000000000000000000000000000000.5', 0, 8);
    t('6', '6.5', 0, 8);
    t('4588255716462680467926051688938462657411100947687253979336100000000000000', '4588255716462680467926051688938462657411100947687253979336100000000000000.5', 0, 8);
    t('300', '300.5', 0, 8);
    t('-8095535225924203212402801', '-8095535225924203212402800.5', 0, 8);
    t('30838279901100000000000000000000000000000000000000000000', '30838279901100000000000000000000000000000000000000000000.5', 0, 8);
    t('-644246265100306600000000000001', '-644246265100306600000000000000.5', 0, 8);
    t('-7949436643533748183697968284137159105717056926000000000000000000000000000000001', '-7949436643533748183697968284137159105717056926000000000000000000000000000000000.5', 0, 8);
    t('5961899981183314783500000000', '5961899981183314783500000000.5', 0, 8);
    t('329320658758032654290364416883005342614380587029114737580920000000000000', '329320658758032654290364416883005342614380587029114737580920000000000000.5', 0, 8);
    t('-70000000000000000000000000001', '-70000000000000000000000000000.5', 0, 8);
    t('-773550954099185953800000000000000000000000000000001', '-773550954099185953800000000000000000000000000000000.5', 0, 8);
    t('-71820001', '-71820000.5', 0, 8);
    t('6824221413055181208880629906440161900000000000000000000000000000000000000000000', '6824221413055181208880629906440161900000000000000000000000000000000000000000000.5', 0, 8);
    t('-293663761783147249577441951060981857156831085100000000000000000000000000000000000000000001', '-293663761783147249577441951060981857156831085100000000000000000000000000000000000000000000.5', 0, 8);
    t('-805331726532205040286379140828190238797602490000000000000000000000000000000000000000001', '-805331726532205040286379140828190238797602490000000000000000000000000000000000000000000.5', 0, 8);
    t('2627542620785923273450000000000000000000', '2627542620785923273450000000000000000000.5', 0, 8);
    t('190055193848119613272218981746041232571688473321679545800000000000000000000000000000000000000000', '190055193848119613272218981746041232571688473321679545800000000000000000000000000000000000000000.5', 0, 8);
    t('-61621063888036802203530857476475979818427083600000000000000000000000000000000000000000000000001', '-61621063888036802203530857476475979818427083600000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-652127458000489992750000000000000000000000000000000000000000000000001', '-652127458000489992750000000000000000000000000000000000000000000000000.5', 0, 8);
    t('7000000000000000000000000', '7000000000000000000000000.5', 0, 8);
    t('6533676582995433778209424794015171896011423181700000000000000', '6533676582995433778209424794015171896011423181700000000000000.5', 0, 8);
    t('27456600275687693800000000000000000000000000000000000000000000000000000000000000000000000000000000', '27456600275687693800000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-194516150000000000000000001', '-194516150000000000000000000.5', 0, 8);
    t('-176140000000001', '-176140000000000.5', 0, 8);
    t('85012661713649137873180295487285610000000000000000000000000000000000000', '85012661713649137873180295487285610000000000000000000000000000000000000.5', 0, 8);
    t('-37523700000000000000000000000000000000000000000000000000000000000000000000000000000001', '-37523700000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-15124465952376999197073295421688223628463231603610811', '-15124465952376999197073295421688223628463231603610810.5', 0, 8);
    t('-5432100777163613256132439687382900000000000000000000000000000000000000000000000001', '-5432100777163613256132439687382900000000000000000000000000000000000000000000000000.5', 0, 8);
    t('652000000', '652000000.5', 0, 8);
    t('-8507892174157676630256932627461950536321146344662200000000000000000000000000000001', '-8507892174157676630256932627461950536321146344662200000000000000000000000000000000.5', 0, 8);
    t('-5780286273116071692453993336522250324675081', '-5780286273116071692453993336522250324675080.5', 0, 8);
    t('2885286893929064654337716318113439600000000000000000000000000000000000000000', '2885286893929064654337716318113439600000000000000000000000000000000000000000.5', 0, 8);
    t('158421702922228687046272106697601990957751203100000000000000000000000000000000000000000000000000', '158421702922228687046272106697601990957751203100000000000000000000000000000000000000000000000000.5', 0, 8);
    t('77845803952870686913386535517781103587210960982825399080400000000000000000000000', '77845803952870686913386535517781103587210960982825399080400000000000000000000000.5', 0, 8);
    t('-65458261744123585088042880333251536411447000000000000000000000000000000000000000000000000001', '-65458261744123585088042880333251536411447000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-5514810000000000001', '-5514810000000000000.5', 0, 8);
    t('-3837413074391000000000000000000000000000001', '-3837413074391000000000000000000000000000000.5', 0, 8);
    t('1891485546514380015490000000000000000000000000', '1891485546514380015490000000000000000000000000.5', 0, 8);
    t('-670001', '-670000.5', 0, 8);
    t('7365874638887372900000000000', '7365874638887372900000000000.5', 0, 8);
    t('80464947550', '80464947550.5', 0, 8);
    t('-913091300000000000000000000000000000000000000000000001', '-913091300000000000000000000000000000000000000000000000.5', 0, 8);
    t('25911127696762546866341398944677062000000000000000000000000', '25911127696762546866341398944677062000000000000000000000000.5', 0, 8);
    t('41448246317918170624465120157939322903000000000000000000000000000000000000000000000000000000000000', '41448246317918170624465120157939322903000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-800000000000000000000000000000000000000000000000000001', '-800000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('577447040809749466752886339212451433404458549958939957382201980490000000000000', '577447040809749466752886339212451433404458549958939957382201980490000000000000.5', 0, 8);
    t('-57572272809174825641085924940000000000000000000000001', '-57572272809174825641085924940000000000000000000000000.5', 0, 8);
    t('31581459400000000000000000000000000000000000000000000000000000000000000000000000000', '31581459400000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-27140000000000000000000000000000000000000000000000000000000000000001', '-27140000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('51603647184997719522882897042563846663899024307208600000000000000000000000', '51603647184997719522882897042563846663899024307208600000000000000000000000.5', 0, 8);
    t('1644323000000', '1644323000000.5', 0, 8);
    t('-9', '-8.5', 0, 8);
    t('-902809102825706989238470350820560000000000000000000000000000000000000000000000000001', '-902809102825706989238470350820560000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-6868477001', '-6868477000.5', 0, 8);
    t('91714927731569000000000000000000000000000000000', '91714927731569000000000000000000000000000000000.5', 0, 8);
    t('-8000000000000000000000000000000000000000000000000000000000000000001', '-8000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-233083085304449000000000000000000000000000000000000000000000000000000000000000000000000000001', '-233083085304449000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('46509073700000', '46509073700000.5', 0, 8);
    t('-1973764439115071357288670461800000000000000001', '-1973764439115071357288670461800000000000000000.5', 0, 8);
    t('-2568500000000000000000001', '-2568500000000000000000000.5', 0, 8);
    t('-24809330903268410000000000000000000000000000000000000000000000000000001', '-24809330903268410000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('7449994364098414600000000000000000000000000000000000000000000', '7449994364098414600000000000000000000000000000000000000000000.5', 0, 8);
    t('2867488032797172375145449433748008158236530896489051275000000000000000000000000000000000', '2867488032797172375145449433748008158236530896489051275000000000000000000000000000000000.5', 0, 8);
    t('-4900000000000000000000000000000000000000000000000000000000000000000000000000001', '-4900000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('4929000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '4929000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('8267546457668258513715771480132070472862216176570000000000000000000000000', '8267546457668258513715771480132070472862216176570000000000000000000000000.5', 0, 8);
    t('593747187059615072947679001676736984922713680000000000000000000000000000000000000000000000', '593747187059615072947679001676736984922713680000000000000000000000000000000000000000000000.5', 0, 8);
    t('2483270425435993889589154244481827665973478012000000000', '2483270425435993889589154244481827665973478012000000000.5', 0, 8);
    t('39543791425933526768174519227303827374238075020570198950767961859654305403206119372345770250000', '39543791425933526768174519227303827374238075020570198950767961859654305403206119372345770250000.5', 0, 8);
    t('67', '67.5', 0, 8);
    t('-219470058205534357335831751487551261713170336001', '-219470058205534357335831751487551261713170336000.5', 0, 8);
    t('-183690000000000000000000000000000000000001', '-183690000000000000000000000000000000000000.5', 0, 8);
    t('3450876609649939508104514905806066940000000000000000000000', '3450876609649939508104514905806066940000000000000000000000.5', 0, 8);
    t('-80540000001', '-80540000000.5', 0, 8);
    t('-62971517498372655372935973087516265926164530906763170287790', '-62971517498372655372935973087516265926164530906763170287789.5', 0, 8);
    t('23409228118870000000', '23409228118870000000.5', 0, 8);
    t('-7000000000000000000000000000000000000001', '-7000000000000000000000000000000000000000.5', 0, 8);
    t('-9031', '-9030.5', 0, 8);
    t('3492817610000000000000000000000000000000000000000000000000000', '3492817610000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-57688287613641931000000000000000000000000001', '-57688287613641931000000000000000000000000000.5', 0, 8);
    t('1984116330670000000000000000000000000000000000000000000000000000000000', '1984116330670000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('6512025216557000000000000000000000000000000000000000000000000000000', '6512025216557000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('12182501403550393384353169696000000000000000000000000000000000000000000000000000000000000000000', '12182501403550393384353169696000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-7425931783170916885202085702943899676988400000000000000000000000000000000000000001', '-7425931783170916885202085702943899676988400000000000000000000000000000000000000000.5', 0, 8);
    t('-4070000000000000000000000000000000000001', '-4070000000000000000000000000000000000000.5', 0, 8);
    t('8647459874624869698149020230000000000000000000000000000000000000000000000000000000000', '8647459874624869698149020230000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('46599000', '46599000.5', 0, 8);
    t('-5010000000000000000000000000000000000000000000000000000000000000000000000000000000000000001', '-5010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('39508000', '39508000.5', 0, 8);
    t('603089057835800884421518508711417801', '603089057835800884421518508711417801.5', 0, 8);
    t('-79471558363526485324781700384768609724195801918000000000000000000000000000001', '-79471558363526485324781700384768609724195801918000000000000000000000000000000.5', 0, 8);
    t('69404910310759646801954680033844865865649000000000000000000000000000000000000000000000000', '69404910310759646801954680033844865865649000000000000000000000000000000000000000000000000.5', 0, 8);
    t('1907980209926030560252116696113309208138159500662562258522400000000000000000000000000000000', '1907980209926030560252116696113309208138159500662562258522400000000000000000000000000000000.5', 0, 8);
    t('329472654458195399229095412173156277813634317805761763485596051933469610596000', '329472654458195399229095412173156277813634317805761763485596051933469610596000.5', 0, 8);
    t('276349000000000', '276349000000000.5', 0, 8);
    t('787126144338547000000000000000000000000000000000000000000000', '787126144338547000000000000000000000000000000000000000000000.5', 0, 8);
    t('1471099043691184058624460905988224213538031110904992774958270704956863896897997000000000000000', '1471099043691184058624460905988224213538031110904992774958270704956863896897997000000000000000.5', 0, 8);
    t('370113010930715428293470648815487051030769793281763667800000000000000000000000', '370113010930715428293470648815487051030769793281763667800000000000000000000000.5', 0, 8);
    t('2065922690807733136485062524396416121715082253637965000000000000000000000000000000000000000000', '2065922690807733136485062524396416121715082253637965000000000000000000000000000000000000000000.5', 0, 8);
    t('-7382816558182407799471782300849610860474300000000000000000000000000000000000000000001', '-7382816558182407799471782300849610860474300000000000000000000000000000000000000000000.5', 0, 8);
    t('-20097560535463220697432958047617508689581603082766010090000000000000000000001', '-20097560535463220697432958047617508689581603082766010090000000000000000000000.5', 0, 8);
    t('628909746633683135845125129915144934623519151508200000', '628909746633683135845125129915144934623519151508200000.5', 0, 8);
    t('-81192352241330340442871946386698010000000000000001', '-81192352241330340442871946386698010000000000000000.5', 0, 8);
    t('6450', '6450.5', 0, 8);
    t('37293916349890940922992467416776788689073657558307952866668243095547190629248951386344800000000', '37293916349890940922992467416776788689073657558307952866668243095547190629248951386344800000000.5', 0, 8);
    t('1718544737998848', '1718544737998848.5', 0, 8);
    t('87231983170377993831727615571315627607167084700000000000', '87231983170377993831727615571315627607167084700000000000.5', 0, 8);
    t('-86061664983179049931693406385826804039300476826004230043833291400000000000001', '-86061664983179049931693406385826804039300476826004230043833291400000000000000.5', 0, 8);
    t('-51891604836621651634017359747812499890000000000000000000000000000000000000000000000000001', '-51891604836621651634017359747812499890000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('8745224135400000000000000000000', '8745224135400000000000000000000.5', 0, 8);
    t('-30620233581205628000000000000000001', '-30620233581205628000000000000000000.5', 0, 8);
    t('14863144633163334418622520000000000000000', '14863144633163334418622520000000000000000.5', 0, 8);
    t('-7820166128000000000000000000000000000000000000000000000000001', '-7820166128000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('490301854395822993260221610000000000000000000000000000000000000000000000000000000000000000000000000', '490301854395822993260221610000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('89435429126192621776997940079101579134477637123767783805081890000', '89435429126192621776997940079101579134477637123767783805081890000.5', 0, 8);
    t('-8', '-7.5', 0, 8);
    t('48028980775647633995683237409146256399109720500000000000000000000000000000000000000', '48028980775647633995683237409146256399109720500000000000000000000000000000000000000.5', 0, 8);
    t('6335331194137189544436447913317232855829543152436874057500', '6335331194137189544436447913317232855829543152436874057500.5', 0, 8);
    t('-628254630769343648758471830148888672377000000000000000000000000000000000000000000000001', '-628254630769343648758471830148888672377000000000000000000000000000000000000000000000000.5', 0, 8);
    t('57839375650970125223236315959022043520630147028794000000000000000000000', '57839375650970125223236315959022043520630147028794000000000000000000000.5', 0, 8);
    t('-771743548959352866414440497741988370454011151128027700000000000001', '-771743548959352866414440497741988370454011151128027700000000000000.5', 0, 8);
    t('-53429592007706646032950000000000001', '-53429592007706646032950000000000000.5', 0, 8);
    t('499380000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '499380000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-641898557705670000001', '-641898557705670000000.5', 0, 8);
    t('600220992462460556877273974257288999278181263177202351996538136181825896120000', '600220992462460556877273974257288999278181263177202351996538136181825896120000.5', 0, 8);
    t('4060100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '4060100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('-8364590709247094923425000000000000000000000000001', '-8364590709247094923425000000000000000000000000000.5', 0, 8);
    t('519409844212193540764606418404780000000000000000000000000000000000000000', '519409844212193540764606418404780000000000000000000000000000000000000000.5', 0, 8);
    t('-3833671650388000000000000000000000000000000000000000000000000000000000000000000000000000000000000001', '-3833671650388000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('48557176761800000000000000000000000000000000000000', '48557176761800000000000000000000000000000000000000.5', 0, 8);
    t('-7711489971', '-7711489970.5', 0, 8);
    t('75000000000000000000', '75000000000000000000.5', 0, 8);
    t('3229580853248988018262985670791998740861538127939000', '3229580853248988018262985670791998740861538127939000.5', 0, 8);
    t('5248316551608552493188922201763221117178358562272408864992232158101702807890000000000000000', '5248316551608552493188922201763221117178358562272408864992232158101702807890000000000000000.5', 0, 8);
    t('-65467380782429716204000000000001', '-65467380782429716204000000000000.5', 0, 8);
    t('-7205690650357445181000000001', '-7205690650357445181000000000.5', 0, 8);
    t('-276409666783477000000000001', '-276409666783477000000000000.5', 0, 8);
    t('-58428012013195193142046078040582490000000000000000000000000000000000001', '-58428012013195193142046078040582490000000000000000000000000000000000000.5', 0, 8);
    t('-751950207597055427000000000000000000000000000001', '-751950207597055427000000000000000000000000000000.5', 0, 8);
    t('-599626209998027538223544864628000000000000000000000000001', '-599626209998027538223544864628000000000000000000000000000.5', 0, 8);
    t('493818820446671117064848278738045865755408340205207514513699818290870000000000000000000000000', '493818820446671117064848278738045865755408340205207514513699818290870000000000000000000000000.5', 0, 8);
    t('72115246495423678068962420259406315700000', '72115246495423678068962420259406315700000.5', 0, 8);
    t('8355575817405490000000000000000', '8355575817405490000000000000000.5', 0, 8);
    t('-760420001', '-760420000.5', 0, 8);
    t('-2637320151000000000000000000000000000000000000000001', '-2637320151000000000000000000000000000000000000000000.5', 0, 8);
    t('365188058248000000000000000000000000000000000000000000000000000000000000000000000000000000', '365188058248000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 0, 8);
    t('24776070252572097097332452000000000', '24776070252572097097332452000000000.5', 0, 8);
    t('-51423316645779393318000000000000001', '-51423316645779393318000000000000000.5', 0, 8);
    t('623144979783067430949482841363102540602541201810000000000000000000000000000000000000000000000', '623144979783067430949482841363102540602541201810000000000000000000000000000000000000000000000.5', 0, 8);
    t('-393701403238457991332600000000000000001', '-393701403238457991332600000000000000000.5', 0, 8);
    t('255574900000000000000000000000000000000000000000000000', '255574900000000000000000000000000000000000000000000000.5', 0, 8);
    t('361300718195583086529006200000000000000', '361300718195583086529006200000000000000.5', 0, 8);
    t('61381935060212901891392491994620938159434302730207737183495000000000000', '61381935060212901891392491994620938159434302730207737183495000000000000.5', 0, 8);
    t('17103494622240782188191669794453543590000000000000000000000000000000', '17103494622240782188191669794453543590000000000000000000000000000000.5', 0, 8);

    t('12.35', '12.345', 2);
    t('12', '12.345', 0);
    t('12', '12.345', -0);

    function tx(fn, msg){
        Test.isException(fn, msg);
    }

    tx(function () {new BigNumber('12.345').dp(NaN)}, ".dp(NaN)");
    tx(function () {new BigNumber('12.345').dp('NaN')}, ".dp('NaN')");
    tx(function () {new BigNumber('12.345').dp([])}, ".dp([])");
    tx(function () {new BigNumber('12.345').dp({})}, ".dp({})");
    tx(function () {new BigNumber('12.345').dp('')}, ".dp('')");
    tx(function () {new BigNumber('12.345').dp(' ')}, ".dp(' ')");
    tx(function () {new BigNumber('12.345').dp('hello')}, ".dp('hello')");
    tx(function () {new BigNumber('12.345').dp('\t')}, ".dp('\t')");
    tx(function () {new BigNumber('12.345').dp(new Date)}, ".dp(new Date)");
    tx(function () {new BigNumber('12.345').dp(new RegExp)}, ".dp(new RegExp)");
    tx(function () {new BigNumber('12.345').dp(7.5)}, ".dp(7.5)");
    tx(function () {new BigNumber('12.345').dp('-1.1e1')}, ".dp('-1.1e1')");

    tx(function () {new BigNumber('12.345').dp('-1')}, ".dp('-1')");
    tx(function () {new BigNumber('12.345').dp(-23)}, ".dp(-23)");
    tx(function () {new BigNumber('12.345').dp(MAX + 1)}, ".dp(MAX + 1)");
    tx(function () {new BigNumber('12.345').dp(MAX + 0.1)}, ".dp(MAX + 1)");
    tx(function () {new BigNumber('12.345').dp('-0.01')}, ".dp('-0.01')");
    tx(function () {new BigNumber('12.345').dp('-1e-1')}, ".dp('-1e-1')");
    tx(function () {new BigNumber('12.345').dp(Infinity)}, ".dp(Infinity)");
    tx(function () {new BigNumber('12.345').dp('-Infinity')}, ".dp('-Infinity')");

    t('13', '12.345', 0, 2);
    t('12', '12.345', 0, null);
    t('12', '12.345', 0);
    t('13', '12.345', 0, 0);
    t('13', '12.345', 0, -0);

    tx(function () {new BigNumber('12.345').dp(0, NaN)}, ".dp(0, NaN)");
    tx(function () {new BigNumber('12.345').dp(0, 'NaN')}, ".dp(0, 'NaN')");
    tx(function () {new BigNumber('12.345').dp(0, [])}, ".dp(0, [])");
    tx(function () {new BigNumber('12.345').dp(0, {})}, ".dp(0, {})");
    tx(function () {new BigNumber('12.345').dp(0, '')}, ".dp(0, '')");
    tx(function () {new BigNumber('12.345').dp(0, ' ')}, ".dp(0, ' ')");
    tx(function () {new BigNumber('12.345').dp(0, 'hello')}, ".dp(0, 'hello')");
    tx(function () {new BigNumber('12.345').dp(0, '\t')}, ".dp(0, '\t')");
    tx(function () {new BigNumber('12.345').dp(0, new Date)}, ".dp(0, new Date)");
    tx(function () {new BigNumber('12.345').dp(0, new RegExp)}, ".dp(0, new RegExp)");
    tx(function () {new BigNumber('12.345').dp(0, 7.5)}, ".dp(0, 7.5)");
    tx(function () {new BigNumber('12.345').dp(0, '-1.1e1')}, ".dp(0, '-1.1e1')");

    tx(function () {new BigNumber('12.345').dp(0, '-1')}, ".dp(0, -1')");
    tx(function () {new BigNumber('12.345').dp(0, -23)}, ".dp(0, -23)");
    tx(function () {new BigNumber('12.345').dp(0, 8.01)}, ".dp(0, 8.01)");
    tx(function () {new BigNumber('12.345').dp(0, 9)}, ".dp(0, 9)");
    tx(function () {new BigNumber('12.345').dp(0, -1)}, ".dp(0, -1)");
    tx(function () {new BigNumber('12.345').dp(0, '-0.01')}, ".dp(0, '-0.01')");
    tx(function () {new BigNumber('12.345').dp(0, '-1e-1')}, ".dp(0, '-1e-1')");
    tx(function () {new BigNumber('12.345').dp(0, Infinity)}, ".dp(0, Infinity)");
    tx(function () {new BigNumber('12.345').dp(0, '-Infinity')}, ".dp(0, '-Infinity')");
});
