/* ----------------------------------------------------------------------------
 * input_manager.c
 * code to distribute input events to modules
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include "pbbinput.h"

#include <glib.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "systems.h"
#include "init.h"
#include "input_manager.h"

extern volatile sig_atomic_t prgexit;

/* --- private module data structure of the input manager --- */

struct moddata_inputmanager {
	GList		*inputsources;

	struct {
		unsigned int shift:1;
		unsigned int ctrl:1;
		unsigned int alt:1;
		unsigned int :0;
	} flags;
	unsigned short lastkey;
	inputqueue_t *iqueues[QUEUECOUNT];
	inputqueue_t kbdqueue[MODULECOUNT];
	inputqueue_t mousequeue[MODULECOUNT];
	inputqueue_t t100queue[MODULECOUNT];
	inputqueue_t t1000queue[MODULECOUNT];
	inputqueue_t queryqueue[MODULECOUNT];
	inputqueue_t configqueue[MODULECOUNT];
	inputqueue_t securequeue[MODULECOUNT];
} modbase_inputmanager;

/* --- interface functions of the inputmanager --- */

/* Init function called by the program init procedure. Function is known by
 * the prginitab. It does module data initialisation.
 */
int
inputmanager_init ()
{
	struct moddata_inputmanager *base = &modbase_inputmanager;
	int x, y, fd;

	base->inputsources = NULL;
	scanEventDevices();

#if 0    /* see mice_handler for comment */
	if ((fd = open("/dev/input/mice", O_RDONLY)) >= 0)  /* open mouse device */
		addInputSource (fd, mice_handler, NULL, TRUE);
#endif

	base->flags.shift = 0;
	base->flags.ctrl = 0;
	base->flags.alt = 0;
	base->lastkey = 0;

	base->iqueues[KBDQUEUE] = base->kbdqueue;
	base->iqueues[MOUSEQUEUE] = base->mousequeue;
	base->iqueues[T100QUEUE] = base->t100queue;
	base->iqueues[T1000QUEUE] = base->t1000queue;
	base->iqueues[QUERYQUEUE] = base->queryqueue;
	base->iqueues[CONFIGQUEUE] = base->configqueue;
	base->iqueues[SECUREQUEUE] = base->securequeue;

	for (y=0; y < QUEUECOUNT; y++)
		for (x=0; x < MODULECOUNT; x++)
			base->iqueues[y][x] = NULL;

	return 0;
}

void
cbFreeInputSource (gpointer data, gpointer user_data)
{
	InputSource *src = data;
	g_source_remove (src->watch);
}

int
inputmanager_exit ()
{
	struct moddata_inputmanager *base = &modbase_inputmanager;

	g_list_foreach (base->inputsources, cbFreeInputSource, NULL);
	return 0;
}

/* ----------- */

int
track_modifier(unsigned short code, unsigned int value)
{
	struct moddata_inputmanager *base = &modbase_inputmanager;

	if (value > 1) value = 1;   /* key repeated = pressed */
	switch (code)
	{
		case KEY_RIGHTSHIFT:
		case KEY_LEFTSHIFT:
			base->flags.shift = value;
			break;
		case KEY_RIGHTCTRL:
		case KEY_LEFTCTRL:
			base->flags.ctrl = value;
			break;
		case KEY_RIGHTALT:
		case KEY_LEFTALT:
			base->flags.alt = value;
			break;
	}
	return (base->flags.ctrl << 2) | (base->flags.alt << 1) | (base->flags.shift << 0);
}

int
get_modifier()
{
  struct moddata_inputmanager *base = &modbase_inputmanager;
  return (base->flags.ctrl << 2) | (base->flags.alt << 1) | (base->flags.shift << 0);
}

/* --- event handler management --- */

gint
cbEventDevices (gconstpointer a, gconstpointer venprod)
{
	const InputSource *src = a;
	
	if (src->user_data == venprod)
		return 0;
	return 1;
}

void
scanEventDevices ()
{
	struct moddata_inputmanager *base = &modbase_inputmanager;
	int n, fd, fcnt = 0;
	char filename[20];
	gpointer evdevid;

	for (n = 0; n < EVDEVCOUNT; n++) {
		sprintf(filename, "/dev/input/event%d", n);
		if ((fd = open(filename, O_RDONLY)) >= 0) {
			evdevid = (gpointer) (0x45460000 | n);
			if ((g_list_find_custom (base->inputsources, evdevid, cbEventDevices)) == NULL)
				addInputSource (fd, event_handler, evdevid, TRUE);
			else
				close(fd);
		} else
			fcnt++;  /* count number of event devices that could not be opened */
	}

	if (fcnt == EVDEVCOUNT)  /* none of the 32 devices can be opened */
		print_msg (PBB_WARN, _("No event devices available. Please check your configuration.\n"));
}

/* --- input source --- */

void
destroyInputSource (gpointer data)
{
	struct moddata_inputmanager *base = &modbase_inputmanager;
	InputSource *src = data;
#ifdef DEBUG
	print_msg (PBB_INFO, "DBG: InputSource removed: %08x\n", src->user_data);
#endif
	base->inputsources = g_list_remove (base->inputsources, src);
	g_free (src);
}

gboolean
handleInputSource (GIOChannel *io, GIOCondition condition, gpointer data)
{
	InputSource *src = data;
	gboolean rc = TRUE;
	
	if (condition & G_IO_IN) {
		/* call the appropriate handler. We assume that io = src->io.
		 * In case of an error the InputSource will be removed to
		 * prevent further problems reading invalid devices. */
		rc = src->handler (g_io_channel_unix_get_fd (io), src->user_data);
	} else
		/* GIOChannel reported an error */
		rc = FALSE;

#ifdef DEBUG
	if (rc == FALSE) {
		print_msg (PBB_INFO, "DBG: InputSource received %s %s %s\n",
				(condition & G_IO_IN)  ? "G_IO_IN"  : "",
		        (condition & G_IO_ERR) ? "G_IO_ERR" : "",
		        (condition & G_IO_HUP) ? "G_IO_HUP" : "");
	}
#endif
	return rc;
}

InputSource*
addInputSource (int fd, void *handler, gpointer user_data, gboolean close_on_exit)
{
	struct moddata_inputmanager *base = &modbase_inputmanager;
	InputSource *src;

	src = g_new0 (InputSource, 1);
	src->io = g_io_channel_unix_new (fd);
	
	/* set channel encoding to binary */
	g_io_channel_set_encoding (src->io, NULL, NULL);

	/* Bound the file handle to the IOChannel. The file handle will be
	 * closed as soon as this IOChannel got obsolete. If this shouldn't
	 * happen, give 'close_on_exit' = FALSE. */
	if (close_on_exit == TRUE)
		g_io_channel_set_close_on_unref (src->io, TRUE);
	
	src->handler   = handler;
	src->user_data = user_data;
	src->watch   = g_io_add_watch_full (src->io, G_PRIORITY_DEFAULT,
			G_IO_IN | G_IO_ERR | G_IO_HUP, handleInputSource, src,
			destroyInputSource);

	/* g_io_channel_unix_new() and g_io_add_watch both increment the
	 * ref counter but we want to get rid of the IOchannel too, when
	 * the source will be removed. */
	g_io_channel_unref (src->io);

	/* Attach the newly created InputSource to the global list */
	base->inputsources = g_list_append (base->inputsources, src);
#ifdef DEBUG
	print_msg (PBB_INFO, "DBG: InputSource added:   %08x\n", src->user_data);
#endif
	return src;
}

/* --- queue managers --- */

int
register_function (int queueid, void *func)
{
  struct moddata_inputmanager *base = &modbase_inputmanager;
  inputqueue_t *queue = base->iqueues[queueid];
  int n;

  for (n=0; n < MODULECOUNT; n++)
    if (queue[n] == NULL) {
      queue[n] = (inputqueue_t) func;
      return 0;
    }
  return -1;
}

long
process_queue_single (int queueid, long tag, long data)
{
  struct tagitem taglist[2];

  taglist_init (taglist);
  taglist_add (taglist, tag, data);
  process_queue (queueid, taglist);
  return taglist->data;
}

int
process_queue (int queueid, struct tagitem *taglist)
{
	struct moddata_inputmanager *base = &modbase_inputmanager;
	inputqueue_t *queue = base->iqueues[queueid];
	int n=0;

	if (queueid == KBDQUEUE) {
		if ((tagfind (taglist, TAG_KEYREPEAT, 0))) {
			base->lastkey = tagfind (taglist, TAG_KEYCODE, 0);
		} else
			base->lastkey = 0;
	}
	while ((queue[n] != NULL) && (n < MODULECOUNT)) {
		queue[n++] (taglist);
	}
	return n;
}

/* --- input handlers --- */

void
ipc_handler ()
{
	struct pbbmessage *msg;
	struct tagitem *source, *dest;
	char msgbuffer[8192];

	if ((ipc_receive (msgbuffer, sizeof(msgbuffer))) == 0) {
		msg = (struct pbbmessage *) msgbuffer;
		switch (msg->action) {
		case READVALUE:
			process_queue (QUERYQUEUE, msg->taglist);
			ipc_send (msg->returnport, CHANGEVALUE, msg->taglist);
			break;
		case CHANGEVALUE:
			process_queue (CONFIGQUEUE, msg->taglist);
			source = dest = msg->taglist;
			while (source->tag != TAG_END) {
				if ((source->tag & FLG_ERROR)) {
					dest->tag = source->tag;
					dest->data = source->data;
					dest++;
				}
				source++;
			}
			dest->tag = TAG_END;
			dest->data = 0;
			ipc_send (msg->returnport, CHANGEERROR, msg->taglist);
			break;
		}
	}
}

gboolean
event_handler (int fd, gpointer user_data)
{
	struct input_event inp;
	struct tagitem taglist[10];
	gboolean rc = FALSE;
	
	taglist_init(taglist);
	
	if (read(fd, &inp, sizeof(inp)) == sizeof(inp)) {
		switch (inp.type) {
		case EV_KEY:
			taglist_add (taglist, TAG_KEYCODE,  (long) inp.code);
			taglist_add (taglist, TAG_KEYREPEAT,(long) inp.value);
			taglist_add (taglist, TAG_MODIFIER, (long) track_modifier (inp.code, inp.value));
			process_queue (KBDQUEUE, taglist);
			break;
		case EV_REL:
			/* TAG_MOUSERELx are currently not used. Only the occourence
			 * of this events will be used */
			if (inp.code)
				taglist_add (taglist, TAG_MOUSERELY, (long) inp.value);
			else
				taglist_add (taglist, TAG_MOUSERELX, (long) inp.value);
    		process_queue (MOUSEQUEUE, taglist);
			break;
		case EV_ABS:
			/* Mouse events are not used. Only the occurence of this events
			 * will currently be used */
    		process_queue (MOUSEQUEUE, taglist);
			break;
		default:
			break; /* catch unknown event types */
		}
		rc = TRUE;
	}
	return rc;
}

/* This function is a tribute to the buggy Xorg synaptics driver.
 * This driver blocks the event device of the trackpad for exclusive
 * use so pbbuttonsd won't get any events from the trackpad. This
 * function uses /dev/input/mice in parallel to the event device
 * so we hopefully tick the synaptics driver.
 *
 * Unfortunately the synaptics driver pushed the mouse events read
 * from the event device directly to X and doesn't mirrot them to
 * /dev/event/mice. This makes it impossible to work around this
 * problem and make the following handler useless. (2006-09-23 MG)
 */
#if 0
gboolean
mice_handler (int fd, gpointer user_data)
{
	signed char buffer[] = {0, 0, 0 ,0};
	struct tagitem taglist[3];
	int count;

	taglist_init(taglist);
	if ((count = read(fd, buffer, sizeof(buffer))) >= 3) {
		taglist_add (taglist, TAG_MOUSERELX, (long) buffer[1]);
		taglist_add (taglist, TAG_MOUSERELY, (long) buffer[2]);
    	process_queue (MOUSEQUEUE, taglist);
	}
	return TRUE;
}
#endif

gboolean
cb_timer100 (gpointer data)
{
  struct moddata_inputmanager *base = &modbase_inputmanager;
  struct tagitem taglist[] = { { TAG_END, 0 } };
  int mod;
  
  process_queue (T100QUEUE, taglist);
 
  mod = get_modifier();
  if ((mod == 0) && (base->lastkey == 0))
    process_queue (SECUREQUEUE, taglist);
  
  return TRUE;
}

gboolean
cb_timer1000 (gpointer data)
{
  struct tagitem taglist[] = { { TAG_END, 0 } };
  process_queue (T1000QUEUE, taglist);
  return TRUE;
}

