// Package filesystem implements the Azure ARM Filesystem service API version
// 2015-10-01-preview.
//
// Creates an Azure Data Lake Store filesystem client.
package filesystem

// Copyright (c) Microsoft and contributors.  All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Code generated by Microsoft (R) AutoRest Code Generator 0.17.0.0
// Changes may cause incorrect behavior and will be lost if the code is
// regenerated.

import (
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"io"
	"net/http"
)

const (
	// APIVersion is the version of the Filesystem
	APIVersion = "2015-10-01-preview"

	// DefaultBaseURI is the default URI used for the service Filesystem
	DefaultBaseURI = "https://localhost"
)

// ManagementClient is the base client for Filesystem.
type ManagementClient struct {
	autorest.Client
	BaseURI                 string
	APIVersion              string
	AdlsFileSystemDNSSuffix string
}

// New creates an instance of the ManagementClient client.
func New(adlsFileSystemDNSSuffix string) ManagementClient {
	return NewWithBaseURI(DefaultBaseURI, adlsFileSystemDNSSuffix)
}

// NewWithBaseURI creates an instance of the ManagementClient client.
func NewWithBaseURI(baseURI string, adlsFileSystemDNSSuffix string) ManagementClient {
	return ManagementClient{
		Client:                  autorest.NewClientWithUserAgent(UserAgent()),
		BaseURI:                 baseURI,
		APIVersion:              APIVersion,
		AdlsFileSystemDNSSuffix: adlsFileSystemDNSSuffix,
	}
}

// Append appends to the specified file. This method does not support multiple
// concurrent appends to the file. NOTE: Concurrent append and normal
// (serial) append CANNOT be used interchangeably. Once a file has been
// appended to using either append option, it can only be appended to using
// that append option. Use the ConcurrentAppend option if you would like
// support for concurrent appends.
//
// directFilePath is the Data Lake Store path (starting with '/') of the file
// to which to append. streamContents is the file contents to include when
// appending to the file. streamContents will be closed upon successful
// return. Callers should ensure closure when receiving an error.op is the
// constant value for the operation. appendParameter is the constant value
// for the operation. transferEncoding is indicates the data being sent to
// the server is being streamed in chunks. offset is the optional offset in
// the stream to begin the append operation. Default is to append at the end
// of the stream.
func (client ManagementClient) Append(directFilePath string, streamContents io.ReadCloser, op string, appendParameter string, transferEncoding string, offset *int64) (result autorest.Response, err error) {
	if streamContents == nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Append", nil, "Required parameter is empty - 'streamContents'")
	}

	req, err := client.AppendPreparer(directFilePath, streamContents, op, appendParameter, transferEncoding, offset)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Append", nil, "Failure preparing request")
	}

	resp, err := client.AppendSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Append", resp, "Failure sending request")
	}

	result, err = client.AppendResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Append", resp, "Failure responding to request")
	}

	return
}

// AppendPreparer prepares the Append request.
func (client ManagementClient) AppendPreparer(directFilePath string, streamContents io.ReadCloser, op string, appendParameter string, transferEncoding string, offset *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"directFilePath": autorest.Encode("path", directFilePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"append":      autorest.Encode("query", appendParameter),
		"op":          autorest.Encode("query", op),
	}
	if offset != nil {
		queryParameters["offset"] = autorest.Encode("query", *offset)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{directFilePath}", pathParameters),
		autorest.WithFile(streamContents),
		autorest.WithQueryParameters(queryParameters),
		autorest.WithHeader("Transfer-Encoding", autorest.String(transferEncoding)))
	return preparer.Prepare(&http.Request{})
}

// AppendSender sends the Append request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) AppendSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// AppendResponder handles the response to the Append request. The method always
// closes the http.Response Body.
func (client ManagementClient) AppendResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// CheckAccess checks if the specified access is available at the given path.
//
// pathParameter is the Data Lake Store path (starting with '/') of the file
// or directory for which to check access. op is the constant value for the
// operation. fsaction is file system operation read/write/execute in string
// form, matching regex pattern '[rwx-]{3}'
func (client ManagementClient) CheckAccess(pathParameter string, op string, fsaction string) (result autorest.Response, err error) {
	req, err := client.CheckAccessPreparer(pathParameter, op, fsaction)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "CheckAccess", nil, "Failure preparing request")
	}

	resp, err := client.CheckAccessSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "CheckAccess", resp, "Failure sending request")
	}

	result, err = client.CheckAccessResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "CheckAccess", resp, "Failure responding to request")
	}

	return
}

// CheckAccessPreparer prepares the CheckAccess request.
func (client ManagementClient) CheckAccessPreparer(pathParameter string, op string, fsaction string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if len(fsaction) > 0 {
		queryParameters["fsaction"] = autorest.Encode("query", fsaction)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// CheckAccessSender sends the CheckAccess request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) CheckAccessSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// CheckAccessResponder handles the response to the CheckAccess request. The method always
// closes the http.Response Body.
func (client ManagementClient) CheckAccessResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Concat concatenates the list of source files into the destination file,
// removing all source files upon success.
//
// destinationPath is the Data Lake Store path (starting with '/') of the
// destination file resulting from the concatenation. sources is a list of
// comma seperated Data Lake Store paths (starting with '/') of the files to
// concatenate, in the order in which they should be concatenated. op is the
// constant value for the operation.
func (client ManagementClient) Concat(destinationPath string, sources []string, op string) (result autorest.Response, err error) {
	if sources == nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Concat", nil, "Required parameter is empty - 'sources'")
	}

	req, err := client.ConcatPreparer(destinationPath, sources, op)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Concat", nil, "Failure preparing request")
	}

	resp, err := client.ConcatSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Concat", resp, "Failure sending request")
	}

	result, err = client.ConcatResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Concat", resp, "Failure responding to request")
	}

	return
}

// ConcatPreparer prepares the Concat request.
func (client ManagementClient) ConcatPreparer(destinationPath string, sources []string, op string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"destinationPath": autorest.Encode("path", destinationPath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
		"sources":     autorest.Encode("query", sources, ","),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{destinationPath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// ConcatSender sends the Concat request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) ConcatSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// ConcatResponder handles the response to the Concat request. The method always
// closes the http.Response Body.
func (client ManagementClient) ConcatResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ConcurrentAppend appends to the specified file. This method supports
// multiple concurrent appends to the file. NOTE: Concurrent append and
// normal (serial) append CANNOT be used interchangeably. Once a file has
// been appended to using either append option, it can only be appended to
// using that append option.
//
// filePath is the Data Lake Store path (starting with '/') of the file to
// which to append using concurrent append. streamContents is the file
// contents to include when appending to the file. streamContents will be
// closed upon successful return. Callers should ensure closure when
// receiving an error.op is the constant value for the operation.
// transferEncoding is indicates the data being sent to the server is being
// streamed in chunks. appendMode is indicates the concurrent append call
// should create the file if it doesn't exist or just open the existing file
// for append. Possible values include: 'autocreate'
func (client ManagementClient) ConcurrentAppend(filePath string, streamContents io.ReadCloser, op string, transferEncoding string, appendMode AppendModeType) (result autorest.Response, err error) {
	if streamContents == nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "ConcurrentAppend", nil, "Required parameter is empty - 'streamContents'")
	}

	req, err := client.ConcurrentAppendPreparer(filePath, streamContents, op, transferEncoding, appendMode)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "ConcurrentAppend", nil, "Failure preparing request")
	}

	resp, err := client.ConcurrentAppendSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "ConcurrentAppend", resp, "Failure sending request")
	}

	result, err = client.ConcurrentAppendResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "ConcurrentAppend", resp, "Failure responding to request")
	}

	return
}

// ConcurrentAppendPreparer prepares the ConcurrentAppend request.
func (client ManagementClient) ConcurrentAppendPreparer(filePath string, streamContents io.ReadCloser, op string, transferEncoding string, appendMode AppendModeType) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"filePath": autorest.Encode("path", filePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if len(string(appendMode)) > 0 {
		queryParameters["appendMode"] = autorest.Encode("query", appendMode)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/WebHdfsExt/{filePath}", pathParameters),
		autorest.WithFile(streamContents),
		autorest.WithQueryParameters(queryParameters),
		autorest.WithHeader("Transfer-Encoding", autorest.String(transferEncoding)))
	return preparer.Prepare(&http.Request{})
}

// ConcurrentAppendSender sends the ConcurrentAppend request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) ConcurrentAppendSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// ConcurrentAppendResponder handles the response to the ConcurrentAppend request. The method always
// closes the http.Response Body.
func (client ManagementClient) ConcurrentAppendResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Create creates a file with optionally specified content.
//
// directFilePath is the Data Lake Store path (starting with '/') of the file
// to create. op is the constant value for the operation. write is the
// constant value for the operation. transferEncoding is indicates the data
// being sent to the server is being streamed in chunks. streamContents is
// the file contents to include when creating the file. This parameter is
// optional, resulting in an empty file if not specified. streamContents will
// be closed upon successful return. Callers should ensure closure when
// receiving an error.overwrite is the indication of if the file should be
// overwritten.
func (client ManagementClient) Create(directFilePath string, op string, write string, transferEncoding string, streamContents io.ReadCloser, overwrite *bool) (result autorest.Response, err error) {
	req, err := client.CreatePreparer(directFilePath, op, write, transferEncoding, streamContents, overwrite)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Create", nil, "Failure preparing request")
	}

	resp, err := client.CreateSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Create", resp, "Failure sending request")
	}

	result, err = client.CreateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Create", resp, "Failure responding to request")
	}

	return
}

// CreatePreparer prepares the Create request.
func (client ManagementClient) CreatePreparer(directFilePath string, op string, write string, transferEncoding string, streamContents io.ReadCloser, overwrite *bool) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"directFilePath": autorest.Encode("path", directFilePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
		"write":       autorest.Encode("query", write),
	}
	if overwrite != nil {
		queryParameters["overwrite"] = autorest.Encode("query", *overwrite)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{directFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters),
		autorest.WithHeader("Transfer-Encoding", autorest.String(transferEncoding)))
	if streamContents != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(streamContents))
	}
	return preparer.Prepare(&http.Request{})
}

// CreateSender sends the Create request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) CreateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// CreateResponder handles the response to the Create request. The method always
// closes the http.Response Body.
func (client ManagementClient) CreateResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusCreated),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Delete deletes the requested file or directory, optionally recursively.
//
// filePath is the Data Lake Store path (starting with '/') of the file or
// directory to delete. op is the constant value for the operation. recursive
// is the optional switch indicating if the delete should be recursive
func (client ManagementClient) Delete(filePath string, op string, recursive *bool) (result FileOperationResult, err error) {
	req, err := client.DeletePreparer(filePath, op, recursive)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Delete", nil, "Failure preparing request")
	}

	resp, err := client.DeleteSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Delete", resp, "Failure sending request")
	}

	result, err = client.DeleteResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Delete", resp, "Failure responding to request")
	}

	return
}

// DeletePreparer prepares the Delete request.
func (client ManagementClient) DeletePreparer(filePath string, op string, recursive *bool) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"filePath": autorest.Encode("path", filePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if recursive != nil {
		queryParameters["recursive"] = autorest.Encode("query", *recursive)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{filePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// DeleteSender sends the Delete request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) DeleteSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// DeleteResponder handles the response to the Delete request. The method always
// closes the http.Response Body.
func (client ManagementClient) DeleteResponder(resp *http.Response) (result FileOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetAclStatus gets Access Control List (ACL) entries for the specified file
// or directory.
//
// aclFilePath is the Data Lake Store path (starting with '/') of the file or
// directory for which to get the ACL. op is the constant value for the
// operation.
func (client ManagementClient) GetAclStatus(aclFilePath string, op string) (result AclStatusResult, err error) {
	req, err := client.GetAclStatusPreparer(aclFilePath, op)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "GetAclStatus", nil, "Failure preparing request")
	}

	resp, err := client.GetAclStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "GetAclStatus", resp, "Failure sending request")
	}

	result, err = client.GetAclStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "GetAclStatus", resp, "Failure responding to request")
	}

	return
}

// GetAclStatusPreparer prepares the GetAclStatus request.
func (client ManagementClient) GetAclStatusPreparer(aclFilePath string, op string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"aclFilePath": autorest.Encode("path", aclFilePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{aclFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetAclStatusSender sends the GetAclStatus request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetAclStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetAclStatusResponder handles the response to the GetAclStatus request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetAclStatusResponder(resp *http.Response) (result AclStatusResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetContentSummary gets the file content summary object specified by the
// file path.
//
// getContentSummaryFilePath is the Data Lake Store path (starting with '/')
// of the file for which to retrieve the summary. op is the constant value
// for the operation.
func (client ManagementClient) GetContentSummary(getContentSummaryFilePath string, op string) (result ContentSummaryResult, err error) {
	req, err := client.GetContentSummaryPreparer(getContentSummaryFilePath, op)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "GetContentSummary", nil, "Failure preparing request")
	}

	resp, err := client.GetContentSummarySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "GetContentSummary", resp, "Failure sending request")
	}

	result, err = client.GetContentSummaryResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "GetContentSummary", resp, "Failure responding to request")
	}

	return
}

// GetContentSummaryPreparer prepares the GetContentSummary request.
func (client ManagementClient) GetContentSummaryPreparer(getContentSummaryFilePath string, op string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"getContentSummaryFilePath": autorest.Encode("path", getContentSummaryFilePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/va/{getContentSummaryFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetContentSummarySender sends the GetContentSummary request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetContentSummarySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetContentSummaryResponder handles the response to the GetContentSummary request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetContentSummaryResponder(resp *http.Response) (result ContentSummaryResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetFileStatus get the file status object specified by the file path.
//
// getFilePath is the Data Lake Store path (starting with '/') of the file or
// directory for which to retrieve the status. op is the constant value for
// the operation.
func (client ManagementClient) GetFileStatus(getFilePath string, op string) (result FileStatusResult, err error) {
	req, err := client.GetFileStatusPreparer(getFilePath, op)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "GetFileStatus", nil, "Failure preparing request")
	}

	resp, err := client.GetFileStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "GetFileStatus", resp, "Failure sending request")
	}

	result, err = client.GetFileStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "GetFileStatus", resp, "Failure responding to request")
	}

	return
}

// GetFileStatusPreparer prepares the GetFileStatus request.
func (client ManagementClient) GetFileStatusPreparer(getFilePath string, op string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"getFilePath": autorest.Encode("path", getFilePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{getFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetFileStatusSender sends the GetFileStatus request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetFileStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetFileStatusResponder handles the response to the GetFileStatus request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetFileStatusResponder(resp *http.Response) (result FileStatusResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ListFileStatus get the list of file status objects specified by the file
// path, with optional pagination parameters
//
// listFilePath is the Data Lake Store path (starting with '/') of the
// directory to list. op is the constant value for the operation. listSize is
// gets or sets the number of items to return. Optional. listAfter is gets or
// sets the item or lexographical index after which to begin returning
// results. For example, a file list of 'a','b','d' and listAfter='b' will
// return 'd', and a listAfter='c' will also return 'd'. Optional. listBefore
// is gets or sets the item or lexographical index before which to begin
// returning results. For example, a file list of 'a','b','d' and
// listBefore='d' will return 'a','b', and a listBefore='c' will also return
// 'a','b'. Optional.
func (client ManagementClient) ListFileStatus(listFilePath string, op string, listSize *int32, listAfter string, listBefore string) (result FileStatusesResult, err error) {
	req, err := client.ListFileStatusPreparer(listFilePath, op, listSize, listAfter, listBefore)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "ListFileStatus", nil, "Failure preparing request")
	}

	resp, err := client.ListFileStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "ListFileStatus", resp, "Failure sending request")
	}

	result, err = client.ListFileStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "ListFileStatus", resp, "Failure responding to request")
	}

	return
}

// ListFileStatusPreparer prepares the ListFileStatus request.
func (client ManagementClient) ListFileStatusPreparer(listFilePath string, op string, listSize *int32, listAfter string, listBefore string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"listFilePath": autorest.Encode("path", listFilePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if listSize != nil {
		queryParameters["listSize"] = autorest.Encode("query", *listSize)
	}
	if len(listAfter) > 0 {
		queryParameters["listAfter"] = autorest.Encode("query", listAfter)
	}
	if len(listBefore) > 0 {
		queryParameters["listBefore"] = autorest.Encode("query", listBefore)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{listFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// ListFileStatusSender sends the ListFileStatus request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) ListFileStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// ListFileStatusResponder handles the response to the ListFileStatus request. The method always
// closes the http.Response Body.
func (client ManagementClient) ListFileStatusResponder(resp *http.Response) (result FileStatusesResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Mkdirs creates a directory.
//
// pathParameter is the Data Lake Store path (starting with '/') of the
// directory to create. op is the constant value for the operation.
func (client ManagementClient) Mkdirs(pathParameter string, op string) (result FileOperationResult, err error) {
	req, err := client.MkdirsPreparer(pathParameter, op)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Mkdirs", nil, "Failure preparing request")
	}

	resp, err := client.MkdirsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Mkdirs", resp, "Failure sending request")
	}

	result, err = client.MkdirsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Mkdirs", resp, "Failure responding to request")
	}

	return
}

// MkdirsPreparer prepares the Mkdirs request.
func (client ManagementClient) MkdirsPreparer(pathParameter string, op string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// MkdirsSender sends the Mkdirs request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) MkdirsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// MkdirsResponder handles the response to the Mkdirs request. The method always
// closes the http.Response Body.
func (client ManagementClient) MkdirsResponder(resp *http.Response) (result FileOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ModifyAclEntries modifies existing Access Control List (ACL) entries on a
// file or folder.
//
// modifyAclFilePath is the Data Lake Store path (starting with '/') of the
// file or directory with the ACL being modified. aclspec is the ACL
// specification included in ACL modification operations in the format
// '[default:]user|group|other::r|-w|-x|-' op is the constant value for the
// operation.
func (client ManagementClient) ModifyAclEntries(modifyAclFilePath string, aclspec string, op string) (result autorest.Response, err error) {
	req, err := client.ModifyAclEntriesPreparer(modifyAclFilePath, aclspec, op)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "ModifyAclEntries", nil, "Failure preparing request")
	}

	resp, err := client.ModifyAclEntriesSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "ModifyAclEntries", resp, "Failure sending request")
	}

	result, err = client.ModifyAclEntriesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "ModifyAclEntries", resp, "Failure responding to request")
	}

	return
}

// ModifyAclEntriesPreparer prepares the ModifyAclEntries request.
func (client ManagementClient) ModifyAclEntriesPreparer(modifyAclFilePath string, aclspec string, op string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"modifyAclFilePath": autorest.Encode("path", modifyAclFilePath),
	}

	queryParameters := map[string]interface{}{
		"aclspec":     autorest.Encode("query", aclspec),
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{modifyAclFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// ModifyAclEntriesSender sends the ModifyAclEntries request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) ModifyAclEntriesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// ModifyAclEntriesResponder handles the response to the ModifyAclEntries request. The method always
// closes the http.Response Body.
func (client ManagementClient) ModifyAclEntriesResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// MsConcat concatenates the list of source files into the destination file,
// deleting all source files upon success. This method accepts more source
// file paths than the Concat method. This method and the parameters it
// accepts are subject to change for usability in an upcoming version.
//
// msConcatDestinationPath is the Data Lake Store path (starting with '/') of
// the destination file resulting from the concatenation. streamContents is a
// list of Data Lake Store paths (starting with '/') of the source files.
// Must be in the format: sources=<comma separated list> streamContents will
// be closed upon successful return. Callers should ensure closure when
// receiving an error.op is the constant value for the operation.
// deleteSourceDirectory is indicates that as an optimization instead of
// deleting each individual source stream, delete the source stream folder if
// all streams are in the same folder instead. This results in a substantial
// performance improvement when the only streams in the folder are part of
// the concatenation operation. WARNING: This includes the deletion of any
// other files that are not source files. Only set this to true when source
// files are the only files in the source directory.
func (client ManagementClient) MsConcat(msConcatDestinationPath string, streamContents io.ReadCloser, op string, deleteSourceDirectory *bool) (result autorest.Response, err error) {
	if streamContents == nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "MsConcat", nil, "Required parameter is empty - 'streamContents'")
	}

	req, err := client.MsConcatPreparer(msConcatDestinationPath, streamContents, op, deleteSourceDirectory)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "MsConcat", nil, "Failure preparing request")
	}

	resp, err := client.MsConcatSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "MsConcat", resp, "Failure sending request")
	}

	result, err = client.MsConcatResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "MsConcat", resp, "Failure responding to request")
	}

	return
}

// MsConcatPreparer prepares the MsConcat request.
func (client ManagementClient) MsConcatPreparer(msConcatDestinationPath string, streamContents io.ReadCloser, op string, deleteSourceDirectory *bool) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"msConcatDestinationPath": autorest.Encode("path", msConcatDestinationPath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if deleteSourceDirectory != nil {
		queryParameters["deleteSourceDirectory"] = autorest.Encode("query", *deleteSourceDirectory)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{msConcatDestinationPath}", pathParameters),
		autorest.WithFile(streamContents),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// MsConcatSender sends the MsConcat request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) MsConcatSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// MsConcatResponder handles the response to the MsConcat request. The method always
// closes the http.Response Body.
func (client ManagementClient) MsConcatResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Open opens and reads from the specified file.
//
// directFilePath is the Data Lake Store path (starting with '/') of the file
// to open. op is the constant value for the operation. read is the constant
// value for the operation.
func (client ManagementClient) Open(directFilePath string, op string, read string, length *int64, offset *int64) (result ReadCloser, err error) {
	req, err := client.OpenPreparer(directFilePath, op, read, length, offset)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Open", nil, "Failure preparing request")
	}

	resp, err := client.OpenSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Open", resp, "Failure sending request")
	}

	result, err = client.OpenResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Open", resp, "Failure responding to request")
	}

	return
}

// OpenPreparer prepares the Open request.
func (client ManagementClient) OpenPreparer(directFilePath string, op string, read string, length *int64, offset *int64) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"directFilePath": autorest.Encode("path", directFilePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
		"read":        autorest.Encode("query", read),
	}
	if length != nil {
		queryParameters["length"] = autorest.Encode("query", *length)
	}
	if offset != nil {
		queryParameters["offset"] = autorest.Encode("query", *offset)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{directFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// OpenSender sends the Open request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) OpenSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// OpenResponder handles the response to the Open request. The method always
// closes the http.Response Body.
func (client ManagementClient) OpenResponder(resp *http.Response) (result ReadCloser, err error) {
	result.Value = &resp.Body
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK))
	result.Response = autorest.Response{Response: resp}
	return
}

// RemoveAclEntries removes existing Access Control List (ACL) entries for a
// file or folder.
//
// removeAclFilePath is the Data Lake Store path (starting with '/') of the
// file or directory with the ACL being removed. aclspec is the ACL spec
// included in ACL removal operations in the format
// '[default:]user|group|other' op is the constant value for the operation.
func (client ManagementClient) RemoveAclEntries(removeAclFilePath string, aclspec string, op string) (result autorest.Response, err error) {
	req, err := client.RemoveAclEntriesPreparer(removeAclFilePath, aclspec, op)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "RemoveAclEntries", nil, "Failure preparing request")
	}

	resp, err := client.RemoveAclEntriesSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "RemoveAclEntries", resp, "Failure sending request")
	}

	result, err = client.RemoveAclEntriesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "RemoveAclEntries", resp, "Failure responding to request")
	}

	return
}

// RemoveAclEntriesPreparer prepares the RemoveAclEntries request.
func (client ManagementClient) RemoveAclEntriesPreparer(removeAclFilePath string, aclspec string, op string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"removeAclFilePath": autorest.Encode("path", removeAclFilePath),
	}

	queryParameters := map[string]interface{}{
		"aclspec":     autorest.Encode("query", aclspec),
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{removeAclFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// RemoveAclEntriesSender sends the RemoveAclEntries request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) RemoveAclEntriesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// RemoveAclEntriesResponder handles the response to the RemoveAclEntries request. The method always
// closes the http.Response Body.
func (client ManagementClient) RemoveAclEntriesResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Rename rename a file or directory.
//
// renameFilePath is the Data Lake Store path (starting with '/') of the file
// or directory to move/rename. destination is the path to move/rename the
// file or folder to op is the constant value for the operation.
func (client ManagementClient) Rename(renameFilePath string, destination string, op string) (result FileOperationResult, err error) {
	req, err := client.RenamePreparer(renameFilePath, destination, op)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Rename", nil, "Failure preparing request")
	}

	resp, err := client.RenameSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Rename", resp, "Failure sending request")
	}

	result, err = client.RenameResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "Rename", resp, "Failure responding to request")
	}

	return
}

// RenamePreparer prepares the Rename request.
func (client ManagementClient) RenamePreparer(renameFilePath string, destination string, op string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"renameFilePath": autorest.Encode("path", renameFilePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"destination": autorest.Encode("query", destination),
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{renameFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// RenameSender sends the Rename request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) RenameSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// RenameResponder handles the response to the Rename request. The method always
// closes the http.Response Body.
func (client ManagementClient) RenameResponder(resp *http.Response) (result FileOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetAcl sets the Access Control List (ACL) for a file or folder.
//
// setAclFilePath is the Data Lake Store path (starting with '/') of the file
// or directory on which to set the ACL. aclspec is the ACL spec included in
// ACL creation operations in the format
// '[default:]user|group|other::r|-w|-x|-' op is the constant value for the
// operation.
func (client ManagementClient) SetAcl(setAclFilePath string, aclspec string, op string) (result autorest.Response, err error) {
	req, err := client.SetAclPreparer(setAclFilePath, aclspec, op)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "SetAcl", nil, "Failure preparing request")
	}

	resp, err := client.SetAclSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "SetAcl", resp, "Failure sending request")
	}

	result, err = client.SetAclResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "SetAcl", resp, "Failure responding to request")
	}

	return
}

// SetAclPreparer prepares the SetAcl request.
func (client ManagementClient) SetAclPreparer(setAclFilePath string, aclspec string, op string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"setAclFilePath": autorest.Encode("path", setAclFilePath),
	}

	queryParameters := map[string]interface{}{
		"aclspec":     autorest.Encode("query", aclspec),
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{setAclFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SetAclSender sends the SetAcl request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) SetAclSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// SetAclResponder handles the response to the SetAcl request. The method always
// closes the http.Response Body.
func (client ManagementClient) SetAclResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// SetOwner sets the owner of a file or directory.
//
// setOwnerFilePath is the Data Lake Store path (starting with '/') of the
// file or directory for which to set the owner. op is the constant value for
// the operation. owner is the AAD Object ID of the user owner of the file or
// directory. If empty, the property will remain unchanged. group is the AAD
// Object ID of the group owner of the file or directory. If empty, the
// property will remain unchanged.
func (client ManagementClient) SetOwner(setOwnerFilePath string, op string, owner string, group string) (result autorest.Response, err error) {
	req, err := client.SetOwnerPreparer(setOwnerFilePath, op, owner, group)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "SetOwner", nil, "Failure preparing request")
	}

	resp, err := client.SetOwnerSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "SetOwner", resp, "Failure sending request")
	}

	result, err = client.SetOwnerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "SetOwner", resp, "Failure responding to request")
	}

	return
}

// SetOwnerPreparer prepares the SetOwner request.
func (client ManagementClient) SetOwnerPreparer(setOwnerFilePath string, op string, owner string, group string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"setOwnerFilePath": autorest.Encode("path", setOwnerFilePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if len(owner) > 0 {
		queryParameters["owner"] = autorest.Encode("query", owner)
	}
	if len(group) > 0 {
		queryParameters["group"] = autorest.Encode("query", group)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{setOwnerFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SetOwnerSender sends the SetOwner request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) SetOwnerSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// SetOwnerResponder handles the response to the SetOwner request. The method always
// closes the http.Response Body.
func (client ManagementClient) SetOwnerResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// SetPermission sets the permission of the file or folder.
//
// setPermissionFilePath is the Data Lake Store path (starting with '/') of
// the file or directory for which to set the permission. op is the constant
// value for the operation. permission is a string representation of the
// permission (i.e 'rwx'). If empty, this property remains unchanged.
func (client ManagementClient) SetPermission(setPermissionFilePath string, op string, permission string) (result autorest.Response, err error) {
	req, err := client.SetPermissionPreparer(setPermissionFilePath, op, permission)
	if err != nil {
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "SetPermission", nil, "Failure preparing request")
	}

	resp, err := client.SetPermissionSender(req)
	if err != nil {
		result.Response = resp
		return result, autorest.NewErrorWithError(err, "filesystem.ManagementClient", "SetPermission", resp, "Failure sending request")
	}

	result, err = client.SetPermissionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.ManagementClient", "SetPermission", resp, "Failure responding to request")
	}

	return
}

// SetPermissionPreparer prepares the SetPermission request.
func (client ManagementClient) SetPermissionPreparer(setPermissionFilePath string, op string, permission string) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"setPermissionFilePath": autorest.Encode("path", setPermissionFilePath),
	}

	queryParameters := map[string]interface{}{
		"api-version": client.APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if len(permission) > 0 {
		queryParameters["permission"] = autorest.Encode("query", permission)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/webhdfs/v1/{setPermissionFilePath}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SetPermissionSender sends the SetPermission request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) SetPermissionSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// SetPermissionResponder handles the response to the SetPermission request. The method always
// closes the http.Response Body.
func (client ManagementClient) SetPermissionResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}
