/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#include "ThumbPane.hpp"

#include "../../gui/Help.hpp"
#include "../../../gnome/gnome.hpp"

#include "wx/gbsizer.h"

#define PADDING 8

using namespace indii;

ThumbPane::ThumbPane(wxWindow* parent) : wxScrolledWindow(parent),
    model(NULL) {
  int i;
  
  /* HSL thumbnails */
  ThumbHSL* thumb;
  for (i = 0; i < 16; i++) {
    thumb = new ThumbHSL(this, i);    
    thumbs.push_back(thumb);
  }  
  
  /* add/remove thumbnail buttons */
  gnome_scale_images(this->GetContentScaleFactor());
  #ifdef __WXMSW__
  /* wxMSW seems to create invisible, but functional, buttons when classic
   * theme is active -- just use text buttons instead */
  butAdd = new wxButton(this, ID_BUT_ADD, _T("1 More"));
  butSub = new wxButton(this, ID_BUT_SUB, _T("1 Less"));
  //butAdd->SetBitmap(*_img_gnome_list_add);
  //butSub->SetBitmap(*_img_gnome_list_remove);
  //butAdd->SetLabel(wxEmptyString);
  //butSub->SetLabel(wxEmptyString);
  #else
  butAdd = new wxBitmapButton(this, ID_BUT_ADD, *_img_gnome_list_add, wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT|wxBORDER_SIMPLE);
  butSub = new wxBitmapButton(this, ID_BUT_SUB, *_img_gnome_list_remove, wxDefaultPosition, wxDefaultSize, wxBU_EXACTFIT|wxBORDER_SIMPLE);
  #endif

  wxGridBagSizer* buttonSizer = new wxGridBagSizer(0,0);
  buttonSizer->Add(butAdd, wxGBPosition(0,0), wxDefaultSpan, wxTOP|wxLEFT|wxBOTTOM, PADDING);
  buttonSizer->Add(butSub, wxGBPosition(0,1), wxDefaultSpan, wxTOP|wxLEFT|wxBOTTOM, PADDING);
  
  /* initial layout */
  wxFlexGridSizer* sizer = new wxFlexGridSizer(1);
  for (i = 0; i < 16; i++) {
    thumbs[i]->Show(false);
  }
  sizer->Add(buttonSizer);
  
  SetScrollRate(10,10);
  SetSizer(sizer);
  int minWidth = thumbs[0]->GetSize().GetWidth() + 30;
  int minHeight = thumbs[0]->GetSize().GetHeight();
  SetClientSize(minWidth, minHeight);
  FitInside();
}

void ThumbPane::setModel(ClusterModel* model) {
  this->model = model;
  ignore();
  for (int i = 0; i < (int)thumbs.size(); ++i) {
    thumbs[i]->setModel(model);
  }

  Enable(model != NULL);
  butAdd->Enable(model != NULL);
  butSub->Enable(model != NULL);
  if (model != NULL) {
    watch(model);
    notifyNumClustersChange();
  }
}

void ThumbPane::updateLayout() {
  const wxSize& size = GetSize();
  const wxSize& thumbSize = thumbs[0]->GetSize();
  int cols = std::max(size.GetWidth() / (thumbSize.GetWidth() + 30), 1);
  
  int i;
  wxFlexGridSizer* sizer = new wxFlexGridSizer(cols);
  wxSizer* oldSizer;
  for (i = 0; i < 16; i++) {
    if (model == NULL || i < model->getNumClusters()) {
      oldSizer = thumbs[i]->GetContainingSizer();
      if (oldSizer != NULL) {
        oldSizer->Detach(thumbs[i]);
      }
      if (model != NULL) {
	      sizer->Add(thumbs[i]);
      }
    }
    thumbs[i]->Show(model != NULL && i < model->getNumClusters());
  }

  wxGridBagSizer* buttonSizer = new wxGridBagSizer(0,0);
  buttonSizer->Add(butAdd, wxGBPosition(0,0), wxDefaultSpan, wxTOP|wxLEFT|wxBOTTOM, PADDING);
  buttonSizer->Add(butSub, wxGBPosition(0,1), wxDefaultSpan, wxTOP|wxLEFT|wxBOTTOM, PADDING);
  sizer->Add(buttonSizer);

  butAdd->Enable(model != NULL && model->getNumClusters() < 16);
  butSub->Enable(model != NULL && model->getNumClusters() > 0);

  SetSizer(sizer);
  sizer->FitInside(this);
}

void ThumbPane::OnSize(wxSizeEvent& evt) {
  updateLayout();
}

void ThumbPane::OnButtonAdd(wxCommandEvent& evt) {
  if (model->getNumClusters() < 16) {
    model->setNumClusters(model->getNumClusters() + 1);
  }
  if (model->getNumClusters() >= 16) {
    butAdd->Disable();
  }
  butSub->Enable();
}

void ThumbPane::OnButtonSub(wxCommandEvent& evt) {
  if (model->getNumClusters() > 1) {
    model->setNumClusters(model->getNumClusters() - 1);
  }
  if (model->getNumClusters() <= 1) {
    butSub->Disable();
  }
  butAdd->Enable();
}

void ThumbPane::OnHelp(wxHelpEvent& evt) {
  Help::show(520);
}

void ThumbPane::notifyNumClustersChange() {
  updateLayout();
}

BEGIN_EVENT_TABLE(ThumbPane, wxScrolledWindow)
EVT_SIZE(ThumbPane::OnSize)
EVT_BUTTON(ID_BUT_ADD, ThumbPane::OnButtonAdd)
EVT_BUTTON(ID_BUT_SUB, ThumbPane::OnButtonSub)
EVT_HELP(wxID_ANY, ThumbPane::OnHelp)
END_EVENT_TABLE()
