/***************************************************************************
    Copyright (C) 2006-2009 Robby Stephenson <robby@periapsis.org>

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU General Public License as        *
 *   published by the Free Software Foundation; either version 2 of        *
 *   the License or (at your option) version 3 or any later version        *
 *   accepted by the membership of KDE e.V. (or its successor approved     *
 *   by the membership of KDE e.V.), which shall act as a proxy            *
 *   defined in Section 14 of version 3 of the license.                    *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 *                                                                         *
 ***************************************************************************/

#include "gamecollection.h"
#include "../tellico_debug.h"

#include <KLocalizedString>

namespace {
  static const char* game_general = I18N_NOOP("General");
  static const char* game_personal = I18N_NOOP("Personal");
}

using Tellico::Data::GameCollection;

GameCollection::GameCollection(bool addDefaultFields_, const QString& title_)
   : Collection(title_.isEmpty() ? i18n("My Games") : title_) {
  setDefaultGroupField(QStringLiteral("platform"));
  if(addDefaultFields_) {
    addFields(defaultFields());
  }
}

Tellico::Data::FieldList GameCollection::defaultFields() {
  FieldList list;
  FieldPtr field;

  list.append(Field::createDefaultField(Field::TitleField));

  QStringList platform;
  platform << platformName(XboxOne) << platformName(Xbox360) << platformName(Xbox)
           << platformName(PlayStation4) << platformName(PlayStation3) << platformName(PlayStation2) << platformName(PlayStation)
           << platformName(PlayStationPortable) << platformName(PlayStationVita)
           << platformName(NintendoSwitch) << platformName(NintendoWiiU)
           << platformName(NintendoWii)  << platformName(Nintendo3DS) << platformName(NintendoDS)
           << platformName(Nintendo64)  << platformName(SuperNintendo) << platformName(Nintendo)
           << platformName(NintendoGameCube) << platformName(Dreamcast)
           << platformName(Windows) << platformName(MacOS) << platformName(Linux);
  field = new Field(QStringLiteral("platform"), i18n("Platform"), platform);
  field->setCategory(i18n(game_general));
  field->setFlags(Field::AllowGrouped);
  list.append(field);

  field = new Field(QStringLiteral("genre"), i18n("Genre"));
  field->setCategory(i18n(game_general));
  field->setFlags(Field::AllowCompletion | Field::AllowMultiple | Field::AllowGrouped);
  field->setFormatType(FieldFormat::FormatPlain);
  list.append(field);

  field = new Field(QStringLiteral("year"), i18n("Release Year"), Field::Number);
  field->setCategory(i18n(game_general));
  field->setFlags(Field::AllowGrouped);
  list.append(field);

  field = new Field(QStringLiteral("publisher"), i18nc("Games - Publisher", "Publisher"));
  field->setCategory(i18n(game_general));
  field->setFlags(Field::AllowCompletion | Field::AllowMultiple | Field::AllowGrouped);
  field->setFormatType(FieldFormat::FormatPlain);
  list.append(field);

  field = new Field(QStringLiteral("developer"), i18n("Developer"));
  field->setCategory(i18n(game_general));
  field->setFlags(Field::AllowCompletion | Field::AllowMultiple | Field::AllowGrouped);
  field->setFormatType(FieldFormat::FormatPlain);
  list.append(field);

  QStringList cert = i18nc("Video game ratings - "
                           "Unrated, Adults Only, Mature, Teen, Everyone 10+, Everyone, Early Childhood, Pending",
                           "Unrated, Adults Only, Mature, Teen, Everyone 10+, Everyone, Early Childhood, Pending")
                     .split(QRegExp(QLatin1String("\\s*,\\s*")), QString::SkipEmptyParts);
  field = new Field(QStringLiteral("certification"), i18n("ESRB Rating"), cert);
  field->setCategory(i18n(game_general));
  field->setFlags(Field::AllowGrouped);
  list.append(field);

  field = new Field(QStringLiteral("description"), i18n("Description"), Field::Para);
  list.append(field);

  field = new Field(QStringLiteral("rating"), i18n("Personal Rating"), Field::Rating);
  field->setCategory(i18n(game_personal));
  field->setFlags(Field::AllowGrouped);
  list.append(field);

  field = new Field(QStringLiteral("completed"), i18n("Completed"), Field::Bool);
  field->setCategory(i18n(game_personal));
  list.append(field);

  field = new Field(QStringLiteral("pur_date"), i18n("Purchase Date"));
  field->setCategory(i18n(game_personal));
  field->setFormatType(FieldFormat::FormatDate);
  list.append(field);

  field = new Field(QStringLiteral("gift"), i18n("Gift"), Field::Bool);
  field->setCategory(i18n(game_personal));
  list.append(field);

  field = new Field(QStringLiteral("pur_price"), i18n("Purchase Price"));
  field->setCategory(i18n(game_personal));
  list.append(field);

  field = new Field(QStringLiteral("loaned"), i18n("Loaned"), Field::Bool);
  field->setCategory(i18n(game_personal));
  list.append(field);

  field = new Field(QStringLiteral("cover"), i18n("Cover"), Field::Image);
  list.append(field);

  field = new Field(QStringLiteral("comments"), i18n("Comments"), Field::Para);
  field->setCategory(i18n(game_personal));
  list.append(field);

  list.append(Field::createDefaultField(Field::IDField));
  list.append(Field::createDefaultField(Field::CreatedDateField));
  list.append(Field::createDefaultField(Field::ModifiedDateField));

  return list;
}


QString GameCollection::normalizePlatform(const QString& platformName_) {
  GamePlatform platform = guessPlatform(platformName_);
  if(platform == UnknownPlatform) {
    QString platformName = platformName_;
    if(platformName.startsWith(QStringLiteral("Microsoft"))) {
      platformName = platformName.mid(sizeof("Microsoft"));
    } else if(platformName.startsWith(QStringLiteral("Sony Playstation"))) {
      // default video game collection has no space between 'PlayStation' and #
      platformName = QStringLiteral("PlayStation") + platformName.mid(sizeof("Sony Playstation"));
    }
    myDebug() << "No default name for" << platformName_ << "; return" << platformName;
    return platformName;
  }
  return platformName(platform);
}

Tellico::Data::GameCollection::GamePlatform GameCollection::guessPlatform(const QString& name_) {
  // try to be smart about guessing the platform from its name
  if(name_.contains(QStringLiteral("PlayStation"), Qt::CaseInsensitive)) {
    if(name_.contains(QStringLiteral("Vita"))) {
      return PlayStationVita;
    } else if(name_.contains(QStringLiteral("Portable"))) {
      return PlayStationPortable;
    } else if(name_.contains(QStringLiteral("4"))) {
      return PlayStation4;
    } else if(name_.contains(QStringLiteral("3"))) {
      return PlayStation3;
    } else if(name_.contains(QStringLiteral("2"))) {
      return PlayStation2;
    } else {
      return PlayStation;
    }
  } else if(name_.contains(QStringLiteral("PSP"))) {
    return PlayStationPortable;
  } else if(name_.contains(QStringLiteral("Xbox"), Qt::CaseInsensitive)) {
    if(name_.contains(QStringLiteral("One"))) {
      return XboxOne;
    } else if(name_.contains(QStringLiteral("360"))) {
      return Xbox360;
    } else {
      return Xbox;
    }
  } else if(name_.contains(QStringLiteral("Switch"))) {
    return NintendoSwitch;
  } else if(name_.contains(QStringLiteral("Wii"))) {
    if(name_.contains(QStringLiteral("U"))) {
      return NintendoWiiU;
    } else {
      return NintendoWii;
    }
  } else if(name_.contains(QStringLiteral("PC")) ||
            name_.contains(QStringLiteral("Windows"))) {
    return Windows;
  } else if(name_.contains(QStringLiteral("Mac"))) {
    return MacOS;
  } else if(name_.contains(QStringLiteral("3DS"))) {
    return Nintendo3DS;
  } else if(name_.contains(QStringLiteral("DS"))) {
    return NintendoDS;
  } else if(name_ == QStringLiteral("Nintendo 64")) {
    return Nintendo64;
  } else if(name_.contains(QStringLiteral("GameCube"), Qt::CaseInsensitive)) {
    return NintendoGameCube;
  } else if(name_.contains(QStringLiteral("Advance"))) {
    return GameBoyAdvance;
  } else if(name_ == QStringLiteral("Game Boy Color")) {
    return GameBoyColor;
  } else if(name_ == QStringLiteral("Game Boy")) {
    return GameBoy;
  } else if(name_.contains(QStringLiteral("SNES")) || name_.contains(QStringLiteral("Super Nintendo"))) {
    return SuperNintendo;
    // nothing left for Nintendo except original
  } else if(name_.contains(QStringLiteral("Nintendo"))) {
    return Nintendo;
  } else if(name_.contains(QStringLiteral("Genesis"))) {
    return Genesis;
  } else if(name_.contains(QStringLiteral("Dreamcast"))) {
    return Dreamcast;
  } else if(name_.contains(QStringLiteral("Linux"))) {
    return Linux;
  } else if(name_.contains(QStringLiteral("ios"), Qt::CaseInsensitive)) {
    return iOS;
  } else if(name_.contains(QStringLiteral("Android"))) {
    return Android;
  }
  myDebug() << "No platform guess for" << name_;
  return UnknownPlatform;
}

QString GameCollection::platformName(GamePlatform platform_) {
  switch(platform_) {
    case Linux:               return i18n("Linux");
    case MacOS:               return i18n("Mac OS");
    case Windows:             return i18nc("Windows Platform", "Windows");
    case iOS:                 return i18nc("iOS Platform", "iOS");
    case Android:             return i18nc("Android Platform", "Android");
    case Xbox:                return i18n("Xbox");
    case Xbox360:             return i18n("Xbox 360");
    case XboxOne:             return i18n("Xbox One");
    case PlayStation:         return i18n("PlayStation");
    case PlayStation2:        return i18n("PlayStation2");
    case PlayStation3:        return i18n("PlayStation3");
    case PlayStation4:        return i18n("PlayStation4");
    case PlayStationPortable: return i18nc("PlayStation Portable", "PSP");
    case PlayStationVita:     return i18n("PlayStation Vita");
    case GameBoy:             return i18n("Game Boy");
    case GameBoyColor:        return i18n("Game Boy Color");
    case GameBoyAdvance:      return i18n("Game Boy Advance");
    case Nintendo:            return i18n("Nintendo");
    case SuperNintendo:       return i18n("Super Nintendo");
    case Nintendo64:          return i18n("Nintendo 64");
    case NintendoGameCube:    return i18n("GameCube");
    case NintendoWii:         return i18n("Nintendo Wii");
    case NintendoWiiU:        return i18n("Nintendo WiiU");
    case NintendoSwitch:      return i18n("Nintendo Switch");
    case NintendoDS:          return i18n("Nintendo DS");
    case Nintendo3DS:         return i18n("Nintendo 3DS");
    case Genesis:             return i18nc("Sega Genesis", "Genesis");
    case Dreamcast:           return i18n("Dreamcast");
    case UnknownPlatform:     break;
    case LastPlatform:        break;
  }
  myDebug() << "Failed to return platform name for" << platform_;
  return QString();
}
