/* Copyright IBM Corp. 2013, 2015 */

#define _GNU_SOURCE
#define _BSD_SOURCE

#include <signal.h>
#include <iconv.h>
#include <fcntl.h>
#include <unistd.h>
#include <endian.h>

#include "query_capacity_int.h"
#include "query_capacity_data.h"


/* we are packing the structures in the header file generated by VM */
#pragma pack(push)
#pragma pack(1)
#include "hcpinfbk_qclib.h"
#pragma pack(pop)


#define STHYI_BUF_SIZE		4096
#define STHYI_BUF_ALIGNMENT	4096
#define STHYI_DATA_FILE_ENV_VAR	"QUERY_CAPACITY_STHYI_DATA_FILE"
#define STHYI_FACILITY_BIT	74

#define STHYI_NA	0
#define STHYI_AVAILABLE	1
struct sthyi_priv {
	char   *data;
	int 	avail;
};

#if defined __s390x__ || __s390__
static void qc_stfle_signal_handler(int signal) {
	qc_debug(NULL, "Signal handler invoked with signal %d\n", signal);

	return;
}
#endif

static int qc_is_sthyi_available(void) {
#if defined __s390x__ || __s390__
	/* initialize for signal handler case */
	unsigned long long stfle_buffer[(STHYI_FACILITY_BIT/64)+1] __attribute__ ((aligned (8))) = { 0,};
	sighandler_t old_handler;

	/* we assume STFLE is available, cannot check, since we are
	 * in problem state and /proc/cpuinfo features might not be present.
	 * Therefore set up signal handler to ignore illegal instructions
	 * on older machines */
	old_handler=signal(SIGILL, qc_stfle_signal_handler);
	{
		register unsigned long reg0 asm("0") =
		STHYI_FACILITY_BIT/64 ;
		asm volatile (".insn s,0xb2b00000,%0"
			      : "=m" (stfle_buffer), "+d" (reg0) :
			      : "cc", "memory");
	}
	signal(SIGILL, old_handler);

	return (stfle_buffer[STHYI_FACILITY_BIT/64] >> (63 - (STHYI_FACILITY_BIT%64))) & 1;
#else
	return 0;
#endif
}

static int qc_sthyi(char *sthyi_buffer) {
#if defined __s390x__ || __s390__
	register unsigned long function_code asm("2") = 0;
	register unsigned long buffer asm("4") = (unsigned long) sthyi_buffer;
	register unsigned long return_code asm("5");
	int cc = -1;

	asm volatile (".insn rre,0xb2560000,%2,%3 \n"
		      "ipm %0\n"
		      "srl %0,28\n"
		      : "=d" (cc), "=d" (return_code)
		      : "d" (function_code), "d" (buffer)
		      : "memory", "cc");
	if (cc == 0) {
		/* buffer was updated */
		return 1;
	}
	/* if cc==-1: exception. never mind, return 0 */
	/* if cc==3: never mind, r carries return code */
#endif

	return 0;
}


static int qc_parse_sthyi_machine(struct qc_handle *cec, iconv_t *cd, struct inf0mac *machine) {
	qc_debug(cec, "Add CEC values from STHYI\n");
	qc_debug_indent_inc();
	if (machine->infmval1 & INFMPROC) {
		qc_debug(cec, "Add processor counts information\n");
		if (qc_set_attr_int(cec, qc_num_cp_dedicated, htobe16(machine->infmdcps), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(cec, qc_num_cp_shared, htobe16(machine->infmscps), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(cec, qc_num_cp_total, htobe16(machine->infmdcps) + htobe16(machine->infmscps), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(cec, qc_num_ifl_dedicated, htobe16(machine->infmdifl), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(cec, qc_num_ifl_shared, htobe16(machine->infmsifl), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(cec, qc_num_ifl_total, htobe16(machine->infmdifl) + htobe16(machine->infmsifl), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(cec, qc_num_cpu_dedicated, htobe16(machine->infmdcps) + htobe16(machine->infmdifl), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(cec, qc_num_cpu_shared, htobe16(machine->infmscps) + htobe16(machine->infmsifl), ATTR_SRC_STHYI))
			return -1;
	}

	if (machine->infmval1 & INFMMID) {
		qc_debug(cec, "Add machine ID information\n");
		if (qc_set_attr_ebcdic_string(cec, qc_type, machine->infmtype, sizeof(machine->infmtype), cd, ATTR_SRC_STHYI) ||
		    qc_set_attr_ebcdic_string(cec, qc_manufacturer, machine->infmmanu, sizeof(machine->infmmanu), cd, ATTR_SRC_STHYI) ||
		    qc_set_attr_ebcdic_string(cec, qc_sequence_code, machine->infmseq, sizeof(machine->infmseq), cd, ATTR_SRC_STHYI) ||
		    qc_set_attr_ebcdic_string(cec, qc_plant, machine->infmpman, sizeof(machine->infmpman), cd, ATTR_SRC_STHYI))
			return -2;
	}
	if (machine->infmval1 & INFMMNAM) {
		qc_debug(cec, "Add machine name information\n");
		if (qc_set_attr_ebcdic_string(cec, qc_layer_name, machine->infmname, sizeof(machine->infmname), cd, ATTR_SRC_STHYI))
			return -3;
	}
	qc_debug_indent_dec();

	return 0;
}

static int qc_parse_sthyi_partition(struct qc_handle *lpar, iconv_t *cd, struct inf0par *partition) {
	qc_debug(lpar, "Add LPAR values from STHYI\n");
	qc_debug_indent_inc();
	if (partition->infpval1 & INFPPROC) {
		qc_debug(lpar, "Add processor counts information\n");
		if (qc_set_attr_int(lpar, qc_num_cp_shared, htobe16(partition->infpscps), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(lpar, qc_num_cp_dedicated, htobe16(partition->infpdcps), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(lpar, qc_num_ifl_shared, htobe16(partition->infpsifl), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(lpar, qc_num_ifl_dedicated, htobe16(partition->infpdifl), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(lpar, qc_num_cpu_shared, htobe16(partition->infpscps) + htobe16(partition->infpsifl), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(lpar, qc_num_cpu_dedicated, htobe16(partition->infpdcps) + htobe16(partition->infpdifl), ATTR_SRC_STHYI))
			return -1;
	}

	if (partition->infpval1 & INFPWBCC) {
		qc_debug(lpar, "Add weight-based capped capacity information\n");
		if (qc_set_attr_int(lpar, qc_cp_weight_capping, htobe32(partition->infpwbcp), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(lpar, qc_ifl_weight_capping, htobe32(partition->infpwbif), ATTR_SRC_STHYI))
			return -2;
	}

	if (partition->infpval1 & INFPACC) {
		qc_debug(lpar, "Add absolute capped capacity information\n");
		if (qc_set_attr_int(lpar, qc_cp_absolute_capping, htobe32(partition->infpabcp), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(lpar, qc_ifl_absolute_capping, htobe32(partition->infpabif), ATTR_SRC_STHYI))
			return -3;
	}

	if (partition->infpval1 & INFPPID) {
		qc_debug(lpar, "Add partition ID information\n");
		if (qc_set_attr_int(lpar, qc_partition_number, htobe16(partition->infppnum), ATTR_SRC_STHYI) ||
		    qc_set_attr_ebcdic_string(lpar, qc_layer_name, partition->infppnam, sizeof(partition->infppnam), cd, ATTR_SRC_STHYI))
			return -4;
	}
	qc_debug_indent_dec();

	return 0;
}

static int qc_parse_sthyi_hypervisor(struct qc_handle *hdl, iconv_t *cd, struct inf0hyp *hv) {
	qc_debug(hdl, "Add HV values from STHYI\n");
	if (hv->infytype != INFYTVM) {
		qc_debug(hdl, "Error: Unsupported hypervisor type %d\n", hv->infytype);
		return -1;
	}
	if (qc_set_attr_int(hdl, qc_hardlimit_consumption, (hv->infyflg1 & INFYLMCN) ? 1 : 0, ATTR_SRC_STHYI) ||
	    qc_set_attr_ebcdic_string(hdl, qc_layer_name, hv->infysyid, sizeof(hv->infysyid), cd, ATTR_SRC_STHYI) ||
	    qc_set_attr_ebcdic_string(hdl, qc_cluster_name, hv->infyclnm, sizeof(hv->infyclnm), cd, ATTR_SRC_STHYI) ||
	    qc_set_attr_int(hdl, qc_num_cpu_total, htobe16(hv->infyscps) + htobe16(hv->infydcps) + htobe16(hv->infysifl) + htobe16(hv->infydifl), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(hdl, qc_num_cpu_shared, htobe16(hv->infyscps) + htobe16(hv->infysifl), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(hdl, qc_num_cpu_dedicated, htobe16(hv->infydcps) + htobe16(hv->infydifl), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(hdl, qc_num_cp_total, htobe16(hv->infyscps) + htobe16(hv->infydcps), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(hdl, qc_num_cp_shared, htobe16(hv->infyscps), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(hdl, qc_num_cp_dedicated, htobe16(hv->infydcps), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(hdl, qc_num_ifl_total, htobe16(hv->infysifl) + htobe16(hv->infydifl), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(hdl, qc_num_ifl_shared, htobe16(hv->infysifl), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(hdl, qc_num_ifl_dedicated, htobe16(hv->infydifl), ATTR_SRC_STHYI))
		return -2;

	return 0;
}

static int qc_parse_sthyi_guest(struct qc_handle *gst, iconv_t *cd, struct inf0gst *guest) {
	struct qc_handle *pool_hdl;
	int rc;

	qc_debug(gst, "Add Guest values from STHYI\n");
	if (qc_set_attr_int(gst, qc_mobility_eligible, (guest->infgflg1 & INFGMOB) ? 1 : 0, ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_has_multiple_cpu_types, (guest->infgflg1 & INFGMCPT) ? 1 : 0, ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_cp_dispatch_limithard, (guest->infgflg1 & INFGCPLH) ? 1 : 0, ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_ifl_dispatch_limithard, (guest->infgflg1 & INFGIFLH) ? 1 : 0, ATTR_SRC_STHYI) ||
	    qc_set_attr_ebcdic_string(gst, qc_layer_name, guest->infgusid, sizeof(guest->infgusid), cd, ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_num_cp_shared, htobe16(guest->infgscps), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_num_cp_dedicated, htobe16(guest->infgdcps), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_cp_dispatch_type, guest->infgcpdt, ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_cp_capped_capacity, htobe32(guest->infgcpcc), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_num_ifl_shared, htobe16(guest->infgsifl), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_num_ifl_dedicated, htobe16(guest->infgdifl), ATTR_SRC_STHYI))
		return -1;

	if ((guest->infgsifl > 0 || guest->infgdifl > 0) &&
	    qc_set_attr_int(gst, qc_ifl_dispatch_type, guest->infgifdt, ATTR_SRC_STHYI))
		return -2;

	if (qc_set_attr_int(gst, qc_ifl_capped_capacity, htobe32(guest->infgifcc), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_num_cp_total, htobe16(guest->infgscps) + htobe16(guest->infgdcps), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_num_ifl_total, htobe16(guest->infgsifl) + htobe16(guest->infgdifl), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_num_cpu_dedicated, htobe16(guest->infgdcps) + htobe16(guest->infgdifl), ATTR_SRC_STHYI) ||
	    qc_set_attr_int(gst, qc_num_cpu_shared, htobe16(guest->infgscps) + htobe16(guest->infgsifl), ATTR_SRC_STHYI))
		return -3;

	if ((qc_is_attr_set_int(gst, qc_num_cpu_total)) &&
	    qc_set_attr_int(gst, qc_num_cpu_total, htobe16(guest->infgscps) + htobe16(guest->infgdcps) + htobe16(guest->infgsifl) + htobe16(guest->infgdifl), ATTR_SRC_STHYI))
		return -4;

	/* if pool name is empty then we're done */
	if ((rc = qc_is_nonempty_ebcdic(gst, guest->infgpnam, sizeof(guest->infgpnam), cd)) > 0) {
		qc_debug(gst, "Add Pool values\n");
		qc_debug(gst, "Layer %2d: z/VM pool\n", gst->layer_no);
		if (qc_insert_handle(gst, &pool_hdl, QC_LAYER_TYPE_ZVM_CPU_POOL)) {
			qc_debug(gst, "Error: Failed to insert pool layer\n");
			return -7;
		}
		if (qc_set_attr_ebcdic_string(pool_hdl, qc_layer_name, guest->infgpnam, sizeof(guest->infgpnam), cd, ATTR_SRC_STHYI) ||
		    qc_set_attr_int(pool_hdl, qc_cp_limithard_cap, (guest->infgpflg & INFGPCLH) ? 1 : 0, ATTR_SRC_STHYI) ||
		    qc_set_attr_int(pool_hdl, qc_cp_capacity_cap, (guest->infgpflg & INFGPCPC) ? 1 : 0, ATTR_SRC_STHYI) ||
		    qc_set_attr_int(pool_hdl, qc_ifl_limithard_cap, (guest->infgpflg & INFGPILH) ? 1 : 0, ATTR_SRC_STHYI) ||
		    qc_set_attr_int(pool_hdl, qc_ifl_capacity_cap, (guest->infgpflg & INFGPIFC) ? 1 : 0, ATTR_SRC_STHYI) ||
		    qc_set_attr_int(pool_hdl, qc_cp_capped_capacity, htobe32(guest->infgpccc), ATTR_SRC_STHYI) ||
		    qc_set_attr_int(pool_hdl, qc_ifl_capped_capacity, htobe32(guest->infgpicc), ATTR_SRC_STHYI))
			return -6;
		rc = 0;
	}

	return rc;
}

static int qc_get_num_vm_layers(struct qc_handle *hdl, int *rc) {
	int i;

	for (hdl = hdl->root, i = 0; hdl != NULL; hdl = hdl->next) {
		if (*(int *)(hdl->layer) == QC_LAYER_TYPE_ZVM_HYPERVISOR)
			i++;
	}

	return i;
}

/* Returns pointer to the n-th VM hypervisor handle. num starts at 0, and handles
   are returned in sequencefrom handle linked list */
static struct qc_handle *qc_get_zvm_layer(struct qc_handle *hdl, int num) {
	struct qc_handle *h = hdl;
	int i;

	for (hdl = hdl->root, i = 0, num++; hdl != NULL; hdl = hdl->next) {
		if (*(int *)(hdl->layer) == QC_LAYER_TYPE_ZVM_HYPERVISOR && ++i == num)
			return hdl;
	}
	qc_debug(h, "Error: Couldn't find z/VM layer %d, only %d layer(s) found\n", num, i);

	return NULL;
}

static int qc_sthyi_process(struct qc_handle *hdl, iconv_t *cd, char *buf) {
	struct sthyi_priv *priv = (struct sthyi_priv *)buf;
	int no_hyp_gst, num_vm_layers, i, rc = 0;
	struct inf0gst *guest[INF0YGMX];
	struct inf0hyp *hv[INF0YGMX];
	struct qc_handle *gst_hdl;
	struct inf0par *partition;
	struct inf0mac *machine;
	struct inf0hdr *header;
	char *sthyi_buffer;

	qc_debug(hdl, "Process STHYI\n");
	qc_debug_indent_inc();
	if (!priv) {
		qc_debug(hdl, "No priv, exiting\n");
		goto out;
	}
	if (priv->avail == STHYI_NA) {
		qc_debug(hdl, "No priv data, exiting\n");
		goto out;
	}
	sthyi_buffer = priv->data;
	if (!sthyi_buffer) {
		qc_debug(hdl, "No data, exiting\n");
		goto out;	// No data: nothing we can do about
	}
	header = (struct inf0hdr *) sthyi_buffer;
	machine = (struct inf0mac *) (sthyi_buffer + htobe16(header->infmoff));
	partition = (struct inf0par *) (sthyi_buffer + htobe16(header->infpoff));
	no_hyp_gst = (int)header->infhygct;
	if (no_hyp_gst > INF0YGMX) {
		qc_debug(hdl, "Error: STHYI reports %d layers, exceeding the supported "
			"maximum of %d\n", no_hyp_gst, INF0YGMX);
		rc = -1;
		goto out;
	}
	if (no_hyp_gst > 0) {
		hv[0] = (struct inf0hyp *)(sthyi_buffer + htobe16(header->infhoff1));
		guest[0] = (struct inf0gst *)(sthyi_buffer + htobe16(header->infgoff1));
	}
	if (no_hyp_gst > 1) {
		hv[1] = (struct inf0hyp *)(sthyi_buffer + htobe16(header->infhoff2));
		guest[1] = (struct inf0gst *)(sthyi_buffer + htobe16(header->infgoff2));
	}
	if (no_hyp_gst > 2) {
		hv[2] = (struct inf0hyp *)(sthyi_buffer + htobe16(header->infhoff3));
		guest[2] = (struct inf0gst *)(sthyi_buffer + htobe16(header->infgoff3));
	}

	num_vm_layers = qc_get_num_vm_layers(hdl, &rc);
	if (rc != 0) {
		rc = -2;
		goto out;
	}

	if (qc_parse_sthyi_machine(hdl, cd, machine)) {
		rc = -3;
		goto out;
	}

	if (hdl->next)
		hdl = hdl->next; /* now we are at the LPAR handle */
	else {
		qc_debug(hdl, "Error: No next handle found\n");
		rc = -4;
		goto out;
	}
	if (qc_parse_sthyi_partition(hdl, cd, partition)) {
		rc = -5;
		goto out;
	}

	if (num_vm_layers != no_hyp_gst) {
		/* STHYI doesn't support more than 3rd level z/VM */
		qc_debug(hdl, "Error: /proc/sysinfo reported %d layers, but STHYI only "
				"covers %d\n", num_vm_layers, no_hyp_gst);
		rc = -6;
		goto out;
	}

	for (i = 0; i < no_hyp_gst; i++) {
		if ((hdl = qc_get_zvm_layer(hdl, i)) == NULL) {
			rc = -7;
			goto out;
		}
		gst_hdl = hdl->next; /* was host layer, now we are at guest layer */
		if (qc_parse_sthyi_hypervisor(hdl, cd, hv[i])) {
			rc = -8;
			goto out;
		}
		if (qc_parse_sthyi_guest(gst_hdl, cd, guest[i])) {
			rc = -9;
			goto out;
		}
	}
out:
	qc_debug_indent_dec();

	return rc;
}

static void qc_sthyi_dump(struct qc_handle *hdl, char *buf) {
	struct sthyi_priv *priv = (struct sthyi_priv *)buf;
	int fd, rc, success = 0;
	char *fname = NULL;

	qc_debug(hdl, "Dump STHYI\n");
	qc_debug_indent_inc();
	if (!priv || priv->avail != STHYI_AVAILABLE) {
		qc_debug(hdl, "No data available\n");
		success = 1;
		goto out;
	}
	if (!priv->data) {
		qc_debug(hdl, "Error: Cannot dump sthyi, since priv->buf == NULL\n");
		goto out;
	}
	if (asprintf(&fname, "%s/sthyi", qc_dbg_dump_dir) == -1) {
		qc_debug(hdl, "Error: Mem alloc error, cannot write dump\n");
		goto out;
	}
	fd = open(fname, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR);
	if (fd == -1) {
		qc_debug(hdl, "Error: Failed to open file '%s' to write dump\n", fname);
		goto out;
	}
	rc = write(fd, priv->data, STHYI_BUF_SIZE);
	close(fd);
	if (rc == -1) {
		qc_debug(hdl, "Error: Failed to write STHYI data into dump: %s\n", strerror(errno));
	} else {
		qc_debug(hdl, "STHYI data dumped to '%s'\n", fname);
		success = 1;
	}

out:
	free(fname);
	if (!success)
		qc_mark_dump_incomplete(hdl, "sthyi");
	qc_debug_indent_dec();
}

static int qc_read_sthyi_dump(struct qc_handle *hdl, char *buf) {
	char *fname = NULL;
	int fd , rc = -1;
	ssize_t lrc;

	if (asprintf(&fname, "%s/sthyi", qc_dbg_use_dump) == -1) {
		qc_debug(hdl, "Error: Mem alloc error, cannot read dump\n");
		goto out;
	}
	if (access(fname, F_OK)) {
		qc_debug(hdl, "No STHYI dump available\n");
		rc = 1;
		goto out;
	}
	if ((fd = open(fname, O_RDONLY)) == -1) {
		qc_debug(hdl, "Error: Failed to open file '%s' to read dump\n", fname);
		goto out;
	}
	lrc = read(fd, buf, STHYI_BUF_SIZE);
	close(fd);
	if (lrc == -1) {
		qc_debug(hdl, "Error: Failed to read STHYI data dump: %s\n", strerror(errno));
	} else {
		qc_debug(hdl, "STHYI data read from '%s'\n", fname);
		rc = 0;
	}

out:
	free(fname);

	return rc;
}

static int qc_sthyi_open(struct qc_handle *hdl, char **buf) {
	struct sthyi_priv *priv = NULL;
	void *p = NULL;
	int rc = 0;

	*buf = NULL;
	qc_debug(hdl, "Retrieve STHYI information\n");
	qc_debug_indent_inc();
	if ((priv = malloc(sizeof(struct sthyi_priv))) == NULL) {
		qc_debug(hdl, "Error: failed to alloc \n");
		rc = -1;
		goto out;
	}
	bzero(priv, sizeof(struct sthyi_priv));
	*buf = (char *)priv;
	if (posix_memalign(&p, STHYI_BUF_ALIGNMENT, STHYI_BUF_SIZE)) {
		qc_debug(hdl, "Error: posix_memalign() failed\n");
		rc = -2;
		goto out;
	}
	priv->data = (char *)p;
	bzero(priv->data, STHYI_BUF_SIZE);

	if (qc_dbg_use_dump) {
		if (qc_read_sthyi_dump(hdl, priv->data) != 0)
			goto out;
		priv->avail = STHYI_AVAILABLE;
	} else {
		if (!qc_is_sthyi_available()) {
			qc_debug(hdl, "STHYI not available\n");
			goto out;
		}
		qc_debug(hdl, "STHYI is available\n");
		/* we assume we are not relocated at this spot, between STFLE and STHYI */
		if (!qc_sthyi(priv->data)) {
			qc_debug(hdl, "Error: STHYI execution failed\n");
			rc = -3;
			goto out;
		}
		priv->avail = STHYI_AVAILABLE;
	}
	goto out;

out:
	qc_debug_indent_dec();

	return rc;
}

static void qc_sthyi_close(struct qc_handle *hdl, char *priv) {
	if (priv) {
		free(((struct sthyi_priv *)priv)->data);
		free(priv);
	}
}

struct qc_data_src sthyi = {qc_sthyi_open,
			    qc_sthyi_process,
			    qc_sthyi_dump,
			    qc_sthyi_close,
			    NULL,
			    NULL};
