/*
  Copyright (C) 2008 Jan Friederich, André Gaul

        This file is part of mmpong.

        mmpong is free software: you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation, either version 3 of the License, or
        (at your option) any later version.

        mmpong is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <CEGUIDefaultResourceProvider.h>
#include <SDL.h>
#include <iostream>
#include "client.h"
#include "gui.h"

using namespace CEGUI;
using namespace std;

#define GUI_ALPHA 0.7
#define GUI_ALPHA_BG 0.3

// Translate a SDLKey to the proper CEGUI::Key
static CEGUI::uint SDLKeyToCEGUIKey(SDLKey key)
{
	switch (key)
	{
		case SDLK_BACKSPACE:    return Key::Backspace;
		case SDLK_TAB:          return Key::Tab;
		case SDLK_RETURN:       return Key::Return;
		case SDLK_PAUSE:        return Key::Pause;
		case SDLK_ESCAPE:       return Key::Escape;
		case SDLK_SPACE:        return Key::Space;
		case SDLK_COMMA:        return Key::Comma;
		case SDLK_MINUS:        return Key::Minus;
		case SDLK_PERIOD:       return Key::Period;
		case SDLK_SLASH:        return Key::Slash;
		case SDLK_0:            return Key::Zero;
		case SDLK_1:            return Key::One;
		case SDLK_2:            return Key::Two;
		case SDLK_3:            return Key::Three;
		case SDLK_4:            return Key::Four;
		case SDLK_5:            return Key::Five;
		case SDLK_6:            return Key::Six;
		case SDLK_7:            return Key::Seven;
		case SDLK_8:            return Key::Eight;
		case SDLK_9:            return Key::Nine;
		case SDLK_COLON:        return Key::Colon;
		case SDLK_SEMICOLON:    return Key::Semicolon;
		case SDLK_EQUALS:       return Key::Equals;
		case SDLK_LEFTBRACKET:  return Key::LeftBracket;
		case SDLK_BACKSLASH:    return Key::Backslash;
		case SDLK_RIGHTBRACKET: return Key::RightBracket;
		case SDLK_a:            return Key::A;
		case SDLK_b:            return Key::B;
		case SDLK_c:            return Key::C;
		case SDLK_d:            return Key::D;
		case SDLK_e:            return Key::E;
		case SDLK_f:            return Key::F;
		case SDLK_g:            return Key::G;
		case SDLK_h:            return Key::H;
		case SDLK_i:            return Key::I;
		case SDLK_j:            return Key::J;
		case SDLK_k:            return Key::K;
		case SDLK_l:            return Key::L;
		case SDLK_m:            return Key::M;
		case SDLK_n:            return Key::N;
		case SDLK_o:            return Key::O;
		case SDLK_p:            return Key::P;
		case SDLK_q:            return Key::Q;
		case SDLK_r:            return Key::R;
		case SDLK_s:            return Key::S;
		case SDLK_t:            return Key::T;
		case SDLK_u:            return Key::U;
		case SDLK_v:            return Key::V;
		case SDLK_w:            return Key::W;
		case SDLK_x:            return Key::X;
		case SDLK_y:            return Key::Y;
		case SDLK_z:            return Key::Z;
		case SDLK_DELETE:       return Key::Delete;
		case SDLK_KP0:          return Key::Numpad0;
		case SDLK_KP1:          return Key::Numpad1;
		case SDLK_KP2:          return Key::Numpad2;
		case SDLK_KP3:          return Key::Numpad3;
		case SDLK_KP4:          return Key::Numpad4;
		case SDLK_KP5:          return Key::Numpad5;
		case SDLK_KP6:          return Key::Numpad6;
		case SDLK_KP7:          return Key::Numpad7;
		case SDLK_KP8:          return Key::Numpad8;
		case SDLK_KP9:          return Key::Numpad9;
		case SDLK_KP_PERIOD:    return Key::Decimal;
		case SDLK_KP_DIVIDE:    return Key::Divide;
		case SDLK_KP_MULTIPLY:  return Key::Multiply;
		case SDLK_KP_MINUS:     return Key::Subtract;
		case SDLK_KP_PLUS:      return Key::Add;
		case SDLK_KP_ENTER:     return Key::NumpadEnter;
		case SDLK_KP_EQUALS:    return Key::NumpadEquals;
		case SDLK_UP:           return Key::ArrowUp;
		case SDLK_DOWN:         return Key::ArrowDown;
		case SDLK_RIGHT:        return Key::ArrowRight;
		case SDLK_LEFT:         return Key::ArrowLeft;
		case SDLK_INSERT:       return Key::Insert;
		case SDLK_HOME:         return Key::Home;
		case SDLK_END:          return Key::End;
		case SDLK_PAGEUP:       return Key::PageUp;
		case SDLK_PAGEDOWN:     return Key::PageDown;
		case SDLK_F1:           return Key::F1;
		case SDLK_F2:           return Key::F2;
		case SDLK_F3:           return Key::F3;
		case SDLK_F4:           return Key::F4;
		case SDLK_F5:           return Key::F5;
		case SDLK_F6:           return Key::F6;
		case SDLK_F7:           return Key::F7;
		case SDLK_F8:           return Key::F8;
		case SDLK_F9:           return Key::F9;
		case SDLK_F10:          return Key::F10;
		case SDLK_F11:          return Key::F11;
		case SDLK_F12:          return Key::F12;
		case SDLK_F13:          return Key::F13;
		case SDLK_F14:          return Key::F14;
		case SDLK_F15:          return Key::F15;
		case SDLK_NUMLOCK:      return Key::NumLock;
		case SDLK_SCROLLOCK:    return Key::ScrollLock;
		case SDLK_RSHIFT:       return Key::RightShift;
		case SDLK_LSHIFT:       return Key::LeftShift;
		case SDLK_RCTRL:        return Key::RightControl;
		case SDLK_LCTRL:        return Key::LeftControl;
		case SDLK_RALT:         return Key::RightAlt;
		case SDLK_LALT:         return Key::LeftAlt;
		case SDLK_LSUPER:       return Key::LeftWindows;
		case SDLK_RSUPER:       return Key::RightWindows;
		case SDLK_SYSREQ:       return Key::SysRq;
		case SDLK_MENU:         return Key::AppMenu;
		case SDLK_POWER:        return Key::Power;
		default:                return 0;
	}
	return 0;
}


GUI::GUI(std::string resbase, int w, int h, bool _visible, ClientState *_state) {
	renderer = new OpenGLRenderer(0, w, h);
	log =  new dummyLogger();
	sys = new System(renderer);
	visible = _visible;
	state = _state;
	try {
		DefaultResourceProvider *rp = static_cast<DefaultResourceProvider *>(sys->getResourceProvider());

		rp->setResourceGroupDirectory("schemes", resbase + "CEGUI/schemes/");
		rp->setResourceGroupDirectory("imagesets", resbase + "CEGUI/imagesets/");
		rp->setResourceGroupDirectory("fonts", resbase + "CEGUI/fonts/");
		rp->setResourceGroupDirectory("layouts", resbase + "CEGUI/layouts/");
		rp->setResourceGroupDirectory("looknfeels", resbase + "CEGUI/looknfeels/");
		if( sys->getDefaultXMLParserName().compare( "XercesParser" ) == 0 )
			rp->setResourceGroupDirectory( "schemas", "CEGUI/gameschemes/");


		Scheme::setDefaultResourceGroup("schemes");
		Imageset::setDefaultResourceGroup("imagesets");
		Font::setDefaultResourceGroup("fonts");
		WindowManager::setDefaultResourceGroup("layouts");
		WidgetLookManager::setDefaultResourceGroup("looknfeels");

		// now get a dummy window

		// ATTENTION (by andré): 
		//    the next line segfaults with CEGUI 0.5.0-4.1 in Debian/Ubuntu
		//    this is fixed in 0.6 (sorry, found no workaround atm)
		SchemeManager::getSingleton().loadScheme( "TaharezLook.scheme" );
		sys->setDefaultMouseCursor( "TaharezLook", "MouseArrow" );
		sys->setDefaultTooltip("TaharezLook/Tooltip");

		WindowManager *wmgr = WindowManager::getSingletonPtr();

		Window *root = wmgr->loadWindowLayout("mmpong-gl.layout");
		root->setAlpha(GUI_ALPHA );
		sys->setGUISheet( root );

		win_config = wmgr->getWindow("WindowConfig");
		win_config->subscribeEvent(FrameWindow::EventCloseClicked, CEGUI::Event::Subscriber(&GUI::on_cancel, this));
		
		edit_server = static_cast<Editbox *> (wmgr->getWindow("EditboxServer"));
		edit_server->setText(state->server);
		
		edit_port = static_cast<Editbox *> (wmgr->getWindow("EditboxPort"));
		edit_port->setText(state->port);
		
		check_sexy = static_cast<Checkbox *> (wmgr->getWindow("CheckboxSexy"));
		check_sexy->setSelected(state->sexy);
		check_sexy->subscribeEvent(Checkbox::EventCheckStateChanged, CEGUI::Event::Subscriber(&GUI::on_sexy, this));
		
		check_sound = static_cast<Checkbox *> (wmgr->getWindow("CheckboxSound"));
		check_sound->setSelected(state->sound);
		check_sound->subscribeEvent(Checkbox::EventCheckStateChanged, CEGUI::Event::Subscriber(&GUI::on_sound, this));

		check_full = static_cast<Checkbox *> (wmgr->getWindow("CheckboxFull"));
		check_full->setSelected(state->renderer->get_full());
		check_full->subscribeEvent(Checkbox::EventCheckStateChanged, CEGUI::Event::Subscriber(&GUI::on_full, this));

		CEGUI::PushButton *btn;
		btn = static_cast<PushButton *> (wmgr->getWindow("ButtonQuit"));
		btn->subscribeEvent(PushButton::EventClicked, CEGUI::Event::Subscriber(&GUI::on_quit, this));

		btn = static_cast<PushButton *> (wmgr->getWindow("ButtonCancel"));
		btn->subscribeEvent(PushButton::EventClicked, CEGUI::Event::Subscriber(&GUI::on_cancel, this));

		btn = static_cast<PushButton *> (wmgr->getWindow("ButtonConnect"));
		btn->subscribeEvent(PushButton::EventClicked, CEGUI::Event::Subscriber(&GUI::on_connect, this));

		// dialog
		win_dialog = wmgr->getWindow("WindowDialog");
		stext_dialog = wmgr->getWindow("StaticTextDialog");

		btn = static_cast<PushButton *> (wmgr->getWindow("ButtonDialogOk"));
		btn->subscribeEvent(PushButton::EventClicked, CEGUI::Event::Subscriber(&GUI::on_dialog_ok, this));


	}
	catch(CEGUI::Exception &ex) {
		printf("Exception: %s\n", ex.getMessage().c_str());
	}
	SDL_ShowCursor(SDL_DISABLE);
	SDL_EnableUNICODE(1);
	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
}


void GUI::resize_init() {
	renderer->grabTextures();
}


void GUI::resize_finish(int w, int h) {
	renderer->restoreTextures();
	renderer->setDisplaySize(Size(w, h));
}


void GUI::handle_mouse_down(Uint8 button) {
	switch (button) {

		case SDL_BUTTON_LEFT:
			sys->injectMouseButtonDown(CEGUI::LeftButton);
			break;

		case SDL_BUTTON_MIDDLE:
			sys->injectMouseButtonDown(CEGUI::MiddleButton);
			break;
		
		case SDL_BUTTON_RIGHT:
			sys->injectMouseButtonDown(CEGUI::RightButton);
			break;

		case SDL_BUTTON_WHEELDOWN:
			sys->injectMouseWheelChange(-1);
			break;

		case SDL_BUTTON_WHEELUP:
			sys->injectMouseWheelChange(1);
			break;
	}
}


void GUI::handle_mouse_up(Uint8 button) {
	switch (button) {

		case SDL_BUTTON_LEFT:
			sys->injectMouseButtonUp(CEGUI::LeftButton);
			break;

		case SDL_BUTTON_MIDDLE:
			sys->injectMouseButtonUp(CEGUI::MiddleButton);
			break;
		
		case SDL_BUTTON_RIGHT:
			sys->injectMouseButtonUp(CEGUI::RightButton);
			break;
	}
}

int GUI::handle_event(SDL_Event &e) {
	CEGUI::uint kc;
	switch(e.type) {

		// mouse
		case SDL_MOUSEMOTION:
			sys->injectMousePosition(e.motion.x, e.motion.y);
			break;

		case SDL_MOUSEBUTTONDOWN:
			handle_mouse_down(e.button.button);
			break;

		case SDL_MOUSEBUTTONUP:
			handle_mouse_up(e.button.button);
			break;

		//keyboard
		case SDL_KEYDOWN:
			kc = SDLKeyToCEGUIKey(e.key.keysym.sym);
			sys->injectKeyDown(kc);
			if (e.key.keysym.unicode)
				sys->injectChar(e.key.keysym.unicode);
			break;

		case SDL_KEYUP:
			sys->injectKeyUp(e.key.keysym.scancode);
			break;

	}

	return 0; 
}


void GUI::render() {
	//inject time
	double t = 0.001 * SDL_GetTicks();
	sys->injectTimePulse( t - last_time );
	last_time = t;
	if (visible)
		sys->renderGUI();
}


bool GUI::on_quit(const CEGUI::EventArgs &e) {
	state->signal_exiting = 1;
	return true;
}


bool GUI::on_cancel(const CEGUI::EventArgs &e) {
	visible = false;
	return true;
}


bool GUI::on_connect(const CEGUI::EventArgs &e) {
	delete state->game;
	state->game = NULL;
	
	try {
		state->server = edit_server->getText().c_str();
		state->port = edit_port->getText().c_str();
		state->game = new NetGame(state->server, state->port, 0);
		set_visible(false);
	} catch (runtime_error &err) {
		cerr << err.what() << endl;
		show_dialog("Error", err.what());
		delete state->game;
		state->game = NULL;
	}
	
	return true;
}


bool GUI::on_sexy(const CEGUI::EventArgs &e) {
	state->sexy = !state->sexy;
	return true;
}


bool GUI::on_sound(const CEGUI::EventArgs &e) {
	state->sound = !state->sound;
	return true;
}


bool GUI::on_full(const CEGUI::EventArgs &e) {
	state->full = !state->full;
	resize_init();
	state->renderer->set_full(state->full);
	resize_finish(state->renderer->get_width(), state->renderer->get_height());

	return true;
}


bool GUI::on_dialog_ok(const CEGUI::EventArgs &e) {
	win_dialog->setModalState(false);
	win_dialog->setVisible(false);
	win_config->setAlpha(GUI_ALPHA);
	return true;
}


void GUI::show_dialog(const std::string &title, const std::string &message) {
	win_config->setAlpha(GUI_ALPHA_BG);
	win_dialog->setText(title);
	stext_dialog->setText(message);
	win_dialog->setVisible(true);
	win_dialog->setModalState(true);
}


void GUI::set_full(bool _full) {
	check_full->setSelected(_full);
/*	if (check_full->isSelected() != state->renderer->get_full())
		on_full(CEGUI::EventArgs());
*/
}


