/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/param.h>
#include <sys/swap.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <errno.h>
#include "utils.h"

pthread_key_t buffer_key;
pthread_key_t buffer_size_key;
pthread_key_t number_swaps_key;

static void create_keys();

int sa_open_swap() {
	pthread_once_t key_once = PTHREAD_ONCE_INIT;
	if (pthread_once(&key_once, create_keys))
		return ENOSYS;

	return 0;
}

int sa_count_swaps(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	*number = (uint16_t) swapctl(SWAP_NSWAP, NULL, 0);
	return 0;
}

int sa_reset_swaps() {
	int number_swaps;
	struct swapent* buffer = (struct swapent*) pthread_getspecific(buffer_key);
	size_t buffer_size = (size_t) pthread_getspecific(buffer_size_key);
	do {
		number_swaps = swapctl(SWAP_NSWAP, NULL, 0);
		if (pthread_setspecific(number_swaps_key, (void *) number_swaps))
			return ENOSYS;

		size_t size = number_swaps * sizeof(struct swapent);
		if (buffer_size < size) {
			safe_realloc((void*) &buffer, &size);
			if (pthread_setspecific(buffer_key, buffer))
				return ENOSYS;
			buffer_size = size;
			if (pthread_setspecific(buffer_size_key, (void *) buffer_size))
				return ENOSYS;
			if (buffer == NULL)
				return ENOMEM;
		}

		errno = 0;
		if (swapctl(SWAP_STATS, buffer, number_swaps) == -1 && errno != ENOMEM)
			return ENOSYS;
	} while (errno == ENOMEM);

	return 0;
}

int sa_get_swap(uint16_t index, struct sa_swap* dst) {
	if (dst == NULL)
		return EINVAL;

	int number_swaps = (int) pthread_getspecific(number_swaps_key);
	if (index >= number_swaps)
		return ENODEV;

	struct swapent* buffer = (struct swapent*) pthread_getspecific(buffer_key);
	strlcpy(dst->name, buffer[index].se_path, sizeof dst->name);
	dst->total = (uint64_t) buffer[index].se_nblks * DEV_BSIZE;
	dst->free = (uint64_t) (buffer[index].se_nblks - buffer[index].se_inuse) * DEV_BSIZE;
	return 0;
}

int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	struct swapent* buffer = (struct swapent*) pthread_getspecific(buffer_key);
	int number_swaps = (int) pthread_getspecific(number_swaps_key);

	*written = 0;
	uint16_t i;
	for (i = 0; i < number_swaps; ++i) {
		if (i == dst_size)
			return ENOMEM;

		strlcpy(dst[i].name, buffer[i].se_path, sizeof dst->name);
		dst[i].total = (uint64_t) buffer[i].se_nblks * DEV_BSIZE;
		dst[i].free = (uint64_t) (buffer[i].se_nblks - buffer[i].se_inuse) * DEV_BSIZE;

		++(*written);
	}
	return 0;
}

static void create_keys() {
	static short keys_created = 0;
	if (keys_created) {
		return;
	}

	pthread_key_create(&buffer_key, free_keys);
	pthread_key_create(&buffer_size_key, NULL);
	pthread_key_create(&number_swaps_key, NULL);
	keys_created = 1;
}
