/***********************************************************************************

    Copyright (C) 2007-2012 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_VIEWS_HEADER
#define LIFEOGRAPH_VIEWS_HEADER


#include "strings.hpp"
#include "diarydata.hpp"
#include "entry.hpp"
#include "widget_chart.hpp"
#include "dialog_export.hpp"
#include "dialog_password.hpp"
#include "widget_textview.hpp"


namespace LIFEO
{

// TAG VIEW ========================================================================================
class TagView : public ElementShower< Tag >
{
    public:
                                TagView();

        Glib::ustring           get_title_str() const
        { return m_ptr2elem->get_name(); }
        Glib::ustring           get_info_str() const
        {
            if( m_ptr2elem->get_type() == DiaryElement::ET_TAG )
                return Glib::ustring::compose( _( STRING::ELEM_WITH_ENTRIES ),
                                               m_ptr2elem->get_type_name(),
                                               m_ptr2elem->get_size() );
            else
                return _( "Filter and theme entries without tags" );
        }

        Gtk::Widget*            get_popover();

        void                    handle_login();
        void                    show( Tag& tag );

        void                    update_combobox_categories();

        enum TagComboType { TCT_NONE, TCT_CATEGORY, TCT_NEW, TCT_SEPARATOR };

    protected:
        bool                    is_title_editable() const
        { return( m_ptr2elem->get_type() == DiaryElement::ET_TAG ); }
        Glib::ustring           get_title_edit_str() const
        { return m_ptr2elem->get_name(); }
        bool                    check_title_applicable( const Glib::ustring& ) const;
        bool                    apply_title( const Glib::ustring& );

        void                    handle_filter_toggled();
        void                    handle_category_changed();
        bool                    is_row_separator( const Glib::RefPtr< Gtk::TreeModel >&,
                                                  const Gtk::TreeIter );

        void                    handle_theme_motion_notify_event( GdkEventMotion* );
        void                    handle_theme_button_release_event( GdkEventButton* );
        void                    add_action_theme_text( const Glib::ustring&,
                                                       const Glib::RefPtr< Gtk::TextTag >& );
        void                    add_tagged_theme_text( const Glib::ustring&,
                                                       const Glib::RefPtr< Gtk::TextTag >& );
        void                    add_plain_theme_text( const Glib::ustring& );
        void                    launch_color_dialog( Gdk::RGBA& );
        void                    launch_font_dialog();

        void                    update_theme_view();
        void                    update_theme_menu();
        void                    reset_theme();
        void                    copy_theme_from( const Tag* );

        void                    dismiss_tag();

        Gtk::Frame*             m_frame_edit;
        Gtk::Button*            m_button_dismiss;
        Gtk::ComboBox*          m_combobox_category;
        WidgetChart*            m_widgetchart;
        Gtk::TextView*          m_textview_theme;
        Gtk::Menu*              m_menu_themes;
        Gtk::MenuButton*        m_button_themes;
        Gtk::Button*            m_button_theme_reset;
        Gtk::Button*            m_B_filtering;

        class ColrecCategory : public Gtk::TreeModel::ColumnRecord
        {
            public:
                ColrecCategory() { add( name ); add( ptr ); add( type ); }
                Gtk::TreeModelColumn< Glib::ustring >       name;
                Gtk::TreeModelColumn< CategoryTags* >       ptr;
                Gtk::TreeModelColumn< TagComboType >        type;
        };

        Glib::RefPtr< Gtk::ListStore >
                                m_liststore_categories;
        ColrecCategory          colrec_category;

        Glib::RefPtr< Gtk::TextTag >     m_tag_action;
        Glib::RefPtr< Gtk::TextTag >     m_tag_heading;
        Glib::RefPtr< Gtk::TextTag >     m_tag_subheading;
        Glib::RefPtr< Gtk::TextTag >     m_tag_normal;
        Glib::RefPtr< Gtk::TextTag >     m_tag_match;
        Glib::RefPtr< Gtk::TextTag >     m_tag_highlight;
        Glib::RefPtr< Gtk::TextTag >     m_tag_comment;
        Glib::RefPtr< Gtk::TextTag >     m_tag_link;
        Glib::RefPtr< Gtk::TextTag >     m_tag_link_broken;
        Glib::RefPtr< Gtk::TextTag >     m_tag_font;
        Glib::RefPtr< Gtk::TextTag >     m_tag_background;
        Glib::RefPtr< Gtk::TextTag >     m_tag_region;
};

// TAG CATEGORY VIEW ===============================================================================
class CategoryTagsView : public ElementShower< CategoryTags >
{
    public:
                                CategoryTagsView();

        Glib::ustring           get_title_str() const
        { return m_ptr2elem->m_name; }
        Glib::ustring           get_info_str() const
        { return Glib::ustring::compose( _( "Tag category with %1 tags" ),
                                         m_ptr2elem->size() ); }

        Gtk::Widget*            get_popover();

        void                    show( CategoryTags& );

    protected:
        bool                    is_title_editable() const
        { return true; }
        Glib::ustring           get_title_edit_str() const
        { return m_ptr2elem->get_name(); }
        bool                    check_title_applicable( const Glib::ustring& ) const;
        bool                    apply_title( const Glib::ustring& );

        void                    dismiss_category();

        Gtk::Button*            m_button_dismiss;
        WidgetChart*            m_widgetchart;
};

// CHAPTER VIEW ====================================================================================
class ChapterView : public ElementShower< Chapter >
{
    public:
                                ChapterView();

        Glib::ustring           get_title_str() const
        {
            Glib::ustring title;

            if( ! m_ptr2elem->get_date().is_hidden() )
            {
                title = m_ptr2elem->get_date().format_string();
                title += "  ";
            }

            title += m_ptr2elem->m_name;

            return title;
        }
        Glib::ustring           get_info_str() const
        {
            if( m_ptr2elem->m_time_span )
                return Glib::ustring::compose( _( "Dated Chapter with %1 entries in %2 days" ),
                                               m_ptr2elem->get_size(), m_ptr2elem->m_time_span );
            else
                return Glib::ustring::compose( _( STRING::ELEM_WITH_ENTRIES ),
                                               m_ptr2elem->get_type_name(),
                                               m_ptr2elem->get_size() );
        }

        Gtk::Widget*            get_popover();

        void                    handle_login();
        void                    show( Chapter& );

        void                    set_todo_status( ElemStatus status );

    protected:
        bool                    is_title_editable() const
        { return true; }
        Glib::ustring           get_title_edit_str() const
        { return m_ptr2elem->m_name; }
        bool                    check_title_applicable( const Glib::ustring& ) const;
        bool                    apply_title( const Glib::ustring& );

        void                    handle_date_changed();
        void                    handle_date_applied();

        void                    update_labels();

        void                    add_new_entry();
        void                    dismiss_chapter();

        Gtk::Image*             m_I_not_todo;
        Gtk::RadioButton*       m_button_not_todo;
        Gtk::RadioButton*       m_button_todo;
        Gtk::RadioButton*       m_button_progressed;
        Gtk::RadioButton*       m_button_done;
        Gtk::RadioButton*       m_button_canceled;
        Gtk::Button*            m_button_dismiss;

        Gtk::Entry*             m_entry_begins;
        Gtk::Button*            m_button_date_apply;
        Gtk::Label*             m_label_ends;
        Gtk::Button*            m_button_new_entry;

        Gtk::Frame*             m_frame_edit;
        Gtk::Frame*             m_frame_actions;
        WidgetChart*            m_widgetchart;
};

// FILTER VIEW =====================================================================================
class FilterView : public ElementShower< Filter >
{
    public:
                                FilterView();

        Glib::ustring           get_title_str() const
        { return m_ptr2elem->get_name(); }
        Glib::ustring           get_info_str() const;

        void                    handle_login();
        void                    show( Filter& );

        void                    update();

    protected:
        bool                    is_title_editable() const
        { return( false ); }
        Glib::ustring           get_title_edit_str() const
        { return m_ptr2elem->get_name(); }
        bool                    check_title_applicable( const Glib::ustring& ) const
        { return false; }
        bool                    apply_title( const Glib::ustring& )
        { return false; }

        void                    make_default();

        void                    handle_trash_changed();
        void                    handle_favored_changed();
        void                    handle_todo_changed();

        void                    clear_tag();
        bool                    handle_tag_dropped(
                                        const Glib::RefPtr< Gdk::DragContext >&,
                                        int, int, guint );

        void                    handle_date_begin_changed();
        void                    handle_date_end_changed();

        void                    update_entries();
        void                    remove_selected_entries();
        void                    remove_all_entries();
        bool                    handle_entry_dropped(
                                        const Glib::RefPtr< Gdk::DragContext >&,
                                        int, int, guint );
        void                    handle_entry_selection_changed();

        Gtk::ToggleButton*      m_TB_todo_not;
        Gtk::ToggleButton*      m_TB_todo_open;
        Gtk::ToggleButton*      m_TB_todo_progressed;
        Gtk::ToggleButton*      m_TB_todo_done;
        Gtk::ToggleButton*      m_TB_todo_canceled;
        Gtk::ToggleButton*      m_TB_not_favored;
        Gtk::ToggleButton*      m_TB_favored;
        Gtk::ToggleButton*      m_TB_not_trashed;
        Gtk::ToggleButton*      m_TB_trashed;

        Gtk::Label*             m_L_tag;
        Gtk::Button*            m_B_clear_tag;

        EntryClear*             m_E_date_begin;
        EntryClear*             m_E_date_end;

        Gtk::TreeView*          m_TV_entries;
        Glib::RefPtr< Gtk::ListStore >      m_LS_entries;
        Gtk::Button*            m_B_remove_entry;
        Gtk::Button*            m_B_rm_all_entries;

        Gtk::Button*            m_B_make_default;
        Gtk::Button*            m_B_revert;

        class ColrecEntries : public Gtk::TreeModel::ColumnRecord
        {
            public:
                ColrecEntries()
                {
                    add( name );
                    add( ptr );
                }
                Gtk::TreeModelColumn< Glib::ustring > name;
                Gtk::TreeModelColumn< Entry* > ptr;
        };

        static ColrecEntries*   colrec_entries;

};

// DIARY VIEW ======================================================================================
class DiaryView : public ElementShower< Diary >
{
    public:
                                DiaryView();
                                ~DiaryView();

        Glib::ustring           get_list_str() const
        { return Glib::ustring::compose( "<b>%1</b>",
                Glib::Markup::escape_text(
                        Glib::filename_display_basename( m_ptr2elem->m_path ) ) ); }
        Glib::ustring           get_title_str() const
        { return m_ptr2elem->m_name; }
        Glib::ustring           get_info_str() const
        {
            int timespan( 0 );
            // FIXME: does not work for some obscure reason..
            for( EntryIterConst iter_entry = m_ptr2elem->m_entries.begin();
                    iter_entry != m_ptr2elem->m_entries.end();
                    ++iter_entry )
            {
                if( ! iter_entry->second->get_date().is_ordinal() )
                {
                    timespan = m_ptr2elem->m_entries.rbegin()->second->get_date().
                            calculate_days_between( iter_entry->second->get_date() );
                    break;
                }
            }
            // TRANSLATORS: %3 stands for "Encrypted diary" or "Unencrypted diary"
            return Glib::ustring::compose( _( "%3 with %1 entries spanning %2 days" ),
                                           m_ptr2elem->m_entries.size(),
                                           timespan,
                                           Diary::d->is_encrypted() ?
                                                   _( "Encrypted diary" ) :
                                                   _( "Unencrypted diary" ) );
        }

        void                    handle_login();
        void                    show( Diary& );

        void                    update_startup_elem();

    protected:
        // DIALOGS
        void                    start_dialog_password();
        void                    start_dialog_import();
        void                    start_dialog_export();
        void                    open_diary_folder();
        void                    handle_language_changed();
        void                    handle_startup_type_changed();
        bool                    handle_startup_elem_drag_motion(
                                        const Glib::RefPtr< Gdk::DragContext >&,
                                        int, int, guint );
        bool                    handle_startup_elem_dropped(
                                        const Glib::RefPtr< Gdk::DragContext >&,
                                        int, int, guint );
        bool                    go_to_startup_elem( const Glib::ustring& );

        void                    update_combobox_chapter_ctg();
        void                    handle_cur_chapter_ctg_changed();
        void                    rename_cur_chapter_ctg();
        void                    dismiss_cur_chapter_ctg();
        bool                    is_row_separator( const Glib::RefPtr< Gtk::TreeModel >&,
                                                  const Gtk::TreeIter );

        Gtk::Button*            m_button_import;
        Gtk::Separator*         m_separator_encryption;
        Gtk::Button*            m_button_encryption;

        Gtk::Frame*             m_frame_startup;
        Gtk::Frame*             m_frame_chapters;

        Gtk::Label*             m_label_path;
        Gtk::ComboBoxText*      m_combobox_spellcheck;
        WidgetChart*            m_widgetchart;

        // STARTUP
        Gtk::ComboBox*          m_combobox_startup_type;
        Gtk::Label*             m_label_startup_elem;
        Gtk::Label*             m_label_startup_elem_drop;

        class ColrecStartupType : public Gtk::TreeModel::ColumnRecord
        {
            public:
                ColrecStartupType() { add( name ); add( type ); }
                Gtk::TreeModelColumn< Glib::ustring >   name;
                Gtk::TreeModelColumn< DEID >            type;
        };

        Glib::RefPtr< Gtk::ListStore >
                                m_liststore_startup_type;
        ColrecStartupType       colrec_startup_type;

        // CHAPTER CATEGORIES & TOPICS
        Gtk::ComboBox*          m_combobox_chapter_ctg;
        Gtk::Button*            m_button_chapter_ctg_dismiss;
        Gtk::Button*            m_button_chapter_ctg_rename;

        enum ChapterCtgComboType { CCCT_CATEGORY, CCCT_NEW, CCCT_SEPARATOR };

        class ColrecChapterCtg : public Gtk::TreeModel::ColumnRecord
        {
            public:
                ColrecChapterCtg() { add( name ); add( ptr ); add( type ); }
                Gtk::TreeModelColumn< Glib::ustring >       name;
                Gtk::TreeModelColumn< CategoryChapters* >   ptr;
                Gtk::TreeModelColumn< ChapterCtgComboType > type;
        };

        Glib::RefPtr< Gtk::ListStore >
                                m_liststore_chapter_ctg;
        ColrecChapterCtg        colrec_chapter_ctg;

        // DIALOGS
        DialogExport*           m_dialog_export;
};

} // end of namespace LIFEO

#endif

