package latexDraw.parsers.pst;

import static java.lang.Math.cos;
import static java.lang.Math.sin;
import static java.lang.Math.toRadians;

import java.awt.Color;
import java.awt.geom.Point2D;
import java.util.Vector;

import latexDraw.parsers.*;
import latexDraw.parsers.pst.PSTParameters.PositionParameters;
import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;

/**
 * Defines an abstract PSTricks parser that parses PSTricks code.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/21/08<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.2<br>
 */
public abstract class PSTParser extends CodeParser
{
	/** The PSTricks parameters that are created, removed and set
	 * during the parsing of the pstricks code. */
	protected Vector<PSTParameters> params;
	
	/** Defines if the parsing of pstricks shapes has began. It
	 * begins when a tag pspicture or begin{pspicture} is parsed. */
	protected boolean started;
	
	/** The parser that parses pstricks parameters. */
	protected PSTParametersParser paramParser;
	
	
	
	/**
	 * Creates and initialises an abstract pstricks parser.
	 * @param code The pstricks code to parse.
	 * @throws IllegalArgumentException If the given code is null.
	 */
	public PSTParser(String code)
	{
		super(code);

		params = new Vector<PSTParameters>();
		paramParser = new PSTParametersParser(code, null);
		paramParser.setCodeCore(codeCore);
	}
	
	
	
	@Override
	public boolean isComment()
	{
		return getChar()=='%';
	}
	
	
	@Override
	public String skipComment()
	{
		String comment;
		
		if(isComment())
		{
			comment = "";
			char c;
			
			while(!isEOL() && !isEOC())
			{
				c = (char)nextChar();
				
				if(!isEOL() && !isEOC())
					comment += c;
			}
			
			nextChar();
			incLinePosition();
			skipWSP();
		}
		else comment = null;
		
		return comment;
	}



	@Override
	public void skipWSP()
	{
		while(getChar()==' ' || isEOL() || getChar()=='\t')
		{
			if(isEOL())
				incLinePosition();
			
			nextChar();
		}
	}
	
	
//	protected void parse_ps()
//	{
//		switch(getChar())
//		{
//			case 'p':  // "\psp"
//				nextChar();
////				parseps();
//				break;
//				
//			default:
//				break;
//		}
//	}
//	
//	
//	
//	protected void parse_p()
//	{
//		switch(getChar())
//		{
//			case 's':  // "\ps"
//				nextChar();
//				parse_ps();
//				break;
//				
//			default:
//				break;
//		}
//	}
//	
//	
//	
//	protected void parseCommand()
//	{
//		switch(getChar())
//		{
//			case '\\': // "\\"
//				break;
//				
//			case 'p':  // "\p"
//				nextChar();
//				parse_p();
//				break;
//				
//			default:
//				break;
//		}
//	}
	
	
	/**
	 * Adds the current character to the parsed text attribute of a pstricks parameters set.
	 * @since 2.0.2
	 */
	protected void addParsedText()
	{
		PSTParameters p = params.lastElement();
		String str      = String.valueOf((char)getChar());
		
		p.textForFrame+=str;
		p.textParsed+=str;
		
		if(p.fromFramedBox)
		{
			boolean again = true;
			int k = params.size()-2;
			while(again && k>=0)
			{
				p = params.elementAt(k);
				p.textForFrame+=str;
				
				if(p.getPosition()!=null)
					again = false;
				else 
					k--;
			}
		}
	}
	
	
	/**
	 * Parses a math expression ($ ... $). The first letter read by this function must be '$'.
	 */
	protected void parseMathText() throws UnexceptedEndOfFileException
	{
		if(started)
		{
			addParsedText();
			
			while(nextChar()!='$' && !isEOC())
				addParsedText();
			
			addParsedText();
		}
		else
		{
			nextChar();
			
			while(getChar()!='$' && !isEOC())
				nextChar();
		}

		if(isEOC()) 
			throw new UnexceptedEndOfFileException();
		
		nextChar();
	}
	
	
	
	@Override
	public void parse()
	{//TODO do not forget to take account of the number of parsed lines when parsing parameters.
		initialise();
//		
//		while(!isEOC())
//			try
//			{
//				switch(getChar())
//				{
//					case '\\':
//						nextChar();
//						parseCommand();
//						break;
//						
//					case '$':
//						nextChar();
//						parseMathText();
//						break;
//						
//					default:
//						break;
//				}
//			}
//		catch(UnexceptedEndOfFileException ex) { ex.printStackTrace(); }
	}
	

	
	@Override
	public void initialise()
	{
		super.initialise();
		
		setPosition(0);
		started  = false;
		PSTParameters.reinitialize();
	}
	
	

	@Override
	public boolean isWSP()
	{
		int c = getChar();
		
		return c==' ' || c=='\n' || c=='\r';
	}
	
	
	
	/**
	 * Actions to do when some text is parsed.
	 * @throws InvalidFormatCommandException If the command is not valid.
	 */
	public abstract void actionsOnText() throws InvalidFormatCommandException;
	
	
	
	/**
	 * Actions to do when a framed box is parsed.
	 * @since 2.0.2
	 * @param type The kind of the framed box:<br>
	 * 0: rectangle<br>
	 * 1: circle<br>
	 * 2: triangle<br>
	 * 3: diamond<br>
	 * 4: ellipse<br>
	 * 5: double rectangle<br>
	 * 6: rectangle with shadow<br>
	 * @throws UnclosedBracketsException If a bracket is not closed.
	 */
	public abstract void actionsOnFramedBox(int type) throws UnclosedBracketsException;
	
	
	
	/**
	 * Actions to do when one or several framed boxes are parsed.
	 * @throws InvalidFormatCommandException If the parsed command is not valid.
	 * @since 1.7
	 */
	public abstract void actionsOnTerminatedFramedBoxes() throws InvalidFormatCommandException;
	
	
	/**
	 * Actions to do when a <code>pscustom</code> command is finished.
	 * @since 0.3
	 */
	public abstract void actionsOnterminatedPsCustom() throws NotFullyManagedException;

	
	
	
	/**
	 * Moves a point according to the PSTricks parameters (Useful for dots, texts).<br>
	 */
	protected Point2D.Double moveCoordinate(Point2D.Double pt)
	{
		if(pt==null)
			return null;
		
		int i, size = params.size();
		double lastAngle=0;
		Point2D.Double newPt = new Point2D.Double(0, 0);
		PSTParameters p;
		PositionParameters pos;
		Vector<Point2D.Double> gcs = new Vector<Point2D.Double>();
		boolean began = false;
		
		if(size==0) return new Point2D.Double(0,0);
		gcs.add(new Point2D.Double(0,0));
		
		for(i=1; i<size; i++)
		{
			p = params.elementAt(i);
			pos = p.getPosition();
			
			if(pos!=null)
			{
				newPt.x+=pos.dx * (p.unit==PSTricksConstants.DEFAULT_UNIT?  p.xUnit : p.unit);
				newPt.y+=pos.dy * (p.unit==PSTricksConstants.DEFAULT_UNIT? p.yUnit : p.unit);

				if(pos.angleRput!=0 || !pos.isSumRotation)
				{
					if(began)	
						newPt = rotatePoint(newPt, gcs.lastElement(), toRadians(lastAngle));
					
					gcs.add((Point2D.Double)newPt.clone());
					
					if(pos.isSumRotation)
						 lastAngle += (360/params.elementAt(i-1).degrees)*pos.angleRput;
					else lastAngle =  (360/params.elementAt(i-1).degrees)*pos.angleRput;
					
					began = true;
				}
			}
		}
		
		if(!gcs.isEmpty() && began)
			newPt = rotatePoint(new Point2D.Double(newPt.x+pt.x, newPt.y+pt.y), gcs.lastElement(), toRadians(lastAngle));
			
		if(!began)
			newPt.setLocation(newPt.x+pt.x, newPt.y+pt.y);
		
		gcs.removeAllElements();
		
		return newPt;
	}

	
	
	
	/**
	 * Rotates a point with as reference an other point.
	 * @param p The point to rotate.
	 * @param gravityC The point of reference.
	 * @param theta The angle of rotation (in radian).
	 * @return The rotated point or null.
	 */
	public static Point2D.Double rotatePoint(Point2D.Double p, Point2D.Double gravityC, double theta)
	{
		if(p==null || gravityC==null)
			return null;
		
		if(theta==0) 
			return (Point2D.Double)p.clone();
		
		Point2D.Double pt = new Point2D.Double();

		pt.x = cos(theta) * (p.x - gravityC.x) - sin(theta)* (p.y - gravityC.y) + gravityC.x;
		pt.y = sin(theta) * (p.x - gravityC.x) + cos(theta)* (p.y - gravityC.y) + gravityC.y;

		return pt;
	}
	
	
	
	/**
	 * Parses a color command.
	 * @throws InvalidFormatCommandException If the format of the command is not valid.
	 * @since 2.0.2
	 */
	public void parse_color() throws InvalidFormatCommandException
	{
		String name;
		Color c;
		
		skipWSPComments(); // Reading the name of the colour.
		
		if(getChar()!='{') throw new InvalidFormatCommandException(getLinePosition());
		
		nextChar();
		name = paramParser.readString(true);
		
		if(name==null || name.length()==0) throw new InvalidFormatCommandException(getLinePosition());
		
		skipWSPComments();
		
		if(getChar()!='}') throw new InvalidFormatCommandException(getLinePosition());
		
		nextChar();
		c = DviPsColors.getColour(name); // We search the colour
		
		if(c!=null && !params.isEmpty()) 
			params.lastElement().textColor = c;
	}
	
	
	
	/**
	 * Parses a definecolor command.
	 * @throws InvalidFormatCommandException If the format of the command is not valid.
	 * @since 2.0.2
	 */
	public void parse_definecolor() throws InvalidFormatCommandException
	{
		String name;
		String type;

		skipWSPComments(); // Reading the name of the colour.
		if(getChar()!='{') throw new InvalidFormatCommandException(getLinePosition());
		nextChar();
		name = paramParser.readString(true);
		if(name==null || name.length()==0) throw new InvalidFormatCommandException(getLinePosition());
		skipWSPComments();
		if(getChar()!='}') throw new InvalidFormatCommandException(getLinePosition());
		nextChar();
		
		skipWSPComments(); // Reading the type of the colour.
		if(getChar()!='{') throw new InvalidFormatCommandException(getLinePosition());
		nextChar();
		type = paramParser.readString(true);
		if(type==null || type.length()==0) throw new InvalidFormatCommandException(getLinePosition());
		skipWSPComments();
		if(getChar()!='}') throw new InvalidFormatCommandException(getLinePosition());
		nextChar();
		
		skipWSPComments(); // Reading the colour.
		if(getChar()!='{') throw new InvalidFormatCommandException(getLinePosition());
		nextChar();
		
		if(type.equals(PSTricksConstants.TOKEN_COLOR_TYPE_NAMED))
		{ // definecolor{alias}{named}{colorName}
			Color c = DviPsColors.getColour(paramParser.readString(true));
			
			if(c==null) throw new InvalidFormatCommandException(getLinePosition());
			DviPsColors.addUserColour(c, name);
		}
		else
		{
			double val1 = paramParser.readReal();
			if(val1<0 || val1>1) throw new InvalidFormatCommandException(getLinePosition());
			
			if(type.equals(PSTricksConstants.TOKEN_COLOR_TYPE_GRAY)) // definecolor{name}{gray}{val}
				DviPsColors.addUserColour(new Color((float)val1, (float)val1, (float)val1), name);	
			else
			{
				skipWSPComments();
				if(getChar()!=',') throw new InvalidFormatCommandException(getLinePosition());
				nextChar();
				double val2 = paramParser.readReal();
				
				skipWSPComments();
				if(getChar()!=',') throw new InvalidFormatCommandException(getLinePosition());
				nextChar();
				double val3 = paramParser.readReal();
				
				if(val2<0 || val2>1 || val3<0 || val3>1) 
					throw new InvalidFormatCommandException(getLinePosition());
				
				if(type.equals(PSTricksConstants.TOKEN_COLOR_TYPE_RGB))// definecolor{name}{rgb}{r g b}
					DviPsColors.addUserColour(new Color((float)val1, (float)val2, (float)val3), name);
				else 
					if(type.equals(PSTricksConstants.TOKEN_COLOR_TYPE_HSB))
						DviPsColors.addUserColour(Color.getHSBColor((float)val1, (float)val2, (float)val3), name);
					else
						if(type.compareTo(PSTricksConstants.TOKEN_COLOR_TYPE_CMYK)==0)
                        {
							skipWSPComments();
							if(getChar()!=',') throw new InvalidFormatCommandException(getLinePosition());
							nextChar();
							double val4 = paramParser.readReal();
							
                            if(val4<0 || val4>1)
                                throw new InvalidFormatCommandException(getLinePosition());
                          
                            DviPsColors.addUserColour(DviPsColors.CMYK2RGB(val1, val2, val3, val4), name);
                         }
                         else
                        	 throw new InvalidFormatCommandException(getLinePosition());
			}//else
		}//else
		
		skipWSPComments();
		if(getChar()!='}') throw new InvalidFormatCommandException(getLinePosition());
		nextChar();
	}
	
	
	
	/**
	 * Parses a 'put' command.
	 * @throws InvalidFormatCommandException If the format of the command is not valid.
	 * @since 2.0.2
	 */
	public void parse_put() throws InvalidFormatCommandException
	{
		skipWSPComments();
		
		if(getChar()=='(')
		{
			Point2D.Double coord = paramParser.readOneCoordinate(true);
			PSTParameters      p = new PSTParameters(params.lastElement());
			
			if(coord==null) throw new InvalidFormatCommandException("put", -1);//$NON-NLS-1$
			
			skipWSPComments();
			
			p.setPosition(coord.x, coord.y, true, 0, 0, PositionParameters.REF_POINT_LEFT+PositionParameters.REF_POINT_BASE,
					0, PositionParameters.REF_POINT_DEFAULT_CENTER);

			if(params.size()>0)
			{
				PSTParameters p2 = params.lastElement();
				p.origin.setLocation(p2.origin.x, p2.origin.y);
			}
			
			params.add(p);
		}
	}
}
