/*

  FITSPNG     FITS to PNG converter
  Copyright (C) 2006-2017  Filip Hroch, Masaryk University, Brno, CZ

  Fitspng is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Fitspng is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Fitspng.  If not, see <http://www.gnu.org/licenses/>.

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <float.h>

#include <assert.h>

void help();
void version();
int fitspng(char *, char *,int,float,float,int,float,float,float,
	    float,float,int,float,float,float,int,float,int,int,int);

int main(int argc,char *argv[])
{
  char *fits = NULL, *png = "fitspng.png";
  int bit_depth, scale, type, verb;
  float u,v,t,s,f,z,st,ss, satur = 1.0;
  int i, set_fl, set_fn, dcspace = 0;
  float xw = 0.2009;  /* for D65 and 2 deg. observer */
  float yw = 0.4610;

  bit_depth = 8;
  scale = 1;
  type = 0;
  verb = 0;
  set_fl = 0; set_fn = 0;

  u = 0.5;
  v = 15.0;
  t = 0.0;
  s = 1.0;
  f = 1.0;
  z = 0.0;

  if( argc == 1 ) {
    help();
    return(0);
  }

  for( i = 1; i < argc; i++) {

    /* help */
    if( strcmp(argv[i],"--help") == 0 || strcmp(argv[i],"-h") == 0 ){
      help();
      return(0);
    }

    /* version */
    if( strcmp(argv[i],"--version") == 0 ){
      version();
      return(0);
    }

    /* verbose */
    else if( strcmp(argv[i],"-v") == 0 || strcmp(argv[i],"--verbose") ==0 ){
      verb = 1;
    }

    /* output name */
    else if( strcmp(argv[i],"-o") == 0 && i++ < argc ){
      png = argv[i];
    }

    /* bit depth */
    else if( strcmp(argv[i],"-B") == 0 && i++ < argc ){

      if( !( sscanf(argv[i],"%d",&bit_depth) == 1 &&
	     (bit_depth == 8 ||  bit_depth == 16)) ) {
        fprintf(stderr,"Set bitpix to 8 or 16.\n");
        return(1);
      }
    }

    else if( strcmp(argv[i],"-f") == 0 && i++ < argc ){

      /* linear */
      if( strcmp(argv[i],"linear") == 0 )
	type = 0;

      /* asinh */
      else if( strcmp(argv[i],"asinh") == 0 )
	type = 1;

      /* magnitude */
      else if( strcmp(argv[i],"log") == 0 )
	type = 2;

      /* gamma */
      else if( strcmp(argv[i],"gamma") == 0 )
	type = 3;

      /* error function */
      else if( strcmp(argv[i],"normal") == 0 )
	type = 4;

      /* square root */
      else if( strcmp(argv[i],"sqrt") == 0 )
	type = 5;

      /* square */
      else if( strcmp(argv[i],"sqr") == 0 )
	type = 6;

      /* logistic */
      else if( strcmp(argv[i],"logistic") == 0 )
	type = 7;

      /* atan */
      else if( strcmp(argv[i],"atan") == 0 )
	type = 8;

      /* ratio */
      else if( strcmp(argv[i],"ratio") == 0 )
	type = 9;

      else {
        fprintf(stderr,"A wrong type of conversion specified.\n");
        return(1);
      }
    }

    /* intensity scale */
    else if( strcmp(argv[i],"-f0") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f",&f) != 1 && f > 0.0 ) {
        fprintf(stderr,"-f0: Specify a positive real number.\n");
        return(1);
      }
    }

    /* zero point */
    else if( strcmp(argv[i],"-fz") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f",&z) != 1 ) {
        fprintf(stderr,"-fz: Specify a real number.\n");
        return(1);
      }
    }

    /* linear scale */
    else if( strcmp(argv[i],"-fl") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f,%f",&t,&s) != 2 && s > 0.0 ) {
        fprintf(stderr,"-fl: Specify a pair of real numbers separated by comma(second positive).\n");
        return(1);
      }
      set_fl = 1;
    }

    /* relative intensity scale */
    else if( strcmp(argv[i],"-fr") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f,%f",&u,&v) != 2 && v > 0.0 ) {
        fprintf(stderr,"-fr: Specify a pair of real numbers separated by comma (second positive).\n");
        return(1);
      }
    }

    /* colour saturation factor */
    else if( strcmp(argv[i],"-fs") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f",&satur) != 1 && satur >= 0.0 ) {
        fprintf(stderr,"-fs: Specify a positive (or zero) real number.\n");
        return(1);
      }
    }

    /* white point */
    else if( strcmp(argv[i],"-fw") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f,%f",&xw,&yw) != 2 ) {
        fprintf(stderr,"-fw: Specify a pair of real numbers separated by comma (both positive).\n");
        return(1);
      }
    }

    /* display colourspace */
    else if( strcmp(argv[i],"-cs") == 0 && i++ < argc ){

      /* sRGB */
      if( strcmp(argv[i],"sRGB") == 0 )
	dcspace = 0;

      /* AdobeRGB */
      else if( strcmp(argv[i],"AdobeRGB") == 0 )
	dcspace = 1;

      else {
        fprintf(stderr,"A wrong type of display colour-space specified.\n");
        return(1);
      }
    }

    /* scotopic threshold */
    else if( strcmp(argv[i],"-fn") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f,%f",&st,&ss) != 2 && ss > 0.0) {
        fprintf(stderr,"-fn: Specify a pair of real numbers separated by comma (second positive).\n");
        return(1);
      }
      set_fn = 1;
    }

    /* rescale */
    else if( strcmp(argv[i],"-s") == 0 && i++ < argc ){
      if( !(sscanf(argv[i],"%d",&scale) == 1 || scale > 0 ) ) {
        fprintf(stderr,"-s: Specify a positive decimal number.\n");
        return(1);
      }
    }

    else {

      if( *argv[i] == '-' )
	fprintf(stderr,"Warning: an unknown parameter? (%s).\n",argv[i]);

      fits = argv[i];
    }

  }

  if( verb )
    fprintf(stderr,"Scaling parameters (type=%d): %f %f\n",type,u,v);

  if( ! fits ) {
    fprintf(stderr,"A FITS file is not specified.\n");
    return(1);
  }

  return(fitspng(fits,png,bit_depth,u,v,scale,f,t,s,st,ss,type,satur,xw,yw,
		 dcspace,z,verb,!set_fl,set_fn));
}


void help()
{
  printf("%s\n\n%s\n\n%s\n%s\n%s\n%s\n %s\n%s\n%s\n %s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n",
	 "FITSPNG    FITS to PNG converter",
	 "Usage: fitspng [options] file",
	 "options:",
	 "\t-f x\t tone function - Func(..): linear (default), asinh, log,",
	 "\t\t gamma, normal, sqrt, sqr (square), logistic, atan, ratio",
	 "\t-f0 f\t parameter: f0*Func(..) (default: 1)",
	 "\t-fz z\t zero parameter: Func(..) + z (default: 0)",
	 "\t-fr u,v\t scale parameters: (median-u*mad)/(mad/v) (default: 3,1/3)",
	 "\t-fl t,s\t scale parameters: (x-t)/s",
	 "\t-fs x\t colour saturation (default: 1)",
	 "\t-fn st,ss threshold and width for night/day vision (no default)",
	 "\t-fw x,y\t coordinates of white point",
	 "\t-cs x\t output colour-space: sRGB (default) or AdobeRGB",
	 "\t-s s\t zoom-out image by 1/s, to create thumbnails (default: 1)",
	 "\t-o\t output filename (default: fitspng.png)",
	 "\t-B\t bits per pixel of output image: 8,16 (default: 8)",
	 "\t-v\t verbose mode",
	 "\t--help\t\t give this help",
	 "\t--version\t display software version");
}


void version()
{
  fprintf(stdout,"FITSPNG, %s, (C) 2006-2017 F. Hroch,"
	  " Masaryk University in Brno, CZ\n",VERSION);
  fprintf(stdout,"FITSPNG comes with ABSOLUTELY NO WARRANTY.\n");
  fprintf(stdout,"You may redistribute copies of FITSPNG\n");
  fprintf(stdout,"under the terms of the GNU General Public License.\n");
  fprintf(stdout,"For more information about these matters, see the file named COPYING.\n");
}
