/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#include "input.h"
#include "curve_editor.h"
#include "curve_library.h"
#include "console.h"
#include "font.h"
#include "utils.h"
#include "random.h"
#include "sine_mixer.h"
#include "beat2value.h"
#include "vector2d.h"
#include "key_consts.h"
#include "ansi_color_codes.h"
#include "fft.h"
#include "din.h"

#include <iostream>
using namespace std;

extern console cons;

extern sine_mixer sinemixer;
extern fft fft0;

extern int lmb, rmb;

console& operator<< (console& c, const hit_t& h) {
  const std::string what [] = {" nothing", " vertex", " L-tangent", " R-tangent"};
  c << h.crv->name << what[h.what] << ' ' << h.id;
  return c;
}

curve_editor::curve_editor (const std::string& settingsf, const std::string& helpf) : helptext (helpf) {

  curcrv = 0;

  use_existing_hit = false;
  hlid = 0;

  todo = DO_NOTHING;

  is_waveform_editor = sine_enabled = fft_enabled = samples_enabled = 0;

  load  (settingsf);

  overlay = 0;

}

curve_editor::~curve_editor () {
  save ();
}

void curve_editor::load (const std::string& fname) {

  // load settings from file

  extern std::string dotdin;
  std::string feditor = dotdin + fname;
  ifstream file (feditor.c_str (), ios::in);
  if (!file) {
    cout << FAIL << "!!! cannot open curve editor settings from: " << fname << " !!!" << ENDL;
    return;
  } else cout << LOAD << "<<< loading curve editor settings from: " << feditor;

  settings_filename = fname;

  std::string ignore;

  basic_editor::load (file);

  int ct; file >> ignore >> ct;
  carry_tangents = (bool) ct;

  int mt; file >> ignore >> mt;
  mirror_tangents = (bool) mt;

  file >> ignore >> sine_enabled;

  file >> ignore >> labelvertex;

  tb.load (file);

  file >> ignore >> fft_enabled;
  file >> ignore >> samples_enabled;
  file >> ignore >> hz >> ignore >> nperiods;
  if (hz <= 0) hz = 440;
  if (nperiods <= 0) nperiods = 2;
  cs.set (hz, nperiods);
  file >> ignore >> is_waveform_editor;

  cout << ", done >>>" << ENDL;

}

void curve_editor::save () {

  // save settings to file
  extern std::string dotdin;
  ofstream file ((dotdin + settings_filename).c_str(), ios::out);
  if (!file) {
    cout << "cannot save " << settings_filename << endl;
    return;
  }

  basic_editor::save (file);

  file << "carry_tangents " << carry_tangents << eol;
  file << "mirror_tangents " << mirror_tangents << eol;
  file << "sine_enabled " << sine_enabled << eol;
  file << "label_vertex " << labelvertex << eol;

  tb.save (file);

  file << "fft_enabled " << fft_enabled << eol;
  file << "samples_enabled " << samples_enabled << eol;
  file << "hz " << hz << eol;
  file << "nperiods " << nperiods << eol;
  file << "is_waveform_editor " << is_waveform_editor << eol;

}

bool curve_editor::handle_input () {

  basic_editor::handle_input (); // pan, zoom, snap etc

  if (keypressed (k_f)) { // move vertex/tangent
    if (todo) {
      todo = DO_NOTHING;
      clear_hit (mov);
      return true;
    } else {
      mov = hittest ();
      if (mov()) {
        undos.push_back (undo_t (mov.crv_id, *mov.crv, win));
        curveinfo[mov.crv_id].lisner->selected (mov.crv_id);
        cons << console::green << "moving " << mov;
        if (keydown (k_lshift)) {
          todo = MOVE_ALL;
          cons << console::green << " and beyond";
        } else todo = MOVE_PICKED;
        cons << eol;
        return true;
      }
    }
  } else {
    switch (todo) {
      case DO_NOTHING:
        break;
      case MOVE_PICKED:
        move ();
        return true;
      case MOVE_ALL:
        move (1);
        return true;
    }
  }

  // sine harmonic composer
  if (sine_enabled) if (sinemixer.handle_input ()) return true;
  if (fft_enabled) if (fft0.lev.handle_input() ) return true;

  if (library) {
    if (library->num_curves ()) {
      if (keypressedd (k_9)) { load_curve (-1); return true; } else
      if (keypressedd (k_0)) { load_curve (+1); return true; } else
      if (keypressed (k_minus)) { library->del (); return true; }
    }
    if (keypressed (k_equals)) { add_curve (); return true; }
  }

  if (keypressed (k_v)) {

    if (keydown (k_lctrl)) { // l_ctrl + v ie paste
      pik = hittest ();
      paste (pik);
    } else { // just v ; delete vertex/tanget
      del = hittest ();
      if (del()) {
        remove ();
        clear_hit (del);
      }
    }
  }

  else if (keypressed (k_i)) { // i to pick, i again to insert
    if (num_curves () == 1) {
      if (!ins()) ins = hit_t (curveinfo[0].curve, 0);
      insert ();
      clear_hit (ins);
    } else {
      if (!ins()) {
        ins = hittest();
        if (ins()) cons << "press i again to insert vertex" << eol; else cons << console::red << "press i on a diamond to insert on that curve" << eol;
      } else {
        insert ();
        clear_hit (ins);
      }

    }
  }

  else if (keypressedd (k_z) && !mov()) { // undo & redo
    if (keydown (k_lshift)) // l_shift + z - redo
      dodo (redos, undos, "no more redos.");
    else
      dodo (undos, redos, "no more undos."); // just z, undo
  }

  else if (keypressedd (k_space) && sine_enabled) sine2curve (); // take sine from sine_mixer and make curve

  else if (keypressed (k_r)) { // create hit list, pick the top
    use_existing_hit = false;
    hittest ();
    if (pik()) {
      pick (pik.crv_id);
      use_existing_hit = true;
      cons << console::green << "picked " << pik << " [" << hlid+1 << '/' << hitlist.size() << ']' << eol;
    }
  }

  else if (keypressed (k_left)) { // browse hit list
    int n = hitlist.size ();
    if (n) {
      int l = n - 1;
      if (--hlid < 0) hlid = l;
      pik = hitlist[hlid];
      calc_hit_params (pik);
      curveinfo [pik.crv_id].lisner->selected (pik.crv_id);
      use_existing_hit = true;
      cons << console::green << "picked " << pik << " [" << hlid+1 << '/' << hitlist.size() << ']' << eol;
    }
  }

  else if (keypressed (k_right)) { // browse hit list
    int n = hitlist.size();
    if (n) {
      int l = n - 1;
      if (++hlid > l) hlid = 0;
      pik = hitlist[hlid];
      calc_hit_params (pik);
      curveinfo [pik.crv_id].lisner->selected (pik.crv_id);
      use_existing_hit = true;
      cons << console::green << "picked " << pik << " [" << hlid+1 << '/' << hitlist.size() << ']' << eol;
    }
  }

  else if (keypressedd (k_t)) { // mirror about x
    mir = hittest ();
    if (mir()) {
      if (keydown (k_lshift))
        mirror (1);  // whole curve
      else
        mirror (); // only hit vertex
      clear_hit (mir);
    }
  }

  else if (keypressed (k_l)) {

    hit_t hit = hittest ();

    if (hit ()) {
      // turn hit curve into polyline by folding tangents into vertex
      // still bezier curve but looks like a polyline ie l_shift + k to go back to bezier curve
      //
      curve_info& ci = curveinfo [hit.crv_id];
      multi_curve& crv = *ci.curve;
      undos.push_back (undo_t (hit.crv_id, crv, win));
      if (keydown (k_lshift)) { // l_shift + l ie fold tangents of hit vertex only
        float vx, vy; crv.get_vertex (hit.id, vx, vy);
        crv.set_left_tangent (hit.id, vx, vy);
        crv.set_right_tangent (hit.id, vx, vy);
        crv.evaluate ();
      } else convert2_polyline (crv);
      ci.lisner->edited (this, hit.crv_id);
    } else labelvertex = !labelvertex; // no curve hit so turn on the labelling of vertices

  }

  else if (keypressed (k_k)) {

    hit_t hit = hittest ();

    if (hit()) {
      curve_info& ci = curveinfo [hit.crv_id];
      multi_curve& crv = *ci.curve;
      undos.push_back (undo_t (hit.crv_id, crv, win));
      const point<float>& obj = get_obj_chunk ();
      if (keydown (k_lshift)) { // unfurl tangents of hit vertex
        float vx, vy; crv.get_vertex (hit.id, vx, vy);
        crv.set_left_tangent (hit.id, vx - obj.x, vy);
        crv.set_right_tangent (hit.id, vx + obj.x, vy);
        crv.evaluate ();
      } else convert2_catmull_rom (crv, max (obj.x, obj.y)); // turn bezier curve into 'catmull-rom' spline; still a bezier curve
      ci.lisner->edited (this, hit.crv_id);
    }
  }

  else if (keypressed (k_m)) { // mirror tangents ie move one tangent and the other mirrors it
    mirror_tangents = !mirror_tangents;
    if (mirror_tangents) cons << console::yellow << "tangents are mirrored." << eol; else cons << console::yellow << "tangents are not mirrored." << eol;
  }

  else if (keypressedd (k_left_bracket)) { // decrease curve resolution
    hit_t hit = hittest ();
    if (hit()) {
      resolution (hit.crv_id, +1);
    }
  }

  else if (keypressedd (k_right_bracket)) { // increase currve resolution
    hit_t hit = hittest ();
    if (hit()) {
      resolution (hit.crv_id, -1);
    }
  }

  else if (keypressedd (k_quote)) { // decrease num sine samples
    sine_mixer::set_sine_samples (sine_mixer::NUM_SINE_SAMPLES - 1);
  }

  else if (keypressedd (k_hash)) { // increase num sine samples
    sine_mixer::set_sine_samples (sine_mixer::NUM_SINE_SAMPLES + 1);
  }

  else if (keypressed (k_g)) { // draw scratch curve
    if (num_curves () == 1) {

      if (!rep()) {
        rep = hit_t (curveinfo[0].curve, 0);
        clear_scratch_curve ();
        cons << console::yellow << "g to add vertex, h to complete or abort" << eol;
      }

      show_scratch_curve = true;
      float sx, sy; snap (sx, sy);
      win_scratch_points.push_back (point<float> (sx, sy));
      float curvx, curvy; win2obj (sx, sy, curvx, curvy);
      curv_scratch_points.push_back (point<float>(curvx, curvy));

    } else {

      if (rep()) {
        show_scratch_curve = true;
        float sx, sy; snap (sx, sy);
        win_scratch_points.push_back (point<float> (sx, sy));
        float curvx, curvy; win2obj (sx, sy, curvx, curvy);
        curv_scratch_points.push_back (point<float>(curvx, curvy));
      } else {
        rep = hittest ();
        if (rep()) {
          clear_scratch_curve ();
          cons << console::yellow << "g to add vertex, h to complete or abort" << eol;
        }
      }
    }

  }

  else if (keypressed (k_h)) { // complete scratch curve
    if (rep()) {
      replace ();
      clear_hit (rep);
      show_scratch_curve = false;
    }
  }

  else if (keypressed (k_c)) {
    if (keydown (k_lctrl)) { // l_ctrl + c - copy curve
      hit_t hit = hittest ();
      if (hit()) {
        copy = *get_curve(hit.crv_id);
        cons << console::green << "copied curve." << eol;
      }
    } else { // carry tangents?
      carry_tangents = !carry_tangents;
      if (carry_tangents) cons << console::yellow << "vertices carry their tangents." << eol; else cons << console::yellow << "vertices desert their tangents" << eol;
    }
  }

  else if (keypressedd (k_semicolon)) { // change curve color
    hit_t hit = hittest ();
    if (hit()) get_curve(hit.crv_id)->set_color ();
  }

  else if (keypressed (k_comma)) { // toggle sine composer
    if (is_waveform_editor) sine_enabled = !sine_enabled;
  }

  else if (keypressed (k_period)) { // toggle FFT display
    if (is_waveform_editor) {
      fft_enabled = !fft_enabled;
      if (fft_enabled) {
        if (num_curves()) fft0.go (curveinfo[0].curve);
      }
    }
  }

  else if (keypressed (k_slash)) { // toggle samples display
    if (is_waveform_editor) {
      samples_enabled = !samples_enabled;
      if (samples_enabled) {
        if (num_curves()) cs.render (curveinfo[0].curve);
      }
    }
  }

  else if (keypressed (k_ralt)) { // mark curve segments
    hit_t hit = hittest ();
    if (hit()) {
      curve_info& ci = curveinfo [hit.crv_id];
      ci.mark_segments = !ci.mark_segments;
    }
  }

  else if (keypressed (k_f8)) { // apply mouse capture
    hit_t hit = hittest ();
    if (mocap0() && hit()) {
      undos.push_back (undo_t (hit.crv_id, *get_curve(hit.crv_id), win));
      macros.push_back (mouse_macro (mocap0, hit));
      cons << console::green << "assigned mouse capture to " << hit << eol;
    }
  }

  else if (keydown (k_f9)) { // remove mouse capture
    hit_t hit = hittest ();
    if (hit()) {
      for (vector<mouse_macro>::iterator i = macros.begin(), j = macros.end(); i != j; ++i) {
        mouse_macro& m = *i;
        if (hit == m.hit) {
          macros.erase (i);
          cons << "detached mouse capture from " << hit << eol;
          break;
        }
      }
    }
  }

  else if (keypressed (k_o)) { // toggle overlay of current din instrument
    overlay = !overlay;
    if (overlay) cons << console::green << "Overlay is ON" << eol; else cons << console::red << "Overlay is OFF" << eol;
  }

  else if (keypressed (k_f1)) helptext(); // show help

  return true;

}

void curve_editor::sine2curve () {
  int cid = 0;
  if (pik()) cid = pik.crv_id;
  extern sine_mixer sinemixer;
  curve_info& ci = curveinfo[cid];
  multi_curve& crv = *ci.curve;
  undos.push_back (undo_t (cid, crv, win));
  sinemixer.make_curve (crv);
  ci.lisner->edited (this, cid);
}

void curve_editor::apply_mocap () {

  float dx, dy;
  for (int i = 0, j = macros.size(); i < j; ++i) {
    mouse_macro& m = macros[i];
    hit_t& h = m.hit;
    m.mo.get (dx, dy);
    move (h, dx, dy);
  }

}

void curve_editor::pick (int i) {
  int ncurves = curveinfo.size ();
  if (i > -1 && i < ncurves) {
    curve_info& ci = curveinfo[i];
    ci.lisner->selected (i);
  }
}

void curve_editor::calc_hit_params (hit_t& hit) {

  if (!hit()) return;

  multi_curve* crv = get_curve(hit.crv_id);
  float vx, vy, lx, ly, rx, ry;
  int id = hit.id;
  crv->get_vertex (id, vx, vy);
  crv->get_left_tangent (id, lx, ly);
  crv->get_right_tangent (id, rx, ry);
  direction (hit.left_tangent.x, hit.left_tangent.y, vx, vy, lx, ly);
  direction (hit.right_tangent.x, hit.right_tangent.y, vx, vy, rx, ry);
  hit.left_tangent_magnitude = magnitude (hit.left_tangent.x, hit.left_tangent.y);
  hit.right_tangent_magnitude = magnitude (hit.right_tangent.x, hit.right_tangent.y);

}

void curve_editor::hittest (multi_curve* crv, int crv_id, const points_array& points, unsigned int what) {

  for (int i = 0, j = points.size (); i < j; ++i) {
    const point<float>& v = points[i];
    float vdx, vdy; obj2win (v, vdx, vdy);
    float d2 = distance2<float> (vdx, vdy, win.mousex, win.mousey);
    if (d2 <= win.handle_radius2) hitlist.push_back (hit_t (crv, crv_id, what, i));
  }
}

hit_t curve_editor::hittest () {

  hit_t hit;

  if (use_existing_hit) {
    use_existing_hit = false;
    return pik;
  } else {
    hitlist.clear ();
    hlid = 0;
  }

  for (int i = 0, j = curveinfo.size (); i < j; ++i) {
    curve_info& ci = curveinfo[i];
    if (ci.disabled == false) {
      multi_curve* crv = ci.curve;
      hittest (crv, i, crv->vertices, hit_t::VERTEX);
      hittest (crv, i, crv->left_tangents, hit_t::LEFT_TANGENT);
      hittest (crv, i, crv->right_tangents, hit_t::RIGHT_TANGENT);
    }
  }

  int n = hitlist.size();
  if (n) {
    pik = hitlist[0];
    calc_hit_params (pik);
  } else pik = hit_t ();

  return pik;

}

bool curve_editor::move (hit_t& hit, float x, float y) {

  curve_info& ci = curveinfo [hit.crv_id];
  multi_curve* crv = ci.curve;

  bool need_eval = false;

  switch (hit.what) {

    case hit_t::VERTEX:
        need_eval = crv->set_vertex (hit.id, x, y);
        if (carry_tangents && need_eval) {
          crv->set_left_tangent (hit.id, x + hit.left_tangent.x, y + hit.left_tangent.y);
          crv->set_right_tangent (hit.id, x + hit.right_tangent.x, y + hit.right_tangent.y);
        }
      break;

    case hit_t::LEFT_TANGENT:
        need_eval = crv->set_left_tangent (hit.id, x, y);
        if (mirror_tangents && need_eval) {
          float vx, vy, lvx, lvy;
          crv->get_vertex (hit.id, vx, vy);
          unit_vector (lvx, lvy, vx, vy, x, y);
          crv->set_right_tangent (hit.id, vx - hit.right_tangent_magnitude * lvx, vy - hit.right_tangent_magnitude * lvy);
        }
      break;

    case hit_t::RIGHT_TANGENT:
        need_eval = crv->set_right_tangent (hit.id, x, y);
        if (mirror_tangents && need_eval) {
          float vx, vy, rvx, rvy;
          crv->get_vertex (hit.id, vx, vy);
          unit_vector (rvx, rvy, vx, vy, x, y);
          crv->set_left_tangent (hit.id, vx - hit.left_tangent_magnitude * rvx, vy - hit.left_tangent_magnitude * rvy);
        }
      break;

  }

  if (need_eval) {
    crv->evaluate();
    ci.lisner->edited (this, hit.crv_id);
  }

  return need_eval;

}

bool curve_editor::move () {
  float sx, sy; snap (sx, sy);
  float cx, cy; win2obj (sx, sy, cx, cy);
  return move (mov, cx, cy);
}

bool curve_editor::move (int) {

  float sx, sy; snap (sx, sy);

  float cx, cy; win2obj (sx, sy, cx, cy);

  curve_info& ci = curveinfo [mov.crv_id];
  multi_curve* crv = ci.curve;

  float vx, vy;
  float dx, dy;

  switch (mov.what) {
    case hit_t::VERTEX:
        crv->get_vertex (mov.id, vx, vy);
        break;
    case hit_t::LEFT_TANGENT:
        crv->get_left_tangent (mov.id, vx, vy);
        break;
    case hit_t::RIGHT_TANGENT:
        crv->get_right_tangent (mov.id, vx, vy);
        break;
  }

  dx = cx - vx;
  dy = cy - vy;

  bool result = false;
  for (int i = mov.id, j = crv->num_vertices (); i < j; ++i) {
    crv->get_vertex (i, vx, vy);
    result |= crv->set_vertex (i, vx + dx, vy + dy);
    crv->get_left_tangent (i, vx, vy);
    result |= crv->set_left_tangent (i, vx + dx, vy + dy);
    crv->get_right_tangent (i, vx, vy);
    result |= crv->set_right_tangent (i, vx + dx, vy + dy);
  }

  if (result) {
    crv->evaluate();
    ci.lisner->edited (this, mov.crv_id);

  }

  return result;

}

void curve_editor::remove () {

  curve_info& ci = curveinfo [del.crv_id];
  multi_curve* crv = ci.curve;

  undos.push_back (undo_t (del.crv_id, *crv, win));

  if (crv->remove (del.id)) {
    crv->evaluate();
    ci.lisner->edited (this, del.crv_id);
  } else {
    cons << console::red << "cant delete curve: " << crv->name << eol;
    undos.pop_back ();
  }

}

void curve_editor::insert () {

  float sx, sy; snap (sx, sy);
  float cx, cy; win2obj (sx, sy, cx, cy);


  curve_info& ci = curveinfo [ins.crv_id];
  multi_curve* crv = ci.curve;
  undos.push_back (undo_t (ins.crv_id, *crv, win));

  const point<float>& obj = get_obj_chunk ();
  if (crv->insert (cx, cy, obj.x, obj.y)) {
    crv->evaluate();
    ci.lisner->edited (this, ins.crv_id);
  } else undos.pop_back ();

}

void curve_editor::replace () {

  if (curv_scratch_points.size () > 1) {

    curve_info& ci = curveinfo [rep.crv_id];
    multi_curve& crv = *ci.curve;
    undos.push_back (undo_t (rep.crv_id, crv, win));

    create_polyline (crv, curv_scratch_points);
    const point<float>& obj = get_obj_chunk ();
    convert2_catmull_rom (crv, max (obj.x, obj.y));

    ci.lisner->edited (this, rep.crv_id);

    cons << console::green << "applied scratch curve." << eol;

  } else cons << console::red << "aborted scratch curve" << eol;


}

void curve_editor::mirror (int whole_curve) {

  // mirrors vertex (or curve) about horizontal - useful when editing waveforms

  curve_info& ci = curveinfo [mir.crv_id];
  multi_curve* crv = ci.curve;
  undos.push_back (undo_t (mir.crv_id, *crv, win));
  float xx, yy;
  if (whole_curve) {
    int num_vertices = crv->num_vertices ();
    for (int i = 0; i < num_vertices; ++i) {
      crv->get_vertex (i, xx, yy); crv->set_vertex (i, xx, -yy);
      crv->get_left_tangent (i, xx, yy); crv->set_left_tangent (i, xx, -yy);
      crv->get_right_tangent (i, xx, yy); crv->set_right_tangent (i, xx, -yy);
    }

    crv->evaluate();
    ci.lisner->edited (this, mir.crv_id);

  } else {
    switch (mir.what) {
      case hit_t::VERTEX:
        crv->get_vertex (mir.id, xx, yy);
        break;
      case hit_t::LEFT_TANGENT:
        crv->get_left_tangent (mir.id, xx, yy);
        break;
      case hit_t::RIGHT_TANGENT:
        crv->get_right_tangent (mir.id, xx, yy);
        break;
    }
    move (mir, xx, -yy);
  }

}

void curve_editor::resolution (int i, int dir) {

  curve_info& ci = curveinfo [i];
  multi_curve* crv = ci.curve;
  float res = crv->get_resolution();
  float newres = res + dir * get_obj_resolution ();
  if (newres > 0) {
    crv->set_resolution (newres);
    crv->evaluate();
    ci.lisner->edited (this, i);
  }

}

void curve_editor::draw () {

  project ();

    draw_common ();
    draw_all ();

    if (sine_enabled) {
      draw_sine ();
      sinemixer.draw ();
    }

    if (fft_enabled) {
      extern fft fft0;
      fft0.lev.draw ();
    }

  unproject ();

  if (overlay) {
    extern din din0;
    din0.draw ();
  }

}

void curve_editor::draw_sine () {

  extern sine_mixer sinemixer;
  int n = sinemixer.norm.size ();
  float ox = 0, oy = 0, dx = 1. / (n - 1);
  glColor3f (0, 0, 1);
  glBegin (GL_LINE_STRIP);
  float wx, wy;
  for (int i = 0; i < n; ++i) {
    oy = sinemixer.norm[i];
    obj2win (ox, oy, wx, wy);
    glVertex2f (wx, wy);
    ox += dx;
  }
  glEnd ();
}

void curve_editor::draw_common () {
  basic_editor::draw ();
  mark_curve_segments ();
  if (show_scratch_curve) draw_scratch_curve ();
  selector.draw ();
}

void curve_editor::draw_scratch_curve () {

  glColor3f (1, 1, 0.7);
  glBegin (GL_LINE_STRIP);

  float x, y; snap (x, y);
  for (int i = 0, j = win_scratch_points.size (); i < j; ++i) {
    const point<float>& p = win_scratch_points[i];
    x = p.x; y = p.y;
    glVertex2f (x, y);
  }

  glVertex2f (x, y);

  float sx, sy; snap (sx, sy);
  glVertex2f (sx, sy);

  glEnd ();

}

void curve_editor::draw_all () {

  // draw labels
  tb.draw ();
  tb.clear ();

  // draw curve profile, vertices, tangents and handles
  for (int i = 0, j = curveinfo.size(); i < j; ++i) {
    curve_info& ci = curveinfo[i];
    multi_curve* crv = ci.curve;
    if (ci.disabled) {
      glColor3f (0.2, 0.2, 0.2);
      draw_curve (crv);
      draw_vertices (crv);
    } else  {
      glColor3f (crv->r, crv->g, crv->b);
      if ((i == pik.crv_id) && (j > 1)) glLineWidth (2);
        draw_curve (crv);
        draw_vertices (crv);
        draw_tangents (crv);
      glLineWidth (1);
    }
  }

  // mark beat/value (gater, fm/am, octave shift)
  int n = bv.size ();
  if (n) {
    float nowx, nowy;
    float x, y;
    for (int i = 0; i < n; ++i) {
      nowx = bv[i]->now;
      nowy = bv[i]->sol (nowx);
      obj2win (nowx, nowy, x, y);
      multi_curve& crv = bv[i]->crv;
      glLineWidth (2);
      glColor3f (crv.r, crv.g, crv.b);
      glBegin (GL_LINES);
        glVertex2i (x - win_chunk.x, y);
        glVertex2i (x + win_chunk.x, y);
        glVertex2i (x, y - win_chunk.y);
        glVertex2i (x, y + win_chunk.y);
      glEnd ();
      glLineWidth (1);
    }
  }

  // label vertices
  if (labelvertex) {
    for (int i = 0, j = curveinfo.size (); i < j; ++i) {
      curve_info& ci = curveinfo[i];
      multi_curve* crv = ci.curve;
      for (int m = 0, n = crv->num_vertices (); m < n; ++m) {
        point<float>& vm = crv->vertices[m];
        sprintf (buf, "%d", m);
        tb.add (text (buf, vm.x + 0.02, vm.y + 0.02, crv->r, crv->g, crv->b));
      }
      tb.refresh (this);
    }
  }

  if (samples_enabled) draw_samples ();

  if (id == KB_KB_ATTACK) {

    glLineWidth (2);
    glEnable (GL_LINE_STIPPLE);
    glLineStipple (1, 0xF000);
    glColor3f (0, 0.5, 1);
    glBegin (GL_LINES);
      glVertex2i (susx, win.top);
      glVertex2i (susx, win.bottom);
    glEnd ();
    glDisable (GL_LINE_STIPPLE);

    glBegin (GL_POLYGON);
      glVertex2i (susbox.left, susbox.top);
      glVertex2i (susbox.midx, susbox.bottom);
      glVertex2i (susbox.right, susbox.top);
    glEnd ();
    glLineWidth (1);

  }

  draw_cursor ();

}

void curve_editor::mark_curve_segments () {

  // drop vertical lines from each curve profile point (dx, dy) to (dx, 0)
  glColor3f (0.2, 0.2, 0.2);
  for (int i = 0, j = curveinfo.size(); i < j; ++i) {
    curve_info& ci = curveinfo[i];
    if (ci.mark_segments) {
      multi_curve* crv = ci.curve;
      vector<curve>& curv = crv->curv;
      for (int i = 0, j = curv.size(); i < j; ++i) {
        vector<crvpt>& vpts = curv[i].vpts;
        glBegin (GL_LINES);
        for (int p = 0, q = vpts.size (); p < q; ++p) {
          float dx, dy; obj2win (vpts[p].x, vpts[p].y, dx, dy);
          glVertex2f (dx, dy);
          glVertex2f (dx, 0);
        }
        glEnd ();
      }
    }
  }

}

void curve_editor::draw_handle (const point<float>& p) {
  float x, y; obj2win (p, x, y);
  float handle [] = {x - win.handle_radius, y, x, y + win.handle_radius, x + win.handle_radius, y, x, y - win.handle_radius};
  glBegin (GL_LINE_LOOP);
    glVertex2f (handle[0], handle[1]);
    glVertex2f (handle[2], handle[3]);
    glVertex2f (handle[4], handle[5]);
    glVertex2f (handle[6], handle[7]);
  glEnd ();
}

void curve_editor::draw_tangent (const point<float>& p, const point<float>& t) {
  float x, y; obj2win (p, x, y);
  float tx, ty; obj2win (t, tx, ty);
  glBegin (GL_LINES);
    glVertex2f (x, y);
    glVertex2f (tx, ty);
  glEnd ();

}

void curve_editor::draw_curve (multi_curve* crv) {

  vector<curve>& curv = crv->curv;
  for (int i = 0, j = curv.size(); i < j; ++i) {
    vector<crvpt>& vpts = curv[i].vpts;
    glBegin (GL_LINE_STRIP);
    for (int p = 0, q = vpts.size (); p < q; ++p) {
      float dx, dy; obj2win (vpts[p].x, vpts[p].y, dx, dy);
      glVertex2f (dx, dy);
    }
    glEnd ();
  }

}

void curve_editor::draw_samples () {

  float x = 0, y = 0;
  float ox, oy;
  obj2win (x, y, ox, oy);

  float r, g, b; curveinfo[0].curve->get_color (r, g, b);
  glColor3f (r, g, b);
  glBegin (GL_LINES);
  for (int i = 0; i < cs.n; ++i) {
    float dx, dy; obj2win (cs.x[i], cs.y[i], dx, dy);
    glVertex2i (dx, dy);
    glVertex2i (dx, oy);
  }
  glEnd ();

}

void curve_editor::draw_vertices (multi_curve* crv) {

  const points_array& vertices = crv->vertices;
  for (int p = 0, q = vertices.size(); p < q; ++p) {
    const point<float>& v = vertices[p];
    draw_handle (v);
  }

}

void curve_editor::draw_tangents (multi_curve* crv) {

  // draw tangents of ith curve

  const points_array& vertices = crv->vertices;
  const points_array& left_tangents = crv->left_tangents;
  const points_array& right_tangents = crv->right_tangents;
  for (int p = 0, q = vertices.size(); p < q; ++p) {
    const point<float>& v = vertices[p];
    const point<float>& lt = left_tangents[p];
    const point<float>& rt = right_tangents[p];
    draw_tangent (v, lt);
    draw_tangent (v, rt);
    draw_handle (lt);
    draw_handle (rt);
  }

}

void curve_editor::dodo (list<undo_t>& do1, list<undo_t>& do2, std::string mesg) {
  if (do1.size() > 0) {
    undo_t& last = do1.back ();
    if (last.i < num_curves ()) {
      curve_info& ci = curveinfo[last.i];
      multi_curve* crv = ci.curve;
      do2.push_back (undo_t (last.i, *crv, win));
        std::string name; float r, g, b;
        crv->get_color (r, g, b);
          *crv = last.curve;
        crv->set_color (r, g, b);
      ci.lisner->edited (this, last.i);
      win = last.win;
      do1.pop_back ();
    }
    hitlist.clear ();
  } else cons << console::red << mesg << eol;

}

void curve_editor::add (multi_curve* crv, curve_listener* lsnr) {
  curveinfo.push_back (curve_info (crv, lsnr));
}

void curve_editor::clear () {
  curveinfo.clear ();
  hitlist.clear ();
  clear_hit (pik);
}

multi_curve* curve_editor::get_curve (int i) {
  if (i > -1 && i < (int) curveinfo.size()) {
    curve_info& ci = curveinfo [i];
    return ci.curve;
  }
  cout << "could not get curve " << i << eol;
  return 0;
}

int curve_editor::num_curves () {
  return curveinfo.size ();
}

curve_info& curve_editor::get_curve_info (int i) {
  return curveinfo[i];
}

void curve_editor::enter () {

  win.calc ();
  basic_editor::calc_visual_params ();

}

void curve_editor::clear_scratch_curve () {
  win_scratch_points.clear ();
  curv_scratch_points.clear ();
  scratch_curve.clear ();
  show_scratch_curve = false;
}

void curve_editor::attach_library (curve_library* lib) {
  library = lib;
}

void curve_editor::bg () {
  apply_mocap ();
}

curve_info::curve_info  (multi_curve* c, curve_listener* l, bool dis) {
  curve = c;
  lisner = l;
  disabled = dis;
  mark_segments = false;
}

curve_info::curve_info () {
  curve = 0;
  lisner = 0;
  disabled = mark_segments = false;
}

hit_t::hit_t (multi_curve* cv, int cid, unsigned int w, int i)  {
  crv = cv;
  crv_id = cid;
  what = w;
  id = i;
}

void hit_t::clear () {
  crv = 0;
  crv_id = -1;
  what = NONE;
  id = -1;
}

int hit_t::operator() () {
  return (crv_id != -1);
}

void curve_editor::clear_hit (hit_t& h) {
  h.clear ();
  use_existing_hit = false;
}

void curve_editor::load_curve (int dir) {
  if (!pik()) pik = hit_t (curveinfo[0].curve, 0);
  multi_curve& crv = *pik.crv;
  undos.push_back (undo_t (pik.crv_id, crv, win));
    float cr, cg, cb; crv.get_color (cr, cg, cb);
      if (dir == 1) crv = library->next (); else crv = library->prev();
    crv.set_color (cr, cg, cb);
  curveinfo[pik.crv_id].lisner->edited (this, pik.crv_id);
  cons.rollup (1);
}

void curve_editor::add_curve () {
  if (!pik()) pik = hit_t (curveinfo[0].curve, 0);
  curve_info& ci = curveinfo [pik.crv_id];
  library->add (*ci.curve);
}

void curve_editor::insert_curve () {
  if (!pik()) pik = hit_t (curveinfo[0].curve, 0);
  library->insert (*get_curve (pik.crv_id));
}

void curve_editor::replace_curve () {
  if (!pik()) pik = hit_t (curveinfo[0].curve, 0);
  library->replace (*get_curve (pik.crv_id));
}

void curve_editor::paste (hit_t& hit) {
  if (hit()) {
    curve_info& ci = curveinfo [hit.crv_id];
    multi_curve* crv = ci.curve;
    if (copy.num_vertices()) {
      undos.push_back (undo_t (hit.crv_id, *crv, win));
        std::string cname; float cr, cg, cb;
        crv->get_color (cr, cg, cb);
          *crv = copy;
        crv->set_color (cr, cg, cb);
      ci.lisner->edited (this, hit.crv_id);
      cons << console::green << "pasted curve." << eol;
    } else cons << console::red << "no copy curve." << eol;
  } else cons << console::red << "pick a curve to paste." << eol;
}

std::string curve_editor::selection () {
  std::stringstream ss;
  const char* what [] = {"invalid", "v", "lt", "rt"};
  for (int i = 0, j = hitlist.size (); i < j; ++i) {
    hit_t& h = hitlist[i];
    multi_curve* c = h.crv;
    float x, y;
    switch (h.what) {
      case hit_t::VERTEX:
        c->get_vertex (h.id, x, y);
        break;
      case hit_t::LEFT_TANGENT:
        c->get_left_tangent (h.id, x, y);
        break;
      case hit_t::RIGHT_TANGENT:
        c->get_right_tangent (h.id, x, y);
        break;
    }
    ss << '{' << c->name << ' ' << what[h.what] << ' ' << h.id << ' ' << x << ' ' << y << "} ";
  }
  return ss.str ();
}

void curve_samples::render (multi_curve* crv) {

  solver ss (crv);
  float xx = -step, dxx = step;
  ss (xx, dxx, n, y);

  float dx = 1;
  for (int i = 0; i < n; ++i) {
    x[i] = dx;
    dx += step;
  }


}

void curve_samples::set (float zh, int p) {

  if (zh <= 0) zh = 440;
  if (p <= 0) p = 2;

  nperiods = p;

  hz = zh;
  hz2step (hz, step);

  int oldn = n;
  n = ceil (p * 1. / step);
  if (x) {
    if (n > oldn) {
      delete[] x; x = new float [n];
      delete[] y; y = new float [n];
    }
  } else {
    x = new float [n];
    y = new float [n];
  }
  int nsz = sizeof (float) * n;
  memset (x, 0, nsz);
  memset (y, 0, nsz);
}
