/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_logger.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref diagnostics::unittest::Test_Logger
 *
 * $Id: test_logger.t.cpp,v 1.11 2005/06/23 09:54:27 esdentem Exp $
 *
 * @author Christian Schallhart
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/unittest/test_system/test_logger.hpp>

#include <diagnostics/unittest/test_system_exception.hpp>

// test support
#include <diagnostics/util/record_vectors.ts.hpp>

#include <sstream>


#define TEST_COMPONENT_NAME Test_Logger
#define TEST_COMPONENT_NAMESPACE diagnostics::unittest

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;


////////////////////////////////////////////////////////////////////////////////

/**
 * @brief [PRO] fct/abnormal: constructing a Test_Logger with @ref LEVEL_TEST. 
 */
void wrong_build_level(Test_Data & test_data)
{
    Test_Logger::Test_Run_Results_t results;
    TEST_THROWING_BLOCK_ENTER;
    Test_Logger logger(LEVEL_TEST,LEVEL_TEST,results);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Logger logger(LEVEL_AUDIT,LEVEL_TEST,results);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Logger logger(LEVEL_TEST,LEVEL_AUDIT,results);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);


    TEST_THROWING_BLOCK_ENTER;
    Test_Logger logger(LEVEL_DEBUG,LEVEL_AUDIT,results);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Logger logger(LEVEL_PROD,LEVEL_AUDIT,results);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    Test_Logger logger(LEVEL_PROD,LEVEL_DEBUG,results);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief [PRO] fct/normal: constructing a Test_Logger with a correct build-level
 */
void correct_build_levels(Test_Data & test_data)
{
    Test_Logger::Test_Run_Results_t results;
    Test_Logger logger1(LEVEL_AUDIT,LEVEL_PROD,results);
    Test_Logger logger2(LEVEL_DEBUG,LEVEL_PROD,results);
    Test_Logger logger3(LEVEL_PROD, LEVEL_PROD,results);

    Test_Logger logger4(LEVEL_AUDIT,LEVEL_DEBUG,results);
    Test_Logger logger5(LEVEL_DEBUG,LEVEL_DEBUG,results);

    Test_Logger logger6(LEVEL_AUDIT,LEVEL_AUDIT,results);
}




////////////////////////////////////////////////////////////////////////////////

/**
 * @brief [PRO] fct/abnormal: just a plain @ref TYPE_TESTCASE_EXIT --
 * produces @ref DIAGNOSTICS_PANIC_LOG
 */
void record_sequence_plain_exit(Test_Data & test_data)
{
    Test_Logger::Test_Run_Results_t results;
    Test_Logger logger(LEVEL_AUDIT,LEVEL_AUDIT,results);
    
    // passive?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE); 
    // no results?
    TEST_ASSERT(results.size()==0);
    
    // TYPE_TESTCASE_EXIT -- INVALID -- causes PANIC LOG
    logger.log(CSTR_RECORD("panic 1",LEVEL_TEST,TYPE_TESTCASE_EXIT));
    
    // passive?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE);
    // no results?
    TEST_ASSERT(results.size()==0);
}

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief [PRO] fct/abnormal: a @ref TYPE_TESTCASE_ENTER twice --
 * produces @ref DIAGNOSTICS_PANIC_LOG
 */
void record_sequence_double_enter(Test_Data & test_data)
{
    Test_Logger::Test_Run_Results_t results;
    Test_Logger logger(LEVEL_PROD,LEVEL_PROD,results);
    
    // passive?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE);
    // no results?
    TEST_ASSERT(results.size()==0);
    
    // TYPE_TESTCASE_ENTER
    logger.log(CSTR_RECORD("ok",LEVEL_TEST,TYPE_TESTCASE_ENTER));
    
    // active?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_ACTIVE);
    // one result?
    TEST_ASSERT(results.size()==1);
    // get if for later comparison
    Test_Run_Result const result(*(results[0]));
    // result incomplete?
    TEST_ASSERT(result.abstract_state()==Test_Run_Result::STATE_INCOMPLETE);
    
    // TYPE_TESTCASE_ENTER -- INVALID -- causes PANIC LOG
    logger.log(CSTR_RECORD("panic 2",LEVEL_TEST,TYPE_TESTCASE_ENTER));
    
    // active?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_ACTIVE); 
    // one result?
    TEST_ASSERT(results.size()==1);
    // no change in the Test_Run_Result?
    TEST_ASSERT(*(results[0])==result);
}

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief [PRO] fct/normal: a @ref TYPE_TESTCASE_ENTER, a @ref
 * TYPE_TESTCASE_EXIT and a @ref TYPE_TESTCASE_ENTER
 */
void record_sequence_enter_exit_enter(Test_Data & test_data)
{
    Test_Logger::Test_Run_Results_t results;
    Test_Logger logger(LEVEL_AUDIT,LEVEL_AUDIT,results);
    
    // passive?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE);
    // no results?
    TEST_ASSERT(results.size()==0);
    
    // TYPE_TESTCASE_ENTER
    logger.log(CSTR_RECORD("ok",LEVEL_TEST,TYPE_TESTCASE_ENTER));
    
    // active?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_ACTIVE);
    // one result?
    TEST_ASSERT(results.size()==1);
    // result incomplete?
    TEST_ASSERT(results[0]->abstract_state()==Test_Run_Result::STATE_INCOMPLETE);

    // TYPE_TESTCASE_EXIT
    logger.log(CSTR_RECORD("ok",LEVEL_TEST,TYPE_TESTCASE_EXIT));

    // passive?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE);
    // one result?
    TEST_ASSERT(results.size()==1);
    // result complete?
    TEST_ASSERT(results[0]->abstract_state()==Test_Run_Result::STATE_COMPLETE);

    Test_Run_Result const result(*(results[0]));

    // TYPE_TESTCASE_ENTER
    logger.log(CSTR_RECORD("ok",LEVEL_TEST,TYPE_TESTCASE_ENTER));
    
    // active?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_ACTIVE);
    // one result?
    TEST_ASSERT(results.size()==2);
    // result incomplete?
    TEST_ASSERT(results[1]->abstract_state()==Test_Run_Result::STATE_INCOMPLETE);
    // first result unchanged?
    TEST_ASSERT(*(results[0])==result);
}

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief helper to compare record sequences fed into the logger and
 * the corresponding result.
 */
void cmp_records(Test_Run_Result::Records_t const & orig,
				 Test_Run_Result::Records_t const & copy,
				 Level_t const level) 
{
    typedef Test_Run_Result::Records_t::const_iterator Const_Iterator_t;
    
    Const_Iterator_t cur(orig.begin());
    Const_Iterator_t const end(orig.end());
    Const_Iterator_t cur2(copy.begin());
    Const_Iterator_t const end2(copy.end());
    for(;cur!=end;++cur) {
		if(cur->type()==TYPE_LOG_OPEN) continue;
		if(cur->type()==TYPE_LOG_CLOSE) continue;

		if(cur->type()==TYPE_FAILED_ASSERTION) {
			TEST_ASSERT(*cur==*cur2++);
		}
		else if(cur->type()==TYPE_UNEXPECTED_EXCEPTION) {
			TEST_ASSERT(*cur==*cur2++);
		}
		else if(cur->level()==LEVEL_TEST) {
			TEST_ASSERT(*cur==*cur2++);
		}
		else if(cur->type()==TYPE_FAILED_CHECK){
			if(cur->level()>level) {
				TEST_ASSERT(*cur==*cur2++);
			}
		}
    }
    TEST_ASSERT(cur2==end2);
}

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief helper to check the passive state of a Test_Logger
 *
 * @arg checks whether the logger is in STATE_PASSIVE  and the
 * results vector has size 1
 * @arg sends in all level/type combinations to be ignored in STATE_PASSIVE
 * @arg checks whether the logger is in STATE_PASSIVE
 */
void check_passive(Test_Logger & logger)
{
#define CHECK_PASSIVE_LOGGING_I(TYPE) do { \
	logger.log(CSTR_RECORD("ignored",LEVEL_PROD,TYPE)); \
        TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE); \
	logger.log(CSTR_RECORD("ignored",LEVEL_DEBUG,TYPE)); \
        TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE); \
	logger.log(CSTR_RECORD("ignored",LEVEL_AUDIT,TYPE)); \
        TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE); \
	logger.log(CSTR_RECORD("ignored",LEVEL_TEST,TYPE)); \
        TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE); } while(false)

#define CHECK_PASSIVE_LOGGING do {\
        CHECK_PASSIVE_LOGGING_I(TYPE_BLOCK_ENTER); \
        CHECK_PASSIVE_LOGGING_I(TYPE_BLOCK_EXIT); \
        CHECK_PASSIVE_LOGGING_I(TYPE_PROCEDURE_ENTER); \
        CHECK_PASSIVE_LOGGING_I(TYPE_PROCEDURE_EXIT); \
        CHECK_PASSIVE_LOGGING_I(TYPE_METHOD_ENTER); \
        CHECK_PASSIVE_LOGGING_I(TYPE_METHOD_EXIT); \
        CHECK_PASSIVE_LOGGING_I(TYPE_TRACE); \
        CHECK_PASSIVE_LOGGING_I(TYPE_TRACE_BINARY); \
        CHECK_PASSIVE_LOGGING_I(TYPE_FAILED_CHECK); \
        CHECK_PASSIVE_LOGGING_I(TYPE_UNEXPECTED_EXCEPTION); \
        CHECK_PASSIVE_LOGGING_I(TYPE_WRONG_EXCEPTION); \
        CHECK_PASSIVE_LOGGING_I(TYPE_MISSING_EXCEPTION); \
        CHECK_PASSIVE_LOGGING_I(TYPE_LOG_OPEN); \
	CHECK_PASSIVE_LOGGING_I(TYPE_LOG_CLOSE); } while(false)

    // passive?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE); 
    // check the passive behavior
    CHECK_PASSIVE_LOGGING;
    // still passive?
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE);
    
#undef CHECK_PASSIVE_LOGGING
#undef CHECK_PASSIVE_LOGGING_I
}


////////////////////////////////////////////////////////////////////////////////
/**
 * @brief [PRO] fct/normal: correct traces and their evaluation. Note
 * that this test is only checking with traces that contain a single
 * execution.
 *
 * The given Record_Vector class must produce a single Test_Case
 * execution, starting with an @ref TYPE_TESTCASE_ENTER and ending
 * with a @ref TYPE_TESTCASE_EXIT. The following steps are executed
 * for target_level in {LEVEL_PROD, LEVEL_DEBUG, LEVEL_AUDIT)
 *
 * @arg A constructs a logger with build_level==LEVEL_AUDIT
 * target_level and an empty results vector (check whether the vector
 * remains empty)
 * @arg B check_passive
 * @arg C logs the original records except the last one and checks after
 * each logged record whether the logger is in STATE_ACTIVE
 * @arg D logs the last record and check for STATE_PASSIVE and the results vector has size 1
 * @arg E checks the records in the result -- whether the right subset
 * has been logged or not.
 * @arg F checks the derived state of the @ref Test_Run_Result and for
 * current build-level in the Test_Run_Result
 * @arg B (since the execution is finished by copying the last record)
 * @arg E-F (since B-D could have changed the state)
 * @arg G check whether there is exactly one result
 * @arg H close logger
 * @arg G
 * @arg E-F
 */
class Correct_Trace_Processing
{
    Record_Vector * const m_record_vector;
public:
    typedef Test_Run_Result::Records_t Records_t;
    typedef Records_t::const_iterator Const_Iterator_t;

    Correct_Trace_Processing(Record_Vector * const r)
		: m_record_vector(r)
    {
    }

    Correct_Trace_Processing(Correct_Trace_Processing const & other)
		: m_record_vector(other.m_record_vector->clone())
    {
    }
    
    ~Correct_Trace_Processing()
    {
		delete m_record_vector;
    }

    void operator()(Test_Data & test_data) const;

private:
    void feed(Records_t const & orig,Test_Data & test_data,Level_t const target_level) const;
};

void Correct_Trace_Processing::operator()(Test_Data & test_data) const
{
    Records_t const records(m_record_vector->records());

    feed(records,test_data,LEVEL_PROD);
    feed(records,test_data,LEVEL_DEBUG);
    feed(records,test_data,LEVEL_AUDIT);
}


void Correct_Trace_Processing::feed(Records_t const & orig,Test_Data & test_data,Level_t const target_level) const
{
    Test_Logger::Test_Run_Results_t results;

    // A constructs a logger with LEVEL_AUDIT and an empty results vector
    // (check whether the vector remains empty)
    Test_Logger logger(LEVEL_AUDIT,target_level,results);
    TEST_ASSERT(results.size()==0);
    
    // B check_passive
    check_passive(logger);

    // C logs the original records except the last one and checks after
    // each logged record whether the logger is in STATE_ACTIVE and the
    // results vector has size 1
    Const_Iterator_t cur(orig.begin());
    Const_Iterator_t const end(orig.end()-1);
    for(;cur!=end;++cur) {
		logger.log(*cur);
		// check incomplete state
		TEST_ASSERT(results.size()==1);
		TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_ACTIVE); 
    }
    // D logs the last record and check for STATE_PASSIVE and the results vector has size 1
    logger.log(*cur);
    TEST_ASSERT(results.size()==1);
    TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_PASSIVE); 
    
    // E checks the records in the result -- whether the right subset
    // has been logged or not.
    cmp_records(orig,results[0]->records(),target_level);
    // F checks the derived state of the @ref Test_Run_Result and for
    // LEVEL_AUDIT in the Test_Run_Result
    m_record_vector->check(*(results[0]),true,target_level);
    TEST_ASSERT(results[0]->build_level()==LEVEL_AUDIT);
    
    // B
    check_passive(logger);
    
    // E
    cmp_records(orig,results[0]->records(),target_level);
    // F
    m_record_vector->check(*(results[0]),true,target_level); 
    TEST_ASSERT(results[0]->build_level()==LEVEL_AUDIT);
    
    // G check whether there is exactly one result
    TEST_ASSERT(results.size()==1);
    // H close logger 
    logger.close();
    // G
    TEST_ASSERT(results.size()==1);
    
    // E
    cmp_records(orig,results[0]->records(),target_level);
    // F
    m_record_vector->check(*(results[0]),true,target_level); 
    TEST_ASSERT(results[0]->build_level()==LEVEL_AUDIT);
}


////////////////////////////////////////////////////////////////////////////////

/**
 * @brief [PRO] fct/normal: multiple correct traces and their
 * evaluation. 8 traces are fed into a single logger and their
 * evaluation is checked.
 *
 * 
 * The following sequence is repeated for each build-level and for
 * each tested Record_Vector -- for each valid
 * build-level/target-level combination a new Test_Logger is used --
 * the same Test_Run_Result vector is used all the time.
 *
 * @arg A check whether logger is passive check_passive(logger);
 * @arg B logs the original records except the last one and checks after
 * each logged record whether the logger is in STATE_ACTIVE and the
 * results vector has original size+1
 * @arg C logs the last record and checks that the results vector has original size+1
 * @arg A check_passive
 * @arg D checks the records in the result -- whether the right subset
 * has been logged or not.
 * @arg E checks the derived state of the @ref Test_Run_Result and for
 * LEVEL_DEBUG in the Test_Run_Result
 *
 * Finally, the number of results is checked for.
 */
class Multiple_Traces_Processing
{
public:
    typedef Test_Logger::Test_Run_Results_t Test_Run_Results_t;
    typedef Test_Run_Result::Records_t Records_t;
    typedef Records_t::const_iterator Const_Iterator_t;
    
    Multiple_Traces_Processing()
    {
    }
    
    void operator()(Test_Data & test_data) const;
    
private:
    void feed(Test_Run_Results_t & results,
			  Test_Logger &logger,
			  Record_Vector & orig_rv,
			  Test_Data & test_data,
			  Level_t const build_level,
			  Level_t const target_level) const;
};

void Multiple_Traces_Processing::operator()(Test_Data & test_data) const
{
    Test_Logger::Test_Run_Results_t results;
    typedef ::std::vector<Record_Vector*> RV_t;
    typedef RV_t::const_iterator RV_Iter_t;
    RV_t rvs;
    rvs.push_back(new Empty_Trace_0sec_0usec());
    rvs.push_back(new Empty_Trace_1sec_1usec_No_Carry());
    rvs.push_back(new Empty_Trace_1sec_1usec_With_Carry());
    rvs.push_back(new Trace_With_One_Failure());
    rvs.push_back(new Trace_With_One_Invalidation());
    rvs.push_back(new Trace_With_One_Failed_Check());
    rvs.push_back(new Trace_With_All_Level_Type_Combinations());
    rvs.push_back(new Trace_With_All_Level_Type_Combinations_Times_2());
    RV_Iter_t cur(rvs.begin());
    RV_Iter_t const end(rvs.end());

    TEST_ASSERT(results.size()==0);
    Test_Logger logger(LEVEL_PROD,LEVEL_PROD,results);
    TEST_ASSERT(results.size()==0);
    for(;cur!=end;++cur) 
		feed(results,logger,**cur,test_data,LEVEL_PROD,LEVEL_PROD);

    TEST_ASSERT(results.size()==8);
    Test_Logger logger2(LEVEL_DEBUG,LEVEL_PROD,results);
    TEST_ASSERT(results.size()==8);
    for(cur=rvs.begin();cur!=end;++cur)
		feed(results,logger2,**cur,test_data,LEVEL_DEBUG,LEVEL_PROD);

    TEST_ASSERT(results.size()==16);
    Test_Logger logger3(LEVEL_DEBUG,LEVEL_DEBUG,results);
    TEST_ASSERT(results.size()==16);
    for(cur=rvs.begin();cur!=end;++cur)
		feed(results,logger3,**cur,test_data,LEVEL_DEBUG,LEVEL_DEBUG);


    TEST_ASSERT(results.size()==24);
    Test_Logger logger4(LEVEL_AUDIT,LEVEL_AUDIT,results);
    TEST_ASSERT(results.size()==24);
    for(cur=rvs.begin();cur!=end;++cur)
		feed(results,logger4,**cur,test_data,LEVEL_AUDIT,LEVEL_AUDIT);

    TEST_ASSERT(results.size()==32);
    Test_Logger logger5(LEVEL_AUDIT,LEVEL_DEBUG,results);
    TEST_ASSERT(results.size()==32);
    for(cur=rvs.begin();cur!=end;++cur)
		feed(results,logger5,**cur,test_data,LEVEL_AUDIT,LEVEL_DEBUG);

    TEST_ASSERT(results.size()==40);
    Test_Logger logger6(LEVEL_AUDIT,LEVEL_PROD,results);
    TEST_ASSERT(results.size()==40);
    for(cur=rvs.begin();cur!=end;++cur)
		feed(results,logger6,**cur,test_data,LEVEL_AUDIT,LEVEL_PROD);

    TEST_ASSERT(results.size()==48);
}


void Multiple_Traces_Processing::feed(Test_Run_Results_t & results,
									  Test_Logger &logger,
									  Record_Vector & orig_rv,
									  Test_Data & test_data,
									  Level_t const build_level,
									  Level_t const target_level) const
{
    Records_t orig(orig_rv.records());
    
    // store size
    int const size(results.size());
    
    // A check whether logger is passive
    check_passive(logger);
 
    // B logs the original records except the last one and checks after
    // each logged record whether the logger is in STATE_ACTIVE and the
    // results vector has original size+1
    Const_Iterator_t cur(orig.begin());
    Const_Iterator_t const end(orig.end()-1);
    for(;cur!=end;++cur) {
		logger.log(*cur);
		// check incomplete state
		TEST_ASSERT(static_cast<int>(results.size())==size+1);
		TEST_ASSERT(logger.abstract_state()==Test_Logger::STATE_ACTIVE); 
    }
    // C logs the last record and checks that the results vector has original size+1
    logger.log(*cur);
    TEST_ASSERT(static_cast<int>(results.size())==size+1);
    // A check_passive
    check_passive(logger);
    
    // D checks the records in the result -- whether the right subset
    // has been logged or not.
    cmp_records(orig,results[size]->records(),target_level);
    // E checks the derived state of the @ref Test_Run_Result and for
    // build_level in the Test_Run_Result
    orig_rv.check(*(results[size]),true,target_level);
    TEST_ASSERT(results[size]->build_level()==build_level);
}

/**
 * @brief [PRO] impl: The implementation for deciding whether to log
 * a TYPE_FAILED_CHECK breaks down if the following two assertions are
 * not met.
 */
void level_odering(Test_Data &)
{
    TEST_ASSERT(LEVEL_PROD<LEVEL_DEBUG);
    TEST_ASSERT(LEVEL_DEBUG<LEVEL_AUDIT);
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(Correct_Trace_Processing(new Empty_Trace_0sec_0usec()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Empty_Trace_1sec_1usec_No_Carry()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Empty_Trace_1sec_1usec_With_Carry()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Trace_With_One_Failure()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Trace_With_One_Invalidation()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Trace_With_One_Failed_Check()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Trace_With_All_Level_Type_Combinations()),LEVEL_PROD);
TEST_NORMAL_CASE(Correct_Trace_Processing(new Trace_With_All_Level_Type_Combinations_Times_2()),LEVEL_PROD);
TEST_NORMAL_CASE(Multiple_Traces_Processing(),LEVEL_PROD);
TEST_NORMAL_CASE(&record_sequence_enter_exit_enter,LEVEL_PROD);
TEST_NORMAL_CASE(&correct_build_levels,LEVEL_PROD);
TEST_ABNORMAL_CASE(&record_sequence_plain_exit,LEVEL_PROD);
TEST_ABNORMAL_CASE(&record_sequence_double_enter,LEVEL_PROD);
TEST_ABNORMAL_CASE(&wrong_build_level,LEVEL_PROD);
TEST_IMPL_CASE(&level_odering);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;
// vim:ts=4:sw=4
