/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "RawImageInfo.h"

#include <QFileInfo>
#include <QMutex>
#include <QPainter>
#include <QImage>

#include <KDebug>

#include <threadweaver/ThreadWeaver.h>

#include "JobThumbnail.h"
#include "PreviewInfo.h"
#include "ProcessingOptions.h"
#include "Icons.h"

struct RawImageInfo::Private {
  Private() : lock(QMutex::Recursive) {}
  QFileInfo fileInfo;
  QImage imagePreview;
  bool previewUptodate;
  QImage pixmapPreview;
  QImage icon;
  bool iconUptodate;
  QMutex lock;
  Status status;
  PreviewInfo* previewInfo;
  ProcessingOptions processingOptions;
  void updatePreview();
  void updateIcon();
};

void RawImageInfo::Private::updatePreview()
{
  if( previewUptodate ) return;
  pixmapPreview = imagePreview;
  previewUptodate = true;
}

void RawImageInfo::Private::updateIcon()
{
  if( iconUptodate ) return;
  icon = pixmapPreview.copy();
  if( status != IDLING )
  {
    QImage statusIconName;
    switch(status)
    {
      case IDLING:
        qFatal("hu?");
        break;
      case LOADING:
        statusIconName = Icons::documentOpen();
        break;
      case DECODING:
      case POSTPROCESSING:
      case PREVIEWING:
        statusIconName = Icons::systemRun();
        break;
      case SAVING:
        statusIconName = Icons::documentSave();
        break;
      case DONE:
        statusIconName = Icons::dialogOkApply();
        break;
      case ERROR:
        statusIconName = Icons::editDelete();
        break;
    }
    QPainter painter( &icon );
    painter.drawImage( 10, 10, statusIconName );
  }
  iconUptodate = true;
}

RawImageInfo::RawImageInfo(const QFileInfo& fileInfo ) : d(new Private)
{
  d->fileInfo = fileInfo;
  d->pixmapPreview = Icons::dcraw();
  d->previewUptodate = true;
  d->iconUptodate = false;
  d->previewInfo = 0;
  JobThumbnail::weaver()->enqueue( new JobThumbnail( RawImageInfoSP(this)) );
  setStatus( LOADING );
}

RawImageInfo::~RawImageInfo()
{
  delete d->previewInfo;
  delete d;
}

const QFileInfo& RawImageInfo::fileInfo() const
{
  return d->fileInfo;
}

const QImage& RawImageInfo::thumbnail() const
{
  QMutexLocker locker(&d->lock);
  d->updatePreview();
  return d->pixmapPreview;
}

const QImage& RawImageInfo::icon() const
{
  QMutexLocker locker(&d->lock);
  d->updatePreview();
  d->updateIcon();
  return d->icon;
}

void RawImageInfo::setThumbnail(const QImage& _image)
{
  QMutexLocker locker(&d->lock);
  d->imagePreview = _image;
  d->previewUptodate = false;
  d->iconUptodate = false;
  setStatus( IDLING );
  emit( previewChanged( ) );
}

void RawImageInfo::setStatus( Status _newStatus )
{
  QMutexLocker locker(&d->lock);
  d->status = _newStatus;
  d->iconUptodate = false;
  emit( iconChanged() );
}

void RawImageInfo::setPreviewInfo( const PreviewInfo* previewInfo)
{
  kDebug() << "============== setPreview info for " << this;
  d->previewInfo = const_cast<PreviewInfo*>(previewInfo);
  if( d->previewInfo )
  {
    d->previewInfo->appendToCache();
  }
}

const PreviewInfo* RawImageInfo::previewInfo() const
{
  return d->previewInfo;
}

const ProcessingOptions& RawImageInfo::processingOptions() const
{
  return d->processingOptions;
}

void RawImageInfo::setProcessingOptions( const ProcessingOptions& _processingOptions )
{
  d->processingOptions = _processingOptions;
}

#include "RawImageInfo.moc"
