// Copyright 2017-2018 - Universite de Strasbourg/CNRS
// The CDS HEALPix library is developped by the Centre de Donnees
// astronomiques de Strasbourgs (CDS) from the following external papers:
//  - [Gorsky2005]     - "HEALPix: A Framework for High-Resolution Discretization and
//                       Fast Analysis of Data Distributed on the Sphere"
//                       http://adsabs.harvard.edu/abs/2005ApJ...622..759G
//  - [Calabretta2004] - "Mapping on the HEALPix grid"
//                       http://adsabs.harvard.edu/abs/2004astro.ph.12607C
//  - [Calabretta2007] - "Mapping on the HEALPix grid"
//                       http://adsabs.harvard.edu/abs/2007MNRAS.381..865C
//  - [Reinecke2015]   - "Efficient data structures for masks on 2D grids"
//                       http://adsabs.harvard.edu/abs/2015A&A...580A.132R
// It is distributed under the terms of the BSD License 2.0
//
// This file is part of the CDS HEALPix library.
//

package cds.healpix;

import static org.junit.Assert.assertEquals;

import org.junit.Test;

import cds.healpix.Healpix;
import cds.healpix.HealpixNestedBMOC;
import cds.healpix.HealpixNestedFixedRadiusConeComputer;

public class NestedConeTest {

  public void coneTest(double coneCenterLonDeg, double coneCenterLatDeg, double radiusDeg, int depth,
      final long[] expectedRes) {
    System.out.println("Cone test. Center "
        + "lon: "  + coneCenterLonDeg + "deg, "
        + "lat: "  + coneCenterLatDeg + "deg, "
        + "radius: " + radiusDeg + " deg, "
        + "depth: " + depth
        );
    final double coneCenterLonRad = Math.toRadians(coneCenterLonDeg);
    final double coneCenterLatRad = Math.toRadians(coneCenterLatDeg);
    final double radiusRad = Math.toRadians(radiusDeg);
    final HealpixNestedFixedRadiusConeComputer cc = Healpix.getNested(depth).newConeComputer(radiusRad);
    System.out.println(cc.getClass().getName());
    
    final HealpixNestedBMOC moc = cc.overlappingCells(coneCenterLonRad, coneCenterLatRad);
    System.out.println("Moc size: " + moc.size());
    int i = 0;
    if (expectedRes != null) {
      for (final HealpixNestedBMOC.CurrentValueAccessor cell : moc) {
        // System.out.println(cell);
        assertEquals(expectedRes[i++], cell.getHash());
      }
      assertEquals(expectedRes.length, i);
    } else {
      /* Decomment to add test (after visual inspection)
      for (final HealpixNestedBMOC.CurrentValueAccessor cell : moc) {
        System.out.println(cell);
      }
      final StringBuilder strBuild = new StringBuilder("new long[]{");
      int n = 1;
      for (final HealpixNestedBMOC.CurrentValueAccessor cell : moc) {
        strBuild.append(cell.getHash()).append("L,");
        if (n++ % 8 == 0) {
          strBuild.append('\n');
        } else {
          strBuild.append(' ');
        }
      }
      System.out.println(strBuild.substring(0, strBuild.length() - 2) + "}");
      */
    }
  }
  
  public void coneCentersTest(double coneCenterLonDeg, double coneCenterLatDeg, double radiusDeg, int depth,
      final long[] expectedRes) {
    System.out.println("Cone test. Center "
        + "lon: "  + coneCenterLonDeg + "deg, "
        + "lat: "  + coneCenterLatDeg + "deg, "
        + "radius: " + radiusDeg + " deg, "
        + "depth: " + depth
        );
    final double coneCenterLonRad = Math.toRadians(coneCenterLonDeg);
    final double coneCenterLatRad = Math.toRadians(coneCenterLatDeg);
    final double radiusRad = Math.toRadians(radiusDeg);
    final HealpixNestedFixedRadiusConeComputer cc = Healpix.getNested(depth).newConeComputer(radiusRad);
    final HealpixNestedBMOC moc = cc.overlappingCenters(coneCenterLonRad, coneCenterLatRad);
    int i = 0;
    if (expectedRes != null) {
      for (final HealpixNestedBMOC.CurrentValueAccessor cell : moc) {
        // System.out.println(cell);
        assertEquals(expectedRes[i++], cell.getHash());
      }
      assertEquals(expectedRes.length, i);
    } else {
      /* Decomment to add test (after visual inspection)
      for (final HealpixNestedBMOC.CurrentValueAccessor cell : moc) {
        System.out.println(cell);
        // if (cell.getHash() == 373286564231L) System.out.println("COUCOU");
      }
      */
    }
  }

  double lonFromSexa(final String lon) {
    final String[] elems = lon.trim().split(" +");
    final double hour = Integer.parseInt(elems[0]);
    final double minh = Integer.parseInt(elems[1]) / 60.0;
    final double sech = Double.parseDouble(elems[2]) / 3600.0;
    return 15 * (hour + minh + sech);
  }

  double latFromSexa(final String lat) {
    final String[] elems = lat.trim().split(" +");
    final double deg = Integer.parseInt(elems[0]);
    final double min = Integer.parseInt(elems[1]) / 60.0;
    final double sec = Double.parseDouble(elems[2]) / 3600.0;
    return deg > 0 ? deg + min + sec : deg - min - sec; 
  }

  @Test
  public void tesConeNP() {
    // draw circle(61.0749875, +89.90546338888889, 5.2403223034772175 deg)
    double lon = lonFromSexa("04 04 17.99700");
    double lat = latFromSexa("+89 54 19.6682");
    double rad = 5.2403223034772175;
    int order = 3;
    coneTest(lon, lat, rad, order, new long[]{63, 127, 191, 255});
  }
  
  @Test
  public void tesConeNP2() {
    // draw circle(37.954541666666664, +89.2641111111111, 0.8410004718589024 deg)
    double lon = lonFromSexa("02 31 49.09000");
    double lat = latFromSexa("+89 15 50.8000");
    double rad = 0.8410004718589024;
    int order = 6;
    coneTest(lon, lat, rad, order, new long[]{4092, 4093, 4094, 4095, 8190, 8191, 12287, 16381, 16383});
  }
  
  @Test
  public void tesCone3() {
    // draw circle(37.954541666666664, +89.2641111111111, 0.8410004718589024 deg)
    double lon = lonFromSexa("06 30 12.40");
    double lat = latFromSexa("+04 50 26.7");
    double rad = 30.5 / 3600d;
    int order = 18;
    coneCentersTest(lon, lat, rad, order, null); //new long[]{4092, 4093, 4094, 4095, 8190, 8191, 12287, 16381, 16383});
    // 18/373286564231 dedans?
  }

  @Test
  public void testCone4() {
    // draw circle(0.0, +0.0, 37.0 deg)
    double lon = 0;
    double lat = 0;
    double rad = 37;
    int order = 3;
   
    coneTest(lon, lat, rad, order, null);
    
    //final HealpixNested hn = Healpix.getNested(order);
    //final HealpixNestedFixedRadiusConeComputer cp = hn.newConeComputer( Math.toRadians(radius) );
    //final HealpixNestedBMOC bmoc = cp.overlappingCells(Math.toRadians(ra), Math.toRadians(dec));
    // long [] out = toFlatArrayOfHash(bmoc);
   
    // System.out.print("overlappingCells ra="+ra+" dec="+dec+" radius="+radius+" order="+order+"\n=> ");
    // for( long a : out ) System.out.print(" "+a);
    // System.out.println();

  }
  
  
  @Test
  public void testCone5() {
    double lon = 012.46214169;
    double lat = -72.49285924;
    double rad = 2.6 * 1.42;
    int order = 4;
    // draw circle(12.46214169,+72.49285924,3.6919999999999997)
    // draw moc 4/2058 2059 2080 2081 2082 2083
    coneTest(lon, -lat, rad, order, new long[]{233, 235, 236, 237, 238, 239, 248, 250});
    // draw circle(12.46214169,-72.49285924,3.6919999999999997)
    // draw moc 4/2058 2059 2080 2081 2082 2083
    coneTest(lon, lat, rad, order, new long[]{2058, 2059, 2080, 2081, 2082, 2083, 2088, 2089});
   
  }

  @Test
  public void testCone6() {
    // double lon = lonFromSexa("00 55 58.86840");
    // double lat = latFromSexa("-73 22 13.6829");
    double lon = lonFromSexa("00 53 35.72");
    double lat = latFromSexa("-72 57 44.8");
    double rad = 4.6;
    int order = 4;
    coneTest(lon, lat, rad, order, new long[]{2058, 2059, 2080, 2081, 2082, 2083, 2088, 2089, 2832, 2833, 2836});
    // draw circle( 00:55:58.86840, -73:22:13.6829, 4.6)
    // draw moc 4/2058 2059 2080 2081 2082 2083 2088 2089 2832 2833 2836*/
   
  }

  
  @Test
  public void testCone7() {
    double lon = 13.158329;
    double lat = -72.80028;
    double rad = 5.64323;
    int order = 3;
    
    // draw circle(13.158329,+72.80028,5.64323)
    // draw circle(103.158329,+72.80028,5.64323)
    coneTest(lon, -lat, rad, order, new long[]{57, 58, 59, 60, 62, 245, 247});
    
    // draw circle(13.158329,-72.80028,5.64323)
    // draw moc 3/ 514 515 520 521 522
    // draw moc 3/ 514 515 520 521 522 708 709
    // draw moc 3/ 514 520 521 522 523 544
    coneTest(lon, lat, rad, order, new long[]{514, 515, 520, 521, 522, 708, 709});
    // 128
    // draw moc 2/ 129 130 131 136 137 176 177 180
  }
  
  @Test
  public void testCone8() {
    double lon = 97.91750000;
    double lat = 5.76952778;
    double rad = 4.45 / 60;
    int order = 9;
   
    // draw circle(97.91750000,+5.76952778, 0.07416666666666666666)
    coneTest(lon, lat, rad, order, new long[]{1424261, 1424262, 1424263, 1424269, 1424274, 1424280});
  }
  
  @Test
  public void testCone9() {
    double lon = 205.0;
    double lat = 30.5;
    double rad = 0.05;
    int order = 18;
    long[] bmocEntries = new long[]{173619134335L, 173619134421L, 173619134422L, 173619134423L, 173619134425L, 173619134427L, 173619134428L, 173619134429L,
        173619134430L, 173619134431L, 173619134445L, 173619134446L, 173619134447L, 173619134448L, 173619134449L, 173619134450L,
        173619134451L, 43404783613L, 173619134456L, 173619134457L, 173619134458L, 173619134459L, 43404783615L, 173619138046L,
        173619138047L, 173619138293L, 173619138294L, 173619138295L, 173619138297L, 173619138298L, 173619138299L, 173619138300L,
        173619138301L, 173619138302L, 173619138303L, 173619138359L, 173619138363L, 173619138364L, 173619138365L, 173619138366L,
        173619138367L, 173619138381L, 173619138382L, 173619138383L, 173619138385L, 173619138386L, 173619138387L, 173619138388L,
        173619138389L, 173619138390L, 173619138391L, 173619138392L, 173619138393L, 173619138394L, 173619138395L, 43404784599L,
        173619138400L, 173619138401L, 173619138402L, 173619138403L, 173619138404L, 173619138405L, 173619138406L, 173619138407L,
        43404784602L, 43404784603L, 10851196151L, 173619138439L, 173619138441L, 173619138442L, 173619138443L, 173619138444L,
        173619138445L, 173619138446L, 173619138447L, 173619138448L, 173619138449L, 173619138450L, 173619138451L, 43404784613L,
        43404784614L, 43404784615L, 173619138464L, 173619138465L, 173619138466L, 173619138467L, 43404784617L, 43404784618L,
        43404784619L, 10851196155L, 2712799039L, 173619139515L, 173619139516L, 173619139517L, 173619139518L, 173619139519L,
        173619139535L, 173619139543L, 173619139545L, 173619139546L, 173619139547L, 173619139548L, 173619139549L, 173619139550L,
        173619139551L, 173619139554L, 173619139555L, 173619139556L, 173619139557L, 173619139558L, 173619139559L, 173619139560L,
        173619139561L, 173619139562L, 173619139563L, 43404784891L, 10851196223L, 173619140011L, 173619140014L, 173619140015L,
        173619140022L, 173619140023L, 173619140024L, 173619140025L, 173619140026L, 173619140027L, 173619140028L, 173619140029L,
        173619140030L, 173619140031L, 173619140046L, 173619140047L, 173619140055L, 173619140057L, 173619140058L, 173619140059L,
        173619140060L, 173619140061L, 173619140062L, 173619140063L, 173619140064L, 173619140065L, 173619140066L, 173619140067L,
        173619140068L, 173619140069L, 173619140070L, 173619140071L, 43404785018L, 43404785019L, 10851196255L, 173619140142L,
        173619140143L, 173619140150L, 173619140151L, 173619140152L, 173619140153L, 173619140154L, 173619140155L, 173619140156L,
        173619140157L, 173619140158L, 173619140159L, 173619140171L, 173619140173L, 173619140174L, 173619140175L, 173619140179L,
        173619140181L, 173619140182L, 173619140183L, 173619140184L, 173619140185L, 173619140186L, 173619140187L, 43404785047L,
        173619140192L, 173619140193L, 173619140194L, 173619140195L, 43404785049L, 43404785050L, 43404785051L, 10851196263L,
        173619140224L, 173619140225L, 173619140226L, 173619140227L, 173619140228L, 173619140229L, 173619140230L, 173619140231L,
        43404785058L, 43404785059L, 10851196265L, 10851196266L, 10851196267L, 2712799067L, 173619140352L, 173619140353L,
        173619140354L, 173619140355L, 43404785089L, 43404785090L, 43404785091L, 10851196273L, 10851196274L, 10851196275L,
        2712799069L, 2712799070L, 2712799071L, 173619140703L, 173619140715L, 173619140716L, 173619140717L, 173619140718L,
        173619140719L, 173619140721L, 173619140722L, 173619140723L, 173619140724L, 173619140725L, 173619140726L, 173619140727L,
        173619140728L, 173619140729L, 173619140730L, 173619140731L, 43404785183L, 173619140758L, 173619140759L, 173619140761L,
        173619140762L, 173619140763L, 173619140764L, 173619140765L, 173619140766L, 173619140767L, 173619140771L, 173619140772L,
        173619140773L, 173619140774L, 173619140775L, 173619140776L, 173619140777L, 173619140778L, 173619140779L, 43404785195L,
        173619140784L, 173619140785L, 173619140786L, 173619140787L, 43404785197L, 43404785198L, 43404785199L, 173619140800L,
        173619140801L, 173619140802L, 173619140803L, 43404785201L, 43404785202L, 43404785203L, 10851196301L, 10851196302L,
        10851196303L, 173619140869L, 173619140870L, 173619140871L, 173619140872L, 173619140873L, 173619140874L, 173619140875L,
        173619140876L, 173619140877L, 173619140878L, 173619140879L, 173619140880L, 173619140881L, 173619140882L, 173619140883L,
        43404785221L, 43404785222L, 43404785223L, 10851196306L, 10851196307L, 2712799077L, 2712799078L, 2712799079L,
        678199770L, 678199771L, 169549943L, 173619143157L, 173619143158L, 173619143159L, 173619143163L, 173619143164L,
        173619143165L, 173619143166L, 173619143167L, 173619143423L, 173619143479L, 173619143485L, 173619143486L, 173619143487L,
        173619143495L, 173619143499L, 173619143500L, 173619143501L, 173619143502L, 173619143503L, 173619143504L, 173619143505L,
        173619143506L, 173619143507L, 43404785877L, 43404785878L, 43404785879L, 173619143520L, 173619143521L, 173619143522L,
        173619143523L, 43404785881L, 43404785882L, 43404785883L, 10851196471L, 173619143565L, 173619143567L, 173619143569L,
        173619143570L, 173619143571L, 173619143572L, 173619143573L, 173619143574L, 173619143575L, 173619143576L, 173619143577L,
        173619143578L, 173619143579L, 43404785895L, 173619143587L, 173619143588L, 173619143589L, 173619143590L, 173619143591L,
        173619143592L, 173619143593L, 173619143594L, 173619143595L, 43404785899L, 10851196475L, 2712799119L, 173619143733L,
        173619143734L, 173619143735L, 173619143737L, 173619143738L, 173619143739L, 173619143740L, 173619143741L, 173619143742L,
        173619143743L, 173619143749L, 173619143750L, 173619143751L, 173619143753L, 173619143754L, 173619143755L, 173619143756L,
        173619143757L, 173619143758L, 173619143759L, 173619143760L, 173619143761L, 173619143762L, 173619143763L, 43404785941L,
        43404785942L, 43404785943L, 173619143776L, 173619143777L, 173619143778L, 173619143779L, 43404785945L, 43404785946L,
        43404785947L, 10851196487L, 173619143813L, 173619143814L, 173619143815L, 173619143817L, 173619143818L, 173619143819L,
        173619143820L, 173619143821L, 173619143822L, 173619143823L, 173619143824L, 173619143825L, 173619143826L, 173619143827L,
        43404785957L, 43404785958L, 43404785959L, 173619143840L, 173619143841L, 173619143842L, 173619143843L, 43404785961L,
        43404785962L, 43404785963L, 10851196491L, 2712799123L, 678199781L, 678199782L, 678199783L, 173619144789L,
        173619144790L, 173619144791L, 173619144793L, 173619144794L, 173619144795L, 173619144796L, 173619144797L, 173619144798L,
        173619144799L, 173619144807L, 173619144812L, 173619144813L, 173619144814L, 173619144815L, 173619144816L, 173619144817L,
        173619144818L, 173619144819L, 43404786205L, 43404786206L, 43404786207L, 173619144861L, 173619144863L, 173619144883L,
        173619144884L, 173619144885L, 173619144886L, 173619144887L, 173619144889L, 173619144890L, 173619144891L, 43404786223L,
        173619144897L, 173619144898L, 173619144899L, 173619144900L, 173619144901L, 173619144902L, 173619144903L, 173619144904L,
        173619144905L, 173619144906L, 173619144907L, 43404786227L, 10851196557L, 10851196558L, 10851196559L, 678199785L,
        173619145221L, 173619145223L, 173619145228L, 173619145229L, 173619145230L, 173619145231L, 173619145232L, 173619145233L,
        173619145234L, 173619145235L, 43404786309L, 43404786310L, 43404786311L, 173619145249L, 173619145250L, 173619145251L,
        173619145252L, 173619145253L, 173619145254L, 173619145255L, 173619145256L, 173619145257L, 173619145258L, 173619145259L,
        43404786315L, 10851196579L, 2712799145L, 2712799146L, 2712799147L, 678199787L, 169549947L, 42387487L,
        173619221815L, 173619221821L, 173619221822L, 173619221823L, 173619221827L, 173619221828L, 173619221829L, 173619221830L,
        173619221831L, 173619221833L, 173619221834L, 173619221835L, 43404805459L, 10851201365L, 173619221856L, 173619221857L,
        173619221858L, 173619221859L, 43404805465L, 43404805466L, 43404805467L, 10851201367L, 173619221907L, 173619221908L,
        173619221909L, 173619221910L, 173619221911L, 173619221913L, 173619221914L, 173619221915L, 43404805479L, 173619221927L,
        173619221933L, 173619221934L, 173619221935L, 173619221936L, 173619221937L, 173619221938L, 173619221939L, 43404805485L,
        43404805486L, 43404805487L, 2712800343L, 173619222141L, 173619222143L, 173619222228L, 173619222229L, 173619222230L,
        173619222231L, 173619222233L, 173619222235L, 173619222236L, 173619222237L, 173619222238L, 173619222239L, 173619222255L,
        173619222257L, 173619222258L, 173619222259L, 43404805565L, 173619222264L, 173619222265L, 173619222266L, 173619222267L,
        43404805567L, 173619222275L, 173619222276L, 173619222277L, 173619222278L, 173619222279L, 173619222281L, 173619222282L,
        173619222283L, 43404805571L, 10851201393L, 173619222304L, 173619222305L, 173619222306L, 173619222307L, 43404805577L,
        173619222312L, 173619222313L, 173619222314L, 173619222315L, 43404805579L, 10851201395L, 2712800349L, 2712800350L,
        2712800351L, 173619223621L, 173619223623L, 173619223628L, 173619223629L, 173619223630L, 173619223631L, 10851201477L,
        173619223651L, 173619223652L, 173619223653L, 173619223654L, 173619223655L, 173619223657L, 173619223658L, 173619223659L,
        43404805915L, 10851201479L, 173619223709L, 173619223711L, 173619223733L, 173619223734L, 173619223735L, 173619223740L,
        173619223741L, 173619223742L, 173619223743L, 173619223744L, 173619223745L, 173619223746L, 173619223747L, 43404805937L,
        173619223752L, 173619223753L, 173619223754L, 173619223755L, 43404805939L, 10851201485L, 10851201486L, 10851201487L,
        678200093L, 173619224081L, 173619224083L, 173619224084L, 173619224085L, 173619224086L, 173619224087L, 173619224089L,
        173619224091L, 43404806023L, 173619224111L, 173619224112L, 173619224113L, 173619224114L, 173619224115L, 43404806029L,
        173619224120L, 173619224121L, 173619224122L, 173619224123L, 43404806031L, 2712800377L, 173619224197L, 173619224199L,
        173619224205L, 173619224206L, 173619224207L, 10851201513L, 173619224228L, 173619224229L, 173619224230L, 173619224231L,
        173619224235L, 173619224236L, 173619224237L, 173619224238L, 173619224239L, 10851201515L, 2712800379L, 678200095L,
        173619229141L, 173619229143L, 173619229149L, 173619229151L, 173619229173L, 173619229174L, 173619229175L, 173619229180L,
        173619229181L, 173619229182L, 173619229183L, 173619229524L, 173619229525L, 173619229526L, 173619229527L, 173619229531L,
        173619229532L, 173619229533L, 173619229534L, 173619229535L, 173619229553L, 173619229555L, 43404807389L, 173619229561L,
        173619229563L, 43404807391L, 173619229649L, 173619229651L, 43404807413L, 173619229656L, 173619229657L, 173619229658L,
        173619229659L, 43404807415L, 173619229680L, 173619229681L, 173619229682L, 173619229683L, 43404807421L, 173619229688L,
        173619229689L, 173619229690L, 173619229691L, 43404807423L, 173619229697L, 173619229699L, 43404807425L, 173619229705L,
        173619229706L, 173619229707L, 43404807427L, 10851201857L, 173619229728L, 173619229729L, 173619229730L, 173619229731L,
        43404807433L, 173619229736L, 173619229737L, 173619229738L, 173619229739L, 43404807435L, 10851201859L, 2712800465L,
        173619229824L, 173619229825L, 173619229826L, 173619229827L, 43404807457L, 43404807458L, 43404807459L, 10851201865L,
        10851201866L, 10851201867L, 2712800467L, 678200117L, 678200118L, 678200119L, 173619231056L, 173619231057L,
        173619231058L, 173619231059L, 43404807765L, 173619231064L, 173619231065L, 173619231066L, 173619231067L, 43404807767L,
        173619231077L, 173619231079L, 173619231085L, 173619231087L, 173619231088L, 173619231089L, 173619231090L, 173619231091L,
        43404807773L, 43404807774L, 43404807775L, 173619231173L, 173619231175L, 173619231181L, 173619231183L, 10851201949L,
        173619231205L, 173619231207L, 173619231213L, 173619231215L, 10851201951L, 173619231557L, 173619231559L, 173619231565L,
        173619231567L, 10851201973L, 173619231589L, 173619231591L, 173619231597L, 173619231599L, 10851201975L, 173619231685L,
        173619231687L, 173619231693L, 173619231695L, 10851201981L, 173619231717L, 173619231719L, 173619231725L, 173619231727L,
        10851201983L, 169550031L, 10596877L, 173619253573L, 173619253575L, 173619253581L, 173619253583L, 10851203349L,
        173619253605L, 173619253607L, 173619253613L, 173619253615L, 43404813404L, 43404813405L, 173619253624L, 173619253625L,
        173619253626L, 173619253627L, 43404813407L, 173619253712L, 173619253713L, 173619253714L, 173619253715L, 43404813429L,
        173619253720L, 173619253721L, 173619253722L, 173619253723L, 43404813431L, 173619253744L, 173619253745L, 173619253746L,
        173619253747L, 43404813437L, 173619253752L, 173619253753L, 173619253754L, 173619253755L, 43404813439L, 173619254096L,
        173619254097L, 173619254098L, 173619254099L, 43404813525L, 173619254105L, 173619254107L, 43404813527L, 173619254129L,
        173619254131L, 43404813533L, 173619254137L, 173619254139L, 43404813535L, 173619254225L, 173619254228L, 173619254229L,
        173619254230L, 173619254231L, 173619254236L, 173619254237L, 173619254238L, 173619254239L, 173619254260L, 173619254261L,
        173619254262L, 173619254263L, 173619254269L, 173619254271L, 169550053L, 173619255637L, 173619255639L, 173619255645L,
        43404814080L, 43404814081L, 173619256328L, 173619256329L, 173619256330L, 173619256331L, 43404814083L, 10851203521L,
        173619256352L, 173619256353L, 173619256354L, 173619256355L, 43404814089L, 173619256360L, 173619256361L, 173619256362L,
        173619256363L, 43404814091L, 10851203523L, 2712800881L, 173619256449L, 173619256451L, 43404814113L, 173619256457L,
        173619256459L, 173619256460L, 173619256461L, 173619256462L, 173619256463L, 10851203529L, 173619256484L, 173619256485L,
        173619256486L, 173619256487L, 173619256492L, 173619256493L, 173619256495L, 10851203531L, 2712800883L, 678200221L,
        173619256837L, 173619256839L, 173619256845L, 43404814212L, 43404814213L, 173619256856L, 173619256857L, 173619256858L,
        173619256859L, 43404814215L, 173619256880L, 173619256881L, 173619256882L, 173619256883L, 43404814221L, 173619256889L,
        173619256891L, 43404814223L, 2712800889L, 173619256977L, 173619256980L, 173619256981L, 173619256982L, 173619256983L,
        173619256988L, 173619256989L, 173619256990L, 173619256991L, 173619257013L, 173619257015L, 173619257021L, 10851203564L,
        10851203565L, 43404814264L, 43404814265L, 173619257064L, 173619257065L, 173619257066L, 173619257067L, 43404814267L,
        10851203567L, 678200223L, 173619262528L, 173619262529L, 173619262531L, 43404815633L, 173619262537L, 173619262539L,
        173619262540L, 173619262541L, 173619262542L, 173619262543L, 10851203909L, 173619262564L, 173619262565L, 173619262566L,
        173619262567L, 173619262573L, 173619262575L, 10851203911L, 173619262661L, 173619262672L, 173619262673L, 173619262674L,
        173619262675L, 43404815669L, 173619262680L, 173619262681L, 173619262683L, 43404815671L, 173619262705L, 173619262708L,
        173619262709L, 173619262710L, 173619262711L, 173619262716L, 173619262717L, 173619262719L, 678200245L, 173619263061L,
        173619263232L, 173619263233L, 173619263234L, 173619263235L, 43404815809L, 173619263240L, 173619263241L, 173619263243L,
        43404815811L, 10851203953L, 173619263265L, 173619263268L, 173619263269L, 173619263270L, 173619263271L, 173619263276L,
        173619263277L, 173619263279L, 10851203955L, 2712800989L, 173619263365L, 173619263376L, 173619263377L, 173619263378L,
        173619263379L, 43404815845L, 173619263384L, 173619263385L, 173619263387L, 173619263388L, 173619263389L, 173619263390L,
        173619263391L, 173619263412L, 173619263413L, 173619263414L, 173619263415L, 173619263421L, 173619263423L, 10851203964L,
        10851203965L, 43404815864L, 43404815865L, 173619263464L, 173619263465L, 173619263466L, 173619263467L, 43404815867L,
        10851203967L, 173619264832L, 173619264833L, 173619264835L, 43404816209L, 173619264841L, 173619264844L, 173619264845L,
        173619264846L, 173619264847L, 10851204053L, 173619264869L, 173619264871L, 173619264880L, 173619264881L, 173619264882L,
        173619264883L, 43404816221L, 173619264888L, 173619264889L, 173619264891L, 43404816223L, 173619264977L, 173619264980L,
        173619264981L, 173619264982L, 173619264983L, 173619264989L, 173619264991L, 42387516L, 42387517L, 169550072L,
        169550073L, 2712801184L, 2712801185L, 43404818976L, 43404818977L, 173619275912L, 173619275913L, 173619275914L,
        173619275915L, 43404818979L, 10851204745L, 173619275936L, 173619275937L, 173619275939L, 173619275940L, 173619275941L,
        173619275942L, 173619275943L, 173619275948L, 173619275949L, 173619275950L, 173619275951L, 10851204747L, 2712801187L,
        678200297L, 173619276293L, 173619276304L, 173619276305L, 173619276306L, 173619276307L, 43404819077L, 173619276313L,
        173619276315L, 173619276316L, 173619276317L, 173619276318L, 173619276319L, 173619276340L, 173619276341L, 173619276343L,
        10851204772L, 10851204773L, 173619276384L, 173619276385L, 173619276386L, 173619276387L, 43404819097L, 173619276392L,
        173619276393L, 173619276395L, 173619276396L, 173619276397L, 173619276398L, 173619276399L, 10851204775L, 173619276484L,
        173619276485L, 173619276487L, 173619276493L, 43404819124L, 43404819125L, 173619276504L, 173619276505L, 173619276506L,
        173619276507L, 43404819127L, 173619276529L, 173619276532L, 173619276533L, 173619276534L, 173619276535L, 173619276541L,
        2712801196L, 2712801197L, 10851204792L, 10851204793L, 43404819176L, 43404819177L, 173619276712L, 173619276713L,
        173619276714L, 173619276715L, 43404819179L, 10851204795L, 2712801199L, 169550075L, 42387519L, 173619293178L,
        173619293179L, 173619293182L, 173619293183L, 173619293864L, 173619293865L, 173619293866L, 173619293867L, 173619293868L,
        173619293869L, 173619293870L, 173619293871L, 173619293874L, 173619293875L, 173619293878L, 173619293879L, 173619293880L,
        173619293881L, 173619293882L, 173619293883L, 43404823471L, 173619293918L, 173619293919L, 173619293921L, 173619293922L,
        173619293923L, 173619293924L, 173619293925L, 173619293926L, 173619293927L, 43404823482L, 43404823483L, 173619293936L,
        173619293937L, 173619293938L, 173619293939L, 173619293940L, 173619293941L, 173619293942L, 173619293943L, 43404823486L,
        43404823487L, 173619294090L, 173619294091L, 173619294093L, 173619294094L, 173619294095L, 173619294104L, 173619294105L,
        173619294106L, 173619294107L, 173619294108L, 173619294109L, 173619294110L, 173619294111L, 10851205882L, 10851205883L,
        173619294150L, 173619294151L, 173619294152L, 173619294153L, 173619294154L, 173619294155L, 173619294156L, 173619294157L,
        173619294158L, 173619294159L, 173619294162L, 173619294163L, 173619294166L, 173619294167L, 43404823542L, 43404823543L,
        10851205886L, 10851205887L, 173619296895L, 173619296898L, 173619296899L, 173619296900L, 173619296901L, 173619296902L,
        173619296903L, 43404824226L, 43404824227L, 173619296912L, 173619296913L, 173619296914L, 173619296915L, 173619296916L,
        173619296917L, 173619296918L, 173619296919L, 43404824230L, 43404824231L, 10851206058L, 10851206059L, 173619296960L,
        173619296961L, 173619296962L, 173619296963L, 173619296964L, 173619296965L, 173619296966L, 173619296967L, 43404824242L,
        43404824243L, 173619296976L, 173619296977L, 173619296978L, 173619296979L, 173619296980L, 173619296981L, 173619296982L,
        173619296983L, 43404824246L, 43404824247L, 10851206062L, 10851206063L, 173619297066L, 173619297067L, 173619297070L,
        173619297071L, 173619297082L, 173619297083L, 173619297086L, 173619297087L, 173619297130L, 173619297131L, 173619297134L,
        173619297135L, 173619297146L, 173619297147L, 173619297150L, 173619297151L, 2712801518L, 2712801519L, 173619297834L,
        173619297835L, 173619297838L, 173619297839L, 43404824480L, 173619297924L, 173619297925L, 173619297926L, 173619297927L,
        43404824482L, 43404824483L, 173619297936L, 173619297937L, 173619297938L, 173619297939L, 173619297940L, 173619297941L,
        173619297942L, 173619297943L, 43404824486L, 43404824487L, 10851206122L, 10851206123L, 173619297984L, 173619297985L,
        173619297986L, 173619297987L, 173619297988L, 173619297989L, 173619297990L, 173619297991L, 43404824498L, 43404824499L,
        173619298000L, 173619298001L, 173619298002L, 173619298003L, 173619298004L, 173619298005L, 173619298006L, 173619298007L,
        43404824502L, 43404824503L, 10851206126L, 10851206127L, 173619298176L, 173619298178L, 173619298179L, 173619298182L,
        173619298183L, 43404824546L, 43404824547L, 173619298194L, 173619298195L, 173619298198L, 173619298199L, 43404824550L,
        43404824551L, 10851206138L, 10851206139L, 173619298242L, 173619298248L, 173619298249L, 173619298250L, 173619298251L,
        173619298252L, 173619298253L, 173619298254L, 173619298255L, 173619298264L, 173619298265L, 173619298266L, 173619298267L,
        173619298268L, 173619298270L, 173619298271L, 10851206142L, 10851206143L, 173619309194L, 173619309195L, 173619309198L,
        173619309199L, 173619309210L, 10851206826L, 173619309232L, 173619309233L, 173619309234L, 173619309235L, 173619309236L,
        173619309237L, 173619309238L, 173619309239L, 43404827310L, 43404827311L, 173619309280L, 173619309282L, 173619309283L,
        173619309286L, 173619309287L, 43404827322L, 43404827323L, 173619309298L, 173619309304L, 173619309305L, 173619309306L,
        173619309307L, 173619309308L, 173619309309L, 173619309310L, 173619309311L, 173619309480L, 173619309482L, 173619309483L,
        173619309486L, 173619309487L, 173619309498L, 173619314746L, 173619314747L, 173619314749L, 173619314750L, 173619314751L,
        173619314783L, 173619314787L, 173619314790L, 173619314791L, 173619314792L, 173619314793L, 173619314794L, 173619314795L,
        43404828699L, 173619314800L, 173619314801L, 173619314802L, 173619314803L, 173619314804L, 173619314805L, 173619314806L,
        173619314807L, 43404828702L, 43404828703L, 173619314817L, 173619314818L, 173619314819L, 173619314820L, 173619314821L,
        173619314822L, 173619314823L, 43404828706L, 43404828707L, 173619314832L, 173619314833L, 173619314834L, 173619314835L,
        43404828709L, 43404828710L, 43404828711L, 10851207178L, 10851207179L, 2712801795L, 173619314954L, 173619314955L,
        173619314956L, 173619314957L, 173619314958L, 173619314959L, 173619314963L, 173619314966L, 173619314967L, 173619314968L,
        173619314969L, 173619314970L, 173619314971L, 43404828743L, 10851207186L, 10851207187L, 173619315009L, 173619315010L,
        173619315011L, 173619315012L, 173619315013L, 173619315014L, 173619315015L, 43404828754L, 43404828755L, 173619315024L,
        173619315025L, 173619315026L, 173619315027L, 43404828757L, 43404828758L, 43404828759L, 10851207190L, 10851207191L,
        2712801798L, 2712801799L, 678200450L, 678200451L, 169550113L, 169550114L, 169550115L, 42387529L,
        42387530L, 42387531L, 678200512L, 10851208208L, 173619331344L, 173619331345L, 173619331346L, 173619331347L,
        173619331348L, 173619331349L, 173619331350L, 173619331351L, 43404832838L, 43404832839L, 10851208210L, 10851208211L,
        173619331394L, 173619331395L, 173619331398L, 43404832850L, 173619331404L, 173619331405L, 173619331406L, 173619331407L,
        173619331416L, 173619331417L, 173619331418L, 173619331419L, 173619331422L, 173619331423L, 10851208214L, 10851208215L,
        2712802054L, 2712802055L, 678200514L, 678200515L, 173619332106L, 173619332128L, 173619332129L, 173619332130L,
        173619332131L, 173619332132L, 173619332133L, 173619332134L, 173619332135L, 43404833034L, 43404833035L, 173619332146L,
        173619332147L, 173619332152L, 173619332153L, 173619332154L, 173619332155L, 173619332156L, 173619332157L, 173619332158L,
        173619332159L, 173619332200L, 173619332202L, 173619332203L, 173619332206L, 2712802066L, 43404833072L, 173619332292L,
        173619332293L, 173619332294L, 173619332295L, 43404833074L, 43404833075L, 173619332304L, 173619332306L, 173619332307L,
        173619332310L, 173619332311L, 43404833078L, 173619332316L, 173619332317L, 173619332318L, 173619332319L, 10851208270L,
        10851208271L, 173619332488L, 173619332489L, 173619332490L, 173619332491L, 173619332494L, 173619332495L, 43404833128L,
        173619332516L, 173619332517L, 173619332518L, 173619332519L, 43404833130L, 43404833131L, 173619332528L, 173619332529L,
        173619332530L, 173619332531L, 173619332534L, 173619332535L, 43404833134L, 173619332540L, 173619332541L, 173619332542L,
        173619332543L, 173619332584L, 173619332585L, 173619332586L, 173619332587L, 173619332590L, 173619332591L, 678200518L,
        2712802076L, 43404833232L, 173619332932L, 173619332933L, 173619332934L, 173619332935L, 43404833234L, 43404833235L,
        173619332944L, 173619332946L, 173619332947L, 173619332950L, 43404833238L, 173619332956L, 173619332957L, 173619332958L,
        173619332959L, 10851208310L, 10851208311L, 2712802078L, 2712802079L, 169550130L, 169550131L, 173619335688L,
        173619335690L, 173619335691L, 173619335694L, 43404833928L, 173619335716L, 173619335717L, 173619335718L, 173619335719L,
        43404833930L, 43404833931L, 173619335730L, 173619335731L, 173619335736L, 173619335737L, 173619335738L, 173619335739L,
        173619335740L, 173619335742L, 173619335743L, 173619335786L, 2712802122L, 173619335872L, 173619335873L, 173619335874L,
        173619335875L, 173619335878L, 173619335879L, 43404833970L, 173619335884L, 173619335885L, 173619335886L, 173619335887L,
        173619335896L, 173619335898L, 173619335899L, 173619335902L, 10851208494L, 43404833980L, 173619335924L, 173619335925L,
        173619335926L, 173619335927L, 43404833982L, 43404833983L, 173619336098L, 173619336104L, 173619336105L, 173619336106L,
        173619336107L, 173619336108L, 173619336110L, 173619336111L, 678200536L, 43404834368L, 173619337476L, 173619337477L,
        173619337478L, 173619337479L, 43404834370L, 43404834371L, 173619337488L, 173619337490L, 173619337491L, 173619337494L,
        43404834374L, 173619337500L, 173619337501L, 173619337502L, 173619337503L, 10851208594L, 10851208595L, 173619337546L,
        173619337568L, 173619337569L, 173619337570L, 173619337571L, 173619337574L, 43404834394L, 173619337580L, 173619337581L,
        173619337582L, 173619337583L, 173619337592L, 173619337594L, 173619337595L, 2712802150L, 10851208604L, 173619337680L,
        173619337681L, 173619337682L, 173619337683L, 173619337684L, 173619337686L, 173619337687L, 43404834422L, 173619337692L,
        173619337693L, 173619337694L, 173619337695L, 10851208606L, 10851208607L, 678200538L, 678200539L, 173619338376L,
        173619338378L, 173619338379L, 173619338400L, 173619338401L, 173619338402L, 173619338403L, 173619338404L, 173619338406L,
        173619338407L, 43404834602L, 173619338412L, 173619338413L, 173619338414L, 173619338415L, 173619338424L, 173619338426L,
        173619338427L, 10851208672L, 173619338768L, 173619338769L, 173619338770L, 173619338771L, 173619338772L, 173619338774L,
        173619338775L, 43404834694L, 173619338780L, 173619338781L, 173619338782L, 173619338783L, 10851208674L, 10851208675L,
        173619338824L, 173619338826L, 173619338827L, 173619338848L, 173619338849L, 173619338850L, 173619338851L, 173619338852L,
        173619338854L, 173619338855L, 43404834714L, 173619338860L, 173619338861L, 173619338862L, 173619338863L, 173619338872L,
        173619338874L, 173619338875L, 2712802170L, 10851208684L, 173619338960L, 173619338961L, 173619338962L, 173619338963L,
        173619338966L, 43404834742L, 173619338972L, 173619338973L, 173619338974L, 173619338975L, 10851208686L, 10851208687L,
        173619339146L, 173619339168L, 173619339169L, 173619339170L, 173619339171L, 173619339174L, 43404834794L, 173619339180L,
        173619339182L, 173619339183L, 42387534L, 169550140L, 678200564L, 43404836160L, 173619344644L, 173619344645L,
        173619344646L, 173619344647L, 43404836162L, 43404836163L, 173619344656L, 173619344658L, 173619344659L, 173619344664L,
        173619344665L, 173619344666L, 173619344667L, 173619344670L, 10851209042L, 43404836172L, 173619344692L, 173619344693L,
        173619344694L, 173619344695L, 43404836174L, 43404836175L, 173619344738L, 173619344744L, 173619344746L, 173619344747L,
        2712802262L, 173619344832L, 173619344833L, 173619344834L, 173619344835L, 173619344836L, 173619344838L, 43404836210L,
        173619344844L, 173619344845L, 173619344846L, 173619344847L, 173619344858L, 10851209054L, 173619344880L, 173619344882L,
        173619344883L, 173619344888L, 173619344889L, 173619344890L, 173619344891L, 173619344892L, 173619344894L, 678200566L,
        2712802268L, 10851209076L, 43404836308L, 173619345236L, 173619345237L, 173619345238L, 173619345239L, 43404836310L,
        43404836311L, 10851209078L, 10851209079L, 2712802270L, 2712802271L, 169550142L, 169550143L, 173619388930L,
        173619388936L, 173619388938L, 173619388939L, 173619388960L, 173619388961L, 173619388962L, 173619388963L, 173619388966L,
        43404847242L, 173619388972L, 173619388973L, 173619388974L, 173619388975L, 43404847264L, 173619389060L, 173619389061L,
        173619389062L, 173619389063L, 43404847266L, 43404847267L, 173619389072L, 173619389074L, 173619389080L, 173619389081L,
        173619389082L, 173619389083L, 10851211818L, 173619389104L, 173619389105L, 173619389106L, 173619389107L, 173619389108L,
        173619389110L, 43404847278L, 173619389116L, 173619389117L, 173619389118L, 173619389119L, 10851211904L, 43404847620L,
        173619390484L, 173619390485L, 173619390486L, 173619390487L, 43404847622L, 43404847623L, 10851211906L, 10851211907L,
        173619390528L, 173619390530L, 173619390536L, 173619390537L, 173619390538L, 173619390539L, 173619390560L, 173619390561L,
        173619390562L, 173619390563L, 173619390564L, 173619390566L, 43404847642L, 173619390572L, 173619390573L, 173619390574L,
        173619390575L, 2712802978L, 43404847664L, 173619390660L, 173619390661L, 173619390662L, 173619390663L, 43404847666L,
        43404847667L, 173619390674L, 173619390680L, 173619390682L, 173619390683L, 10851211918L, 173619390704L, 173619390705L,
        173619390706L, 173619390707L, 173619390710L, 43404847678L, 173619390716L, 173619390718L, 2712802984L, 10851211940L,
        43404847764L, 173619391060L, 173619391061L, 173619391062L, 173619391063L, 43404847766L, 173619391068L, 173619391069L,
        173619391070L, 173619391071L, 10851211942L, 10851211943L, 2712802986L, 2712802987L, 173619391242L, 173619391264L,
        173619391266L, 173619391267L, 173619391272L, 173619391273L, 173619391274L, 173619391275L, 173619391360L, 173619391361L,
        173619391362L, 173619391363L, 173619391364L, 173619391366L, 43404847842L, 173619391372L, 173619391374L, 173619391375L,
        43404847848L, 173619391396L, 173619391397L, 173619391398L, 173619391399L, 43404847850L, 173619391404L, 173619391405L,
        173619391406L, 173619391407L, 173619391418L, 10596888L, 10596889L, 10596890L, 42387564L, 169550260L,
        169550261L, 169550262L, 678201052L, 2712804212L, 2712804213L, 2712804214L, 10851216860L, 43404867444L,
        43404867445L, 43404867446L, 173619469788L, 173619469789L, 173619469790L, 173619469791L, 10851216862L, 43404867452L,
        173619469812L, 173619469813L, 173619469814L, 173619469816L, 173619469817L, 173619469818L, 173619469819L, 173619469820L,
        678201054L, 2712804220L, 43404867536L, 173619470148L, 173619470149L, 173619470150L, 173619470151L, 43404867538L,
        173619470156L, 173619470157L, 173619470158L, 173619470159L, 173619470160L, 173619470161L, 173619470162L, 173619470176L,
        173619470177L, 173619470178L, 173619470179L, 173619470180L, 173619470182L, 173619470184L, 173619470185L, 173619470186L,
        173619470187L, 10851216888L, 43404867556L, 173619470228L, 173619470229L, 173619470230L, 173619470231L, 173619470232L,
        173619470233L, 173619470234L, 173619470235L, 173619470236L, 173619470237L, 173619470238L, 43404867560L, 43404867561L,
        43404867562L, 173619470252L, 173619470253L, 173619470254L, 173619470255L, 173619470256L, 173619470257L, 173619470258L,
        173619470259L, 173619470264L, 173619470266L, 173619470272L, 169550264L, 169550265L, 169550266L, 678201068L,
        678201069L, 678201070L, 2712804284L, 2712804285L, 2712804286L, 10851217148L, 43404868596L, 43404868597L,
        43404868598L, 173619474396L, 173619474397L, 173619474398L, 173619474399L, 43404868600L, 173619474404L, 173619474405L,
        173619474406L, 173619474407L, 173619474408L, 173619474409L, 173619474410L, 173619474411L, 173619474412L, 173619474413L,
        173619474414L, 173619474416L, 173619474417L, 173619474418L, 173619474419L, 173619474420L, 678201072L, 678201073L,
        678201074L, 2712804300L, 2712804301L, 2712804302L, 10851217212L, 43404868852L, 43404868853L, 43404868854L,
        173619475420L, 173619475421L, 173619475422L, 173619475423L, 43404868856L, 43404868857L, 43404868858L, 173619475436L,
        173619475437L, 173619475438L, 173619475439L, 173619475440L, 173619475441L, 173619475442L, 173619475443L, 173619475444L,
        173619475448L, 2712804304L, 10851217220L, 10851217221L, 10851217222L, 43404868892L, 173619475572L, 173619475573L,
        173619475574L, 173619475575L, 43404868894L, 173619475580L, 173619475581L, 173619475582L, 2712804306L, 43404868912L,
        43404868913L, 43404868914L, 173619475660L, 173619475661L, 173619475662L, 173619475663L, 173619475664L, 173619475665L,
        173619475666L, 173619475667L, 173619475668L, 173619475672L, 173619475680L, 173619475681L, 173619475682L, 173619475683L,
        173619475684L, 173619475685L, 173619475686L, 173619475688L, 173619475689L, 173619475690L, 43404868928L, 173619475716L,
        173619475717L, 173619475718L, 173619475720L, 173619475721L, 173619475722L, 173619475723L, 173619475724L, 173619475744L,
        173619475746L, 10851217248L, 43404868996L, 173619475988L, 173619475989L, 173619475990L, 173619475991L, 173619475992L,
        173619475993L, 173619475994L, 173619475995L, 173619475996L, 43404869000L, 173619476004L, 173619476005L, 173619476006L,
        173619476007L, 173619476008L, 173619476009L, 173619476010L, 173619476011L, 173619476012L, 173619476013L, 173619476014L,
        173619476016L, 173619476017L, 173619476018L, 173619476032L, 173619476096L, 173619476097L, 173619476098L, 173619476099L,
        173619476104L, 2712804320L, 2712804321L, 2712804322L, 10851217292L, 43404869172L, 43404869173L, 43404869174L,
        173619476700L, 173619476701L, 173619476702L, 173619476703L, 43404869176L, 43404869177L, 43404869178L, 173619476716L,
        173619476717L, 173619476718L, 173619476719L, 173619476720L, 173619476721L, 173619476722L, 173619476723L, 173619476724L,
        173619476725L, 173619476726L, 173619476728L, 173619476729L, 173619476730L, 10851217296L, 43404869188L, 43404869189L,
        43404869190L, 173619476764L, 173619476765L, 173619476766L, 173619476767L, 43404869192L, 43404869193L, 43404869194L,
        173619476780L, 173619476781L, 173619476782L, 173619476783L, 173619476784L, 173619476785L, 173619476786L, 173619476787L,
        173619476788L, 173619476789L, 173619476790L, 173619476792L, 173619476793L, 173619476794L, 173619476800L, 173619476801L,
        173619476802L, 173619476803L, 173619476804L, 173619476805L, 173619476806L, 173619476808L, 173619476809L, 173619476810L,
        173619476864L, 173619476865L, 173619476866L, 173619476867L, 173619476868L, 173619476869L, 173619476870L, 173619476872L,
        173619476873L, 173619476874L, 10851217312L, 43404869252L, 43404869253L, 43404869254L, 173619477020L, 173619477021L,
        173619477022L, 173619477023L, 43404869256L, 43404869257L, 43404869258L, 173619477036L, 173619477037L, 173619477038L,
        173619477039L, 173619477040L, 173619477041L, 173619477042L, 173619477043L, 173619477044L, 173619477045L, 173619477046L,
        173619477048L, 173619477049L, 173619477050L, 173619477056L, 173619477057L, 173619477058L, 173619477059L, 173619477060L,
        173619477061L, 173619477062L, 173619477064L, 173619477065L, 173619477066L, 173619477120L, 173619477121L, 173619477122L,
        173619477123L, 173619477124L, 173619477125L, 173619477128L, 678201088L, 10851217424L, 173619478800L, 173619478802L,
        173619478808L, 173619478809L, 173619478810L, 173619478811L, 10851217426L, 173619478832L, 173619478833L, 173619478834L,
        173619478835L, 173619478840L, 173619478841L, 173619478842L, 173619478843L, 173619478844L, 173619478846L, 10851217432L,
        43404869732L, 173619478932L, 173619478934L, 43404869734L, 173619478940L, 173619478941L, 173619478942L, 173619478943L,
        10851217434L, 43404869740L, 173619478964L, 173619478965L, 173619478966L, 173619478967L, 43404869742L, 173619478972L,
        173619478973L, 173619478974L, 173619478975L, 173619479016L, 173619479018L, 678201090L, 2712804364L, 173619479360L,
        173619479362L, 173619479368L, 173619479370L, 173619479392L, 173619479393L, 173619479394L, 173619479395L, 173619479400L,
        173619479401L, 173619479402L, 173619479403L, 2712804366L, 173619479488L, 173619479489L, 173619479490L, 173619479491L,
        173619479494L, 43404869874L, 173619479500L, 173619479502L, 43404869880L, 173619479524L, 173619479526L, 43404869882L,
        173619479532L, 173619479534L, 678201096L, 2712804388L, 43404870224L, 173619480900L, 173619480902L, 173619480903L,
        43404870226L, 173619480908L, 173619480909L, 173619480910L, 173619480911L, 43404870232L, 173619480932L, 173619480933L,
        173619480934L, 173619480935L, 43404870234L, 173619480940L, 173619480941L, 173619480942L, 173619480943L, 2712804390L,
        43404870256L, 173619481028L, 173619481029L, 173619481030L, 173619481031L, 43404870258L, 173619481036L, 173619481037L,
        173619481038L, 173619481039L, 43404870264L, 173619481060L, 173619481061L, 173619481062L, 173619481063L, 43404870266L,
        43404870267L, 173619481072L, 173619481074L, 173619481080L, 173619481082L, 678201098L, 2712804396L, 10851217588L,
        173619481424L, 173619481426L, 173619481432L, 173619481434L, 10851217590L, 173619481456L, 173619481458L, 173619481464L,
        173619481466L, 2712804398L, 10851217596L, 173619481552L, 173619481554L, 173619481560L, 173619481562L, 10851217598L,
        173619481584L, 173619481586L, 173619481592L, 173619481594L, 678201120L, 2712804484L, 10851217940L, 173619487056L,
        173619487058L, 173619487064L, 173619487066L, 10851217942L, 173619487088L, 173619487090L, 173619487096L, 173619487098L,
        2712804486L, 43404871792L, 173619487172L, 173619487173L, 173619487174L, 173619487175L, 43404871794L, 173619487180L,
        173619487181L, 173619487182L, 173619487183L, 173619487184L, 43404871800L, 173619487204L, 173619487205L, 173619487206L,
        173619487207L, 43404871802L, 173619487212L, 173619487213L, 173619487214L, 173619487215L, 678201122L, 2712804492L,
        43404871888L, 173619487556L, 173619487557L, 173619487558L, 173619487559L, 43404871890L, 173619487564L, 173619487565L,
        173619487566L, 173619487567L, 43404871896L, 173619487588L, 173619487590L, 43404871898L, 173619487596L, 173619487598L,
        2712804494L, 43404871920L, 173619487684L, 173619487686L, 173619487688L, 173619487689L, 173619487690L, 173619487691L,
        173619487692L, 173619487694L, 173619487712L, 173619487713L, 173619487714L, 173619487715L, 173619487720L, 173619487721L,
        173619487722L, 173619487723L, 678201128L, 10851218064L, 10851218065L, 10851218066L, 43404872268L, 43404872269L,
        43404872270L, 173619489084L, 173619489085L, 173619489086L, 173619489087L, 173619489088L, 173619489089L, 173619489090L,
        173619489091L, 173619489096L, 173619489098L, 173619489120L, 173619489122L, 173619489128L, 10851218072L, 43404872292L,
        173619489172L, 173619489173L, 173619489174L, 173619489175L, 43404872294L, 173619489180L, 173619489181L, 173619489182L,
        173619489183L, 10851218074L, 43404872300L, 173619489204L, 173619489206L, 43404872302L, 173619489212L, 173619489214L,
        678201130L, 10851218096L, 173619489552L, 173619489553L, 173619489554L, 173619489555L, 173619489556L, 173619489560L,
        173619489561L, 173619489562L, 173619489563L, 43404872392L, 43404872393L, 43404872394L, 173619489580L, 173619489581L,
        173619489582L, 173619489583L, 173619489584L, 173619489585L, 173619489586L, 173619489592L, 173619489594L, 43404872416L,
        173619489668L, 173619489669L, 173619489670L, 173619489671L, 43404872418L, 173619489676L, 173619489677L, 173619489678L,
        173619489679L, 43404872424L, 173619489700L, 173619489702L, 173619489704L, 173619489705L, 173619489706L, 173619489707L,
        173619489708L, 2712804864L, 10851219460L, 10851219461L, 10851219462L, 43404877852L, 173619511412L, 173619511413L,
        173619511414L, 173619511415L, 43404877854L, 173619511420L, 173619511421L, 173619511422L, 173619511423L, 2712804866L,
        10851219468L, 43404877876L, 173619511508L, 173619511510L, 173619511512L, 173619511513L, 173619511514L, 173619511515L,
        173619511516L, 10851219470L, 173619511536L, 173619511537L, 173619511538L, 173619511539L, 173619511544L, 173619511546L,
        173619511552L, 173619511553L, 173619511554L, 173619511555L, 173619511560L, 173619511562L, 173619511584L, 173619511586L,
        2712804872L, 43404877968L, 173619511876L, 173619511877L, 173619511878L, 173619511879L, 43404877970L, 173619511884L,
        173619511885L, 173619511886L, 173619511888L, 173619511904L, 173619511905L, 173619511906L, 173619511907L, 173619511908L,
        173619511910L, 173619511912L, 173619511913L, 173619511914L, 173619511915L, 10851219496L, 43404877988L, 173619511956L,
        173619511957L, 173619511958L, 173619511959L, 43404877990L, 173619511964L, 173619511965L, 173619511966L, 173619511967L,
        10851219498L, 43404877996L, 173619511988L, 173619511989L, 173619511990L, 173619511992L, 173619511993L, 173619511994L,
        173619511995L, 173619511996L, 173619512000L, 173619512002L, 43404878336L, 43404878337L, 43404878338L, 173619513356L,
        173619513357L, 173619513358L, 173619513359L, 173619513360L, 173619513361L, 173619513362L, 173619513368L, 43404878344L,
        173619513380L, 173619513381L, 173619513382L, 173619513384L, 173619513385L, 173619513386L, 173619513387L, 173619513388L,
        173619513472L, 173619513473L, 173619513474L, 173619513480L, 173619626241L, 173619626244L, 173619626245L, 173619626246L,
        173619626247L, 173619626253L, 43404906564L, 43404906565L, 173619626264L, 173619626265L, 173619626266L, 173619626267L,
        43404906567L, 173619626289L, 173619626292L, 173619626293L, 173619626294L, 173619626295L, 173619626301L, 10851226644L,
        10851226645L, 43404906584L, 43404906585L, 173619626344L, 173619626345L, 173619626346L, 173619626347L, 43404906587L,
        10851226647L, 173619626433L, 173619626436L, 173619626437L, 173619626438L, 173619626439L, 173619626448L, 173619626449L,
        173619626450L, 173619626451L, 43404906613L, 173619626456L, 173619626457L, 173619626459L, 173619626460L, 173619626461L,
        173619626462L, 173619626463L, 173619626484L, 173619626485L, 173619626487L, 2712806672L, 2712806673L, 10851226696L,
        10851226697L, 173619627168L, 173619627169L, 173619627170L, 173619627171L, 43404906793L, 173619627176L, 173619627177L,
        173619627180L, 173619627181L, 173619627182L, 173619627183L, 10851226699L, 2712806675L, 678201669L, 173619627525L,
        173619627536L, 173619627537L, 173619627538L, 173619627539L, 43404906885L, 173619627545L, 173619627548L, 173619627549L,
        173619627551L, 43404906896L, 43404906897L, 173619627592L, 173619627593L, 173619627594L, 173619627595L, 43404906899L,
        10851226725L, 173619627616L, 173619627617L, 173619627620L, 173619627621L, 173619627622L, 173619627623L, 173619627629L,
        43404906908L, 43404906909L, 173619627640L, 173619627641L, 173619627642L, 173619627643L, 173619627644L, 173619627645L,
        173619627646L, 173619627647L, 173619627732L, 173619627733L, 173619627735L, 2712806684L, 2712806685L, 173619627904L,
        173619627905L, 173619627906L, 173619627907L, 43404906977L, 173619627913L, 173619627916L, 173619627917L, 173619627919L,
        43404906980L, 43404906981L, 173619627928L, 173619627929L, 173619627930L, 173619627931L, 43404906983L, 173619627952L,
        173619627953L, 173619627956L, 173619627957L, 173619627958L, 173619627959L, 173619627965L, 10851226748L, 10851226749L,
        43404907000L, 43404907001L, 173619628008L, 173619628009L, 173619628011L, 173619628012L, 173619628013L, 173619628014L,
        173619628015L, 10851226751L, 173619629381L, 173619629392L, 173619629393L, 173619629395L, 173619629396L, 173619629397L,
        173619629398L, 173619629399L, 173619629405L, 169550420L, 169550421L, 43404908032L, 43404908033L, 173619632136L,
        173619632137L, 173619632139L, 173619632140L, 173619632141L, 173619632142L, 173619632143L, 10851227009L, 173619632165L,
        173619632176L, 173619632177L, 173619632179L, 173619632180L, 173619632181L, 173619632182L, 173619632183L, 173619632189L,
        10851227012L, 10851227013L, 43404908056L, 43404908057L, 173619632232L, 173619632233L, 173619632235L, 173619632236L,
        173619632237L, 173619632238L, 173619632239L, 10851227015L, 173619632325L, 173619632336L, 173619632337L, 173619632339L,
        173619632340L, 173619632341L, 173619632342L, 173619632343L, 2712806756L, 2712806757L, 173619632512L, 173619632513L,
        173619632514L, 173619632515L, 43404908129L, 173619632520L, 173619632521L, 173619632524L, 173619632525L, 173619632526L,
        173619632527L, 43404908132L, 43404908133L, 173619632536L, 173619632537L, 173619632538L, 173619632539L, 43404908135L,
        173619632561L, 173619632564L, 173619632565L, 10851227036L, 10851227037L, 173619632608L, 173619632609L, 173619632610L,
        173619632611L, 173619632612L, 173619632613L, 173619632614L, 173619632615L, 173619632620L, 173619632621L, 43404908156L,
        43404908157L, 173619632632L, 173619632633L, 173619632635L, 173619632636L, 173619632637L, 173619632638L, 173619632639L,
        2712806768L, 2712806769L, 10851227080L, 10851227081L, 43404908328L, 43404908329L, 173619633320L, 173619633321L,
        173619633322L, 173619633323L, 43404908331L, 10851227083L, 2712806771L, 678201693L, 173619633665L, 173619633668L,
        173619633669L, 173619633680L, 173619633681L, 173619633682L, 173619633683L, 173619633684L, 173619633685L, 173619633686L,
        173619633687L, 173619633728L, 173619633729L, 173619633730L, 173619633731L, 43404908433L, 173619633736L, 173619633737L,
        173619633740L, 173619633741L, 43404908436L, 43404908437L, 173619633752L, 173619633753L, 173619633754L, 173619633755L,
        173619633756L, 173619633757L, 173619633758L, 173619633759L, 43404908480L, 43404908481L, 173619633928L, 173619633929L,
        173619633930L, 173619633931L, 43404908483L, 10851227121L, 173619633952L, 173619633953L, 173619633956L, 173619633957L,
        173619633968L, 173619633969L, 173619633971L, 173619633972L, 173619633973L, 173619633974L, 173619633975L, 10851227124L,
        10851227125L, 173619634016L, 173619634017L, 173619634018L, 173619634019L, 173619634020L, 173619634021L, 173619634022L,
        173619634023L, 173619634028L, 173619634029L, 43404908508L, 43404908509L, 173619634040L, 173619634041L, 173619634044L,
        173619634045L, 169550592L, 169550593L, 678202376L, 678202377L, 10851238048L, 10851238049L, 43404952200L,
        43404952201L, 173619808808L, 173619808809L, 173619808811L, 173619808812L, 173619808813L, 173619808814L, 173619808815L,
        43404952204L, 43404952205L, 173619808824L, 173619808825L, 173619808826L, 173619808827L, 173619808828L, 173619808829L,
        173619808830L, 173619808831L, 10851238052L, 10851238053L, 43404952216L, 43404952217L, 173619808872L, 173619808873L,
        173619808874L, 173619808875L, 173619808876L, 173619808877L, 173619808878L, 173619808879L, 10851238055L, 173619808964L,
        173619808965L, 173619808976L, 173619808977L, 173619808980L, 173619808981L, 2712809516L, 2712809517L, 173619809152L,
        173619809153L, 173619809156L, 173619809157L, 173619809168L, 173619809169L, 173619809172L, 173619809173L, 173619809216L,
        173619809217L, 173619809220L, 173619809221L, 173619809232L, 173619809233L, 173619809236L, 173619809237L, 678202380L,
        678202381L, 2712809528L, 2712809529L, 173619809920L, 173619809921L, 173619809924L, 173619809925L, 173619809936L,
        173619809937L, 173619809940L, 173619809941L, 173619809984L, 173619809985L, 173619809988L, 173619809989L, 173619810000L,
        173619810001L, 173619810004L, 173619810005L, 10851238128L, 10851238129L, 43404952520L, 43404952521L, 173619810088L,
        173619810089L, 173619810090L, 173619810091L, 173619810092L, 173619810093L, 173619810094L, 173619810095L, 43404952524L,
        43404952525L, 173619810104L, 173619810105L, 173619810106L, 173619810107L, 173619810108L, 173619810109L, 173619810110L,
        173619810111L, 10851238132L, 10851238133L, 43404952536L, 43404952537L, 173619810152L, 173619810153L, 173619810154L,
        173619810155L, 173619810156L, 173619810157L, 173619810158L, 43404952540L, 43404952541L, 173619810168L, 173619810169L,
        173619810172L, 173619810173L, 173619810176L, 173619810177L, 169550596L, 169550597L, 678202392L, 2712809572L,
        2712809573L, 2712809574L, 10851238300L, 10851238301L, 10851238302L, 43404953212L, 43404953213L, 173619812856L,
        173619812857L, 173619812858L, 173619812859L, 173619812860L, 173619812861L, 173619812862L, 173619812863L, 10851238304L,
        10851238305L, 173619812896L, 173619812897L, 173619812898L, 173619812899L, 173619812900L, 173619812901L, 173619812902L,
        173619812903L, 173619812904L, 173619812905L, 173619812912L, 173619812913L, 173619812914L, 173619812915L, 173619812916L,
        173619812917L, 173619812918L, 173619812919L, 10851238308L, 43404953236L, 43404953237L, 173619812952L, 173619812953L,
        173619812954L, 173619812955L, 173619812956L, 173619812957L, 173619812958L, 173619812959L, 173619812960L, 173619812961L,
        173619812964L, 173619812965L, 173619812976L, 43404953280L, 43404953281L, 173619813128L, 173619813129L, 173619813130L,
        173619813131L, 173619813132L, 173619813133L, 173619813136L, 173619813137L, 173619813138L, 173619813139L, 173619813140L,
        173619813141L, 173619813142L, 173619813143L, 173619813144L, 173619813145L, 173619813184L, 173619813185L, 173619813186L,
        173619813187L, 173619813188L, 173619813189L, 173619813200L, 173619813201L, 2712809584L, 2712809585L, 10851238344L,
        10851238345L, 43404953384L, 173619813540L, 173619813541L, 173619813542L, 173619813543L, 173619813544L, 173619813545L,
        173619813546L, 173619813548L, 173619813549L, 173619813552L, 173619813553L, 173619813554L, 173619813555L, 173619813556L,
        173619813557L, 173619813558L, 43404953392L, 43404953393L, 173619813576L, 173619813577L, 173619813578L, 173619813579L,
        173619813580L, 173619813581L, 173619813582L, 173619813583L, 43404953396L, 173619813588L, 173619813589L, 173619813590L,
        173619813591L, 173619813592L, 173619813593L, 173619813594L, 173619813596L, 173619813597L, 173619813600L, 173619813601L,
        10851238352L, 10851238353L, 10851238354L, 43404953420L, 43404953421L, 173619813688L, 173619813689L, 173619813690L,
        173619813691L, 173619813692L, 173619813693L, 173619813694L, 173619813695L, 10851238356L, 43404953428L, 43404953429L,
        43404953430L, 173619813724L, 173619813725L, 173619813726L, 173619813727L, 173619813728L, 173619813729L, 173619813730L,
        173619813731L, 173619813732L, 173619813733L, 173619813734L, 173619813735L, 173619813736L, 173619813737L, 173619813744L,
        173619813745L, 173619813746L, 173619813748L, 173619813760L, 173619813761L, 173619813762L, 173619813763L, 173619813764L,
        173619813765L, 173619813766L, 173619813776L, 678202432L, 678202433L, 2712809736L, 2712809737L, 10851238952L,
        10851238953L, 10851238954L, 43404955820L, 43404955821L, 43404955822L, 173619823292L, 173619823293L, 173619823294L,
        173619823295L, 10851238956L, 43404955828L, 43404955829L, 43404955830L, 173619823324L, 173619823325L, 173619823326L,
        173619823327L, 43404955832L, 173619823332L, 173619823333L, 173619823334L, 173619823335L, 173619823336L, 173619823337L,
        173619823338L, 173619823339L, 173619823340L, 173619823341L, 173619823344L, 173619823345L, 173619823346L, 173619823347L,
        173619823348L, 10851238960L, 10851238961L, 10851238962L, 43404955852L, 43404955853L, 43404955854L, 173619823420L,
        173619823421L, 173619823422L, 173619823423L, 10851238964L, 43404955860L, 43404955861L, 173619823448L, 173619823449L,
        173619823450L, 173619823451L, 173619823452L, 173619823453L, 173619823454L, 173619823456L, 173619823457L, 173619823458L,
        173619823459L, 173619823460L, 173619823461L, 173619823462L, 173619823463L, 173619823464L, 173619823465L, 173619823472L,
        43404955872L, 173619823492L, 173619823493L, 173619823494L, 173619823495L, 173619823496L, 173619823497L, 173619823498L,
        173619823500L, 173619823504L, 173619823505L, 173619823506L, 173619823508L, 2712809744L, 10851238980L, 10851238981L,
        10851238982L, 43404955932L, 43404955933L, 43404955934L, 173619823740L, 173619823741L, 173619823742L, 173619823743L,
        10851238984L, 10851238985L, 43404955944L, 43404955945L, 43404955946L, 173619823788L, 173619823789L, 173619823790L,
        173619823791L, 43404955948L, 173619823796L, 173619823797L, 173619823798L, 173619823799L, 173619823800L, 173619823801L,
        173619823802L, 173619823804L, 43404955952L, 43404955953L, 173619823816L, 173619823817L, 173619823818L, 173619823819L,
        173619823820L, 173619823821L, 173619823822L, 173619823824L, 173619823825L, 173619823826L, 173619823827L, 173619823828L,
        173619823829L, 173619823832L, 173619823840L, 173619823841L, 10851238992L, 43404955972L, 173619823892L, 173619823893L,
        173619823894L, 173619823895L, 173619823896L, 173619823897L, 173619823898L, 173619823899L, 173619823900L, 173619823904L,
        173619823905L, 173619823906L, 173619823907L, 173619823908L, 173619823909L, 173619823910L, 173619823911L, 173619823912L,
        173619823913L, 173619823914L, 173619823920L, 173619823936L, 173619823937L, 173619823938L, 173619824128L, 173619824129L,
        173619824130L, 173619824131L, 173619824132L, 173619824133L, 173619824136L, 43404956160L, 173619824644L, 173619824645L,
        173619824646L, 173619824647L, 173619824648L, 173619824649L, 173619824650L, 173619824652L, 173619824656L, 173619824657L,
        173619824658L, 173619824660L, 173619824661L};
    coneTest(lon, lat, rad, order, bmocEntries);
    System.out.println("MOC size: " + bmocEntries.length);
    lat = 30.6;
    
    bmocEntries = new long[]{173619474423L, 173619474427L, 173619474428L, 173619474429L, 173619474430L, 173619474431L, 173619475446L, 173619475447L,
        173619475449L, 173619475450L, 173619475451L, 173619475452L, 173619475453L, 173619475454L, 173619475455L, 173619475831L,
        173619475833L, 173619475834L, 173619475835L, 173619475836L, 173619475837L, 173619475838L, 173619475839L, 173619475869L,
        173619475870L, 173619475871L, 173619475879L, 173619475882L, 173619475883L, 173619475884L, 173619475885L, 173619475886L,
        173619475887L, 173619475888L, 173619475889L, 173619475890L, 173619475891L, 173619475892L, 173619475893L, 173619475894L,
        173619475895L, 43404868974L, 43404868975L, 173619475907L, 173619475908L, 173619475909L, 173619475910L, 173619475911L,
        173619475912L, 173619475913L, 173619475914L, 173619475915L, 43404868979L, 173619475920L, 173619475921L, 173619475922L,
        173619475923L, 43404868981L, 43404868982L, 43404868983L, 10851217246L, 10851217247L, 173619476029L, 173619476030L,
        173619476031L, 173619476047L, 173619476051L, 173619476053L, 173619476054L, 173619476055L, 173619476056L, 173619476057L,
        173619476058L, 173619476059L, 43404869015L, 173619476065L, 173619476066L, 173619476067L, 173619476068L, 173619476069L,
        173619476070L, 173619476071L, 173619476072L, 173619476073L, 173619476074L, 173619476075L, 43404869019L, 10851217255L,
        173619476103L, 173619476107L, 173619476108L, 173619476109L, 173619476110L, 173619476111L, 173619476112L, 173619476113L,
        173619476114L, 173619476115L, 173619476116L, 173619476117L, 173619476118L, 173619476119L, 43404869030L, 43404869031L,
        173619476128L, 173619476129L, 173619476130L, 173619476131L, 43404869033L, 43404869034L, 43404869035L, 10851217259L,
        2712804315L, 173619476224L, 173619476225L, 173619476226L, 173619476227L, 43404869057L, 43404869058L, 43404869059L,
        10851217265L, 10851217266L, 10851217267L, 2712804317L, 2712804318L, 2712804319L, 173619476703L, 173619476719L,
        173619476723L, 173619476724L, 173619476725L, 173619476726L, 173619476727L, 173619476728L, 173619476729L, 173619476730L,
        173619476731L, 43404869183L, 173619476783L, 173619476787L, 173619476789L, 173619476790L, 173619476791L, 173619476792L,
        173619476793L, 173619476794L, 173619476795L, 43404869199L, 173619476805L, 173619476806L, 173619476807L, 173619476809L,
        173619476810L, 173619476811L, 173619476812L, 173619476813L, 173619476814L, 173619476815L, 173619476816L, 173619476817L,
        173619476818L, 173619476819L, 43404869205L, 43404869206L, 43404869207L, 173619476832L, 173619476833L, 173619476834L,
        173619476835L, 43404869209L, 43404869210L, 43404869211L, 10851217303L, 173619476867L, 173619476868L, 173619476869L,
        173619476870L, 173619476871L, 173619476872L, 173619476873L, 173619476874L, 173619476875L, 43404869219L, 10851217305L,
        10851217306L, 10851217307L, 2712804327L, 173619477045L, 173619477046L, 173619477047L, 173619477049L, 173619477050L,
        173619477051L, 173619477052L, 173619477053L, 173619477054L, 173619477055L, 173619477059L, 173619477060L, 173619477061L,
        173619477062L, 173619477063L, 173619477065L, 173619477066L, 173619477067L, 43404869267L, 10851217317L, 173619477088L,
        173619477089L, 173619477090L, 173619477091L, 43404869273L, 43404869274L, 43404869275L, 10851217319L, 173619477125L,
        173619477126L, 173619477127L, 173619477129L, 173619477131L, 173619477132L, 173619477133L, 173619477134L, 173619477135L,
        173619477136L, 173619477137L, 173619477138L, 173619477139L, 43404869285L, 43404869286L, 43404869287L, 173619477152L,
        173619477153L, 173619477154L, 173619477155L, 43404869289L, 43404869290L, 43404869291L, 10851217323L, 2712804331L,
        678201083L, 169550271L, 173619514091L, 173619514093L, 173619514094L, 173619514095L, 173619514099L, 173619514100L,
        173619514101L, 173619514102L, 173619514103L, 173619514104L, 173619514105L, 173619514106L, 173619514107L, 43404878527L,
        173619514218L, 173619514219L, 173619514220L, 173619514221L, 173619514222L, 173619514223L, 173619514227L, 173619514229L,
        173619514230L, 173619514231L, 173619514232L, 173619514233L, 173619514234L, 173619514235L, 43404878559L, 173619514250L,
        173619514251L, 173619514252L, 173619514253L, 173619514254L, 173619514255L, 173619514258L, 173619514259L, 173619514260L,
        173619514261L, 173619514262L, 173619514263L, 173619514264L, 173619514265L, 173619514266L, 173619514267L, 43404878567L,
        173619514272L, 173619514273L, 173619514274L, 173619514275L, 43404878569L, 43404878570L, 43404878571L, 10851219643L,
        173619514304L, 173619514305L, 173619514306L, 173619514307L, 43404878577L, 43404878578L, 43404878579L, 10851219645L,
        10851219646L, 10851219647L, 173619514607L, 173619514617L, 173619514618L, 173619514619L, 173619514620L, 173619514621L,
        173619514622L, 173619514623L, 173619514782L, 173619514783L, 173619514786L, 173619514787L, 173619514789L, 173619514790L,
        173619514791L, 173619514792L, 173619514793L, 173619514794L, 173619514795L, 43404878699L, 173619514800L, 173619514801L,
        173619514802L, 173619514803L, 173619514804L, 173619514805L, 173619514806L, 173619514807L, 43404878702L, 43404878703L,
        173619514825L, 173619514826L, 173619514827L, 173619514828L, 173619514829L, 173619514830L, 173619514831L, 173619514834L,
        173619514835L, 173619514837L, 173619514838L, 173619514839L, 173619514840L, 173619514841L, 173619514842L, 173619514843L,
        43404878711L, 10851219678L, 10851219679L, 173619514891L, 173619514893L, 173619514894L, 173619514895L, 173619514902L,
        173619514903L, 173619514904L, 173619514905L, 173619514906L, 173619514907L, 173619514908L, 173619514909L, 173619514910L,
        173619514911L, 173619514912L, 173619514913L, 173619514914L, 173619514915L, 43404878729L, 43404878730L, 43404878731L,
        10851219683L, 173619514944L, 173619514945L, 173619514946L, 173619514947L, 173619514948L, 173619514949L, 173619514950L,
        173619514951L, 43404878738L, 43404878739L, 10851219685L, 10851219686L, 10851219687L, 2712804922L, 2712804923L,
        678201231L, 173619517183L, 173619517354L, 173619517355L, 173619517358L, 173619517359L, 173619517370L, 173619517371L,
        173619517374L, 173619517375L, 173619517418L, 173619517419L, 173619517422L, 173619517423L, 173619517434L, 173619517435L,
        173619517438L, 173619517439L, 173619517487L, 173619517498L, 173619517499L, 173619517500L, 173619517501L, 173619517502L,
        173619517503L, 173619517542L, 173619517543L, 173619517544L, 173619517545L, 173619517546L, 173619517547L, 173619517548L,
        173619517549L, 173619517550L, 173619517551L, 173619517554L, 173619517555L, 173619517556L, 173619517557L, 173619517558L,
        173619517559L, 43404879390L, 43404879391L, 173619517568L, 173619517569L, 173619517570L, 173619517571L, 173619517572L,
        173619517573L, 173619517574L, 173619517575L, 43404879394L, 43404879395L, 10851219849L, 10851219850L, 10851219851L,
        2712804963L, 173619517711L, 173619517722L, 173619517723L, 173619517726L, 173619517727L, 173619517728L, 173619517729L,
        173619517730L, 173619517731L, 173619517732L, 173619517733L, 173619517734L, 173619517735L, 43404879434L, 43404879435L,
        10851219859L, 173619517768L, 173619517769L, 173619517770L, 173619517771L, 173619517772L, 173619517773L, 173619517774L,
        173619517775L, 173619517782L, 173619517783L, 173619517784L, 173619517785L, 173619517786L, 173619517787L, 173619517788L,
        173619517789L, 173619517790L, 173619517791L, 10851219862L, 10851219863L, 2712804966L, 2712804967L, 678201242L,
        678201243L, 173619518466L, 173619518467L, 173619518470L, 173619518471L, 43404879618L, 43404879619L, 173619518481L,
        173619518482L, 173619518483L, 173619518484L, 173619518485L, 173619518486L, 173619518487L, 43404879622L, 43404879623L,
        10851219906L, 10851219907L, 173619518528L, 173619518529L, 173619518530L, 173619518531L, 173619518532L, 173619518533L,
        173619518534L, 173619518535L, 43404879634L, 43404879635L, 173619518544L, 173619518545L, 173619518546L, 173619518547L,
        173619518548L, 173619518549L, 173619518550L, 173619518551L, 43404879638L, 43404879639L, 10851219910L, 10851219911L,
        2712804978L, 2712804979L, 678201245L, 678201246L, 678201247L, 173619519502L, 173619519503L, 173619519509L,
        173619519510L, 173619519511L, 173619519512L, 173619519513L, 173619519514L, 173619519515L, 173619519516L, 173619519517L,
        173619519518L, 173619519519L, 173619519521L, 173619519522L, 173619519523L, 173619519524L, 173619519525L, 173619519526L,
        173619519527L, 43404879882L, 43404879883L, 10851219971L, 173619519552L, 173619519553L, 173619519554L, 173619519555L,
        43404879889L, 43404879890L, 43404879891L, 10851219973L, 10851219974L, 10851219975L, 2712804994L, 2712804995L,
        678201249L, 678201250L, 678201251L, 169550313L, 169550314L, 169550315L, 42387579L, 173619528362L,
        173619528363L, 173619528366L, 173619528367L, 173619528378L, 173619528379L, 173619528382L, 173619528383L, 173619528426L,
        173619528427L, 173619528430L, 173619528431L, 173619528442L, 173619528443L, 173619528446L, 173619528447L, 173619528618L,
        173619528619L, 173619528622L, 173619528623L, 678201288L, 43404882496L, 173619529988L, 173619529989L, 173619529990L,
        173619529991L, 43404882498L, 43404882499L, 173619530000L, 173619530001L, 173619530002L, 173619530003L, 173619530004L,
        173619530005L, 173619530006L, 173619530007L, 43404882502L, 43404882503L, 10851220626L, 10851220627L, 173619530048L,
        173619530049L, 173619530050L, 173619530051L, 173619530052L, 173619530053L, 173619530054L, 173619530055L, 43404882514L,
        43404882515L, 173619530064L, 173619530065L, 173619530066L, 173619530067L, 173619530070L, 173619530071L, 43404882518L,
        43404882519L, 10851220630L, 10851220631L, 2712805158L, 2712805159L, 678201290L, 678201291L, 173619530754L,
        173619530755L, 173619530758L, 173619530759L, 43404882690L, 173619530764L, 173619530765L, 173619530766L, 173619530767L,
        173619530776L, 173619530777L, 173619530778L, 173619530779L, 173619530780L, 173619530781L, 173619530782L, 173619530783L,
        10851220674L, 10851220675L, 173619530824L, 173619530825L, 173619530826L, 173619530827L, 173619530830L, 173619530831L,
        173619530842L, 173619530843L, 173619530846L, 10851220678L, 43404882716L, 173619530868L, 173619530869L, 173619530870L,
        173619530871L, 43404882718L, 43404882719L, 2712805170L, 2712805171L, 173619531040L, 173619531041L, 173619531042L,
        173619531043L, 173619531044L, 173619531046L, 173619531047L, 43404882762L, 43404882763L, 173619531058L, 173619531059L,
        173619531062L, 43404882766L, 173619531068L, 173619531069L, 173619531070L, 173619531071L, 173619531112L, 173619531113L,
        173619531114L, 173619531115L, 173619531116L, 173619531118L, 173619531119L, 173619531130L, 173619531131L, 173619531134L,
        2712805174L, 10851220700L, 43404882804L, 173619531220L, 173619531221L, 173619531222L, 173619531223L, 43404882806L,
        43404882807L, 10851220702L, 10851220703L, 678201294L, 678201295L, 173619533952L, 173619533953L, 173619533954L,
        173619533955L, 173619533958L, 173619533959L, 43404883490L, 43404883491L, 173619533970L, 173619533976L, 173619533977L,
        173619533978L, 173619533979L, 173619533980L, 173619533981L, 173619533982L, 173619533983L, 10851220874L, 10851220875L,
        173619534026L, 173619534027L, 173619534048L, 173619534049L, 173619534050L, 173619534051L, 173619534052L, 173619534053L,
        173619534054L, 173619534055L, 43404883514L, 43404883515L, 173619534064L, 173619534066L, 173619534067L, 173619534070L,
        173619534071L, 43404883518L, 173619534076L, 173619534077L, 173619534078L, 173619534079L, 173619534248L, 173619534249L,
        173619534250L, 173619534251L, 173619534254L, 173619534255L, 678201306L, 43404883648L, 173619534596L, 173619534597L,
        173619534598L, 173619534599L, 43404883650L, 43404883651L, 173619534608L, 173619534609L, 173619534610L, 173619534611L,
        173619534612L, 173619534614L, 173619534615L, 43404883654L, 43404883655L, 10851220914L, 10851220915L, 173619534658L,
        173619534664L, 173619534665L, 173619534666L, 173619534667L, 173619534668L, 173619534670L, 173619534671L, 173619534682L,
        10851220918L, 173619534704L, 173619534705L, 173619534706L, 173619534707L, 173619534708L, 173619534710L, 173619534711L,
        43404883678L, 43404883679L, 2712805230L, 2712805231L, 173619535394L, 173619535400L, 173619535401L, 173619535402L,
        173619535403L, 173619535404L, 173619535406L, 173619535407L, 43404883872L, 173619535492L, 173619535493L, 173619535494L,
        173619535495L, 43404883874L, 43404883875L, 173619535504L, 173619535505L, 173619535506L, 173619535507L, 173619535510L,
        173619535511L, 43404883878L, 173619535516L, 173619535517L, 173619535518L, 173619535519L, 10851220970L, 10851220971L,
        173619535560L, 173619535561L, 173619535562L, 173619535563L, 173619535566L, 43404883896L, 173619535588L, 173619535589L,
        173619535590L, 173619535591L, 43404883898L, 43404883899L, 173619535600L, 173619535602L, 173619535603L, 173619535608L,
        173619535609L, 173619535610L, 173619535611L, 173619535612L, 173619535613L, 173619535614L, 173619535615L, 173619535786L,
        173619535787L, 42387582L, 169550332L, 678201332L, 173619541248L, 173619541249L, 173619541250L, 173619541251L,
        173619541252L, 173619541254L, 173619541255L, 43404885314L, 173619541260L, 173619541261L, 173619541262L, 173619541263L,
        173619541272L, 173619541273L, 173619541274L, 173619541275L, 173619541278L, 10851221330L, 43404885324L, 173619541300L,
        173619541301L, 173619541302L, 173619541303L, 43404885326L, 43404885327L, 173619541344L, 173619541346L, 173619541347L,
        173619541352L, 173619541353L, 173619541354L, 173619541355L, 173619541356L, 173619541358L, 173619541359L, 173619541370L,
        2712805334L, 10851221340L, 173619541456L, 173619541457L, 173619541458L, 173619541459L, 173619541462L, 43404885366L,
        173619541468L, 173619541469L, 173619541470L, 173619541471L, 10851221342L, 10851221343L, 678201334L, 678201335L,
        169550334L, 169550335L, 173619823935L, 173619823949L, 173619823950L, 173619823951L, 173619823952L, 173619823953L,
        173619823954L, 173619823955L, 43404955989L, 173619823960L, 173619823961L, 173619823962L, 173619823963L, 43404955991L,
        173619823969L, 173619823970L, 173619823971L, 173619823972L, 173619823973L, 173619823974L, 173619823975L, 173619823976L,
        173619823977L, 173619823978L, 173619823979L, 43404955995L, 10851238999L, 173619824019L, 173619824020L, 173619824021L,
        173619824022L, 173619824023L, 173619824025L, 173619824026L, 173619824027L, 43404956007L, 173619824037L, 173619824038L,
        173619824039L, 173619824043L, 173619824044L, 173619824045L, 173619824046L, 173619824047L, 173619824048L, 173619824049L,
        173619824050L, 173619824051L, 43404956013L, 43404956014L, 43404956015L, 2712809751L, 173619824221L, 173619824222L,
        173619824223L, 173619824243L, 173619824244L, 173619824245L, 173619824246L, 173619824247L, 173619824248L, 173619824249L,
        173619824250L, 173619824251L, 43404956063L, 173619824317L, 173619824319L, 173619824325L, 173619824326L, 173619824327L,
        173619824331L, 173619824332L, 173619824333L, 173619824334L, 173619824335L, 173619824336L, 173619824337L, 173619824338L,
        173619824339L, 43404956085L, 43404956086L, 43404956087L, 173619824352L, 173619824353L, 173619824354L, 173619824355L,
        43404956089L, 173619824360L, 173619824361L, 173619824362L, 173619824363L, 43404956091L, 10851239023L, 173619824384L,
        173619824385L, 173619824386L, 173619824387L, 43404956097L, 173619824392L, 173619824393L, 173619824394L, 173619824395L,
        43404956099L, 10851239025L, 10851239026L, 10851239027L, 2712809757L, 2712809758L, 2712809759L, 173619825141L,
        173619825143L, 173619825148L, 173619825149L, 173619825150L, 173619825151L, 173619825489L, 173619825491L, 173619825492L,
        173619825493L, 173619825494L, 173619825495L, 173619825496L, 173619825497L, 173619825498L, 173619825499L, 43404956375L,
        173619825509L, 173619825511L, 173619825516L, 173619825517L, 173619825518L, 173619825519L, 173619825520L, 173619825521L,
        173619825522L, 173619825523L, 43404956381L, 43404956382L, 43404956383L, 173619825589L, 173619825591L, 173619825597L,
        173619825598L, 173619825599L, 173619825601L, 173619825603L, 173619825604L, 173619825605L, 173619825606L, 173619825607L,
        173619825608L, 173619825609L, 173619825610L, 173619825611L, 43404956403L, 10851239101L, 173619825632L, 173619825633L,
        173619825634L, 173619825635L, 43404956409L, 43404956410L, 43404956411L, 10851239103L, 173619825683L, 173619825684L,
        173619825685L, 173619825686L, 173619825687L, 173619825689L, 173619825690L, 173619825691L, 43404956423L, 173619825703L,
        173619825708L, 173619825709L, 173619825710L, 173619825711L, 173619825712L, 173619825713L, 173619825714L, 173619825715L,
        43404956429L, 43404956430L, 43404956431L, 2712809777L, 173619825793L, 173619825795L, 173619825796L, 173619825797L,
        173619825798L, 173619825799L, 173619825800L, 173619825801L, 173619825802L, 173619825803L, 43404956451L, 10851239113L,
        173619825824L, 173619825825L, 173619825826L, 173619825827L, 43404956457L, 43404956458L, 43404956459L, 10851239115L,
        2712809779L, 678202445L, 678202446L, 678202447L, 42387653L, 173619831059L, 173619831060L, 173619831061L,
        173619831062L, 173619831063L, 173619831065L, 173619831066L, 173619831067L, 43404957767L, 173619831085L, 173619831087L,
        173619831088L, 173619831089L, 173619831090L, 173619831091L, 43404957773L, 173619831096L, 173619831097L, 173619831098L,
        173619831099L, 43404957775L, 2712809861L, 173619831172L, 173619831173L, 173619831174L, 173619831175L, 173619831179L,
        173619831180L, 173619831181L, 173619831182L, 173619831183L, 10851239449L, 173619831201L, 173619831203L, 43404957801L,
        173619831208L, 173619831209L, 173619831210L, 173619831211L, 43404957803L, 10851239451L, 2712809863L, 173619831383L,
        173619831389L, 173619831391L, 173619831412L, 173619831413L, 173619831414L, 173619831415L, 173619831419L, 173619831420L,
        173619831421L, 173619831422L, 173619831423L, 173619831505L, 173619831507L, 43404957877L, 173619831512L, 173619831513L,
        173619831514L, 173619831515L, 43404957879L, 173619831527L, 173619831533L, 173619831535L, 173619831536L, 173619831537L,
        173619831538L, 173619831539L, 43404957885L, 43404957886L, 43404957887L, 173619831552L, 173619831553L, 173619831554L,
        173619831555L, 43404957889L, 43404957890L, 43404957891L, 10851239473L, 10851239474L, 10851239475L, 2712809869L,
        2712809870L, 2712809871L, 169550617L, 173619832901L, 173619832902L, 173619832903L, 173619832908L, 173619832909L,
        173619832910L, 173619832911L, 10851239557L, 173619832931L, 173619832932L, 173619832933L, 173619832934L, 173619832935L,
        173619832937L, 173619832939L, 43404958235L, 10851239559L, 173619833015L, 173619833021L, 173619833023L, 173619833025L,
        173619833026L, 173619833027L, 43404958257L, 173619833032L, 173619833033L, 173619833034L, 173619833035L, 43404958259L,
        10851239565L, 173619833056L, 173619833057L, 173619833058L, 173619833059L, 43404958265L, 43404958266L, 43404958267L,
        10851239567L, 678202473L, 173619833365L, 173619833367L, 173619833372L, 173619833373L, 173619833374L, 173619833375L,
        173619833396L, 173619833397L, 173619833398L, 173619833399L, 173619833404L, 173619833405L, 173619833406L, 173619833407L,
        2712809897L, 173619833489L, 173619833491L, 173619833492L, 173619833493L, 173619833494L, 173619833495L, 173619833497L,
        173619833499L, 43404958375L, 173619833521L, 173619833523L, 43404958381L, 173619833528L, 173619833529L, 173619833530L,
        173619833531L, 43404958383L, 2712809899L, 678202475L, 169550619L, 42387655L, 173619855376L, 173619855377L,
        173619855378L, 173619855379L, 43404963845L, 173619855384L, 173619855385L, 173619855386L, 173619855387L, 43404963847L,
        173619855407L, 173619855408L, 173619855409L, 173619855410L, 173619855411L, 43404963853L, 173619855416L, 173619855417L,
        173619855418L, 173619855419L, 43404963855L, 2712810241L, 173619855493L, 173619855495L, 173619855501L, 173619855503L,
        10851240969L, 173619855525L, 173619855527L, 173619855533L, 173619855535L, 10851240971L, 2712810243L, 678202561L,
        173619855877L, 173619855879L, 173619855885L, 173619855887L, 10851240993L, 173619855909L, 173619855910L, 173619855911L,
        173619855916L, 173619855917L, 173619855918L, 173619855919L, 10851240995L, 2712810249L, 173619856004L, 173619856005L,
        173619856006L, 173619856007L, 173619856012L, 173619856013L, 173619856014L, 173619856015L, 10851241001L, 173619856036L,
        173619856037L, 173619856038L, 173619856039L, 173619856044L, 173619856045L, 173619856046L, 173619856047L, 10851241003L,
        2712810251L, 678202563L, 169550641L, 173619857412L, 173619857413L, 173619857414L, 173619857415L, 173619857420L,
        173619857421L, 173619857422L, 173619857423L, 10851241089L, 173619857444L, 173619857445L, 173619857446L, 173619857447L,
        173619857452L, 173619857453L, 173619857454L, 173619857455L, 10851241091L, 2712810273L, 173619857541L, 173619857543L,
        173619857549L, 173619857551L, 10851241097L, 173619857573L, 173619857575L, 173619857581L, 173619857583L, 10851241099L,
        2712810275L, 678202569L, 173619857925L, 173619857927L, 173619857933L, 173619857935L, 10851241121L, 173619857957L,
        173619857968L, 173619857969L, 173619857970L, 173619857971L, 43404964493L, 173619857976L, 173619857977L, 173619857978L,
        173619857979L, 43404964495L, 2712810281L, 173619858064L, 173619858065L, 173619858066L, 173619858067L, 43404964517L,
        173619858072L, 173619858073L, 173619858074L, 173619858075L, 43404964519L, 173619858096L, 173619858097L, 173619858099L,
        43404964525L, 173619858105L, 173619858107L, 43404964527L, 2712810283L, 678202571L, 169550643L, 42387661L,
        173619863569L, 173619863571L, 43404965893L, 173619863577L, 173619863579L, 173619863580L, 173619863581L, 173619863582L,
        173619863583L, 173619863604L, 173619863605L, 173619863606L, 173619863607L, 173619863612L, 173619863613L, 173619863614L,
        173619863615L, 2712810369L, 173619863700L, 173619863701L, 173619863703L, 173619863709L, 173619863711L, 173619863733L,
        10851241484L, 10851241485L, 173619863776L, 173619863777L, 173619863778L, 173619863779L, 43404965945L, 173619863784L,
        173619863785L, 173619863786L, 173619863787L, 43404965947L, 10851241487L, 678202593L, 173619864128L, 173619864129L,
        173619864130L, 173619864131L, 43404966033L, 173619864137L, 173619864139L, 43404966035L, 10851241509L, 173619864161L,
        173619864163L, 173619864164L, 173619864165L, 173619864166L, 173619864167L, 173619864172L, 173619864173L, 173619864174L,
        173619864175L, 10851241511L, 173619864260L, 173619864261L, 173619864263L, 173619864269L, 173619864271L, 43404966068L,
        43404966069L, 173619864280L, 173619864281L, 173619864282L, 173619864283L, 43404966071L, 173619864304L, 173619864305L,
        173619864306L, 173619864307L, 43404966077L, 173619864312L, 173619864313L, 173619864314L, 173619864315L, 43404966079L,
        678202595L, 169550649L, 173619865681L, 173619865683L, 173619865684L, 173619865685L, 173619865686L, 173619865687L,
        173619865692L, 173619865693L, 173619865694L, 173619865695L, 173619865716L, 173619865717L, 173619865719L, 173619865725L,
        173619865727L, 10851241616L, 10851241617L, 43404966472L, 43404966473L, 173619865896L, 173619865897L, 173619865898L,
        173619865899L, 43404966475L, 10851241619L, 2712810405L, 173619865984L, 173619865985L, 173619865986L, 173619865987L,
        43404966497L, 173619865992L, 173619865993L, 173619865995L, 43404966499L, 10851241625L, 173619866017L, 173619866020L,
        173619866021L, 173619866022L, 173619866023L, 173619866028L, 173619866029L, 173619866031L, 10851241627L, 2712810407L,
        173619866373L, 173619866375L, 173619866384L, 173619866385L, 173619866386L, 173619866387L, 43404966597L, 173619866392L,
        173619866393L, 173619866394L, 173619866395L, 43404966599L, 173619866417L, 173619866419L, 173619866420L, 173619866421L,
        173619866422L, 173619866423L, 173619866428L, 173619866429L, 173619866430L, 173619866431L, 2712810413L, 173619866517L,
        173619866519L, 173619866560L, 173619866561L, 173619866562L, 173619866563L, 43404966641L, 173619866568L, 173619866569L,
        173619866570L, 173619866571L, 43404966643L, 10851241661L, 173619866593L, 173619866595L, 173619866596L, 173619866597L,
        173619866598L, 173619866599L, 173619866604L, 173619866605L, 173619866606L, 173619866607L, 10851241663L, 169550651L,
        42387663L, 2649229L, 173619953989L, 173619953991L, 173619954000L, 173619954001L, 173619954002L, 173619954003L,
        43404988501L, 173619954008L, 173619954009L, 173619954011L, 43404988503L, 173619954033L, 173619954036L, 173619954037L,
        173619954038L, 173619954039L, 173619954044L, 173619954045L, 173619954047L, 10851247168L, 10851247169L, 43404988680L,
        43404988681L, 173619954728L, 173619954729L, 173619954730L, 173619954731L, 43404988683L, 10851247171L, 2712811793L,
        173619954816L, 173619954817L, 173619954818L, 173619954819L, 43404988705L, 173619954825L, 173619954827L, 173619954828L,
        173619954829L, 173619954830L, 173619954831L, 10851247177L, 173619954852L, 173619954853L, 173619954855L, 173619954861L,
        43404988716L, 43404988717L, 173619954872L, 173619954873L, 173619954874L, 173619954875L, 43404988719L, 2712811795L,
        678202949L, 173619955217L, 173619955219L, 173619955220L, 173619955221L, 173619955222L, 173619955223L, 173619955228L,
        173619955229L, 173619955231L, 43404988816L, 43404988817L, 173619955272L, 173619955273L, 173619955274L, 173619955275L,
        43404988819L, 10851247205L, 173619955296L, 173619955297L, 173619955298L, 173619955299L, 43404988825L, 173619955305L,
        173619955308L, 173619955309L, 173619955310L, 173619955311L, 10851247207L, 173619955397L, 173619955399L, 173619955408L,
        173619955409L, 173619955410L, 173619955411L, 43404988853L, 173619955416L, 173619955417L, 173619955419L, 173619955420L,
        173619955421L, 173619955422L, 173619955423L, 173619955444L, 173619955445L, 173619955447L, 2712811804L, 2712811805L,
        10851247224L, 10851247225L, 173619955616L, 173619955617L, 173619955618L, 173619955619L, 43404988905L, 173619955624L,
        173619955625L, 173619955626L, 173619955627L, 43404988907L, 10851247227L, 2712811807L, 173619956993L, 173619956996L,
        173619956997L, 173619956998L, 173619956999L, 173619957005L, 43404989252L, 43404989253L, 173619957016L, 173619957017L,
        173619957018L, 173619957019L, 43404989255L, 173619957041L, 173619957044L, 173619957045L, 173619957046L, 173619957047L,
        173619957053L, 10851247316L, 10851247317L, 43404989272L, 43404989273L, 173619957096L, 173619957097L, 173619957098L,
        173619957099L, 43404989275L, 10851247319L, 173619957185L, 173619957188L, 173619957189L, 173619957190L, 173619957191L,
        173619957197L, 43404989300L, 43404989301L, 173619957208L, 173619957209L, 173619957210L, 173619957211L, 43404989303L,
        173619957233L, 173619957236L, 173619957237L, 173619957238L, 173619957239L, 173619957245L, 169550740L, 169550741L,
        2712811872L, 2712811873L, 10851247496L, 10851247497L, 43404989992L, 43404989993L, 173619959976L, 173619959977L,
        173619959978L, 173619959979L, 43404989995L, 10851247499L, 2712811875L, 678202969L, 173619960321L, 173619960324L,
        173619960325L, 173619960326L, 173619960327L, 173619960333L, 43404990084L, 43404990085L, 173619960344L, 173619960345L,
        173619960346L, 173619960347L, 173619960348L, 173619960349L, 173619960350L, 173619960351L, 173619960372L, 173619960373L,
        173619960375L, 10851247524L, 10851247525L, 173619960416L, 173619960417L, 173619960418L, 173619960419L, 43404990105L,
        173619960424L, 173619960425L, 173619960427L, 173619960428L, 173619960429L, 173619960430L, 173619960431L, 10851247527L,
        173619960517L, 173619960528L, 173619960529L, 173619960530L, 173619960531L, 43404990133L, 173619960537L, 173619960540L,
        173619960541L, 173619960543L, 2712811884L, 2712811885L, 43404990176L, 43404990177L, 173619960712L, 173619960713L,
        173619960714L, 173619960715L, 43404990179L, 10851247545L, 173619960736L, 173619960737L, 173619960739L, 173619960740L,
        173619960741L, 173619960742L, 173619960743L, 173619960749L, 43404990188L, 43404990189L, 173619960760L, 173619960761L,
        173619960762L, 173619960763L, 173619960764L, 173619960765L, 173619960766L, 173619960767L, 2712811887L, 169550743L,
        173619966228L, 173619966229L, 173619966231L, 173619966272L, 173619966273L, 173619966274L, 173619966275L, 43404991569L,
        173619966281L, 173619966284L, 173619966285L, 173619966286L, 173619966287L, 43404991572L, 43404991573L, 173619966296L,
        173619966297L, 173619966298L, 173619966299L, 43404991575L, 173619966320L, 173619966321L, 173619966324L, 173619966325L,
        173619966326L, 173619966327L, 173619966333L, 10851247936L, 10851247937L, 43404991752L, 43404991753L, 173619967016L,
        173619967017L, 173619967019L, 173619967020L, 173619967021L, 173619967022L, 173619967023L, 10851247939L, 2712811985L,
        173619967109L, 173619967120L, 173619967121L, 173619967123L, 173619967124L, 173619967125L, 173619967126L, 173619967127L,
        173619967133L, 43404991792L, 43404991793L, 173619967176L, 173619967177L, 173619967179L, 173619967180L, 173619967181L,
        173619967182L, 173619967183L, 10851247949L, 173619967205L, 173619967216L, 173619967217L, 173619967219L, 173619967220L,
        173619967221L, 173619967222L, 173619967223L, 173619967229L, 2712811988L, 2712811989L, 10851247960L, 10851247961L,
        43404991848L, 43404991849L, 173619967400L, 173619967401L, 173619967403L, 173619967404L, 173619967405L, 173619967406L,
        173619967407L, 10851247963L, 2712811991L, 173619967749L, 173619967760L, 173619967761L, 173619967764L, 173619967765L,
        173619967766L, 173619967767L, 173619967808L, 173619967809L, 173619967810L, 173619967811L, 43404991953L, 173619967816L,
        173619967817L, 173619967820L, 173619967821L, 173619967823L, 43404991956L, 43404991957L, 173619967832L, 173619967833L,
        173619967834L, 173619967835L, 43404991959L, 173619967857L, 173619967860L, 173619967861L, 42387696L, 42387697L,
        678203168L, 678203169L, 10851250720L, 10851250721L, 173620011552L, 173620011553L, 173620011554L, 173620011555L,
        173620011556L, 173620011557L, 173620011558L, 173620011559L, 173620011565L, 43405002892L, 43405002893L, 173620011576L,
        173620011577L, 173620011580L, 173620011581L, 173620011582L, 173620011583L, 10851250724L, 10851250725L, 43405002904L,
        43405002905L, 173620011624L, 173620011625L, 173620011626L, 173620011627L, 43405002907L, 10851250727L, 173620011713L,
        173620011716L, 173620011717L, 173620011728L, 173620011729L, 173620011731L, 173620011732L, 173620011733L, 173620011734L,
        173620011735L, 2712812684L, 2712812685L, 173620011904L, 173620011905L, 173620011906L, 173620011907L, 43405002977L,
        173620011913L, 173620011916L, 173620011917L, 43405002980L, 43405002981L, 173620011928L, 173620011929L, 173620011931L,
        173620011932L, 173620011933L, 173620011934L, 173620011935L, 43405002992L, 43405002993L, 173620011976L, 173620011977L,
        173620011978L, 173620011979L, 43405002995L, 10851250749L, 173620012001L, 173620012004L, 173620012005L, 173620012016L,
        173620012017L, 173620012020L, 173620012021L, 173620012022L, 173620012023L, 678203172L, 678203173L, 2712812696L,
        2712812697L, 10851250792L, 10851250793L, 173620012704L, 173620012705L, 173620012706L, 173620012707L, 173620012708L,
        173620012709L, 173620012710L, 173620012711L, 173620012720L, 173620012721L, 173620012722L, 173620012723L, 43405003181L,
        173620012728L, 173620012729L, 173620012732L, 173620012733L, 10851250796L, 10851250797L, 43405003192L, 43405003193L,
        173620012776L, 173620012777L, 173620012780L, 173620012781L, 43405003196L, 43405003197L, 173620012792L, 173620012793L,
        173620012794L, 173620012795L, 173620012796L, 173620012797L, 173620012798L, 173620012799L, 2712812700L, 2712812701L,
        10851250808L, 10851250809L, 43405003240L, 43405003241L, 173620012968L, 173620012969L, 173620012970L, 173620012971L,
        173620012972L, 173620012973L, 173620012974L, 173620012975L, 43405003244L, 43405003245L, 173620012984L, 173620012985L,
        173620012986L, 173620012987L, 173620012988L, 173620012989L, 173620012990L, 173620012991L, 2712812703L, 173620014357L,
        173620014400L, 173620014401L, 173620014404L, 173620014405L, 173620014416L, 173620014417L, 173620014420L, 173620014421L,
        678203184L, 678203185L, 678203186L, 2712812748L, 2712812749L, 10851251000L, 10851251001L, 43405004008L,
        43405004009L, 43405004010L, 173620016044L, 173620016045L, 173620016046L, 173620016047L, 43405004012L, 43405004013L,
        173620016056L, 173620016057L, 173620016058L, 173620016059L, 173620016060L, 173620016061L, 173620016062L, 173620016063L,
        10851251004L, 10851251005L, 43405004024L, 43405004025L, 173620016104L, 173620016105L, 173620016106L, 173620016107L,
        173620016108L, 173620016109L, 173620016110L, 173620016111L, 43405004028L, 43405004029L, 173620016120L, 173620016121L,
        173620016122L, 173620016123L, 173620016124L, 173620016125L, 173620016126L, 678203188L, 678203189L, 2712812760L,
        2712812761L, 10851251048L, 10851251049L, 43405004200L, 43405004201L, 173620016808L, 173620016809L, 173620016812L,
        173620016813L, 173620016816L, 173620016817L, 173620016818L, 173620016819L, 173620016820L, 173620016821L, 173620016822L,
        173620016823L, 173620016824L, 173620016825L, 10851251052L, 10851251053L, 173620016864L, 173620016865L, 173620016866L,
        173620016867L, 173620016868L, 173620016869L, 173620016870L, 173620016871L, 173620016880L, 173620016881L, 173620016882L,
        173620016884L, 173620016885L, 2712812764L, 2712812765L, 43405004256L, 43405004257L, 173620017032L, 173620017033L,
        173620017034L, 173620017035L, 173620017036L, 173620017037L, 173620017038L, 173620017039L, 43405004260L, 43405004261L,
        173620017048L, 173620017049L, 173620017050L, 173620017051L, 173620017052L, 173620017053L, 173620017054L, 173620017056L,
        173620017057L, 43405004272L, 173620017092L, 173620017093L, 173620017094L, 173620017095L, 173620017096L, 173620017097L,
        173620017100L, 173620017104L, 173620017105L, 173620017106L, 173620017107L, 173620017108L, 173620017109L, 173620017110L,
        173620017152L, 173620017153L, 173620017156L, 173620017157L, 173620017168L, 173620017169L, 173620017172L, 173620017173L,
        173620017216L, 173620017217L, 173620017220L, 173620017221L, 173620017232L, 173620017233L, 173620017236L, 173620017237L,
        173620017408L, 173620017409L, 173620017412L, 169550800L, 169550801L, 169550802L, 678203212L, 678203213L,
        678203214L, 2712812860L, 2712812861L, 2712812862L, 10851251452L, 10851251453L, 10851251454L, 43405005820L,
        173620023284L, 173620023285L, 173620023286L, 173620023287L, 173620023288L, 173620023289L, 173620023290L, 173620023291L,
        173620023292L, 173620023293L, 173620023294L, 169550804L, 678203220L, 2712812884L, 2712812885L, 2712812886L,
        10851251548L, 43405006196L, 43405006197L, 43405006198L, 173620024796L, 173620024797L, 173620024798L, 173620024799L,
        43405006200L, 43405006201L, 43405006202L, 173620024812L, 173620024813L, 173620024814L, 173620024815L, 173620024816L,
        173620024817L, 173620024818L, 173620024819L, 173620024820L, 173620024824L, 2712812888L, 2712812889L, 2712812890L,
        10851251564L, 43405006260L, 43405006261L, 43405006262L, 173620025052L, 173620025053L, 173620025054L, 173620025055L,
        43405006264L, 173620025060L, 173620025061L, 173620025062L, 173620025063L, 173620025064L, 173620025065L, 173620025066L,
        173620025067L, 173620025068L, 173620025069L, 173620025070L, 173620025072L, 173620025073L, 173620025074L, 173620025075L,
        173620025076L, 10851251568L, 43405006276L, 43405006277L, 43405006278L, 173620025116L, 173620025117L, 173620025118L,
        173620025119L, 43405006280L, 43405006281L, 43405006282L, 173620025132L, 173620025133L, 173620025134L, 173620025135L,
        173620025136L, 173620025137L, 173620025138L, 173620025139L, 173620025140L, 173620025144L, 173620025152L, 173620025153L,
        173620025154L, 173620025155L, 173620025156L, 173620025160L, 173620025216L, 173620025217L, 173620025218L, 173620025219L,
        173620025220L, 173620025224L, 678203224L, 2712812900L, 2712812901L, 10851251608L, 10851251609L, 10851251610L,
        43405006444L, 43405006445L, 173620025784L, 173620025785L, 173620025786L, 173620025787L, 173620025788L, 173620025789L,
        173620025790L, 43405006448L, 43405006449L, 43405006450L, 173620025804L, 173620025805L, 173620025806L, 173620025807L,
        173620025808L, 173620025809L, 173620025810L, 173620025811L, 173620025812L, 173620025813L, 173620025814L, 173620025816L,
        173620025817L, 173620025818L, 173620025824L, 173620025825L, 173620025826L, 173620025827L, 173620025828L, 173620025832L,
        2712812904L, 10851251620L, 43405006484L, 43405006485L, 43405006486L, 173620025948L, 173620025949L, 173620025950L,
        173620025951L, 43405006488L, 43405006489L, 173620025960L, 173620025961L, 173620025962L, 173620025963L, 173620025964L,
        173620025965L, 173620025966L, 173620025967L, 173620025968L, 173620025969L, 173620025970L, 173620025971L, 173620025972L,
        173620025973L, 173620025974L, 173620025976L, 43405006496L, 43405006497L, 43405006498L, 173620025996L, 173620025997L,
        173620025998L, 173620025999L, 43405006500L, 173620026004L, 173620026005L, 173620026006L, 173620026007L, 173620026008L,
        173620026009L, 173620026010L, 173620026012L, 173620026016L, 173620026017L, 173620026018L, 173620026019L, 173620026020L,
        173620026021L, 173620026048L, 173620026049L, 173620026050L, 43405006528L, 173620026116L, 173620026117L, 173620026118L,
        173620026119L, 173620026120L, 173620026121L, 173620026122L, 173620026123L, 173620026124L, 173620026128L, 173620026129L,
        173620026130L, 10851251648L, 43405006596L, 173620026388L, 173620026389L, 173620026390L, 173620026391L, 173620026392L,
        173620026393L, 173620026394L, 173620026395L, 173620026396L, 173620026397L, 173620026398L, 43405006600L, 173620026404L,
        173620026405L, 173620026406L, 173620026407L, 173620026408L, 173620026409L, 173620026410L, 173620026411L, 173620026412L,
        173620026416L, 173620026432L, 173620026433L, 173620026434L, 173620026496L, 678203232L, 2712812932L, 2712812933L,
        2712812934L, 10851251740L, 10851251741L, 43405006968L, 43405006969L, 43405006970L, 173620027884L, 173620027885L,
        173620027886L, 173620027887L, 43405006972L, 173620027892L, 173620027893L, 173620027894L, 173620027895L, 173620027896L,
        173620027897L, 173620027898L, 173620027899L, 173620027900L, 173620027901L, 10851251744L, 10851251745L, 43405006984L,
        43405006985L, 43405006986L, 173620027948L, 173620027949L, 173620027950L, 173620027951L, 43405006988L, 43405006989L,
        173620027960L, 173620027961L, 173620027962L, 173620027963L, 173620027964L, 173620027965L, 173620027966L, 10851251748L,
        43405006996L, 43405006997L, 43405006998L, 173620027996L, 173620027997L, 173620027998L, 173620027999L, 173620028000L,
        173620028001L, 173620028002L, 173620028003L, 173620028004L, 173620028005L, 173620028006L, 173620028007L, 173620028008L,
        173620028009L, 173620028016L, 173620028017L, 173620028018L, 173620028020L, 173620028021L, 173620028032L, 173620028033L,
        173620028036L, 43405007040L, 43405007041L, 173620028168L, 173620028169L, 173620028170L, 173620028171L, 173620028172L,
        173620028173L, 173620028174L, 173620028176L, 173620028177L, 173620028178L, 173620028179L, 173620028180L, 173620028181L,
        173620028182L, 173620028183L, 173620028184L, 173620028185L, 173620028224L, 173620028225L, 173620028226L, 173620028228L,
        2712812944L, 10851251780L, 10851251781L, 10851251782L, 43405007132L, 43405007133L, 173620028536L, 173620028537L,
        173620028538L, 173620028539L, 173620028540L, 173620028541L, 173620028542L, 173620028543L, 10851251784L, 43405007140L,
        43405007141L, 173620028568L, 173620028569L, 173620028570L, 173620028571L, 173620028572L, 173620028573L, 173620028574L,
        173620028576L, 173620028577L, 173620028578L, 173620028579L, 173620028580L, 173620028581L, 173620028582L, 173620028592L,
        173620028608L, 173620028609L, 173620028610L, 173620028611L, 173620028612L, 173620028613L, 173620028614L, 173620028615L,
        173620028616L, 173620028624L, 173620028625L, 10851251792L, 43405007172L, 43405007173L, 173620028696L, 173620028697L,
        173620028698L, 173620028699L, 173620028700L, 173620028701L, 173620028702L, 173620028703L, 173620028704L, 173620028705L,
        173620028706L, 173620028707L, 173620028708L, 173620028709L, 173620028710L, 173620028711L, 173620028712L, 173620028713L,
        173620028720L, 173620028721L, 173620028736L, 173620028737L, 173620028738L, 173620028739L, 173620028740L, 173620028741L,
        173620028742L, 173620028744L, 173620028752L, 173620240522L, 173620240544L, 173620240545L, 173620240546L, 173620240547L,
        173620240548L, 173620240550L, 173620240551L, 43405060138L, 173620240556L, 173620240557L, 173620240558L, 173620240559L,
        173620240568L, 173620240570L, 173620240571L, 10851265056L, 173620240912L, 173620240913L, 173620240914L, 173620240915L,
        173620240916L, 173620240918L, 173620240919L, 43405060230L, 173620240924L, 173620240925L, 173620240926L, 173620240927L,
        10851265058L, 10851265059L, 173620240968L, 173620240970L, 173620240971L, 173620240992L, 173620240993L, 173620240994L,
        173620240995L, 173620240996L, 173620240998L, 173620240999L, 43405060250L, 173620241004L, 173620241005L, 173620241006L,
        173620241007L, 173620241016L, 173620241018L, 173620241019L, 2712816266L, 10851265068L, 173620241104L, 173620241105L,
        173620241106L, 173620241107L, 173620241108L, 173620241110L, 173620241111L, 43405060278L, 173620241116L, 173620241117L,
        173620241118L, 173620241119L, 10851265070L, 10851265071L, 173620241288L, 173620241290L, 173620241291L, 173620241312L,
        173620241313L, 173620241314L, 173620241315L, 173620241316L, 173620241318L, 173620241319L, 43405060330L, 173620241324L,
        173620241325L, 173620241326L, 173620241327L, 173620241336L, 173620241338L, 173620241339L, 678204072L, 10851265168L,
        173620242704L, 173620242705L, 173620242706L, 173620242707L, 173620242708L, 173620242710L, 173620242711L, 43405060678L,
        173620242716L, 173620242717L, 173620242718L, 173620242719L, 10851265170L, 10851265171L, 173620242762L, 173620242784L,
        173620242785L, 173620242786L, 173620242787L, 173620242790L, 43405060698L, 173620242796L, 173620242797L, 173620242798L,
        173620242799L, 2712816294L, 43405060720L, 173620242884L, 173620242885L, 173620242886L, 173620242887L, 43405060722L,
        43405060723L, 173620242896L, 173620242898L, 173620242899L, 173620242904L, 173620242905L, 173620242906L, 173620242907L,
        173620242908L, 173620242910L, 10851265182L, 43405060732L, 173620242932L, 173620242933L, 173620242934L, 173620242935L,
        43405060734L, 43405060735L, 678204074L, 678204075L, 173620243618L, 173620243624L, 173620243625L, 173620243626L,
        173620243627L, 173620243968L, 173620243969L, 173620243970L, 173620243971L, 173620243972L, 173620243974L, 173620243975L,
        43405060994L, 173620243980L, 173620243981L, 173620243982L, 173620243983L, 173620243994L, 10851265250L, 173620244016L,
        173620244017L, 173620244018L, 173620244019L, 173620244024L, 173620244025L, 173620244026L, 173620244027L, 173620244028L,
        173620244030L, 10851265256L, 43405061028L, 173620244116L, 173620244117L, 173620244118L, 173620244119L, 43405061030L,
        43405061031L, 10851265258L, 10851265259L, 173620244162L, 173620244168L, 173620244170L, 173620244171L, 173620244192L,
        173620244193L, 173620244194L, 173620244195L, 173620244198L, 43405061050L, 173620244204L, 173620244205L, 173620244206L,
        173620244207L, 169551360L, 2712821776L, 43405148432L, 173620593732L, 173620593733L, 173620593734L, 173620593735L,
        43405148434L, 43405148435L, 173620593744L, 173620593746L, 173620593752L, 173620593753L, 173620593754L, 173620593755L,
        10851287110L, 173620593776L, 173620593777L, 173620593778L, 173620593779L, 173620593780L, 173620593782L, 43405148446L,
        173620593788L, 173620593789L, 173620593790L, 173620593791L, 2712821778L, 10851287116L, 43405148468L, 173620593876L,
        173620593877L, 173620593878L, 173620593879L, 43405148470L, 43405148471L, 10851287118L, 10851287119L, 173620594048L,
        173620594050L, 173620594056L, 173620594057L, 173620594058L, 173620594059L, 173620594080L, 173620594081L, 173620594082L,
        173620594083L, 173620594084L, 173620594086L, 43405148522L, 173620594092L, 173620594093L, 173620594094L, 173620594095L,
        678205446L, 43405148608L, 173620594436L, 173620594437L, 173620594438L, 173620594439L, 43405148610L, 43405148611L,
        173620594450L, 173620594456L, 173620594458L, 173620594459L, 10851287154L, 173620594480L, 173620594481L, 173620594482L,
        173620594483L, 173620594488L, 173620594489L, 173620594490L, 173620594491L, 173620594492L, 173620594494L, 10851287160L,
        43405148644L, 173620594580L, 173620594581L, 173620594582L, 173620594583L, 43405148646L, 173620594588L, 173620594589L,
        173620594590L, 173620594591L, 10851287162L, 10851287163L, 173620594634L, 173620594656L, 173620594658L, 173620594664L,
        173620594665L, 173620594666L, 173620594667L, 169551362L, 678205452L, 2712821812L, 173620596032L, 173620596033L,
        173620596034L, 173620596035L, 173620596038L, 43405149010L, 173620596044L, 173620596046L, 43405149016L, 173620596068L,
        173620596069L, 173620596070L, 173620596071L, 43405149018L, 173620596076L, 173620596077L, 173620596078L, 173620596079L,
        173620596090L, 2712821814L, 10851287260L, 173620596176L, 173620596178L, 173620596184L, 173620596186L, 173620596187L,
        10851287262L, 173620596208L, 173620596209L, 173620596210L, 173620596211L, 173620596216L, 173620596217L, 173620596218L,
        173620596219L, 173620596222L, 678205454L, 2712821820L, 10851287284L, 43405149140L, 173620596564L, 173620596566L,
        43405149142L, 173620596572L, 173620596574L, 173620596575L, 10851287286L, 43405149148L, 173620596596L, 173620596597L,
        173620596598L, 173620596599L, 43405149150L, 173620596604L, 173620596605L, 173620596606L, 173620596607L, 2712821822L,
        10851287292L, 43405149172L, 173620596692L, 173620596693L, 173620596694L, 173620596695L, 43405149174L, 43405149175L,
        10851287294L, 10851287295L, 173620599424L, 173620599426L, 173620599432L, 173620599434L, 173620599456L, 173620599458L,
        173620599459L, 173620599464L, 173620599465L, 173620599466L, 173620599467L, 42387842L, 173620604928L, 173620604929L,
        173620604930L, 173620604931L, 173620604936L, 173620604937L, 173620604938L, 173620604939L, 173620604960L, 173620604961L,
        173620604962L, 173620604963L, 173620604964L, 173620604966L, 43405151242L, 173620604972L, 173620604974L, 43405151264L,
        173620605060L, 173620605062L, 43405151266L, 173620605068L, 173620605070L, 43405151272L, 173620605092L, 173620605093L,
        173620605094L, 173620605095L, 43405151274L, 173620605100L, 173620605101L, 173620605102L, 173620605103L, 43405151360L,
        173620605444L, 173620605445L, 173620605446L, 173620605447L, 43405151362L, 173620605452L, 173620605453L, 173620605454L,
        173620605455L, 43405151368L, 173620605476L, 173620605477L, 173620605478L, 173620605479L, 43405151370L, 173620605484L,
        173620605485L, 173620605486L, 173620605487L, 43405151392L, 173620605572L, 173620605573L, 173620605574L, 173620605575L,
        43405151394L, 173620605580L, 173620605581L, 173620605582L, 173620605583L, 43405151400L, 173620605604L, 173620605605L,
        173620605606L, 173620605607L, 43405151402L, 173620605612L, 173620605613L, 173620605614L, 173620605615L, 43405151744L,
        173620606980L, 173620606981L, 173620606982L, 173620606983L, 43405151746L, 173620606988L, 173620606989L, 173620606990L,
        173620606991L, 43405151752L, 173620607012L, 173620607013L, 173620607014L, 173620607015L, 43405151754L, 173620607020L,
        173620607021L, 173620607022L, 173620607023L, 43405151776L, 173620607108L, 173620607109L, 173620607110L, 173620607111L,
        43405151778L, 173620607116L, 173620607117L, 173620607118L, 173620607119L, 43405151784L, 173620607140L, 173620607141L,
        173620607142L, 173620607143L, 43405151786L, 173620607148L, 173620607149L, 173620607150L, 173620607151L, 43405151872L,
        173620607492L, 173620607493L, 173620607494L, 173620607495L, 43405151874L, 173620607500L, 173620607501L, 173620607502L,
        173620607503L, 43405151880L, 173620607524L, 173620607525L, 173620607526L, 173620607527L, 43405151882L, 173620607532L,
        173620607533L, 173620607534L, 43405151904L, 173620607620L, 173620607622L, 43405151906L, 173620607628L, 173620607630L,
        43405151912L, 173620607652L, 173620607654L, 173620607656L, 173620607657L, 173620607658L, 173620607659L, 173620607660L,
        173620607662L, 169551392L, 678205572L, 2712822292L, 2712822293L, 2712822294L, 10851289180L, 43405156724L,
        43405156725L, 43405156726L, 173620626908L, 173620626909L, 173620626910L, 173620626911L, 10851289182L, 43405156732L,
        173620626932L, 173620626933L, 173620626934L, 173620626935L, 43405156734L, 173620626940L, 173620626941L, 173620626942L,
        173620626943L, 678205574L, 2712822300L, 10851289204L, 43405156820L, 173620627284L, 173620627285L, 173620627286L,
        43405156822L, 173620627292L, 173620627294L, 10851289206L, 173620627312L, 173620627313L, 173620627314L, 173620627315L,
        173620627316L, 173620627318L, 173620627320L, 173620627321L, 173620627322L, 173620627323L, 2712822302L, 10851289212L,
        173620627408L, 173620627409L, 173620627410L, 173620627411L, 173620627416L, 173620627418L, 43405156856L, 173620627428L,
        173620627429L, 173620627430L, 173620627431L, 43405156858L, 173620627436L, 173620627437L, 173620627438L, 173620627439L,
        173620627440L, 173620627442L, 169551394L, 678205580L, 2712822324L, 43405157200L, 173620628804L, 173620628805L,
        173620628806L, 43405157202L, 173620628812L, 173620628814L, 173620628832L, 173620628833L, 173620628834L, 173620628835L,
        173620628836L, 173620628840L, 173620628841L, 173620628842L, 173620628843L, 10851289304L, 43405157220L, 43405157221L,
        43405157222L, 173620628892L, 173620628893L, 173620628894L, 173620628895L, 10851289306L, 43405157228L, 173620628916L,
        173620628917L, 173620628918L, 173620628919L, 43405157230L, 173620628924L, 173620628926L, 173620628928L, 173620628930L,
        173620628936L, 2712822328L, 2712822329L, 2712822330L, 10851289324L, 10851289325L, 10851289326L, 43405157308L,
        43405157309L, 43405157310L, 173620629244L, 173620629245L, 173620629246L, 173620629247L, 10851289328L, 173620629264L,
        173620629265L, 173620629266L, 173620629267L, 173620629268L, 173620629272L, 173620629273L, 173620629274L, 173620629275L,
        43405157320L, 173620629284L, 173620629285L, 173620629286L, 173620629287L, 43405157322L, 173620629292L, 173620629293L,
        173620629294L, 173620629295L, 173620629296L, 173620629298L, 43405157344L, 173620629380L, 173620629381L, 173620629382L,
        173620629384L, 173620629385L, 173620629386L, 173620629387L, 173620629388L, 173620629390L, 173620629408L, 173620629409L,
        173620629410L, 173620629411L, 173620629416L, 173620629418L, 173620629504L, 173620629505L, 173620629506L, 173620629507L,
        173620629512L, 173620629513L, 173620629514L, 173620629515L, 173620629536L, 173620629537L, 173620629538L, 173620629539L,
        173620629544L, 173620629546L, 173620629632L, 173620629634L, 173620629640L, 173620629642L, 678205600L, 678205601L,
        678205602L, 2712822412L, 2712822413L, 2712822414L, 10851289660L, 10851289661L, 10851289662L, 43405158652L,
        43405158653L, 43405158654L, 173620634620L, 173620634621L, 173620634622L, 173620634623L, 2712822416L, 10851289668L,
        43405158676L, 173620634708L, 173620634709L, 173620634710L, 173620634711L, 43405158678L, 173620634716L, 173620634717L,
        173620634718L, 10851289670L, 173620634736L, 173620634737L, 173620634738L, 173620634739L, 173620634740L, 173620634744L,
        173620634745L, 173620634746L, 2712822418L, 43405158704L, 173620634820L, 173620634821L, 173620634822L, 173620634823L,
        43405158706L, 173620634828L, 173620634829L, 173620634830L, 173620634832L, 173620634848L, 173620634849L, 173620634850L,
        173620634851L, 173620634852L, 173620634856L, 173620634857L, 173620634858L, 10851289696L, 43405158788L, 173620635156L,
        173620635157L, 173620635158L, 173620635159L, 43405158790L, 173620635164L, 173620635165L, 173620635166L, 10851289698L,
        173620635184L, 173620635185L, 173620635186L, 173620635187L, 173620635188L, 173620635192L, 173620635193L, 173620635194L,
        173620635200L, 43405158816L, 173620635268L, 173620635269L, 173620635270L, 173620635271L, 173620635272L, 173620635273L,
        173620635274L, 173620635275L, 173620635276L, 173620635278L, 173620635280L, 173620635296L, 173620635297L, 173620635298L,
        173620635299L, 173620635304L, 173620635306L, 678205608L, 2712822436L, 10851289748L, 43405158996L, 173620635988L,
        173620635989L, 173620635990L, 173620635992L, 173620635993L, 173620635994L, 173620635995L, 173620635996L, 43405159000L,
        173620636004L, 173620636005L, 173620636006L, 173620636007L, 43405159002L, 173620636012L, 173620636013L, 173620636014L,
        173620636015L, 173620636016L, 173620636018L, 10851289752L, 43405159012L, 43405159013L, 43405159014L, 173620636060L,
        173620636061L, 173620636062L, 173620636063L, 10851289754L, 43405159020L, 173620636084L, 173620636085L, 173620636086L,
        173620636087L, 173620636088L, 173620636089L, 173620636090L, 173620636091L, 173620636092L, 173620636096L, 173620636097L,
        173620636098L, 173620636099L, 173620636100L, 173620636104L, 173620636105L, 173620636106L, 2712822440L, 10851289764L,
        10851289765L, 10851289766L, 43405159068L, 43405159069L, 43405159070L, 173620636284L, 173620636285L, 173620636286L,
        173620636287L, 2712822442L, 43405159088L, 43405159089L, 43405159090L, 173620636364L, 173620636365L, 173620636366L,
        173620636367L, 173620636368L, 173620636369L, 173620636370L, 173620636371L, 173620636372L, 173620636373L, 173620636374L,
        173620636376L, 173620636377L, 173620636378L, 43405159096L, 173620636388L, 173620636389L, 173620636390L, 173620636391L,
        173620636392L, 173620636393L, 173620636394L, 173620636395L, 173620636396L, 43405159104L, 173620636420L, 173620636421L,
        173620636422L, 173620636423L, 43405159106L, 173620636428L, 173620636429L, 173620636430L, 173620636432L, 173620636434L,
        173620636448L, 173620636449L, 173620636450L, 173620636451L, 173620636452L, 173620636456L, 10851295232L, 43405180932L,
        173620723732L, 173620723733L, 173620723734L, 173620723735L, 173620723736L, 173620723737L, 173620723738L, 173620723739L,
        173620723740L, 173620723741L, 173620723742L, 43405180936L, 173620723748L, 173620723749L, 173620723750L, 173620723751L,
        173620723752L, 173620723753L, 173620723754L, 173620723755L, 173620723756L, 173620723757L, 173620723758L, 173620723760L,
        173620723761L, 173620723762L, 173620723776L, 173620723840L, 173620723841L, 173620723842L, 173620723848L};
    coneTest(lon, lat, rad, order, bmocEntries);
    
    
    /*For an nside : 262144(order 18 for size 0.05 degrees), ra=  205.0 and radius = 8.726646259971648E-4 radians (0.05 in degrees) and inclusive = true.
        For dec inputs : 30.5 and 30.6
            query_disc(262144, 205.0, 30.5 ,8.726646259971648E-4 , true); //works
            query_disc(262144, 205.0, 30.6 ,8.726646259971648E-4 , true); //does not work. throws an ArrayIndexOutOfBoundsException
    */
    // // draw circle(97.91750000,+5.76952778, 0.07416666666666666666)
    
  }
  
  @Test
  public void testCone10() {
    double lon = 36.80105218;
    double lat = 56.78028536;
    double rad = 14.93;
    int order = 3;
    
    // draw circle(36.80105218,+56.78028536, 14.93)    
    coneTest(lon, lat, rad, order, new long[]{26L, 27L, 30L, 37L, 38L, 39L, 44L, 45L, 46L, 47L, 48L,
        49L, 50L, 51L, 52L, 54L, 56L, 57L, 58L,
        59L, 60L});

  }
  

  @Test
  public void testCone11() {
    double lon = 160.771389;
    double lat = 64.3813;
    double rad = 0.8962;
    int order = 6;
    
    // draw circle(160.771389, 64.3813, 0.8962)
    coneTest(lon, lat, rad, order, new long[]{7477L, 7479L, 7498L, 7499L, 7520L, 7521L, 7522L, 7523L});
    
    // draw circle(160.771389, -64.3813, 0.8962)
    coneTest(lon, -lat, rad, order, new long[]{37228L, 37229L, 37230L, 37231L, 37240L, 37242L, 37316L, 37317L});
  }
  
  
  @Test
  public void testCone12() {
    double lon = 210.89895009;
    double lat = 54.29995006;
    double rad = 6.9444; //Math.toDegrees(0.1212026445754942);
    int order = 8; // new long[]{167L, 172L, 173L, 178L, 179L, 184L, 185L}
    
    // draw circle(210.89895009, 54.29995006, 6.9444)
    coneTest(lon, lat, rad, order, null /*new long[]{7477L, 7479L, 7498L, 7499L, 7520L, 7521L, 7522L, 7523L}*/);
    
    // draw circle(160.771389, -64.3813, 0.8962)
    // coneTest(lon, -lat, rad, order, new long[]{37228L, 37229L, 37230L, 37231L, 37240L, 37242L, 37316L, 37317L});
  }
  
  @Test
  public void testCone13() {
	double lon = Math.toDegrees(4.646003386280478);
	double lat = Math.toDegrees(-0.41602905223485265);
	double rad = Math.toDegrees(0.9886484667476922);
	int order = 6;
	coneTest(lon, lat, rad, order, null);
  }
  
  @Test
  public void testCone14() {
    HealpixNestedBMOC bmoc = cds.healpix.Healpix.getNested(11)
        .newConeComputerApprox(5./3600.)
        .overlappingCells(0.5*Math.PI, -1.1617);
    System.out.println("Moc size: " + bmoc.size()); 
  }
  
  
  
  public static void main(String[] args) {
    new NestedConeTest().testCone14();
  }
  
}
