/*
cdcd - Command Driven CD player
Copyright (C) 1998-99 Tony Arcieri
Copyright (C) 2001, 2002, 2003 Fabrice Bauzac

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*

  IMPORTANT THINGS TO FIX:

  - Look at cmd_sites_add;

  - Review cmd_sites_refresh;

*/

#include <stdio.h>

#include "cmd_sites.h"
#include "cmdline.h"
#include "str.h"
#include "rlhist.h"
#include "interface.h"

Execfunc cmd_sites_cdcd, cmd_sites_help, cmd_sites_refresh,
  cmd_sites_display, cmd_sites_add, cmd_sites_delete, cmd_sites_move,
  cmd_sites_quit;

struct Command sites_cmds[] = {
  ALIAS ("?", "help"),
  {
   "..", "", "Go back to the cdcd command-line.",
   cmd_sites_cdcd, NULL, 0},
  {
   "help", NULL, "Ha, ha, ha.  This is an truly excellent joke.",
   cmd_sites_help, NULL, 0},
  {
   "refresh", "[URL]", "Update the CDDB server list from the default "
   "CDDB server or the specified URL.", cmd_sites_refresh, NULL, 1},
  {
   "display", "", "Display the current CDDB server list.",
   cmd_sites_display, NULL, 1},
  {
   "add", "LOCATION url=URL TYPE",
   "Add an entry to the CDDB server list.  `LOCATION' is a position "
   "in the list.",
   cmd_sites_add, NULL, 1},
  {
   "del", "[LOCATION]", "Delete an entry from the CDDB server list.",
   cmd_sites_delete, NULL, 1},
  {
   "move", "SOURCE DESTINATION",
   "Move a CDDB server list entry from `SOURCE' to `DESTINATION'.",
   cmd_sites_move, NULL, 1},
  {
   "quit", "", "Quit cdcd.", cmd_sites_quit, NULL, 0},
  ALIAS ("exit", "quit"),
  {NULL, NULL, NULL, NULL, NULL, 0}
};

int
cmd_sites_quit (char **argv)
{
  return XRET_QUIT;
}

int
cmd_sites_cdcd (char **argv)
{
  return XRET_BACK;
}

int
cmd_sites_display (char **argv)
{
  struct cddb_serverlist list;
  struct cddb_conf conf;
  struct cddb_server proxy;
  char *p;

  cddb_read_serverlist (&conf, &list, &proxy);
  p = cddb_serverlist_prettystr (&list);
  fputs (p, stdout);
  free (p);
  return XRET_NULL;
}

/* FIXME: This has not been tested even once yet */
int
cmd_sites_refresh (char **argv)
{
  struct cddb_host host;
  struct cddb_serverlist list;
  struct cddb_conf conf;
  struct cddb_server proxy;

  cddb_read_serverlist (&conf, &list, &proxy);

  if (argv[1])
    cddb_process_url (&host, argv[1]);
  else
    {
      host.host_protocol = CDDB_MODE_CDDBP;
      strncpy (host.host_server.server_name, cdcd_cddb_host, 256);
      host.host_server.server_port = CDDBP_DEFAULT_PORT;
    }

  cddb_get_siteslist (&list, host);	/* FIXME: Isn't this a memory leak?  */

  if (list.list_len < 1)
    {
      puts ("Retrieved empty server list... not stored");
      return XRET_NULL;
    }

  conf.conf_access = CDDB_ACCESS_REMOTE;
  cddb_write_serverlist (conf, list, proxy);
  return XRET_NULL;
}

int
cmd_sites_add (char **argv)
{
  int location;
  int type;
  char *url;
  struct cddb_serverlist list;
  struct cddb_conf conf;
  struct cddb_server proxy;
  int loc_provided = 0, type_provided = 0, url_provided = 0;

  cddb_read_serverlist (&conf, &list, &proxy);

  for (++argv; *argv; argv++)
    {
      if (!loc_provided && !read_int (*argv, &location))
	{
	  loc_provided = 1;
	}
      else if (!type_provided && !read_type (*argv, &type))
	{
	  type_provided = 1;
	}
      else if (!url_provided && !read_url (*argv, &url))
	{
	  url_provided = 1;
	}
      else
	{
	  if (url_provided)
	    free (url);
	  puts ("Errors during arguments parsing");
	  return XRET_NULL;
	}
    }

  if (!loc_provided || !url_provided || !type_provided)
    {
      if (url_provided)
	free (url);
      puts ("You must supply all the arguments.");
      return XRET_NULL;
    }

  --location;

  if (location < 0 || location > list.list_len)
    {
      free (url);
      puts ("Invalid location");
      return XRET_NULL;
    }

  {
    int i;
    list.list_len++;
    for (i = list.list_len; i > location; i--)
      memcpy (&list.list_host[i],
	      &list.list_host[i - 1], sizeof (struct cddb_host));
    cddb_process_url (&list.list_host[location], url);
  }
  free (url);

  /* FIXME NOW: Since I don't understand how it worked, I don't know
     how to make it work here.  -- fb.  */
  if (type == TYPE_CDDB)
    list.list_host[location].host_protocol = CDDB_MODE_CDDBP;
  else
    list.list_host[location].host_protocol = CDINDEX_MODE_HTTP;

  cddb_write_serverlist (conf, list, proxy);
  return XRET_NULL;
}

int
cmd_sites_delete (char **argv)
{
  int loc_provided = 0;
  int location;
  struct cddb_serverlist list;
  struct cddb_conf conf;
  struct cddb_server proxy;

  cddb_read_serverlist (&conf, &list, &proxy);

  /* You may think this is useless, but it's much easier to maintain
     if we need, say, an additional option */
  for (++argv; *argv; argv++)
    {
      if (!loc_provided && !read_int (*argv, &location))
	loc_provided = 1;
      else
	{
	  puts ("Errors during arguments parsing");
	  return XRET_NULL;
	}
    }

  if (!loc_provided)
    location = 1;

  if (location < 1 || location > list.list_len)
    {
      puts ("Invalid location");
      return XRET_NULL;
    }

  {
    int i;
    for (i = location; i < list.list_len - 1; i++)
      memcpy (&list.list_host[i],
	      &list.list_host[i + 1], sizeof (struct cddb_host));
    list.list_len--;
  }

  cddb_write_serverlist (conf, list, proxy);
  return XRET_NULL;
}

int
cmd_sites_move (char **argv)
{
  struct cddb_serverlist list;
  struct cddb_conf conf;
  struct cddb_server proxy;
  struct cddb_host host;
  int l1, l2;
  int l1_prov = 0, l2_prov = 0;

  cddb_read_serverlist (&conf, &list, &proxy);

  {
    char **arg;
    for (arg = argv + 1; *arg; arg++)
      {
	if (!l1_prov && !read_int (*arg, &l1))
	  l1_prov = 1;
	else if (!l2_prov && !read_int (*arg, &l2))
	  l2_prov = 1;
	else
	  {
	    puts ("Errors during argument parsing");
	    return 0;
	  }
      }
  }

  if (!l1_prov || !l2_prov)
    {
      puts ("Not enough parameters");
      return 0;
    }

  l1--;
  l2--;
  if (l1 < 0 || l2 < 0 || l1 > list.list_len || l2 > list.list_len)
    {
      puts ("Arguments out of range");
      return 0;
    }

  /* This has been essentially copy-pasted */
  {
    int i;
    memcpy (&host, &list.list_host[l1], sizeof (struct cddb_host));
    if (l2 > l1)
      {
	for (i = l1; i < l2; i++)
	  memcpy (&list.list_host[i],
		  &list.list_host[i + 1], sizeof (struct cddb_host));
      }
    else if (l2 < l1)
      {
	for (i = l1; i > l2; i--)
	  memcpy (&list.list_host[i],
		  &list.list_host[i - 1], sizeof (struct cddb_host));
      }
    memcpy (&list.list_host[l2], &host, sizeof (struct cddb_host));
  }

  cddb_write_serverlist (conf, list, proxy);
  cddb_process_url (&host, argv[0]);
  return 0;
}

int
cmd_sites_help (char **argv)
{
  cmdhelp (sites_cmds, argv, 0);
  return 0;
}

char *
sites_command_matcher (const char *text, int state)
{
  return command_matcher (sites_cmds, text, state);
}

int
cmd_sites_mainloop ()
{
  return cmd_mainloop (sites_command_matcher, "cdcd/sites> ", sites_cmds);
}

void
init_cmd_sites ()
{
  sort_commands (sites_cmds);
}
