{
  Copyright 2014-2019 Michalis Kamburelis.

  This file is part of "Castle Game Engine".

  "Castle Game Engine" is free software; see the file COPYING.txt,
  included in this distribution, for details about the copyright.
  Parts of this file are based on FPC packages/fpmkunit/src/fpmkunit.pp unit,
  which conveniently uses *exactly* the same license as Castle Game Engine.

  "Castle Game Engine" is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ----------------------------------------------------------------------------
}

{ Generating resources (res) files. }
unit ToolResources;

interface

uses CastleUtils, CastleStringUtils,
  ToolProject, ToolArchitectures;

{ Maybe make castle-auto-generated-resources.res (depends on platform).
  Returns if resource file was created. }
function MakeAutoGeneratedResources(const Project: TCastleProject;
  const FinalOutputResourcePath: string;
  const OS: TOS; const CPU: TCpu; const Plugin: Boolean): Boolean;

implementation

uses SysUtils,
  CastleURIUtils, CastleLog, CastleFilesUtils,
  ToolCommonUtils, ToolUtils, ToolCompilerInfo;

function MakeAutoGeneratedResources(const Project: TCastleProject;
  const FinalOutputResourcePath: string;
  const OS: TOS; const CPU: TCpu; const Plugin: Boolean): Boolean;
const
  RcTemplate: array [boolean { plugin? }] of string = (
    {$I ../embedded_templates/windows/castle-auto-generated-resources.rc.inc},
    {$I ../embedded_templates/windows/castle-plugin-auto-generated-resources.rc.inc}
  );
  RcName: array [boolean { plugin? }] of string = (
    'castle-auto-generated-resources.rc',
    'castle-plugin-auto-generated-resources.rc'
  );
  DeprecatedRcName: array [boolean { plugin? }] of string = (
    'automatic-windows-resources.rc',
    'plugin-automatic-windows-resources.rc'
  );
  ManifestTemplate = {$I ../embedded_templates/windows/castle-automatic-windows.manifest.inc};
var
  IcoPath, OutputRc, OutputManifest: string;
  WindresOutput, WindresExe, RcFilename, ManifestFilename, ResName: string;
  WindresStatus: Integer;
  OutputResourcesPath, FullIcoPath: string;
begin
  // For now, the .res files are only used on Windows
  Result := OS in AllWindowsOSes;
  if not Result then Exit;

  OutputResourcesPath := TempOutputPath(Project.Path) + 'windows' + PathDelim;
  CheckForceDirectories(OutputResourcesPath);

  OutputRc := Project.ReplaceMacros(RcTemplate[Plugin]);

  IcoPath := Project.Icons.FindExtension(['.ico']);
  FullIcoPath := CombinePaths(Project.Path, IcoPath);
  {$ifdef MSWINDOWS}
  { use only / on Windows, to avoid "unrecognized escape sequence" messages from windres }
  FullIcoPath := StringReplace(FullIcoPath, '\', '/', [rfReplaceAll]);
  {$endif}
  if IcoPath <> '' then
    OutputRc := 'MainIcon ICON "' + FullIcoPath + '"' + NL + OutputRc else
    WritelnWarning('Windows Resources', 'Icon in format suitable for Windows (.ico) not found. Exe file will not have icon.');

  RcFilename := OutputResourcesPath + RcName[Plugin];
  StringToFile(RcFilename, OutputRc);

  ManifestFilename := OutputResourcesPath + 'castle-automatic-windows.manifest';
  OutputManifest := Project.ReplaceMacros(ManifestTemplate);
  StringToFile(ManifestFilename, OutputManifest);

  // Using FindExeFpc, because on Windows windres is installed along with FPC
  WindresExe := FindExeFpc('windres');
  // Try alternative names of windres from Debian packages
  if WindresExe = '' then
    case CPU of
      i386  : WindresExe := FindExe('i586-mingw32msvc-windres');
      x86_64: WindresExe := FindExe('amd64-mingw32msvc-windres');
      else raise Exception.Create('CPU not supported on Windows');
    end;
  { try new names, https://packages.debian.org/search?searchon=contents&keywords=windres&mode=path&suite=stable&arch=any }
  if WindresExe = '' then
    case CPU of
      i386  : WindresExe := FindExe('i686-w64-mingw32-windres');
      x86_64: WindresExe := FindExe('x86_64-w64-mingw32-windres');
      else raise Exception.Create('CPU not supported on Windows');
    end;
  if WindresExe = '' then
    raise Exception.Create('Cannot find "windres" executable on $PATH. On Windows, it should be installed along with FPC (Free Pascal Compiler), so just make sure FPC is installed and available on $PATH. On Linux, "windres" is usually available as part of MinGW, so install the package named like "mingw*-binutils".');

  ResName := ChangeFileExt(RcName[Plugin], '.res');
  RunCommandIndirPassthrough(OutputResourcesPath, WindresExe,
    ['-i', RcName[Plugin], '-o', ResName],
    WindresOutput, WindresStatus);
  if WindresStatus <> 0 then
    raise Exception.Create('windres failed, cannot create Windows resource');

  CheckRenameFile(
    OutputResourcesPath + ResName,
    FinalOutputResourcePath + ResName);

  // copy the .res file to a deprecated name, only for backward compatibility
  CheckCopyFile(
    FinalOutputResourcePath + ResName,
    FinalOutputResourcePath + ChangeFileExt(DeprecatedRcName[Plugin], '.res'));

  WritelnVerbose('Generated ' + ResName + ', make sure you include it in your .lpr source file like this:');
  WritelnVerbose('  {$ifdef CASTLE_AUTO_GENERATED_RESOURCES} {$R ' + ResName + '} {$endif}');
end;

end.
