{
  Copyright 2002-2018 Michalis Kamburelis.

  This file is part of "Castle Game Engine".

  "Castle Game Engine" is free software; see the file COPYING.txt,
  included in this distribution, for details about the copyright.

  "Castle Game Engine" is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ----------------------------------------------------------------------------
}

{$ifdef read_interface}
  { Base for all nodes which specify texture coordinates. }
  TAbstractTextureCoordinateNode = class(TAbstractGeometricPropertyNode)
  public
    procedure CreateNode; override;

    {$I auto_generated_node_helpers/x3dnodes_x3dtexturecoordinatenode.inc}
  end;

  { Base for all nodes which specify a texture coordinate, but not MultiTextureCoordinate. }
  TAbstractSingleTextureCoordinateNode = class(TAbstractTextureCoordinateNode)
  public
    constructor Create(const AX3DName: string = ''; const ABaseUrl: string = ''); override;
    {$I auto_generated_node_helpers/x3dnodes_x3dsingletexturecoordinatenode.inc}
  end;

  { Base for all nodes which specify a texture, possibly multi-texture. }
  TAbstractTextureNode = class(TAbstractAppearanceChildNode)
  strict protected
    { Alpha channel detected from image contents (or children nodes),
      ignoring our FdAlphaChannel field.
      Actually set only when image data is loaded
      (e.g. TAbstractTexture2DNode.IsTextureLoaded in case of TAbstractTexture2DNode
      descendant). }
    function AlphaChannelData: TAlphaChannel; virtual;
  public
    procedure CreateNode; override;

    { Short description how texture is defined, is it inline or loaded
      from URL, is it video of simple image texture.
      "none" if it's not defined at all.

      Calling this @italic(may not) cause automatically loading
      the texture data (for exampe, from file in case of TAbstractTexture2DNode).
      So it cannot describe the actually loaded data.

      In this class, simply returns X3DType.
      Override to say something more descriptive. }
    function TextureDescription: string; virtual;

    { Alpha channel of the loaded texture data.
      Looks at loaded texture data, and at alphaChannel field
      (see https://castle-engine.io/x3d_extensions.php#section_ext_alpha_channel_detection ).
      In case of MultiTexture node, looks at children. }
    function AlphaChannelFinal: TAlphaChannel; virtual;
  end;

  { Base for all nodes which specify a texture, but not multi-texture. }
  TAbstractSingleTextureNode = class(TAbstractTextureNode)
  strict private
    function GetAlphaChannelField: TAutoAlphaChannel;
    procedure SetAlphaChannelField(const Value: TAutoAlphaChannel);
  public
    procedure CreateNode; override;

    strict private FFdEffects: TMFNode;
    public property FdEffects: TMFNode read FFdEffects;

    {$ifndef CASTLE_SLIM_NODES}
    strict private FFdCrossOrigin: TSFString;
    public property FdCrossOrigin: TSFString read FFdCrossOrigin;
    {$endif}

    strict private FFdAlphaChannel: TSFStringEnum;
    public property FdAlphaChannel: TSFStringEnum read FFdAlphaChannel;
    property AlphaChannelField: TAutoAlphaChannel read GetAlphaChannelField write SetAlphaChannelField;

    function AlphaChannelFinal: TAlphaChannel; override;

    {$I auto_generated_node_helpers/x3dnodes_x3dsingletexturenode.inc}
  end;

  { VRML/X3D texture that is 2D and can be loaded (from file or some other stream).

    For X3D, this descends from X3DTextureNode and is an ancestor
    for X3DTexture2DNode, so X3D hierarchy is nicely preserved. }
  TAbstractTexture2DNode = class(TAbstractSingleTextureNode)
  private
    { Together we call FTextureImage, FTextureComposite,
      FTextureVideo, FAlphaChannelData as "texture data". }

    { FTextureImage is <> nil if texture is currently loaded (IsTextureLoaded)
      and it was loaded to an image (not video).

      Note that this may still have zero size (IsEmpty = @true),
      IsTextureImage checks this also. }
    FTextureImage: TEncodedImage;

    { Only if FTextureImage is <> nil, then FTextureComposite may also be <> nil,
      it this image is part of Composite file. }
    FTextureComposite: TCompositeImage;

    { Analogous to FTextureImage, this is the loaded video file.
      Assigned here, should always have TVideo.Loaded = @true. }
    FTextureVideo: TVideo;

    FAlphaChannelData: TAlphaChannel;

    { Non-nil only if FTextureImage, FTextureComposite or FTextueVideo
      should be freed using
      UsedCache (TextureImage_DecReference or Video_DecReference).
      Also loaded FTextureVideo should always have it's own Cache property set
      to this. }
    UsedCache: TTexturesVideosCache;

    FIsTextureLoaded: boolean;
    procedure SetIsTextureLoaded(Value: boolean);
    procedure FreeAndNilTextureData;
  strict protected
    FTextureUsedFullUrl: string;

    { Loads texture data (image or video file).

      It should set either FTextureImage or FTextureVideo to something non-nil
      (don't care here about the previous value of these fields --- it's for
      sure @nil). If FTextureImage is set, you can also set FTextureComposite.
      If you leave them as @nil, this means that loading failed
      (and WritelnWarning didn't cause an exception).
      You should also set FAlphaChannelData.

      You do not care in this method about things like
      IsImageLoaded --- this method should just always,
      unconditionally, make everything it can do to load texture data from
      file(s).

      You can use WritelnWarning inside,
      so we're prepared that this may even exit with exception
      (since WritelnWarning can raise exception).

      If you set FTextureImage, you have to set it to something
      returned by LoadTextureImage. See TextureImage docs.

      Set WasCacheUsed here. @true means you loaded the data using Cache.
      For FTextureVideo, it's Cache property should also be set to
      our Cache, this happens automatically in Video_IncReference.

      Also, set FTextureUsedFullUrl here.

      In this class, this simply produces WritelnWarning with
      "not implemented" message and returns @nil. It's not declared
      as abstract, because there may be classes descending from this,
      and we want to at least be able to parse them
      and then ignore (while not overriden abstract method would cause
      simple crashes). }
    procedure LoadTextureData(out WasCacheUsed: boolean); virtual;

    function GetRepeatS: boolean; virtual; abstract;
    function GetRepeatT: boolean; virtual; abstract;
    procedure SetRepeatS(const Value: boolean); virtual; abstract;
    procedure SetRepeatT(const Value: boolean); virtual; abstract;

    function AlphaChannelData: TAlphaChannel; override;
  public
    procedure CreateNode; override;
    destructor Destroy; override;

    { TextureImage, TextureComposite and TextureVideo contain actual texture data.
      TextureImage may come from inline VRML texture or could be loaded
      from file (including from some URL), this doesn't concern you here.

      Calls to TextureImage, TextureVideo, IsTextureImage, IsTextureVideo,
      TextureComposite
      will automatically load the data, so in simple situations you really
      don't need to do anything. Just check and use them when you want,
      and things will just work. See IsTextureLoaded for more control
      about loading / unloading.

      Note that either one of TextureImage or TextureVideo may be @nil,
      if the other one is loaded. Or when loading failed
      (warning will be reported by WritelnWarning).
      IsTextureImage checks that TextureImage is non-nil (so it's successfully
      loaded) and additionally that texture size is not zero.
      Similar for IsTextureVideo.

      Note that when image failed to load, or URL was just empty,
      we can have IsTextureLoaded = @true but both IsTextureImage = @false
      and IsTextureVideo = @false.

      TextureImage may have any class allowed by LoadTextureImage.

      @groupBegin }
    function TextureImage: TEncodedImage;
    function IsTextureImage: boolean;
    function TextureComposite: TCompositeImage;
    function TextureVideo: TVideo;
    function IsTextureVideo: boolean;
    { @groupEnd }

    { Is the texture data already loaded.
      Since the texture will be loaded automatically, you're usually
      not interested in this property. You can read it to e.g. predict
      if next TextureImage / TextureVideo call may take a long time.
      (You know that if IsTextureLoaded = @true then TextureImage
      just returns ready image instantly).

      You can also set IsTextureLoaded.
      Setting to @true means that you request the texture to be loaded @italic(now),
      if it's not loaded already. Setting to @false may be useful if you want
      to release resources (e.g. when you want to keep TTextureNode instance
      loaded but you know that you will not need
      TextureImage / TextureComposite / TextureVideo anymore).
      You can also set it to @false and then back to @true if you want to
      request reloading the texture from URL (e.g. if you suspect that
      the URL contents changed).

      Note that IsTextureLoaded is set to @true, even if actual loading
      failed. You still have to check afterwards IsTextureImage and
      IsTextureVideo to know if loading was actually successful.
      This is deliberate --- it means that each call to TextureImage etc.
      will not unnecessarily read the disk (or even connect to internet)
      when the file does not exist. Also, the loading errors reported
      by WritelnWarning will not be repeated --- they will
      occur only once, when IsTextureLoaded changes from @false to @true. }
    property IsTextureLoaded: boolean
      read FIsTextureLoaded write SetIsTextureLoaded;

    { Whether the texture repeats or clamps in given direction.
      Getting or setting this is the most comfortable way to change underlying
      node fields, setting this automatically does all necessary (sends events
      etc., see TVRMLField.Send).
      @groupBegin }
    property RepeatS: boolean read GetRepeatS write SetRepeatS;
    property RepeatT: boolean read GetRepeatT write SetRepeatT;
    { @groupEnd }

    { Once the texture data (image or video) is loaded,
      this is set to the URL that was used to load,
      or '' if no URL was used. "No URL was used" may mean that
      no URL was valid, or inlined image was used.

      This is always a full, expanded (i.e. not relative) URL.

      In case of data: URLs, this doesn't contain actual data (it would
      be too long then, and TextureUsedFullUrl is mainly for showing to the
      user), it's cutted. }
    property TextureUsedFullUrl: string read FTextureUsedFullUrl;

    strict private FFdTextureProperties: TSFNode;
    public property FdTextureProperties: TSFNode read FFdTextureProperties;

    {$I auto_generated_node_helpers/x3dnodes_x3dtexture2dnode.inc}
  end;

  { X3D texture that is 2D and can be loaded (from file or some other stream). }
  TAbstractX3DTexture2DNode = class(TAbstractTexture2DNode)
  strict protected
    function GetRepeatS: boolean; override;
    function GetRepeatT: boolean; override;
    procedure SetRepeatS(const Value: boolean); override;
    procedure SetRepeatT(const Value: boolean); override;
  public
    procedure CreateNode; override;

    strict private FFdRepeatS: TSFBool;
    public property FdRepeatS: TSFBool read FFdRepeatS;

    strict private FFdRepeatT: TSFBool;
    public property FdRepeatT: TSFBool read FFdRepeatT;
  end;

  { Base for all nodes which specify a transformation of texture coordinates. }
  TAbstractTextureTransformNode = class(TAbstractAppearanceChildNode)
  public
    procedure CreateNode; override;
    function TransformMatrix: TMatrix4; virtual; abstract;
    {$I auto_generated_node_helpers/x3dnodes_x3dtexturetransformnode.inc}
  end;

  { Base for all nodes which specify a texture coordinate transform, but not MultiTextureTransform. }
  TAbstractSingleTextureTransformNode = class(TAbstractTextureTransformNode)
  public
    constructor Create(const AX3DName: string = ''; const ABaseUrl: string = ''); override;
    {$I auto_generated_node_helpers/x3dnodes_x3dsingletexturetransformnode.inc}
  end;

  { Texture image loaded from a file. }
  TImageTextureNode = class(TAbstractX3DTexture2DNode, IAbstractUrlObject)
  strict protected
    procedure LoadTextureData(out WasCacheUsed: boolean); override;
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdUrl: TMFString;
    public property FdUrl: TMFString read FFdUrl;

    strict private FFdFlipVertically: TSFBool;
    public property FdFlipVertically: TSFBool read FFdFlipVertically;

    function TextureDescription: string; override;

    {$I auto_generated_node_helpers/x3dnodes_imagetexture.inc}
  end;

  { Movie file, that can be played and displayed as a texture. }
  TMovieTextureNode = class(TAbstractX3DTexture2DNode, IAbstractSoundSourceNode,
    IAbstractUrlObject, IAbstractTimeDependentNode)
  strict private
    FDuration: TFloatTime;
    FInternalTimeDependentHandler: TInternalTimeDependentHandler;
    function GetCycleInterval: TFloatTime;
  private
    function GetInternalTimeDependentHandler: TInternalTimeDependentHandler;
  strict protected
    procedure LoadTextureData(out WasCacheUsed: boolean); override;
  public
    procedure CreateNode; override;
    destructor Destroy; override;

    class function ClassX3DType: string; override;

    strict private FFdDescription: TSFString;
    public property FdDescription: TSFString read FFdDescription;

    strict private FFdLoop: TSFBool;
    public property FdLoop: TSFBool read FFdLoop;

    strict private FFdPauseTime: TSFTime;
    public property FdPauseTime: TSFTime read FFdPauseTime;

    strict private FFdResumeTime: TSFTime;
    public property FdResumeTime: TSFTime read FFdResumeTime;

    strict private FFdSpeed: TSFFloat;
    public property FdSpeed: TSFFloat read FFdSpeed;

    strict private FFdStartTime: TSFTime;
    public property FdStartTime: TSFTime read FFdStartTime;

    strict private FFdStopTime: TSFTime;
    public property FdStopTime: TSFTime read FFdStopTime;

    strict private FFdUrl: TMFString;
    public property FdUrl: TMFString read FFdUrl;

    strict private FFdFlipVertically: TSFBool;
    public property FdFlipVertically: TSFBool read FFdFlipVertically;

    { Event out } { }
    strict private FEventDuration_changed: TSFTimeEvent;
    public property EventDuration_changed: TSFTimeEvent read FEventDuration_changed;

    { Event out } { }
    strict private FEventElapsedTime: TSFTimeEvent;
    public property EventElapsedTime: TSFTimeEvent read FEventElapsedTime;

    { Event out } { }
    strict private FEventIsActive: TSFBoolEvent;
    public property EventIsActive: TSFBoolEvent read FEventIsActive;

    { Event out } { }
    strict private FEventIsPaused: TSFBoolEvent;
    public property EventIsPaused: TSFBoolEvent read FEventIsPaused;

    function TextureDescription: string; override;

    { Internal time-dependent logic handler. }
    property InternalTimeDependentHandler: TInternalTimeDependentHandler
      read FInternalTimeDependentHandler;

    { Duration for this time-dependent node.
      Duration is initialized from loaded video length (default is -1).
      So it's automatically initialized when you call IsTextureVideo,
      TextureVideo methods.

      cycleInterval is just set to duration scaled by 1/Abs(speed),
      like required by X3D spec.

      Duration (and so, also cycleInterval) is not reset when video
      is freed (like when you set
      IsTextureLoaded to @false, maybe implicitly by calling
      TCastleSceneCore.FreeResources with frTextureDataInNodes).
      This way this is available even you freed the texture video data to
      save memory. }
    property Duration: TFloatTime read FDuration;

    function IsActive: boolean;
    function IsPaused: boolean;
    function ElapsedTime: TFloatTime;
    function ElapsedTimeInCycle: TFloatTime;

    {$I auto_generated_node_helpers/x3dnodes_movietexture.inc}
  end;

  { Application of several individual textures on top of each other,
    used instead of a single texture when desired. }
  TMultiTextureNode = class(TAbstractTextureNode)
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdAlpha: TSFFloat;
    public property FdAlpha: TSFFloat read FFdAlpha;

    strict private FFdColor: TSFColor;
    public property FdColor: TSFColor read FFdColor;

    strict private FFdFunction: TMFString;
    public property FdFunction: TMFString read FFdFunction;

    strict private FFdMode: TMFString;
    public property FdMode: TMFString read FFdMode;

    strict private FFdSource: TMFString;
    public property FdSource: TMFString read FFdSource;

    strict private FFdTexture: TMFNode;
    public property FdTexture: TMFNode read FFdTexture;

    function AlphaChannelData: TAlphaChannel; override;

    {$I auto_generated_node_helpers/x3dnodes_multitexture.inc}
  end;

  { Multiple texture coordinates per vertex,
    to be used with multi-texturing by @link(TMultiTextureNode). }
  TMultiTextureCoordinateNode = class(TAbstractTextureCoordinateNode)
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdTexCoord: TMFNode;
    public property FdTexCoord: TMFNode read FFdTexCoord;

    {$I auto_generated_node_helpers/x3dnodes_multitexturecoordinate.inc}
  end;

  { Multiple texture transforms,
    to be used with multi-texturing by @link(TMultiTextureNode). }
  TMultiTextureTransformNode = class(TAbstractTextureTransformNode)
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdTextureTransform: TMFNode;
    public property FdTextureTransform: TMFNode read FFdTextureTransform;

    { For MultiTextureTransform, this always raises an internal error.
      Reason: you cannot get single texture transform matrix from
      MultiTextureTransform.

      @raises(EInternalError Always, since this method has no sense
        for MultiTextureTransform.) }
    function TransformMatrix: TMatrix4; override;

    {$I auto_generated_node_helpers/x3dnodes_multitexturetransform.inc}
  end;

  { Texture specified as an explicit array of pixel values (see @link(FdImage) field).

    This is useful when authoring X3D, as you can place texture contents directly
    inside the X3D file (without the need for any external file).

    PixelTexture is also comfortable when using Object Pascal to construct
    the X3D graph. In this case, it allows to provide an image as a ready
    TCastleImage instance, by changing the value of the
    @code(TPixelTextrueNode.FdImage.Value).

    An alternative to PixelTexture is to use "data URI".
    With "data URI", you can embed any file contents inside an URL.
    This way, you can use @link(TImageTextureNode ImageTexture),
    and inside @code(ImageTexture.url) field
    you can place an embedded image contents (instead of a normal URL or filename).
    To convert your image file to a "data URI", you can our tool "to_data_uri".
    The image contents stay compressed this way (although expressed in base64),
    so it may be more compact than PixelTexture.
  }
  TPixelTextureNode = class(TAbstractX3DTexture2DNode)
  strict protected
    procedure LoadTextureData(out WasCacheUsed: boolean); override;
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdImage: TSFImage;
    public property FdImage: TSFImage read FFdImage;

    function TextureDescription: string; override;

    {$I auto_generated_node_helpers/x3dnodes_pixeltexture.inc}
  end;

  { 2D texture coordinates used by vertex-based geometry nodes. }
  TTextureCoordinateNode = class(TAbstractSingleTextureCoordinateNode)
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdPoint: TMFVec2f;
    public property FdPoint: TMFVec2f read FFdPoint;

    {$I auto_generated_node_helpers/x3dnodes_texturecoordinate.inc}
  end;

  { Automatic generation of texture coordinates. }
  TTextureCoordinateGeneratorNode = class(TAbstractSingleTextureCoordinateNode)
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdMode: TSFString;
    public property FdMode: TSFString read FFdMode;

    strict private FFdParameter: TMFFloat;
    public property FdParameter: TMFFloat read FFdParameter;

    strict private FFdProjectedLight: TSFNode;
    public property FdProjectedLight: TSFNode read FFdProjectedLight;

    {$I auto_generated_node_helpers/x3dnodes_texturecoordinategenerator.inc}
  end;

  { Old BS Contact name for TextureCoordinateGenerator.
    See examples from [http://www.bitmanagement.de/developer/contact/relnotes6.html] }
  TTextureCoordGenNode = class(TTextureCoordinateGeneratorNode)
  public
    class function ClassX3DType: string; override;
  end;

  { Adjust the texture application properties. }
  TTexturePropertiesNode = class(TAbstractNode)
  strict private
    function GetMinificationFilter: TAutoMinificationFilter;
    procedure SetMinificationFilter(const Value: TAutoMinificationFilter);
    function GetMagnificationFilter: TAutoMagnificationFilter;
    procedure SetMagnificationFilter(const Value: TAutoMagnificationFilter);
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdAnisotropicDegree: TSFFloat;
    public property FdAnisotropicDegree: TSFFloat read FFdAnisotropicDegree;

    strict private FFdBorderColor: TSFColorRGBA;
    public property FdBorderColor: TSFColorRGBA read FFdBorderColor;

    strict private FFdBorderWidth: TSFInt32;
    public property FdBorderWidth: TSFInt32 read FFdBorderWidth;

    strict private FFdBoundaryModeS: TSFString;
    public property FdBoundaryModeS: TSFString read FFdBoundaryModeS;

    strict private FFdBoundaryModeT: TSFString;
    public property FdBoundaryModeT: TSFString read FFdBoundaryModeT;

    strict private FFdBoundaryModeR: TSFString;
    public property FdBoundaryModeR: TSFString read FFdBoundaryModeR;

    strict private FFdMagnificationFilter: TSFString;
    public property FdMagnificationFilter: TSFString read FFdMagnificationFilter;

    strict private FFdMinificationFilter: TSFString;
    public property FdMinificationFilter: TSFString read FFdMinificationFilter;

    strict private FFdTextureCompression: TSFString;
    public property FdTextureCompression: TSFString read FFdTextureCompression;

    strict private FFdTexturePriority: TSFFloat;
    public property FdTexturePriority: TSFFloat read FFdTexturePriority;

    strict private FFdGenerateMipMaps: TSFBool;
    public property FdGenerateMipMaps: TSFBool read FFdGenerateMipMaps;

    strict private FFdGUITexture: TSFBool;
    public property FdGUITexture: TSFBool read FFdGUITexture;

    property MagnificationFilter: TAutoMagnificationFilter
      read GetMagnificationFilter write SetMagnificationFilter;

    property MinificationFilter: TAutoMinificationFilter
      read GetMinificationFilter write SetMinificationFilter;

    {$I auto_generated_node_helpers/x3dnodes_textureproperties.inc}
  end;

  { 2D transformation that can be applied to texture coordinates (e.g. from
    @link(TTextureCoordinateNode). }
  TTextureTransformNode = class(TAbstractSingleTextureTransformNode)
  public
    procedure CreateNode; override;
    class function ClassX3DType: string; override;

    strict private FFdCenter: TSFVec2f;
    public property FdCenter: TSFVec2f read FFdCenter;

    strict private FFdRotation: TSFFloat;
    public property FdRotation: TSFFloat read FFdRotation;

    strict private FFdScale: TSFVec2f;
    public property FdScale: TSFVec2f read FFdScale;

    strict private FFdTranslation: TSFVec2f;
    public property FdTranslation: TSFVec2f read FFdTranslation;

    function TransformMatrix: TMatrix4; override;

    {$I auto_generated_node_helpers/x3dnodes_texturetransform.inc}
  end;

{$endif read_interface}

{$ifdef read_implementation}

{ TAbstractSingleTextureCoordinateNode --------------------------------------- }

constructor TAbstractSingleTextureCoordinateNode.Create(const AX3DName: string; const ABaseUrl: string);
begin
  inherited;
  FdMapping.ChangeAlways := chTextureCoordinate;
end;

{ TAbstractTextureCoordinateNode ---------------------------------------------- }

procedure TAbstractTextureCoordinateNode.CreateNode;
begin
  inherited;

  { X3D XML spec doesn't specify containerField for abstract X3D classes.
    texCoord seems most sensible for this case. }
  DefaultContainerField := 'texCoord';
end;

{ TAbstractTextureNode ------------------------------------------------------- }

procedure TAbstractTextureNode.CreateNode;
begin
  inherited;
  DefaultContainerField := 'texture';
end;

function TAbstractTextureNode.AlphaChannelData: TAlphaChannel;
begin
  Result := acNone;
end;

function TAbstractTextureNode.AlphaChannelFinal: TAlphaChannel;
begin
  Result := AlphaChannelData;
end;

function TAbstractTextureNode.TextureDescription: string;
begin
  Result := X3DType;
end;

{ TAbstractSingleTextureNode -------------------------------------------------------- }

procedure TAbstractSingleTextureNode.CreateNode;
begin
  inherited;

  FFdEffects := TMFNode.Create(Self, false, 'effects', [TEffectNode]);
   FdEffects.ChangeAlways := chEverything;
  AddField(FFdEffects);

  {$ifndef CASTLE_SLIM_NODES}
  FFdCrossOrigin := TSFString.Create(Self, false, 'crossOrigin', '');
   FdCrossOrigin.ChangeAlways := chNone;
  AddField(FFdCrossOrigin);
  {$endif}

  FFdAlphaChannel := TSFStringEnum.Create(Self, false, 'alphaChannel', AlphaToString, Ord(acAuto));
   FdAlphaChannel.ChangeAlways := chVisibleVRML1State;
  AddField(FFdAlphaChannel);
end;

function TAbstractSingleTextureNode.GetAlphaChannelField: TAutoAlphaChannel;
begin
  Result := TAutoAlphaChannel(FdAlphaChannel.EnumValue);
end;

procedure TAbstractSingleTextureNode.SetAlphaChannelField(const Value: TAutoAlphaChannel);
begin
  FdAlphaChannel.SendEnumValue(Ord(Value));
end;

function TAbstractSingleTextureNode.AlphaChannelFinal: TAlphaChannel;
var
  Res: TAutoAlphaChannel;
begin
  Res := AlphaChannelField;
  if Res = acAuto then
    Result := AlphaChannelData else
    Result := Res;
end;

{ TAbstractTexture2DNode ----------------------------------------------------------- }

procedure TAbstractTexture2DNode.CreateNode;
begin
  inherited;

  UsedCache := nil;
  FIsTextureLoaded := false;

  FFdTextureProperties := TSFNode.Create(Self, false, 'textureProperties', [TTexturePropertiesNode]);
   FdTextureProperties.ChangeAlways := chTextureRendererProperties;
  AddField(FFdTextureProperties);
end;

destructor TAbstractTexture2DNode.Destroy;
begin
  FreeAndNilTextureData;
  inherited;
end;

procedure TAbstractTexture2DNode.FreeAndNilTextureData;
begin
  if FTextureImage <> nil then
  begin
    if UsedCache <> nil then
    begin
      UsedCache.TextureImage_DecReference(FTextureImage, FTextureComposite);
      UsedCache := nil;
    end else
    begin
      FreeAndNil(FTextureImage);
      FreeAndNil(FTextureComposite);
    end;
  end;

  if FTextureVideo <> nil then
  begin
    if UsedCache <> nil then
    begin
      UsedCache.Video_DecReference(FTextureVideo);
      UsedCache := nil;
    end else
      FreeAndNil(FTextureVideo);
  end;
end;

function TAbstractTexture2DNode.TextureImage: TEncodedImage;
begin
  { Setting IsTextureLoaded property will initialize FTextureImage. }
  IsTextureLoaded := true;

  Result := FTextureImage;
end;

function TAbstractTexture2DNode.IsTextureImage: boolean;
begin
  Result := (TextureImage <> nil) and (not TextureImage.IsEmpty);
end;

function TAbstractTexture2DNode.TextureComposite: TCompositeImage;
begin
  { Setting IsTextureLoaded property will initialize
    FTextureImage, FTextureComposite. }
  IsTextureLoaded := true;

  Result := FTextureComposite;
end;

function TAbstractTexture2DNode.TextureVideo: TVideo;
begin
  { Setting IsTextureLoaded property will initialize FTextureVideo. }
  IsTextureLoaded := true;

  Result := FTextureVideo;
end;

function TAbstractTexture2DNode.IsTextureVideo: boolean;
begin
  Result := (TextureVideo <> nil) and
    (TextureVideo.Width <> 0) and
    (TextureVideo.Height <> 0);
end;

procedure TAbstractTexture2DNode.SetIsTextureLoaded(Value: boolean);

  procedure DoLoadTexture;
  var
    WasCacheUsed: boolean;
  begin
    FreeAndNilTextureData;

    LoadTextureData(WasCacheUsed);
    if WasCacheUsed then
      UsedCache := X3DCache;
  end;

begin
  if Value <> FIsTextureLoaded then
  begin
    if Value then
    begin
      { actually load the texture }
      DoLoadTexture;
    end else
    begin
      { unload the texture }
      FreeAndNilTextureData;
    end;

    FIsTextureLoaded := Value;
  end;
end;

procedure TAbstractTexture2DNode.LoadTextureData(out WasCacheUsed: boolean);
begin
  WasCacheUsed := false;
  FTextureUsedFullUrl := '';

  WritelnWarning('VRML/X3D', Format('Loading textures from "%s" node not implemented', [X3DType]));
end;

function TAbstractTexture2DNode.AlphaChannelData: TAlphaChannel;
begin
  Result := FAlphaChannelData;
end;

{ TAbstractX3DTexture2DNode ------------------------------------------------------ }

procedure TAbstractX3DTexture2DNode.CreateNode;
begin
  inherited;

  FFdRepeatS := TSFBool.Create(Self, false, 'repeatS', true);
   FdRepeatS.ChangeAlways := chTextureRendererProperties;
  AddField(FFdRepeatS);

  FFdRepeatT := TSFBool.Create(Self, false, 'repeatT', true);
   FdRepeatT.ChangeAlways := chTextureRendererProperties;
  AddField(FFdRepeatT);
end;

function TAbstractX3DTexture2DNode.GetRepeatS: boolean;
begin
  Result := FdRepeatS.Value;
end;

function TAbstractX3DTexture2DNode.GetRepeatT: boolean;
begin
  Result := FdRepeatT.Value;
end;

procedure TAbstractX3DTexture2DNode.SetRepeatS(const Value: boolean);
begin
  FdRepeatS.Send(Value);
end;

procedure TAbstractX3DTexture2DNode.SetRepeatT(const Value: boolean);
begin
  FdRepeatT.Send(Value);
end;

procedure TAbstractTextureTransformNode.CreateNode;
begin
  inherited;

  DefaultContainerField := 'textureTransform';
end;

{ TAbstractSingleTextureTransformNode --------------------------------------- }

constructor TAbstractSingleTextureTransformNode.Create(const AX3DName: string; const ABaseUrl: string);
begin
  inherited;
  FdMapping.ChangeAlways := chTextureTransform; // TODO is this enough? chTextureTransform will not free proxy?
end;

{ TImageTextureNode ---------------------------------------------------------- }

procedure TImageTextureNode.CreateNode;
begin
  inherited;

  FFdUrl := TMFString.Create(Self, true, 'url', []);
   FdUrl.ChangeAlways := chTextureImage;
  AddField(FFdUrl);
  { X3D specification comment: [URI] }

  FFdFlipVertically := TSFBool.Create(Self, false, 'flipVertically', false);
  AddField(FFdFlipVertically);
end;

class function TImageTextureNode.ClassX3DType: string;
begin
  Result := 'ImageTexture';
end;

procedure TImageTextureNode.LoadTextureData(out WasCacheUsed: boolean);

  function LoadOptions: TLoadImageOptions;
  begin
    Result := [];
    if FlipVertically then
      Include(Result, liFlipVertically);
  end;

var
  I: Integer;
  FullUrl, ErrorMessage: string;
  Success: boolean;
begin
  WasCacheUsed := false;
  FTextureUsedFullUrl := '';
  Success := false;
  ErrorMessage := '';

  for I := 0 to FdUrl.Count - 1 do
    if FdUrl.Items[I] = '' then
    begin
      { Special error message in this case, otherwise empty URL
        would be expanded by PathFromBaseUrl to directory
        and produce unclear error message. }
      ErrorMessage := ErrorMessage + NL + '  Empty URL ignored';
    end else
    begin
      FullUrl := PathFromBaseUrl(FdUrl.Items[I]);
      try
        FTextureImage := X3DCache.TextureImage_IncReference(
          FullUrl, FTextureComposite, FAlphaChannelData, LoadOptions);
        WasCacheUsed := true;
        if WarnAboutAbsoluteFilenames and
           AbsoluteFileURI(FdUrl.Items[I]) and
           { AbsoluteFileURI mistakes Blender relative paths '//xxx' as absolute,
             so eliminate them below. }
           not BlenderRelativePath(FdUrl.Items[I]) then
          WritelnWarning('VRML/X3D', Format('Loaded data from an absolute filename "%s", this makes the data possibly unportable (it will probably not work on other systems/locations). Always use relative paths.',
            [FdUrl.Items[I]]));
        FTextureUsedFullUrl := URIDisplay(FullUrl);
        Success := true;
        Break;
      except
        on E: Exception do
          { Silence the Exception, continue trying to load more textures. }
          ErrorMessage := ErrorMessage + NL + '  ' + Format(SLoadError,
            [E.ClassName, 'texture', URIDisplay(FullUrl), E.Message]);
      end;
    end;

  if ErrorMessage <> '' then
  begin
    if Success then
      WritelnLog('Texture', 'Some of the texture URLs failed to load, although we found one URL that succeeded:' +
        ErrorMessage)
    else
      WritelnWarning('Texture', 'Cannot load the texture:' + ErrorMessage);
  end;
  { Note that when Success = false but ErrorMessage = '', then we're quiet
    about it. This happens when Urls.Count = 0. }
end;

function TImageTextureNode.TextureDescription: string;
begin
  if TextureUsedFullUrl <> '' then
    Result := 'image from file "' + TextureUsedFullUrl + '"' else
    Result := 'none';
end;

procedure TMovieTextureNode.CreateNode;
begin
  inherited;

  FFdDescription := TSFString.Create(Self, true, 'description', '');
  AddField(FFdDescription);

  FFdLoop := TSFBool.Create(Self, true, 'loop', false);
  AddField(FFdLoop);

  FFdPauseTime := TSFTime.Create(Self, true, 'pauseTime', 0);
   FdPauseTime.ChangeAlways := chTimeStopStart;
  AddField(FFdPauseTime);
  { X3D specification comment: (-Inf,Inf) }

  FFdResumeTime := TSFTime.Create(Self, true, 'resumeTime', 0);
   FdResumeTime.ChangeAlways := chTimeStopStart;
  AddField(FFdResumeTime);
  { X3D specification comment: (-Inf,Inf) }

  FFdSpeed := TSFFloat.Create(Self, true, 'speed', 1.0);
  AddField(FFdSpeed);
  { X3D specification comment: (-Inf,Inf) }

  FFdStartTime := TSFTimeIgnoreWhenActive.Create(Self, true, 'startTime', 0);
   FdStartTime.ChangeAlways := chTimeStopStart;
  AddField(FFdStartTime);
  { X3D specification comment: (-Inf,Inf) }

  FFdStopTime := TSFStopTime.Create(Self, true, 'stopTime', 0);
   FdStopTime.ChangeAlways := chTimeStopStart;
  AddField(FFdStopTime);
  { X3D specification comment: (-Inf,Inf) }

  FFdUrl := TMFString.Create(Self, true, 'url', []);
   FdUrl.ChangeAlways := chTextureImage;
  AddField(FFdUrl);
  { X3D specification comment: [URI] }

  FFdFlipVertically := TSFBool.Create(Self, false, 'flipVertically', false);
  AddField(FFdFlipVertically);

  FEventDuration_changed := TSFTimeEvent.Create(Self, 'duration_changed', false);
  AddEvent(FEventDuration_changed);

  FEventElapsedTime := TSFTimeEvent.Create(Self, 'elapsedTime', false);
  AddEvent(FEventElapsedTime);

  FEventIsActive := TSFBoolEvent.Create(Self, 'isActive', false);
  AddEvent(FEventIsActive);

  FEventIsPaused := TSFBoolEvent.Create(Self, 'isPaused', false);
  AddEvent(FEventIsPaused);

  FDuration := -1;

  FInternalTimeDependentHandler := TInternalTimeDependentHandler.Create;
  FInternalTimeDependentHandler.Node := Self;
  FInternalTimeDependentHandler.Fdloop := FdLoop;
  FInternalTimeDependentHandler.FdpauseTime := FdPauseTime;
  FInternalTimeDependentHandler.FdresumeTime := FdResumeTime;
  FInternalTimeDependentHandler.FdstartTime := FdStartTime;
  FInternalTimeDependentHandler.FdstopTime := FdStopTime;
  FInternalTimeDependentHandler.EventisActive:= EventisActive;
  FInternalTimeDependentHandler.EventisPaused := EventisPaused;
  FInternalTimeDependentHandler.EventelapsedTime := EventelapsedTime;
  FInternalTimeDependentHandler.OnCycleInterval :=
    {$ifdef CASTLE_OBJFPC}@{$endif} GetCycleInterval;
end;

function TMovieTextureNode.GetInternalTimeDependentHandler: TInternalTimeDependentHandler;
begin
  Result := FInternalTimeDependentHandler;
end;

destructor TMovieTextureNode.Destroy;
begin
  FreeAndNil(FInternalTimeDependentHandler);
  inherited;
end;

class function TMovieTextureNode.ClassX3DType: string;
begin
  Result := 'MovieTexture';
end;

procedure TMovieTextureNode.LoadTextureData(out WasCacheUsed: boolean);

  function LoadOptions: TLoadImageOptions;
  begin
    Result := [];
    if FlipVertically then
      Include(Result, liFlipVertically);
  end;

var
  I: Integer;
  FullUrl: string;
begin
  WasCacheUsed := true;
  FTextureUsedFullUrl := '';

  for I := 0 to FdUrl.Count - 1 do
  begin
    FullUrl := PathFromBaseUrl(FdUrl.Items[I]);
    try
      FTextureVideo := X3DCache.Video_IncReference(
        FullUrl, FAlphaChannelData, LoadOptions);

      { if loading succeded, set WasCacheUsed and others and break. }
      WasCacheUsed := true;
      FTextureUsedFullUrl := FullUrl;
      FDuration := FTextureVideo.TimeDuration;
      if Scene <> nil then
        EventDuration_Changed.Send(FDuration, Scene.NextEventTime);
      Break;
    except
      on E: Exception do
        { Remember that WritelnWarning *may* raise an exception. }
        WritelnWarning('Video', Format(SLoadError,
          [E.ClassName, 'video', URIDisplay(FullUrl), E.Message]));
    end;
  end;
end;

function TMovieTextureNode.GetCycleInterval: TFloatTime;
var
  Sp: Single;
begin
  Sp := Speed;
  if Sp <> 0 then
    Result := FDuration / Abs(Sp)
  else
    Result := 0;
end;

function TMovieTextureNode.IsActive: boolean;
begin
  Result := InternalTimeDependentHandler.IsActive;
end;

function TMovieTextureNode.IsPaused: boolean;
begin
  Result := InternalTimeDependentHandler.IsPaused;
end;

function TMovieTextureNode.ElapsedTime: TFloatTime;
begin
  Result := InternalTimeDependentHandler.ElapsedTime;
end;

function TMovieTextureNode.ElapsedTimeInCycle: TFloatTime;
begin
  Result := InternalTimeDependentHandler.ElapsedTimeInCycle;
end;

function TMovieTextureNode.TextureDescription: string;
begin
  if TextureUsedFullUrl <> '' then
    Result := 'video from file "' + TextureUsedFullUrl + '"' else
    Result := 'none';
end;

procedure TMultiTextureNode.CreateNode;
begin
  inherited;

  FFdAlpha := TSFFloat.Create(Self, true, 'alpha', 1);
   FdAlpha.ChangeAlways := chTextureRendererProperties;
  AddField(FFdAlpha);
  { X3D specification comment: [0,1] }

  FFdColor := TSFColor.Create(Self, true, 'color', Vector3(1, 1, 1));
   FdColor.ChangeAlways := chTextureRendererProperties;
  AddField(FFdColor);
  { X3D specification comment: [0,1] }

  FFdFunction := TMFString.Create(Self, true, 'function', []);
   FdFunction.ChangeAlways := chTextureRendererProperties;
  AddField(FFdFunction);

  FFdMode := TMFString.Create(Self, true, 'mode', []);
   FdMode.ChangeAlways := chTextureRendererProperties;
  AddField(FFdMode);

  FFdSource := TMFString.Create(Self, true, 'source', []);
   FdSource.ChangeAlways := chTextureRendererProperties;
  AddField(FFdSource);

  FFdTexture := TMFNode.Create(Self, true, 'texture', [TAbstractSingleTextureNode]);
   FdTexture.ChangeAlways := chTextureRendererProperties;
  AddField(FFdTexture);
end;

class function TMultiTextureNode.ClassX3DType: string;
begin
  Result := 'MultiTexture';
end;

function TMultiTextureNode.AlphaChannelData: TAlphaChannel;
var
  ChildTex: TX3DNode;
  I: Integer;
begin
  Result := acNone;
  for I := 0 to FdTexture.Count - 1 do
  begin
    ChildTex := FdTexture[I];
    if (ChildTex <> nil) and
       (ChildTex is TAbstractSingleTextureNode) then
      AlphaMaxVar(Result, TAbstractSingleTextureNode(ChildTex).AlphaChannelFinal);
  end;
end;

procedure TMultiTextureCoordinateNode.CreateNode;
begin
  inherited;

  FFdTexCoord := TMFNode.Create(Self, true, 'texCoord', [TAbstractSingleTextureCoordinateNode]);
   FdTexCoord.ChangeAlways := chTextureCoordinate;
  AddField(FFdTexCoord);
end;

class function TMultiTextureCoordinateNode.ClassX3DType: string;
begin
  Result := 'MultiTextureCoordinate';
end;

procedure TMultiTextureTransformNode.CreateNode;
begin
  inherited;

  FFdTextureTransform := TMFNode.Create(Self, true, 'textureTransform', [TAbstractSingleTextureTransformNode]);
   FdTextureTransform.ChangeAlways := chEverything;
  AddField(FFdTextureTransform);
end;

class function TMultiTextureTransformNode.ClassX3DType: string;
begin
  Result := 'MultiTextureTransform';
end;

function TMultiTextureTransformNode.TransformMatrix: TMatrix4;
begin
  raise EInternalError.Create('You cannot get single TransformMatrix from MultiTextureTransform node');
  Result := TMatrix4.Identity; { avoid warnings that result not set }
end;

procedure TPixelTextureNode.CreateNode;
begin
  inherited;

  FFdImage := TSFImage.Create(Self, true, 'image', nil);
   FdImage.ChangeAlways := chTextureImage;
  AddField(FFdImage);
  { X3D specification comment: 0 0 }
  end;

class function TPixelTextureNode.ClassX3DType: string;
begin
  Result := 'PixelTexture';
end;

procedure TPixelTextureNode.LoadTextureData(out WasCacheUsed: boolean);
begin
  WasCacheUsed := false;

  if not FdImage.Value.IsEmpty then
  begin
    FTextureImage := FdImage.Value.MakeCopy;
    FAlphaChannelData := FdImage.Value.AlphaChannel;
  end;
end;

function TPixelTextureNode.TextureDescription: string;
begin
  if not FdImage.Value.IsEmpty then
    result := Format('inlined image (width = %d; height = %d; with alpha = %s)',
      [ FdImage.Value.Width,
        FdImage.Value.Height,
        BoolToStr(FdImage.Value.HasAlpha, true) ]) else
    result := 'none';
end;

procedure TTextureCoordinateNode.CreateNode;
begin
  inherited;

  FFdPoint := TMFVec2f.Create(Self, true, 'point', []);
   FdPoint.ChangeAlways := chTextureCoordinate;
  AddField(FFdPoint);
  { X3D specification comment: (-Inf,Inf) }
end;

class function TTextureCoordinateNode.ClassX3DType: string;
begin
  Result := 'TextureCoordinate';
end;

procedure TTextureCoordinateGeneratorNode.CreateNode;
begin
  inherited;

  FFdMode := TSFString.Create(Self, true, 'mode', 'SPHERE');
   FdMode.ChangeAlways := chTextureCoordinate;
  AddField(FFdMode);
  { X3D specification comment: [see Table 18.6] }

  FFdParameter := TMFFloat.Create(Self, true, 'parameter', []);
   FdParameter.ChangeAlways := chTextureCoordinate;
  AddField(FFdParameter);
  { X3D specification comment: [see Table 18.6] }

  { Note that projectedLight node is not enumerated as an active node
    for traversing (in DirectEnumerateActive), because the light doesn't
    shine here. We don't want
    to override it's transform with transformation of this
    TextureCoordinateGenerator. }
  FFdProjectedLight := TSFNode.Create(Self, true, 'projectedLight',
    [TSpotLightNode_1, TDirectionalLightNode_1,
     TSpotLightNode  , TDirectionalLightNode  ]);
   FdProjectedLight.ChangeAlways := chTextureCoordinate;
  AddField(FFdProjectedLight);
end;

class function TTextureCoordinateGeneratorNode.ClassX3DType: string;
begin
  Result := 'TextureCoordinateGenerator';
end;

class function TTextureCoordGenNode.ClassX3DType: string;
begin
  Result := 'TextureCoordGen';
end;

procedure TTexturePropertiesNode.CreateNode;
begin
  inherited;

  FFdAnisotropicDegree := TSFFloat.Create(Self, true, 'anisotropicDegree', 1.0);
   FdAnisotropicDegree.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdAnisotropicDegree);
  { X3D specification comment: [1,Inf) }

  FFdBorderColor := TSFColorRGBA.Create(Self, true, 'borderColor', Vector4(0, 0, 0, 0));
   FdBorderColor.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdBorderColor);
  { X3D specification comment: [0,1] }

  FFdBorderWidth := TSFInt32.Create(Self, true, 'borderWidth', 0);
   FdBorderWidth.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdBorderWidth);
  { X3D specification comment: [0,1] }

  FFdBoundaryModeS := TSFString.Create(Self, true, 'boundaryModeS', 'REPEAT');
   FdBoundaryModeS.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdBoundaryModeS);
  { X3D specification comment: [see Table 18.7] }

  FFdBoundaryModeT := TSFString.Create(Self, true, 'boundaryModeT', 'REPEAT');
   FdBoundaryModeT.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdBoundaryModeT);
  { X3D specification comment: [see Table 18.7] }

  FFdBoundaryModeR := TSFString.Create(Self, true, 'boundaryModeR', 'REPEAT');
   FdBoundaryModeR.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdBoundaryModeR);
  { X3D specification comment: [see Table 18.7] }

  FFdMagnificationFilter := TSFString.Create(Self, true, 'magnificationFilter', 'FASTEST');
   FdMagnificationFilter.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdMagnificationFilter);
  { X3D specification comment: [see Table 18.8] }

  FFdMinificationFilter := TSFString.Create(Self, true, 'minificationFilter', 'FASTEST');
   FdMinificationFilter.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdMinificationFilter);
  { X3D specification comment: [see Table 18.9] }

  FFdTextureCompression := TSFString.Create(Self, true, 'textureCompression', 'FASTEST');
   FdTextureCompression.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdTextureCompression);
  { X3D specification comment: [see Table 18.10] }

  FFdTexturePriority := TSFFloat.Create(Self, true, 'texturePriority', 0);
   FdTexturePriority.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdTexturePriority);
  { X3D specification comment: [0,1] }

  FFdGenerateMipMaps := TSFBool.Create(Self, false, 'generateMipMaps', false);
   FdGenerateMipMaps.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdGenerateMipMaps);

  FFdGUITexture := TSFBool.Create(Self, false, 'guiTexture', false);
   FdGUITexture.ChangeAlways := chTexturePropertiesNode;
  AddField(FFdGUITexture);

  { X3D XML spec (edition 2) mistakenly claims it should be
    "lineProperties", which is nonsense... I set this to "textureProperties". }
  DefaultContainerField := 'textureProperties';
end;

class function TTexturePropertiesNode.ClassX3DType: string;
begin
  Result := 'TextureProperties';
end;

function TTexturePropertiesNode.GetMinificationFilter: TAutoMinificationFilter;
var
  S: string;
begin
  S := UpperCase(FdMinificationFilter.Value);

  if S = 'AVG_PIXEL' then Result := minLinear else
  if S = 'AVG_PIXEL_AVG_MIPMAP' then Result := minLinearMipmapLinear else
  if S = 'AVG_PIXEL_NEAREST_MIPMAP' then Result := minLinearMipmapNearest else
  if S = 'NEAREST_PIXEL_AVG_MIPMAP' then Result := minNearestMipmapLinear else
  if S = 'NEAREST_PIXEL_NEAREST_MIPMAP' then Result := minNearestMipmapNearest else
  if S = 'NEAREST_PIXEL' then Result := minNearest else

  if S = 'DEFAULT' then Result := minDefault else
  if S = 'FASTEST' then Result := minFastest else
  if S = 'NICEST' then Result := minNicest else

  if S = 'NEAREST' then
  begin
    WritelnWarning('VRML/X3D', Format('"%s" is not allowed texture minification, this is an InstantReality extension, please fix to "NEAREST_PIXEL"', [S]));
    Result := minNearest;
  end else

  if S = 'LINEAR' then
  begin
    WritelnWarning('VRML/X3D', Format('"%s" is not allowed texture minification, this is an InstantReality extension, please fix to "AVG_PIXEL"', [S]));
    Result := minLinear;
  end else

  begin
    Result := minDefault;
    WritelnWarning('VRML/X3D', Format('Unknown texture minification filter "%s"', [S]));
  end;
end;

procedure TTexturePropertiesNode.SetMinificationFilter(const Value: TAutoMinificationFilter);
const
  Names: array [TAutoMinificationFilter] of string = (
    'NEAREST_PIXEL',
    'AVG_PIXEL',
    'NEAREST_PIXEL_NEAREST_MIPMAP',
    'NEAREST_PIXEL_AVG_MIPMAP',
    'AVG_PIXEL_NEAREST_MIPMAP',
    'AVG_PIXEL_AVG_MIPMAP',
    'DEFAULT',
    'FASTEST',
    'NICEST'
  );
begin
  FdMinificationFilter.Send(Names[Value]);
end;

function TTexturePropertiesNode.GetMagnificationFilter: TAutoMagnificationFilter;
var
  S: string;
begin
  S := UpperCase(FdMagnificationFilter.Value);

  if S = 'AVG_PIXEL' then Result := magLinear else
  if S = 'NEAREST_PIXEL' then Result := magNearest else

  if S = 'DEFAULT' then Result := magDefault else
  if S = 'FASTEST' then Result := magFastest else
  if S = 'NICEST' then Result := magNicest else

  if S = 'NEAREST' then
  begin
    WritelnWarning('VRML/X3D', Format('"%s" is not allowed texture magnification, this is an InstantReality extension, please fix to "NEAREST_PIXEL"', [S]));
    Result := magNearest;
  end else

  if S = 'LINEAR' then
  begin
    WritelnWarning('VRML/X3D', Format('"%s" is not allowed texture magnification, this is an InstantReality extension, please fix to "AVG_PIXEL"', [S]));
    Result := magLinear;
  end else

  begin
    Result := magDefault;
    WritelnWarning('VRML/X3D', Format('Unknown texture minification filter "%s"', [S]));
  end;
end;

procedure TTexturePropertiesNode.SetMagnificationFilter(const Value: TAutoMagnificationFilter);
const
  Names: array [TAutoMagnificationFilter] of string = (
    'NEAREST_PIXEL',
    'AVG_PIXEL',
    'DEFAULT',
    'FASTEST',
    'NICEST'
  );
begin
  FdMagnificationFilter.Send(Names[Value]);
end;

procedure TTextureTransformNode.CreateNode;
begin
  inherited;

  FFdCenter := TSFVec2f.Create(Self, true, 'center', Vector2(0, 0));
   FdCenter.ChangeAlways := chTextureTransform;
  AddField(FFdCenter);
  { X3D specification comment: (-Inf,Inf) }

  FFdRotation := TSFFloat.Create(Self, true, 'rotation', 0);
   FFdrotation.Angle := true;
   FFdrotation.ChangeAlways := chTextureTransform;
  AddField(FFdRotation);
  { X3D specification comment: (-Inf,Inf) }

  FFdScale := TSFVec2f.Create(Self, true, 'scale', Vector2(1, 1));
   FdScale.ChangeAlways := chTextureTransform;
  AddField(FFdScale);
  { X3D specification comment: (-Inf,Inf) }

  FFdTranslation := TSFVec2f.Create(Self, true, 'translation', Vector2(0, 0));
   FdTranslation.ChangeAlways := chTextureTransform;
  AddField(FFdTranslation);
  { X3D specification comment: (-Inf,Inf) }
end;

class function TTextureTransformNode.ClassX3DType: string;
begin
  Result := 'TextureTransform';
end;

function TTextureTransformNode.TransformMatrix: TMatrix4;
begin
  { Yes, VRML 2 and X3D specs say in effect that the order of operations
    is *reversed* with regards to VRML 1 spec.
          VRML 1 spec says it's (in order) scale, rotation, translation.
    VRML 2 / X3D spec say  it's (in order) translation, rotation, scale.

    Moreover, VRML 2 / X3D spec give explicit formula which we follow:
      Tc' = -C * S * R * C * T * Tc

    To test that this order matters, check e.g. VRML NIST test suite,
    "Appearance -> TextureTransform" tests 15, 16 (ImageTexture),
    31, 32 (MovieTexture) and 47, 48 (PixelTexture).
    Current implementation passes it (results match images).
    Also, results match Xj3d and OpenVRML results.

    Other links:

    - At least looking at source code, some old version of
      FreeWRL had it reversed (matching VRML 1):
      [http://search.cpan.org/src/LUKKA/FreeWRL-0.14/VRMLFunc.xs]
      function TextureTransform_Rend.

    - Other: [http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4320634] }

  Result :=
    TranslationMatrix(Vector3( -FdCenter.Value[0], -FdCenter.Value[1], 0 )) *
    ScalingMatrix(Vector3( FdScale.Value[0], FdScale.Value[1], 1 )) *
    RotationMatrixRad(FdRotation.Value, Vector3(0, 0, 1)) *
    TranslationMatrix( Vector3(
      FdTranslation.Value[0] + FdCenter.Value[0],
      FdTranslation.Value[1] + FdCenter.Value[1], 0));
end;

procedure RegisterTexturingNodes;
begin
  NodesManager.RegisterNodeClasses([
    TImageTextureNode,
    TMovieTextureNode,
    TMultiTextureNode,
    TMultiTextureCoordinateNode,
    TMultiTextureTransformNode,
    TPixelTextureNode,
    TTextureCoordinateNode,
    TTextureCoordinateGeneratorNode,
    TTextureCoordGenNode,
    TTexturePropertiesNode,
    TTextureTransformNode
  ]);
end;

{$endif read_implementation}
