//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Axis/BasicAxisItem.cpp
//! @brief     Implements various axis items
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Axis/BasicAxisItem.h"
#include "Base/Axis/MakeScale.h"
#include "Base/Axis/Scale.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString IsVisible("IsVisible");
const QString Nbins("Nbins");
const QString MinDeg("MinDeg");
const QString MaxDeg("MaxDeg");
const QString Title("Title");
const QString BaseData("BaseData");
const QString LockMinMax("LockMinMax");
const QString LogScale("LogScale");

} // namespace Tag
} // namespace

BasicAxisItem::BasicAxisItem(QObject* parent)
    : QObject(parent)
    , m_visible(true)
    , m_nbins(100)
    , m_min(0)
    , m_max(-1)
    , m_title("")
{
}

BasicAxisItem::~BasicAxisItem() = default;

int BasicAxisItem::binCount() const
{
    return m_nbins;
}

void BasicAxisItem::setBinCount(size_t value)
{
    m_nbins = int(value);
}


double BasicAxisItem::min() const
{
    return m_min;
}

void BasicAxisItem::setMin(double value)
{
    m_min = value;
    emit axisRangeChanged();
}

double BasicAxisItem::max() const
{
    return m_max;
}

void BasicAxisItem::setMax(double value)
{
    m_max = value;
    emit axisRangeChanged();
}

QString BasicAxisItem::title() const
{
    return m_title;
}

void BasicAxisItem::setTitle(const QString& title)
{
    m_title = title;
    emit axisTitleChanged();
}

std::unique_ptr<Scale> BasicAxisItem::itemToRegularAxis(double scale) const
{
    return uniqueEquiDivision(title().toStdString(), binCount(), min() * scale, max() * scale);
}

std::unique_ptr<Scale> BasicAxisItem::itemToAxis(double scale,
                                                 const Scale& /*converted_axis*/) const
{
    return itemToRegularAxis(scale);
}

bool BasicAxisItem::isVisible() const
{
    return m_visible;
}

void BasicAxisItem::setVisible(bool b)
{
    m_visible = b;
    emit axisVisibilityChanged();
}

void BasicAxisItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // visibility
    w->writeStartElement(Tag::IsVisible);
    XML::writeAttribute(w, XML::Attrib::value, m_visible);
    w->writeEndElement();

    // nbins
    w->writeStartElement(Tag::Nbins);
    XML::writeAttribute(w, XML::Attrib::value, m_nbins);
    w->writeEndElement();

    // min
    w->writeStartElement(Tag::MinDeg);
    XML::writeAttribute(w, XML::Attrib::value, m_min);
    w->writeEndElement();

    // max
    w->writeStartElement(Tag::MaxDeg);
    XML::writeAttribute(w, XML::Attrib::value, m_max);
    w->writeEndElement();

    // title
    w->writeStartElement(Tag::Title);
    XML::writeAttribute(w, XML::Attrib::value, m_title);
    w->writeEndElement();
}

void BasicAxisItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // visibility?
        if (tag == Tag::IsVisible) {
            XML::readAttribute(r, XML::Attrib::value, &m_visible);
            XML::gotoEndElementOfTag(r, tag);

            // nbins
        } else if (tag == Tag::Nbins) {
            XML::readAttribute(r, XML::Attrib::value, &m_nbins);
            XML::gotoEndElementOfTag(r, tag);

            // min
        } else if (tag == Tag::MinDeg) {
            XML::readAttribute(r, XML::Attrib::value, &m_min);
            XML::gotoEndElementOfTag(r, tag);

            // max
        } else if (tag == Tag::MaxDeg) {
            XML::readAttribute(r, XML::Attrib::value, &m_max);
            XML::gotoEndElementOfTag(r, tag);

            // title
        } else if (tag == Tag::Title) {
            XML::readAttribute(r, XML::Attrib::value, &m_title);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}
