//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Base/Types/OwningVector.h
//! @brief     Defines and implements templated class OwningVector.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifdef SWIG
#error no need to expose this header to Swig
#endif // SWIG
#ifndef BORNAGAIN_BASE_TYPES_OWNINGVECTOR_H
#define BORNAGAIN_BASE_TYPES_OWNINGVECTOR_H

#include <cstddef>
#include <utility>
#include <vector>

//! A vector of unique pointers to objects.
//!
//! Cannot be copied. For a copyable vector of cloneable objects, use CloneableVector.

template <class T>
class OwningVector {
public:
    OwningVector() = default;
    //! Constructor that takes over ownership of elements in given vector
    OwningVector(std::vector<T*>&& v)
    {
        m_v.reserve(v.size());
        for (T* e : v)
            m_v.emplace_back(e);
    }
    OwningVector(const OwningVector& other) = delete;
    OwningVector(OwningVector&& other) = default;
    ~OwningVector() { clear(); }

    OwningVector& operator=(const OwningVector& other) = delete;
    OwningVector& operator=(OwningVector&& other) = default;

    void reserve(size_t n) { m_v.reserve(n); }
    void emplace_back(T* e) { m_v.emplace_back(e); }
    void insert_at(size_t index, T* e) { m_v.insert(m_v.begin() + index, e); }

    void clear()
    {
        for (T* e : *this)
            delete e;
        m_v.clear();
    }

    size_t size() const { return m_v.size(); }
    bool empty() const { return m_v.empty(); }
    T* const& operator[](int i) const { return m_v.operator[](i); }
    T* const& at(int i) const { return m_v.at(i); }
    T* const& front() const { return m_v.front(); }
    T* const& back() const { return m_v.back(); }

    void delete_element(T* e)
    {
        if (!e)
            return;
        for (size_t i = 0; i < m_v.size(); i++)
            if (m_v[i] == e) {
                delete m_v[i];
                m_v.erase(m_v.begin() + i);
                return;
            }
    }

    T* release_at(size_t i)
    {
        if (i >= m_v.size())
            return nullptr;
        T* result = m_v.at(i);
        m_v.erase(m_v.begin() + i);
        return result;
    }

    T* release_back()
    {
        if (m_v.empty())
            return nullptr;
        T* result = back();
        m_v.pop_back();
        return result;
    }

    T* release_front()
    {
        if (m_v.empty())
            return nullptr;
        T* result = m_v.front();
        m_v.erase(m_v.begin());
        return result;
    }

    const std::vector<const T*>& reference() const { return m_v; }
    std::vector<const T*> const_vector() const
    {
        const std::vector<const T*> result(m_v.begin(), m_v.end());
        return result;
    }
    std::vector<T*> cloned_vector() const
    {
        std::vector<T*> result;
        for (T* e : m_v)
            result.emplace_back(e->clone());
        return result;
    }

    using Iterator = typename std::vector<T*>::iterator; // "typename" can be dropped under C++20
    using ConstIterator = typename std::vector<T*>::const_iterator;

    ConstIterator begin() const { return m_v.cbegin(); }
    ConstIterator end() const { return m_v.cend(); }
    Iterator begin() { return m_v.begin(); }
    Iterator end() { return m_v.end(); }

protected:
    std::vector<T*> m_v;
};

#endif // BORNAGAIN_BASE_TYPES_OWNINGVECTOR_H
