# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from gtk.gdk import Pixbuf as gPixbuf, pixbuf_new_from_file
from gobject import TYPE_STRING as gString, TYPE_BOOLEAN as gBoolean
from gtk import (WIN_POS_CENTER_ALWAYS, ListStore, CellRendererText,
                 CellRendererPixbuf, CellRendererToggle, TreeViewColumn,
                 ICON_SIZE_LARGE_TOOLBAR, Label)
from gtk.glade import XML as glade_XML
from os.path import join, isfile
from pickle import dump

from gui.aboutdialog import AboutDialog
from common.functions import Functions
from common.config import ConfigLoader
config = ConfigLoader()
functions = Functions()

class ExtensionsConfig(object):
    def __init__(self, extensions):
        self.extensions = extensions
        self.is_in_config = False

        # Create the configuration GUI
        glade_file = join(functions.datadir, 'glade', 'prefswindow.glade')
        self.widgets = glade_XML(glade_file, 'window1', domain='bluemindo')

        window = self.widgets.get_widget('window1')
        window.set_position(WIN_POS_CENTER_ALWAYS)
        window.set_title(_('Preferences'))
        window.connect('delete_event', self.on_window_close)

        validate_button = self.widgets.get_widget('button-validate')
        validate_button.connect('clicked', self.close, window)

        self.widgets.get_widget('button-about').connect('clicked', self.about)

        # Show modules and plugins
        found_extensions = self.extensions.get_extensions()
        activated_plugins = self.extensions.get_actived_plugins()

        # Create treeviews
        modules_tree = self.widgets.get_widget('treeview1')
        plugins_tree = self.widgets.get_widget('treeview2')

        render_pixbuf = CellRendererPixbuf()
        render_text = CellRendererText()
        render_bool = CellRendererToggle()
        render_bool.set_property('activatable', True)
        render_bool.set_property('inconsistent', False)

        modules_list = ListStore(gPixbuf, gString, gString)
        modules_tree.set_model(modules_list)
        modules_tree.set_rules_hint(True)
        modules_column = TreeViewColumn()

        modules_column.pack_start(render_pixbuf, expand=False)
        modules_column.add_attribute(render_pixbuf, 'pixbuf', 0)

        modules_column.pack_start(render_text, expand=True)
        modules_column.add_attribute(render_text, 'markup', 1)

        modules_tree.append_column(modules_column)

        plugins_list = ListStore(gPixbuf, gString, gBoolean, gString)
        plugins_tree.set_model(plugins_list)
        plugins_tree.set_rules_hint(True)

        plugins_column0 = TreeViewColumn()
        plugins_column0.pack_start(render_pixbuf, expand=False)
        plugins_column0.add_attribute(render_pixbuf, 'pixbuf', 0)
        plugins_column0.set_expand(True)
        plugins_column0.pack_start(render_text, expand=True)
        plugins_column0.add_attribute(render_text, 'markup', 1)
        plugins_tree.append_column(plugins_column0)

        plugins_column1 = TreeViewColumn()
        plugins_column1.pack_start(render_bool, expand=False)
        plugins_column1.add_attribute(render_bool, 'active', 2)
        plugins_tree.append_column(plugins_column1)

        # GTK signals
        render_bool.connect_object('toggled', self.on_toggle, plugins_list)
        m = 'modules'
        p = 'plugins'
        modules_tree.get_selection().connect('changed', self.openconf, m)
        plugins_tree.get_selection().connect('changed', self.openconf, p)

        for module in found_extensions['modules']:
            if isfile(module['logo']):
                icon = pixbuf_new_from_file(module['logo'])
            else:
                icon = plugins_tree.render_icon(stock_id=module['logo'],
                                                size=ICON_SIZE_LARGE_TOOLBAR,
                                                detail=None)

            modules_list.append((icon, '<b>%s</b>' % module['name'],
                                                     module['name']))

        for plugin in found_extensions['plugins']:
            if plugin['name'].lower() in activated_plugins:
                act = True
            else:
                act = False

            if isfile(plugin['logo']):
                icon = pixbuf_new_from_file(plugin['logo'])
            else:
                icon = plugins_tree.render_icon(stock_id=plugin['logo'],
                                                size=ICON_SIZE_LARGE_TOOLBAR,
                                                detail=None)

            plugins_list.append((icon, '<b>%s</b>\n<small>%s</small>' %
                                (plugin['name'], plugin['description']),
                                 act, plugin['name']))


    def on_window_close(self, widget, event):
        """Save configuration after the window close."""

        # Configure current extension
        if self.is_in_config:
            self.extensions.load_event('OnModuleConfigurationSave',
                                        self.is_in_config)
            self.is_in_config = False

    def close(self, widget, window):
        """Save configuration on button click."""

        # Configure current extension
        if self.is_in_config:
            self.extensions.load_event('OnModuleConfigurationSave',
                                        self.is_in_config)
            self.is_in_config = False

        window.destroy()

    def openconf(self, selection, exttype):
        """Open the configuration for a given extension."""
        # Configure current extension
        if self.is_in_config:
            self.extensions.load_event('OnModuleConfigurationSave',
                                        self.is_in_config)
            self.is_in_config = False

        # Start new configuration
        (mod, iter_) = selection.get_selected()

        if iter_:
            pname = mod.get_value(iter_, 1).split('\n')
            name = functions.clear_html(pname[0])

            extensions = self.extensions.get_extensions()
            actived = self.extensions.get_actived_plugins()

            if exttype != 'modules':
                exttype = 'plugins'

            for ext in extensions[exttype]:
                if ext['name'] == name:

                    if not ext['configurable']:
                        # This extension is not configurable
                        self.is_in_config = False

                        if exttype == 'modules':
                            hbox = self.widgets.get_widget('hbox1')
                        else:
                            hbox = self.widgets.get_widget('hbox2')

                        try:
                            kids = hbox.get_children()
                            hbox.remove(kids[2])
                        except IndexError:
                            pass

                        lbl = Label('<b>%s</b>' % _('This extension is not '
                                                    'configurable.'))
                        lbl.set_use_markup(True)
                        hbox.add(lbl)
                        lbl.show()
                    elif name.lower() not in actived and exttype == 'plugins':
                        # This plugin is deactivated
                        self.is_in_config = False
                        hbox = self.widgets.get_widget('hbox2')

                        try:
                            kids = hbox.get_children()
                            hbox.remove(kids[2])
                        except IndexError:
                            pass

                        lbl = Label('<b>%s</b>' % _('You should start the '
                                                    'plugin before trying to '
                                                    'configure it.'))
                        lbl.set_use_markup(True)
                        hbox.add(lbl)
                        lbl.show()
                    else:
                        # Open the configuration
                        self.extensions.load_event('OnModuleConfiguration',
                                                  (name, self.widgets))
                        self.is_in_config = name

                    return

    def about(self, widget):
        """Show an AboutDialog for the selected extension."""
        notepad = self.widgets.get_widget('notebook1')
        current = notepad.get_current_page()

        if current == 0:
            # If the extension is a module, show the Bluemindo's AboutDialog
            # instead of a specialized one
            AboutDialog()
        elif current == 1:
            # We've got a plugin now!
            treeview = self.widgets.get_widget('treeview2')
            (mod, iter_) = treeview.get_selection().get_selected()

            if iter_:
                # Get plugin's informations
                pname = mod.get_value(iter_, 1).split('\n')
                name = functions.clear_html(pname[0])

                extensions = self.extensions.get_extensions()
                for plugin in extensions['plugins']:
                    if plugin['name'] == name:

                        if isfile(plugin['logo']):
                            logopixbuf = pixbuf_new_from_file(plugin['logo'])
                        else:
                            logopixbuf = None

                        # Construct the AboutDialog
                        about_widgets = glade_XML(join(functions.datadir,
                                            'glade', 'prefswindow.glade'),
                                            'aboutdialog1', domain='bluemindo')
                        about_dialog = about_widgets.get_widget('aboutdialog1')

                        # Add datas
                        about_dialog.set_name(plugin['name'])
                        about_dialog.set_version(str(plugin['version']))
                        about_dialog.set_comments(plugin['description'])
                        about_dialog.set_logo(logopixbuf)
                        about_dialog.set_copyright('%s\n%s' % (
                                         plugin['authors'], plugin['license']))

                        # Show the AboutDialog and exit
                        about_dialog.run()
                        about_dialog.destroy()
                        return

    def on_toggle(self, liststore, pid):
        value = liststore[pid][2]
        if value:
            liststore[pid][2] = False
            status = False
        else:
            liststore[pid][2] = True
            status = True

        nm = liststore[pid][1].split('\n')
        name = functions.clear_html(nm[0]).lower()

        # Active or de-active a plugin
        actived = self.extensions.get_actived_plugins()

        if status and name not in actived:
            actived.append(name)
            self.extensions.activate_plugin(name)
        elif not status and name in actived:
            actived.remove(name)
            self.extensions.shutdown_plugin(name)

        # Write the new Plugins.cfg file
        dump(actived, open(join(config.confdir, 'Plugins.cfg'), 'w'))