# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
#
# Autopilot Functional Test Tool
# Copyright (C) 2013,2017 Canonical
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

from contextlib import ExitStack
from gi.repository import GLib
import signal
import subprocess
from testtools import TestCase
from testtools.matchers import (
    Contains,
    Equals,
    GreaterThan,
    HasLength,
    IsInstance,
    MatchesListwise,
    Not,
    raises,
)
from testtools.content import text_content
from unittest.mock import MagicMock, Mock, patch

from autopilot.application import (
    NormalApplicationLauncher,
)
from autopilot.application._environment import (
    GtkApplicationEnvironment,
    QtApplicationEnvironment,
)
import autopilot.application._launcher as _l
from autopilot.application._launcher import (
    ApplicationLauncher,
    get_application_launcher_wrapper,
    launch_process,
    _attempt_kill_pid,
    _get_app_env_from_string_hint,
    _get_application_environment,
    _get_application_path,
    _is_process_running,
    _kill_process,
)
from autopilot.utilities import sleep


class ApplicationLauncherTests(TestCase):
    def test_raises_on_attempt_to_use_launch(self):
        self.assertThat(
            lambda: ApplicationLauncher(self.addDetail).launch(),
            raises(
                NotImplementedError("Sub-classes must implement this method.")
            )
        )

    def test_init_uses_default_values(self):
        launcher = ApplicationLauncher()
        self.assertEqual(launcher.caseAddDetail, launcher.addDetail)
        self.assertEqual(launcher.proxy_base, None)
        self.assertEqual(launcher.dbus_bus, 'session')

    def test_init_uses_passed_values(self):
        case_addDetail = self.getUniqueString()
        emulator_base = self.getUniqueString()
        dbus_bus = self.getUniqueString()

        launcher = ApplicationLauncher(
            case_addDetail=case_addDetail,
            emulator_base=emulator_base,
            dbus_bus=dbus_bus,
        )
        self.assertEqual(launcher.caseAddDetail, case_addDetail)
        self.assertEqual(launcher.proxy_base, emulator_base)
        self.assertEqual(launcher.dbus_bus, dbus_bus)

    @patch('autopilot.application._launcher.fixtures.EnvironmentVariable')
    def test_setUp_patches_environment(self, ev):
        self.useFixture(ApplicationLauncher(dbus_bus=''))
        ev.assert_called_with('DBUS_SESSION_BUS_ADDRESS', '')


class NormalApplicationLauncherTests(TestCase):

    def test_kill_process_and_attach_logs(self):
        mock_addDetail = Mock()
        app_launcher = NormalApplicationLauncher(mock_addDetail)

        with patch.object(
            _l, '_kill_process', return_value=("stdout", "stderr", 0)
        ):
            app_launcher._kill_process_and_attach_logs(0, 'app')

            self.assertThat(
                mock_addDetail.call_args_list,
                MatchesListwise([
                    Equals(
                        [('process-return-code (app)', text_content('0')), {}]
                    ),
                    Equals(
                        [('process-stdout (app)', text_content('stdout')), {}]
                    ),
                    Equals(
                        [('process-stderr (app)', text_content('stderr')), {}]
                    ),
                ])
            )

    def test_setup_environment_returns_prepare_environment_return_value(self):
        app_launcher = self.useFixture(NormalApplicationLauncher())
        with patch.object(_l, '_get_application_environment') as gae:
            self.assertThat(
                app_launcher._setup_environment(
                    self.getUniqueString(), None, []),
                Equals(gae.return_value.prepare_environment.return_value)
            )

    @patch('autopilot.application._launcher.'
           'get_proxy_object_for_existing_process')
    @patch('autopilot.application._launcher._get_application_path')
    def test_launch_call_to_get_application_path(self, gap, _):
        """Test that NormalApplicationLauncher.launch calls
        _get_application_path with the arguments it was passed,"""
        launcher = NormalApplicationLauncher()
        with patch.object(launcher, '_launch_application_process'):
            with patch.object(launcher, '_setup_environment') as se:
                se.return_value = ('', [])
                token = self.getUniqueString()
                launcher.launch(token)
                gap.assert_called_once_with(token)

    @patch('autopilot.application._launcher.'
           'get_proxy_object_for_existing_process')
    @patch('autopilot.application._launcher._get_application_path')
    def test_launch_call_to_setup_environment(self, gap, _):
        """Test the NornmalApplicationLauncher.launch calls
        self._setup_environment with the correct application path from
        _get_application_path and the arguments passed to it."""
        launcher = NormalApplicationLauncher()
        with patch.object(launcher, '_launch_application_process'):
            with patch.object(launcher, '_setup_environment') as se:
                se.return_value = ('', [])
                token_a = self.getUniqueString()
                token_b = self.getUniqueString()
                token_c = self.getUniqueString()
                launcher.launch(token_a, arguments=[token_b, token_c])
                se.assert_called_once_with(
                    gap.return_value,
                    None,
                    [token_b, token_c],
                )

    @patch('autopilot.application._launcher.'
           'get_proxy_object_for_existing_process')
    @patch('autopilot.application._launcher._get_application_path')
    def test_launch_call_to_launch_application_process(self, _, __):
        """Test that NormalApplicationLauncher.launch calls
        launch_application_process with the return values of
        setup_environment."""
        launcher = NormalApplicationLauncher()
        with patch.object(launcher, '_launch_application_process') as lap:
            with patch.object(launcher, '_setup_environment') as se:
                token_a = self.getUniqueString()
                token_b = self.getUniqueString()
                token_c = self.getUniqueString()
                se.return_value = (token_a, [token_b, token_c])
                launcher.launch('', arguments=['', ''])
                lap.assert_called_once_with(
                    token_a,
                    True,
                    None,
                    [token_b, token_c],
                )

    @patch('autopilot.application._launcher.'
           'get_proxy_object_for_existing_process')
    @patch('autopilot.application._launcher._get_application_path')
    def test_launch_gets_correct_proxy_object(self, _, gpofep):
        """Test that NormalApplicationLauncher.launch calls
        get_proxy_object_for_existing_process with the correct return values of
        other functions."""
        launcher = NormalApplicationLauncher()
        with patch.object(launcher, '_launch_application_process') as lap:
            with patch.object(launcher, '_setup_environment') as se:
                se.return_value = ('', [])
                launcher.launch('')
                gpofep.assert_called_once_with(process=lap.return_value,
                                               pid=lap.return_value.pid,
                                               emulator_base=None,
                                               dbus_bus='session')

    @patch('autopilot.application._launcher.'
           'get_proxy_object_for_existing_process')
    @patch('autopilot.application._launcher._get_application_path')
    def test_launch_sets_process_of_proxy_object(self, _, gpofep):
        """Test that NormalApplicationLauncher.launch returns the proxy object
        returned by get_proxy_object_for_existing_process."""
        launcher = NormalApplicationLauncher()
        with patch.object(launcher, '_launch_application_process') as lap:
            with patch.object(launcher, '_setup_environment') as se:
                se.return_value = ('', [])
                launcher.launch('')
                set_process = gpofep.return_value.set_process
                set_process.assert_called_once_with(lap.return_value)

    @patch('autopilot.application._launcher.'
           'get_proxy_object_for_existing_process')
    @patch('autopilot.application._launcher._get_application_path')
    def test_launch_returns_proxy_object(self, _, gpofep):
        """Test that NormalApplicationLauncher.launch returns the proxy object
        returned by get_proxy_object_for_existing_process."""
        launcher = NormalApplicationLauncher()
        with patch.object(launcher, '_launch_application_process'):
            with patch.object(launcher, '_setup_environment') as se:
                se.return_value = ('', [])
                result = launcher.launch('')
                self.assertEqual(result, gpofep.return_value)

    def test_launch_application_process(self):
        """The _launch_application_process method must return the process
        object, must add the _kill_process_and_attach_logs method to the
        fixture cleanups, and must call the launch_process function with the
        correct arguments.
        """
        launcher = NormalApplicationLauncher(self.addDetail)
        launcher.setUp()

        expected_process_return = self.getUniqueString()
        with patch.object(
            _l, 'launch_process', return_value=expected_process_return
        ) as patched_launch_process:
            process = launcher._launch_application_process(
                "/foo/bar", False, None, [])

            self.assertThat(process, Equals(expected_process_return))
            self.assertThat(
                [f[0] for f in launcher._cleanups._cleanups],
                Contains(launcher._kill_process_and_attach_logs)
            )
            patched_launch_process.assert_called_with(
                "/foo/bar",
                [],
                False,
                cwd=None
            )


class ApplicationLauncherInternalTests(TestCase):

    def test_get_app_env_from_string_hint_returns_qt_env(self):
        self.assertThat(
            _get_app_env_from_string_hint('QT'),
            IsInstance(QtApplicationEnvironment)
        )

    def test_get_app_env_from_string_hint_returns_gtk_env(self):
        self.assertThat(
            _get_app_env_from_string_hint('GTK'),
            IsInstance(GtkApplicationEnvironment)
        )

    def test_get_app_env_from_string_hint_raises_on_unknown(self):
        self.assertThat(
            lambda: _get_app_env_from_string_hint('FOO'),
            raises(ValueError("Unknown hint string: FOO"))
        )

    def test_get_application_environment_uses_app_type_argument(self):
        with patch.object(_l, '_get_app_env_from_string_hint') as from_hint:
            _get_application_environment(app_type="app_type")
            from_hint.assert_called_with("app_type")

    def test_get_application_environment_uses_app_path_argument(self):
        with patch.object(
            _l, 'get_application_launcher_wrapper'
        ) as patched_wrapper:
            _get_application_environment(app_path="app_path")
            patched_wrapper.assert_called_with("app_path")

    def test_get_application_environment_raises_runtime_with_no_args(self):
        self.assertThat(
            lambda: _get_application_environment(),
            raises(
                ValueError(
                    "Must specify either app_type or app_path."
                )
            )
        )

    def test_get_application_environment_raises_on_app_type_error(self):
        unknown_app_type = self.getUniqueString()
        with patch.object(
            _l, '_get_app_env_from_string_hint',
            side_effect=ValueError()
        ):
            self.assertThat(
                lambda: _get_application_environment(
                    app_type=unknown_app_type
                ),
                raises(RuntimeError(
                    "Autopilot could not determine the correct introspection "
                    "type to use. You can specify this by providing app_type."
                ))
            )

    def test_get_application_environment_raises_on_app_path_error(self):
        unknown_app_path = self.getUniqueString()
        with patch.object(
            _l, 'get_application_launcher_wrapper', side_effect=RuntimeError()
        ):
            self.assertThat(
                lambda: _get_application_environment(
                    app_path=unknown_app_path
                ),
                raises(RuntimeError(
                    "Autopilot could not determine the correct introspection "
                    "type to use. You can specify this by providing app_type."
                ))
            )

    @patch.object(_l.os, 'killpg')
    def test_attempt_kill_pid_logs_if_process_already_exited(self, killpg):
        killpg.side_effect = OSError()

        with patch.object(_l, '_logger') as patched_log:
            _attempt_kill_pid(0)
            patched_log.info.assert_called_with(
                "Appears process has already exited."
            )

    @patch.object(_l, '_attempt_kill_pid')
    def test_kill_process_succeeds(self, patched_kill_pid):
        mock_process = Mock()
        mock_process.returncode = 0
        mock_process.communicate.return_value = ("", "",)

        with patch.object(
            _l, '_is_process_running', return_value=False
        ):
            self.assertThat(_kill_process(mock_process), Equals(("", "", 0)))

    @patch.object(_l, '_attempt_kill_pid')
    def test_kill_process_tries_again(self, patched_kill_pid):
        with sleep.mocked():
            mock_process = Mock()
            mock_process.pid = 123
            mock_process.communicate.return_value = ("", "",)

            with patch.object(
                _l, '_is_process_running', return_value=True
            ) as proc_running:
                _kill_process(mock_process)

                self.assertThat(proc_running.call_count, GreaterThan(1))
                self.assertThat(patched_kill_pid.call_count, Equals(2))
                patched_kill_pid.assert_called_with(123, signal.SIGKILL)

    @patch.object(_l.subprocess, 'Popen')
    def test_launch_process_uses_arguments(self, popen):
        launch_process("testapp", ["arg1", "arg2"])

        self.assertThat(
            popen.call_args_list[0][0],
            Contains(['testapp', 'arg1', 'arg2'])
        )

    @patch.object(_l.subprocess, 'Popen')
    def test_launch_process_default_capture_is_false(self, popen):
        launch_process("testapp", [])

        self.assertThat(
            popen.call_args[1]['stderr'],
            Equals(None)
        )
        self.assertThat(
            popen.call_args[1]['stdout'],
            Equals(None)
        )

    @patch.object(_l.subprocess, 'Popen')
    def test_launch_process_can_set_capture_output(self, popen):
        launch_process("testapp", [], capture_output=True)

        self.assertThat(
            popen.call_args[1]['stderr'],
            Not(Equals(None))
        )
        self.assertThat(
            popen.call_args[1]['stdout'],
            Not(Equals(None))
        )

    @patch.object(_l.subprocess, 'check_output')
    def test_get_application_launcher_wrapper_finds_qt(self, check_output):
        check_output.return_value = "LIBQTCORE"
        self.assertThat(
            get_application_launcher_wrapper("/fake/app/path"),
            IsInstance(QtApplicationEnvironment)
        )

    @patch.object(_l.subprocess, 'check_output')
    def test_get_application_launcher_wrapper_finds_gtk(self, check_output):
        check_output.return_value = "LIBGTK"
        self.assertThat(
            get_application_launcher_wrapper("/fake/app/path"),
            IsInstance(GtkApplicationEnvironment)
        )

    @patch.object(_l.subprocess, 'check_output')
    def test_get_application_path_returns_stripped_path(self, check_output):
        check_output.return_value = "/foo/bar   "

        self.assertThat(_get_application_path("bar"), Equals('/foo/bar'))
        check_output.assert_called_with(
            ['which', 'bar'], universal_newlines=True
        )

    def test_get_application_path_raises_when_cant_find_app(self):
        test_path = self.getUniqueString()
        expected_error = "Unable to find path for application {app}: Command"\
                         " '['which', '{app}']' returned non-zero exit "\
                         "status 1.".format(app=test_path)
        with patch.object(_l.subprocess, 'check_output') as check_output:
            check_output.side_effect = subprocess.CalledProcessError(
                1,
                ['which', test_path]
            )

            self.assertThat(
                lambda: _get_application_path(test_path),
                raises(ValueError(expected_error))
            )

    def test_get_application_launcher_wrapper_raises_runtimeerror(self):
        test_path = self.getUniqueString()
        expected_error = "Command '['ldd', '%s']' returned non-zero exit"\
                         " status 1." % test_path
        with patch.object(_l.subprocess, 'check_output') as check_output:
            check_output.side_effect = subprocess.CalledProcessError(
                1,
                ['ldd', test_path]
            )

            self.assertThat(
                lambda: get_application_launcher_wrapper(test_path),
                raises(RuntimeError(expected_error))
            )

    def test_get_application_launcher_wrapper_returns_none_for_unknown(self):
        with patch.object(_l.subprocess, 'check_output') as check_output:
            check_output.return_value = self.getUniqueString()
            self.assertThat(
                get_application_launcher_wrapper(""), Equals(None)
            )

    @patch.object(_l.psutil, 'pid_exists')
    def test_is_process_running_checks_with_pid(self, pid_exists):
        pid_exists.return_value = True
        self.assertThat(_is_process_running(123), Equals(True))
        pid_exists.assert_called_with(123)
