/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * ragator - argus record aggregator  with flow model manipulation.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <ragator.h>
#include <signal.h>
#include <math.h>

int RaLabelCounter = 0;

void RaProcessRecord (struct ArgusRecord *argus)
{
   if (argus->ahdr.type & ARGUS_MAR)
      RaProcessManRecord (argus);

   else {
      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:
                  RaProcessTCPRecord (argus);
                  break;

               case IPPROTO_UDP:
                  RaProcessUDPRecord (argus);
                  break;

               case IPPROTO_ICMP:
                  RaProcessICMPRecord (argus);
                  break;

               default:
                  RaProcessIPRecord (argus);
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            RaProcessARPRecord (argus);
            break;

         default:
            RaProcessNonIPRecord (argus);
            break;
      }
   }

   if (Nflag > 0) {
      if (Nflag-- == 1) {
         RaParseComplete(SIGINT);
      }
   }
}



void RaProcessSrvRecord (struct ArgusServiceRecord *);
 
int RaInitialized = 0;
int RaTimeout     = 0x1FFFFFFF;
int RaIdleTimeout = 0;
int RaPrintSVGOutput = 0;

struct RaFlowModelStruct *RaFlowModel = NULL;

void
ArgusClientInit ()
{
   struct ArgusModeStruct *mode = NULL;

   if (!(RaInitialized)) {
      RaWriteOut = 0;
      RaCumulativeMerge = 1;

      if ((mode = ArgusModeList) != NULL) {
         while (mode) {
            if (!(strcasecmp (mode->mode, "svg")))
               RaPrintSVGOutput++;
  
            mode = mode->nxt;
         }
      }
   
      if (ArgusFlowModelFile) {
         if ((RaFlowModel = RaReadFlowModelFile(NULL)) == NULL)
            ArgusLog (LOG_ERR, "RaReadFlowModelFile failed %s\n", strerror(errno));

         if (aflag)
            RaFlowModel->report = 1;
      }
   
      if ((RaModelerQueue = RaNewQueue()) == NULL)
            ArgusLog (LOG_ERR, "RaNewQueue failed %s\n", strerror(errno));
   
      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RA_HASHTABLESIZE;
      }

      if (Hflag) {
         if (!(RaHistoTimeParse (Hstr)))
            usage();
      }

      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);

      RaInitialized++;
   }
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if ((sig >= 0) && (!RaParseCompleting)) {
      RaParseCompleting++;
      if (RaModelerQueue)
         RaProcessQueue (RaModelerQueue, ARGUS_STOP);
   }

   if (sig == SIGINT)
      _exit(0);

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete: returning\n");
#endif
}

void
ArgusClientTimeout ()
{
   RaProcessQueue (RaModelerQueue, ARGUS_STATUS);

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusMinuteUpdate++ == 60) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s HashHdrs %d Hash %d Queue %d\n",
                     print_time(&ArgusGlobalTime), RaAllocHashTableHeaders,
                     RaHashTable.count, RaModelerQueue->count);
      ArgusMinuteUpdate = 1;
   }

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}

void
parse_arg (int argc, char**argv)
{ 
}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Ragator Version %s\n", version);
   fprintf (stderr, "usage:  %s [-f flowfile]\n", ArgusProgramName);
   fprintf (stderr, "usage:  %s [-f flowfile] [ra-options] [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "options:    -f <flowfile>      read flow model from <flowfile>.\n");
   fprintf (stderr, "ra-options: -a                 print record summaries on termination.\n");
   fprintf (stderr, "            -A                 print application bytes.\n");
   fprintf (stderr, "            -b                 dump packet-matching code.\n");
   fprintf (stderr, "            -C                 treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>         specify debug level\n");
   fprintf (stderr, "            -F <conffile>      read configuration from <conffile>.\n");
   fprintf (stderr, "            -h                 print help.\n");
   fprintf (stderr, "            -n                 don't convert numbers to names.\n");
   fprintf (stderr, "            -p <digits>        print fractional time with <digits> precision.\n");
   fprintf (stderr, "            -q                 quiet mode. don't print record outputs.\n");
   fprintf (stderr, "            -r <file>          read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -R                 print out response data when availabile.\n");
   fprintf (stderr, "            -s [-][+[#]]field  specify fields to print.\n");
   fprintf (stderr, "            -S <host[:port]>   specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "            -t <timerange>     specify <timerange> for reading records.\n");
   fprintf (stderr, "                      format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                               timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                   mm/dd[/yy]\n");
   fprintf (stderr, "                                                   -%%d{yMhdms}\n");
   fprintf (stderr, "            -T <secs>          attach to remote server for T seconds.\n");
   fprintf (stderr, "            -u                 print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth>     specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>          write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "            -z                 print Argus TCP state changes.\n");
   fprintf (stderr, "            -Z <s|d|b>         print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}

void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;
   struct ArgusRecordStore *store;
   struct RaPolicyStruct *rap;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   extern int RaHistoStart;
   int i = 0;

   if (xflag && (srv->status == RA_SVCFAILED))
      return;

   RaThisActiveDuration = RaGetActiveDuration(argus);
   RaThisActiveIndex = RaHistoTimeSeries;

   if (Hflag) {
      if (RaThisActiveDuration < RaHistoStart)
         return;

      for (i = 0; i < (RaHistoTimeSeries + 1); i++) {
         if (RaThisActiveDuration < RaHistoTimeValues[i]) {
            RaThisActiveIndex = i;
            break;
         }
      }

      if (i == RaHistoTimeSeries)
         return;

   } else
      RaThisActiveIndex = 0;

   bcopy((char *) flow, (char *) RaArgusFlow, sizeof(struct ArgusFlow));

   if ((rap = RaFlowModelOverRides(argus, RaFlowModel)) != NULL) {
      RaModifyFlow(rap, argus, RaFlowModel);
      RaTimeout = rap->ArgusTimeout;
      RaIdleTimeout = rap->ArgusIdleTimeout;
   } else {
      RaModifyDefaultFlow(argus);
   }

   if ((store = RaFindArgusRecord(&RaHashTable, argus)) == NULL)
      if ((store = RaFindRevArgusRecord(&RaHashTable, argus)) != NULL) {
      }

   if (store) {
      RaThisArgusStore = store;

      RaCheckTimeout (store, argus);

      if ((srv->status & RA_SVCTEST) != (store->status & RA_SVCTEST)) {
         RaSendArgusRecord(store);
         store->status &= ~RA_SVCTEST;
         store->status |= (srv->status & RA_SVCTEST);
      }

      if (!(store->data[RaThisActiveIndex])) {
         struct ArgusRecordData *data = NULL;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;
            data->status |= srv->status & RA_SVCTEST;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.meanval * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.meanval * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[RaThisActiveIndex] = data;
         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

      } else
         RaMergeArgusRecord(argus, store, RaThisActiveIndex);

      RaUpdateArgusStore(argus, store);

   } else {
      if ((store = RaNewArgusStore(argus)) != NULL) {
         struct ArgusRecordData *data = NULL;

         store->ArgusTimeout     = RaTimeout;
         store->ArgusIdleTimeout = RaIdleTimeout;

         RaThisArgusStore = store;

         store->status |= RA_MODIFIED;
         store->status |= srv->status & RA_SVCTEST;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.meanval * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.meanval * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[RaThisActiveIndex] = data;

            if ((store->rahtblhdr = RaAddHashEntry (&RaHashTable, store)) != NULL)
               RaAddToQueue(RaModelerQueue, &store->qhdr);

         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));
      } else
         ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
   }

   RaIdleTimeout = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaProcessSrvRecord: done\n");
#endif
}


void
RaProcessManRecord (struct ArgusRecord *argus)
{
}


struct ArgusServiceRecord ArgusThisSrv;

void
RaProcessTCPRecord (struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;
   int srccount, dstcount;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   srccount = argus->argus_far.src.count;
   dstcount = argus->argus_far.dst.count;

   RaIdleTimeout = 7200;

   if (Vflag) {
      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
         if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
            if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT)) {
               if (tcp->state & ARGUS_RESET) {
                  if (tcp->state & ARGUS_DST_RESET)
                     if (argus->argus_far.src.count && argus->argus_far.dst.count)
                        if (!(argus->argus_far.src.bytes && argus->argus_far.dst.bytes))
                           ArgusThisSrv.status = RA_SVCFAILED;
       
                  if (tcp->state & ARGUS_SRC_RESET)
                     if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                        ArgusThisSrv.status = RA_SVCFAILED;
               }
            }
    
            if (tcp->state & ARGUS_TIMEOUT)
               if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                  ArgusThisSrv.status = RA_SVCFAILED;
         }
      }
   }

   RaProcessSrvRecord (&ArgusThisSrv);
   RaIdleTimeout = 0;
}


void
RaProcessICMPRecord (struct ArgusRecord *argus)
{
   int srccount, dstcount;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   srccount = argus->argus_far.src.count;
   dstcount = argus->argus_far.dst.count;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessUDPRecord (struct ArgusRecord *argus)
{
   int srccount, dstcount;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   srccount = argus->argus_far.src.count;
   dstcount = argus->argus_far.dst.count;

   if (Vflag)
      if ((srccount == 0) || (dstcount == 0))
         ArgusThisSrv.status = RA_SVCFAILED;

   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   if (Rflag)
      if (!((srccount > 0) && (dstcount > 0)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessIPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;

      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessARPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessNonIPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


#include <stdio.h>
#include <errno.h>


#define RA_MAXQSCAN  25600
#define RA_MAXQSIZE  250000
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordStore *obj = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP:
         while ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL)
            RaTimeoutArgusStore(obj);

         break;

      default:
         while (queue->count > RA_MAXQSIZE) {
            obj = (struct ArgusRecordStore *) RaRemoveFromQueue(RaModelerQueue, RaModelerQueue->start->prv);
            RaTimeoutArgusStore(obj);
         }

         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  if (RaCheckTimeout(obj, NULL))
                     RaTimeoutArgusStore(obj);
                  else
                     RaAddToQueue(queue, &obj->qhdr);

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}



int
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   unsigned char buf[MAXSTRLEN];
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   struct ArgusFarHeaderStruct *farhdr;
   struct ArgusAGRStruct *agr = NULL;
   int i, retn = 0;

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if ((data = store->data[i]) != NULL) {

         argus = data->argus;

         if ((Lflag > 0) && !(qflag)) {
            if (RaLabel == NULL)
               RaLabel = RaGenerateLabel(argus);

            if (!(RaLabelCounter++ % Lflag)) {
               if (!(wfile))
                  printf ("%s\n", RaLabel);
               fflush (stdout);
            }

            if (Lflag < 0)
               Lflag = 0;
         }

         if (argus && (data->status & RA_MODIFIED)) {
            if (data->act.n > 0) {
               data->agr.act.n = data->act.n;
               data->agr.act.meanval = data->act.sumtime/data->act.n;
               data->agr.act.stdev = sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.n = data->idle.n;
               data->agr.idle.meanval = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL)
               bcopy ((char *)&data->agr, (char *)agr, data->agr.length);

            bcopy ((char *) argus, buf, argus->ahdr.length);
            argus = (struct ArgusRecord *) buf;

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) == NULL) {
               
               if ((RaFlowModel && (RaFlowModel->report > 0))  || (Hflag != 0)) {
                  bcopy ((char *)&data->agr, &buf[argus->ahdr.length], data->agr.length);
                  argus->ahdr.length += data->agr.length;
                  argus->ahdr.status |= ARGUS_MERGED;
                  ArgusFree (data->argus);
                  data->argus = RaCopyArgusRecord(argus);
                  ArgusThisFarStatus = ArgusIndexRecord(data->argus, ArgusThisFarHdrs);

               } else
                  argus->ahdr.status &= ~(ARGUS_MERGED);
            }

            if (RaPrintActualValues) {
               switch (argus->ahdr.status & 0xFFFF) {
                  case ETHERTYPE_IP:
                     if (!(data->status & ARGUS_FAR_SRCADDR_MODIFIED))
                        if (argus->argus_far.flow.ip_flow.ip_src != data->flow.ip_flow.ip_src)
                           argus->argus_far.flow.ip_flow.ip_src = data->flow.ip_flow.ip_src;
                        
                     if (!(data->status & ARGUS_FAR_DSTADDR_MODIFIED))
                        if (argus->argus_far.flow.ip_flow.ip_dst != data->flow.ip_flow.ip_dst)
                           argus->argus_far.flow.ip_flow.ip_dst = data->flow.ip_flow.ip_dst;

                     if (!(data->status & ARGUS_FAR_PROTO_MODIFIED)) {
                        if (argus->argus_far.flow.ip_flow.ip_p != data->flow.ip_flow.ip_p)
                           argus->argus_far.flow.ip_flow.ip_p = data->flow.ip_flow.ip_p;

                        switch (argus->argus_far.flow.ip_flow.ip_p) {
                           case IPPROTO_TCP:
                           case IPPROTO_UDP:
                              if (!(data->status & ARGUS_FAR_SRCPORT_MODIFIED)) {
                                 if (argus->argus_far.flow.ip_flow.sport != data->flow.ip_flow.sport)
                                    argus->argus_far.flow.ip_flow.sport = data->flow.ip_flow.sport;
                              }

                              if (!(data->status & ARGUS_FAR_DSTPORT_MODIFIED)) {
                                 if (argus->argus_far.flow.ip_flow.dport != data->flow.ip_flow.dport)
                                    argus->argus_far.flow.ip_flow.dport = data->flow.ip_flow.dport;
                              }
                              break;
                        }
                     }
               }
            }
      
            if ((ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusWfileList)))) {
               struct ArgusWfileStruct *wfile = NULL, *start = NULL;
 
               if ((wfile = ArgusFrontList(ArgusWfileList)) != NULL) { 
                  start = wfile; 
                  do { 
                     if ((exceptfile == NULL) || strcmp(wfile->filename, exceptfile)) { 
#ifdef _LITTLE_ENDIAN
                        ArgusHtoN(argus); 
#endif
                        ArgusWriteNewLogfile (wfile, argus); 
 
#ifdef _LITTLE_ENDIAN
                        ArgusNtoH(argus); 
#endif
                     }   
                     ArgusPopFrontList(ArgusWfileList); 
                     ArgusPushBackList(ArgusWfileList, wfile); 
                     wfile = ArgusFrontList(ArgusWfileList); 
 
                  } while (wfile != start); 
               }   

            } else {
      
               if (argus->ahdr.type & ARGUS_MAR)
                  printf ("%s\n", get_man_string (argus));
                  
               else {
                  ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

                  switch (argus->ahdr.status & 0xFFFF) {
                     case ETHERTYPE_IP:
                        switch (argus->argus_far.flow.ip_flow.ip_p) {
                           case IPPROTO_TCP:              
                              printf ("%s", get_tcp_string (argus));
                              break;
            
                           case IPPROTO_ICMP:              
                              printf ("%s", get_icmp_string (argus));
                              break;
            
                           default:
                              printf ("%s", get_ip_string (argus));
                              break;
                        }
                        break;
            
                     case ETHERTYPE_ARP:
                     case ETHERTYPE_REVARP:
                        printf ("%s", get_arp_string (argus));
                        break;
            
                     default:
                        printf ("%s", get_nonip_string (argus));
                        break;
                  }

                  printf ("\n");
               }
               fflush (stdout);
            }
      
            argus = data->argus;

            if (argus->ahdr.type & ARGUS_FAR) {
               int farlen, length = argus->ahdr.length - sizeof(argus->ahdr);
               farhdr = (struct ArgusFarHeaderStruct *)((char *)argus + sizeof(argus->ahdr));
      
               while (length > 0) {
                  switch (farhdr->type) {
                     case ARGUS_FAR: {
                        struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;
                        far->time.start.tv_sec = 0x7FFFFFFF; far->time.start.tv_usec = 0;
                        far->time.last.tv_sec = 0; far->time.last.tv_usec = 0;
                        far->src.count = 0; far->src.bytes = 0;
                        far->dst.count = 0; far->dst.bytes = 0;
                        break;
                     }
      
                     case ARGUS_TCP_DSR: {
                        struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr;
      
                        tcp->state = 0;
                        tcp->src.seqbase = 0; tcp->src.ackbytes = 0;
                        tcp->src.rpkts = 0; tcp->src.win = 0; tcp->src.flags = 0;
                        tcp->dst.seqbase = 0; tcp->dst.ackbytes = 0;
                        tcp->dst.rpkts = 0; tcp->dst.win = 0; tcp->dst.flags = 0;
      
                        break;
                     }
      
                     case ARGUS_TIME_DSR: {
                        struct ArgusTimeStruct *time = (struct ArgusTimeStruct *) farhdr;

                        time->src.act.n = 0;
                        time->src.act.meanval = 0;
                        time->src.act.stdev = 0;
                        time->src.act.maxval = 0;
                        time->src.act.minval = 0x7FFFFFFF;
                        time->src.idle.n = 0;
                        time->src.idle.meanval = 0;
                        time->src.idle.stdev = 0;
                        time->src.idle.maxval = 0;
                        time->src.idle.minval = 0x7FFFFFFF;

                        time->dst.act.n = 0;
                        time->dst.act.meanval = 0;
                        time->dst.act.stdev = 0;
                        time->dst.act.maxval = 0;
                        time->dst.act.minval = 0x7FFFFFFF;
                        time->dst.idle.n = 0;
                        time->dst.idle.meanval = 0;
                        time->dst.idle.stdev = 0;
                        time->dst.idle.maxval = 0;
                        time->dst.idle.minval = 0x7FFFFFFF;

                        break;
                     }

                     case ARGUS_VLAN_DSR: {
                        struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) farhdr;
                        vlan->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_MPLS_DSR: {
                        struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) farhdr;
                        mpls->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_AGR_DSR: {
                        struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;
      
                        agr->count = 0;
                        agr->act.n = 0;
                        agr->act.minval = 0x7FFFFFFF;   agr->act.meanval = 0;
                        agr->act.stdev = 0;  agr->act.maxval = 0;
                        agr->idle.n = 0;
                        agr->idle.minval = 0x7FFFFFFF; agr->idle.meanval = 0;
                        agr->idle.stdev = 0; agr->idle.maxval = 0;
                        break;
                     }
                  }

                  if ((farlen = farhdr->length) == 0)
                     break;
                  if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
                      (farhdr->type == ARGUS_DSTUSRDATA_DSR))
                     farlen = farlen * 4;
                  length -= farlen;
                  farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
               }
            }

            data->agr.count = 0;
            data->agr.act.n = 0;
            data->agr.act.meanval = 0;
            data->agr.act.stdev = 0;
            data->agr.act.maxval = 0;
            data->agr.act.minval = 0x7FFFFFFF;
            data->agr.idle.n = 0;
            data->agr.idle.meanval = 0;
            data->agr.idle.stdev = 0;
            data->agr.idle.maxval = 0;
            data->agr.idle.minval = 0x7FFFFFFF;
      
            data->act.n = 0;
            data->act.sumtime = 0;
            data->act.sumsqrd = 0;
            data->idle.n = 0;
            data->idle.sumtime = 0;
            data->idle.sumtime = 0;
      
            data->argus->ahdr.status &= ~ARGUS_MERGED;
            data->status &= ~RA_MODIFIED;
         }
      }
   }

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
   return (retn);
}
