/*
 * Copyright (C) 2012 Tobias Brunner
 * Copyright (C) 2005-2008 Martin Willi
 * Copyright (C) 2005 Jan Hutter
 * Hochschule fuer Technik Rapperswil
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

/**
 * @defgroup mac mac
 * @{ @ingroup crypto
 */

#ifndef MAC_H_
#define MAC_H_

typedef struct mac_t mac_t;

#include <library.h>

/**
 * Generic interface for message authentication codes.
 *
 * Classes implementing this interface can use the PRF and signer wrappers.
 */
struct mac_t {

	/**
	 * Generate message authentication code.
	 *
	 * If out is NULL, no result is given back.  A next call will
	 * append the data to already supplied data.  If out is not NULL,
	 * the mac of all apended data is calculated, written to out and the
	 * internal state is reset.
	 *
	 * @param data		chunk of data to authenticate
	 * @param out		pointer where the generated bytes will be written
	 * @return			TRUE if mac generated successfully
	 */
	bool (*get_mac)(mac_t *this, chunk_t data,
					uint8_t *out) __attribute__((warn_unused_result));

	/**
	 * Get the size of the resulting MAC.
	 *
	 * @return			block size in bytes
	 */
	size_t (*get_mac_size)(mac_t *this);

	/**
	 * Set the key to be used for the MAC.
	 *
	 * Any key length must be accepted.
	 *
	 * @param key		key to set
	 * @return			TRUE if key set successfully
	 */
	bool (*set_key)(mac_t *this,
					chunk_t key) __attribute__((warn_unused_result));

	/**
	 * Destroys a mac_t object.
	 */
	void (*destroy) (mac_t *this);
};

#endif /** MAC_H_ @}*/
