# Copyright (C) 2017 Red Hat, Inc., Marcus Linden <mlinden@redhat.com>
# This file is part of the sos project: https://github.com/sosreport/sos
#
# This copyrighted material is made available to anyone wishing to use,
# modify, copy, or redistribute it subject to the terms and conditions of
# version 2 of the GNU General Public License.
#
# See the LICENSE file in the source distribution for further information.

from sos.plugins import Plugin, RedHatPlugin, DebianPlugin, UbuntuPlugin, \
    SuSEPlugin


class Conntrack(Plugin, RedHatPlugin, DebianPlugin, UbuntuPlugin, SuSEPlugin):
    """conntrack - netfilter connection tracking
    """

    plugin_name = 'conntrack'
    profiles = ('network', 'cluster')

    packages = ('conntrack-tools', 'conntrack', 'conntrackd')

    def setup(self):
        self.add_copy_spec("/etc/conntrackd/conntrackd.conf")
        self.add_cmd_output([
            "conntrackd -s network",
            "conntrackd -s cache",
            "conntrackd -s runtime",
            "conntrackd -s link",
            "conntrackd -s rsqueue",
            "conntrackd -s queue",
            "conntrackd -s ct",
            "conntrackd -s expect",
        ])

        # Collect info from conntrack
        self.add_cmd_output([
            "conntrack -L -o extended",
            "conntrack -S",
        ])

        # Capture additional data from namespaces; each command is run
        # per-namespace
        ip_netns = self.exec_cmd("ip netns")
        cmd_prefix = "ip netns exec "
        if ip_netns['status'] == 0:
            out_ns = []
            for line in ip_netns['output'].splitlines():
                # If there's no namespaces, no need to continue
                if line.startswith("Object \"netns\" is unknown") \
                        or line.isspace() \
                        or line[:1].isspace():
                    continue
                out_ns.append(line.partition(' ')[0])
            for namespace in out_ns:
                ns_cmd_prefix = cmd_prefix + namespace + " "
                self.add_cmd_output(ns_cmd_prefix + "conntrack -L -o extended")
                self.add_cmd_output(ns_cmd_prefix + "conntrack -S")

# vim: set et ts=4 sw=4 :
