;;;
;;; scmail - a mail filter written in Scheme
;;;
;;; Copyright (C) 2002-2004 Satoru Takabayashi <satoru@namazu.org> 
;;;     All rights reserved.
;;;     This is free software with ABSOLUTELY NO WARRANTY.
;;;
;;; Permission to use, copy, modify, distribute this software and
;;; accompanying documentation for any purpose is hereby granted,
;;; provided that existing copyright notices are retained in all
;;; copies and that this notice is included verbatim in all
;;; distributions.
;;; This software is provided as is, without express or implied
;;; warranty.  In no circumstances the author(s) shall be liable
;;; for any damages arising out of the use of this software.
;;;

(define-module scmail.mh
  (use srfi-1)
  (use file.util)
  (use scmail.mail)
  (use scmail.mailbox)
  )

(select-module scmail.mh)

;;
;; mailbox object
;;
(define-class <mh-mailbox> (<mailbox>) ())

(define-method scmail-mailbox-mail-list ((mailbox <mh-mailbox>) folder)
  (let1 directory (scmail-mailbox-folder->path mailbox folder)
        (if (file-is-directory? directory)
            (map (lambda (id)
                   (build-path directory (number->string id)))
                 (sort
                  (map string->number
                       (filter (lambda (x) (let1 y (build-path directory x)
                                                 (and (file-is-regular? y)
                                                      (file-is-readable? y))))
                               (directory-list directory
                                               :children? #t
                                               :filter #/^[0-9]+$/)))))
            '())))

(scmail-mailbox-add-type! 'mh <mh-mailbox>)


;;
;; mail object
;;
(define-class <mh-mail> (<mail>)
  ((mailbox :init-value #f
            :init-keyword :mailbox)))

;; (mh-file 123 "~/Mail/inbox") => "/home/foobar/Mail/inbox/123"
(define (mh-file id directory)
  (build-path directory (number->string id)))

;; for dry-run mode.
(define mh-dry-run-last-id
  (let1 id-table (make-hash-table 'string=?)
        (lambda (directory)
          (let1 id (hash-table-get id-table directory 0)
                (hash-table-put! id-table directory (+ id 1))
                id))))

;; (mh-last-mail-id "~/Mail/inbox") => 8
(define-method mh-last-mail-id ((mail <mh-mail>) folder)
  (let1 mailbox (ref mail 'mailbox)
        (if (file-is-directory? 
             (scmail-mailbox-folder->path mailbox folder))
            (let ((mlist (scmail-mailbox-mail-list mailbox folder)))
              (if (null? mlist)
                  0
                  (string->number (sys-basename (car (last-pair mlist))))))
            (mh-dry-run-last-id folder))))

(define-method scmail-mail-prepare ((mail <mh-mail>) folder)
  (let* ((mailbox (ref mail 'mailbox))
         (dest-directory (scmail-mailbox-make-folder mailbox folder))
         (new-id (+ 1 (mh-last-mail-id mail folder))))
    (mh-file new-id dest-directory)))

(define-method scmail-mail-copy ((mail <mh-mail>) folder)
  (let1 new-file (scmail-mail-prepare mail folder)
    (scmail-mail-write mail new-file)
    new-file))


(scmail-mail-add-type! 'mh <mh-mail>)

(provide "scmail/mh")

