//===-- OptionValueChar.cpp -----------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#include "lldb/Interpreter/OptionValueChar.h"

#include "lldb/Interpreter/OptionArgParser.h"
#include "lldb/Interpreter/OptionValue.h"
#include "lldb/Utility/Stream.h"
#include "lldb/Utility/StringList.h"
#include "llvm/ADT/STLExtras.h"

using namespace lldb;
using namespace lldb_private;

static void DumpChar(Stream &strm, char value) {
  if (value != '\0')
    strm.PutChar(value);
  else
    strm.PutCString("(null)");
}

void OptionValueChar::DumpValue(const ExecutionContext *exe_ctx, Stream &strm,
                                uint32_t dump_mask) {
  if (dump_mask & eDumpOptionType)
    strm.Printf("(%s)", GetTypeAsCString());

  if (dump_mask & eDumpOptionValue) {
    if (dump_mask & eDumpOptionType)
      strm.PutCString(" = ");
    DumpChar(strm, m_current_value);
    if (dump_mask & eDumpOptionDefaultValue &&
        m_current_value != m_default_value) {
      DefaultValueFormat label(strm);
      DumpChar(strm, m_default_value);
    }
  }
}

Status OptionValueChar::SetValueFromString(llvm::StringRef value,
                                           VarSetOperationType op) {
  Status error;
  switch (op) {
  case eVarSetOperationClear:
    Clear();
    break;

  case eVarSetOperationReplace:
  case eVarSetOperationAssign: {
    bool success = false;
    char char_value = OptionArgParser::ToChar(value, '\0', &success);
    if (success) {
      m_current_value = char_value;
      m_value_was_set = true;
    } else
      return Status::FromErrorStringWithFormatv(
          "'{0}' cannot be longer than 1 character", value);
  } break;

  default:
    error = OptionValue::SetValueFromString(value, op);
    break;
  }
  return error;
}
