#!/usr/bin/env python

"""
A Web interface to an event calendar.

Copyright (C) 2014, 2015, 2016, 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from datetime import datetime, timedelta
from imiptools.data import get_address, get_uri, get_verbose_address, make_uid, \
                           uri_parts
from imiptools.dates import format_datetime, get_date, get_datetime, \
                            get_datetime_item, get_end_of_day, get_start_of_day, \
                            get_start_of_next_day, get_timestamp, ends_on_same_day, \
                            to_date, to_timezone
from imiptools.period import add_day_start_points, add_empty_days, add_slots, \
                             get_scale, get_slots, get_spans, partition_by_day, \
                             remove_end_slot, Period, Point
from imipweb.resource import FormUtilities, ResourceClient

class CalendarPage(ResourceClient, FormUtilities):

    "A request handler for the calendar page."

    # Request logic methods.

    def handle_newevent(self):

        """
        Handle any new event operation, creating a new event and redirecting to
        the event page for further activity.
        """

        _ = self.get_translator()

        # Check the validation token.

        if not self.check_validation_token():
            return False

        # Handle a submitted form.

        args = self.env.get_args()

        for key in args.keys():
            if key.startswith("newevent-"):
                i = key[len("newevent-"):]
                break
        else:
            return False

        # Create a new event using the available information.

        slots = args.get("slot", [])
        participants = args.get("participants", [])
        summary = args.get("summary-%s" % i, [None])[0]

        if not slots:
            return False

        # Obtain the user's timezone.

        tzid = self.get_tzid()

        # Coalesce the selected slots.

        slots.sort()
        coalesced = []
        last = None

        for slot in slots:
            start, end = (slot.split("-", 1) + [None])[:2]
            start = get_datetime(start, {"TZID" : tzid})
            end = end and get_datetime(end, {"TZID" : tzid}) or get_start_of_next_day(start, tzid)

            if last:
                last_start, last_end = last

                # Merge adjacent dates and datetimes.

                if start == last_end or \
                    not isinstance(start, datetime) and \
                    get_start_of_day(last_end, tzid) == get_start_of_day(start, tzid):

                    last = last_start, end
                    continue

                # Handle datetimes within dates.
                # Datetime periods are within single days and are therefore
                # discarded.

                elif not isinstance(last_start, datetime) and \
                    get_start_of_day(start, tzid) == get_start_of_day(last_start, tzid):

                    continue

                # Add separate dates and datetimes.

                else:
                    coalesced.append(last)

            last = start, end

        if last:
            coalesced.append(last)

        # Invent a unique identifier.

        uid = make_uid(self.user)

        # Create a calendar object and store it as a request.

        record = []
        rwrite = record.append

        # Define a single occurrence if only one coalesced slot exists.

        start, end = coalesced[0]
        start_value, start_attr = get_datetime_item(start, tzid)
        end_value, end_attr = get_datetime_item(end, tzid)
        user_attr = self.get_user_attributes()

        utcnow = get_timestamp()

        rwrite(("UID", {}, uid))
        rwrite(("SUMMARY", {}, summary or (_("New event at %s") % utcnow)))
        rwrite(("DTSTAMP", {}, utcnow))
        rwrite(("DTSTART", start_attr, start_value))
        rwrite(("DTEND", end_attr, end_value))
        rwrite(("ORGANIZER", user_attr, self.user))

        cn_participants = uri_parts(filter(None, participants))
        participants = []

        for cn, participant in cn_participants:
            d = {"RSVP" : "TRUE", "PARTSTAT" : "NEEDS-ACTION"}
            if cn:
                d["CN"] = cn
            rwrite(("ATTENDEE", d, participant))
            participants.append(participant)

        if self.user not in participants:
            d = {"PARTSTAT" : "ACCEPTED"}
            d.update(user_attr)
            rwrite(("ATTENDEE", d, self.user))

        # Define additional occurrences if many slots are defined.

        rdates = []

        for start, end in coalesced[1:]:
            start_value, start_attr = get_datetime_item(start, tzid)
            end_value, end_attr = get_datetime_item(end, tzid)
            rdates.append("%s/%s" % (start_value, end_value))

        if rdates:
            rwrite(("RDATE", {"VALUE" : "PERIOD", "TZID" : tzid}, rdates))

        node = ("VEVENT", {}, record)

        self.store.set_event(self.user, uid, None, node=node)
        self.store.queue_request(self.user, uid)

        # Redirect to the object (or the first of the objects), where instead of
        # attendee controls, there will be organiser controls.

        self.redirect(self.link_to(uid, args=self.get_time_navigation_args()))
        return True

    def update_participants(self):

        "Update the participants used for scheduling purposes."

        args = self.env.get_args()
        participants = args.get("participants", [])

        try:
            for name, value in args.items():
                if name.startswith("remove-participant-"):
                    i = int(name[len("remove-participant-"):])
                    del participants[i]
                    break
        except ValueError:
            pass

        # Trim empty participants.

        while participants and not participants[-1].strip():
            participants.pop()

        return participants

    # Page fragment methods.

    def show_user_navigation(self):

        "Show user-specific navigation."

        page = self.page
        user_attr = self.get_user_attributes()

        page.p(id_="user-navigation")
        page.a(get_verbose_address(self.user, user_attr), href=self.link_to("profile"), class_="username")
        page.p.close()

    def show_requests_on_page(self):

        "Show requests for the current user."

        _ = self.get_translator()

        page = self.page
        view_period = self.get_view_period()
        duration = view_period and view_period.get_duration() or timedelta(1)

        # NOTE: This list could be more informative, but it is envisaged that
        # NOTE: the requests would be visited directly anyway.

        requests = self._get_requests()

        page.div(id="pending-requests")

        if requests:
            page.p(_("Pending requests:"))

            page.ul()

            for uid, recurrenceid, request_type in requests:
                obj = self._get_object(uid, recurrenceid)
                if obj:

                    # Provide a link showing the request in context.

                    periods = self.get_periods(obj)
                    if periods:
                        start = to_date(periods[0].get_start())
                        end = max(to_date(periods[0].get_end()), start + duration)
                        d = {"start" : format_datetime(start), "end" : format_datetime(end)}
                        page.li()
                        page.a(obj.get_value("SUMMARY"), href="%s#request-%s-%s" % (self.link_to(args=d), uid, recurrenceid or ""))
                        page.li.close()

            page.ul.close()

        else:
            page.p(_("There are no pending requests."))

        page.div.close()

    def show_participants_on_page(self, participants):

        "Show participants for scheduling purposes."

        _ = self.get_translator()

        page = self.page

        # Show any specified participants together with controls to remove and
        # add participants.

        page.div(id="participants")

        page.p(_("Participants for scheduling:"))

        for i, participant in enumerate(participants):
            page.p()
            page.input(name="participants", type="text", value=participant)
            page.input(name="remove-participant-%d" % i, type="submit", value=_("Remove"))
            page.p.close()

        page.p()
        page.input(name="participants", type="text")
        page.input(name="add-participant", type="submit", value=_("Add"))
        page.p.close()

        page.div.close()

    def show_calendar_controls(self):

        """
        Show controls for hiding empty days and busy slots in the calendar.

        The positioning of the controls, paragraph and table are important here:
        the CSS file describes the relationship between them and the calendar
        tables.
        """

        _ = self.get_translator()

        page = self.page
        args = self.env.get_args()

        self.control("showdays", "checkbox", "show", ("show" in args.get("showdays", [])), id="showdays", accesskey="D")
        self.control("hidebusy", "checkbox", "hide", ("hide" in args.get("hidebusy", [])), id="hidebusy", accesskey="B")

        page.p(id_="calendar-controls", class_="controls")
        page.span(_("Select days or periods for a new event."))
        page.label(_("Hide busy time periods"), for_="hidebusy", class_="hidebusy enable")
        page.label(_("Show busy time periods"), for_="hidebusy", class_="hidebusy disable")
        page.label(_("Show empty days"), for_="showdays", class_="showdays disable")
        page.label(_("Hide empty days"), for_="showdays", class_="showdays enable")
        page.input(name="reset", type="submit", value=_("Clear selections"), id="reset")
        page.p.close()

    def show_time_navigation(self, freebusy, view_period):

        """
        Show the calendar navigation links for the schedule defined by
        'freebusy' and for the period defined by 'view_period'.
        """

        _ = self.get_translator()

        page = self.page
        view_start = view_period.get_start()
        view_end = view_period.get_end()
        duration = view_period.get_duration()

        preceding_events = view_start and freebusy.get_overlapping([Period(None, view_start, self.get_tzid())]) or []
        following_events = view_end and freebusy.get_overlapping([Period(view_end, None, self.get_tzid())]) or []

        last_preceding = preceding_events and to_date(preceding_events[-1].get_end()) + timedelta(1) or None
        first_following = following_events and to_date(following_events[0].get_start()) or None

        page.p(id_="time-navigation")

        if view_start:
            page.input(name="start", type="hidden", value=format_datetime(view_start))

            if last_preceding:
                preceding_start = last_preceding - duration
                page.label(_("Show earlier events"), for_="earlier-events", class_="earlier-events")
                page.input(name="earlier-events", id_="earlier-events", type="submit")
                page.input(name="earlier-events-start", type="hidden", value=format_datetime(preceding_start))
                page.input(name="earlier-events-end", type="hidden", value=format_datetime(last_preceding))

            earlier_start = view_start - duration
            page.label(_("Show earlier"), for_="earlier", class_="earlier")
            page.input(name="earlier", id_="earlier", type="submit")
            page.input(name="earlier-start", type="hidden", value=format_datetime(earlier_start))
            page.input(name="earlier-end", type="hidden", value=format_datetime(view_start))

        if view_end:
            page.input(name="end", type="hidden", value=format_datetime(view_end))

            later_end = view_end + duration
            page.label(_("Show later"), for_="later", class_="later")
            page.input(name="later", id_="later", type="submit")
            page.input(name="later-start", type="hidden", value=format_datetime(view_end))
            page.input(name="later-end", type="hidden", value=format_datetime(later_end))

            if first_following:
                following_end = first_following + duration
                page.label(_("Show later events"), for_="later-events", class_="later-events")
                page.input(name="later-events", id_="later-events", type="submit")
                page.input(name="later-events-start", type="hidden", value=format_datetime(first_following))
                page.input(name="later-events-end", type="hidden", value=format_datetime(following_end))

        page.p.close()

    def get_time_navigation(self):

        "Return the start and end dates for the calendar view."

        for args in [self.env.get_args(), self.env.get_query()]:
            if args.has_key("earlier"):
                start_name, end_name = "earlier-start", "earlier-end"
                break
            elif args.has_key("earlier-events"):
                start_name, end_name = "earlier-events-start", "earlier-events-end"
                break
            elif args.has_key("later"):
                start_name, end_name = "later-start", "later-end"
                break
            elif args.has_key("later-events"):
                start_name, end_name = "later-events-start", "later-events-end"
                break
            elif args.has_key("start") or args.has_key("end"):
                start_name, end_name = "start", "end"
                break
        else:
            return None, None

        view_start = self.get_date_arg(args, start_name)
        view_end = self.get_date_arg(args, end_name)
        return view_start, view_end

    def get_time_navigation_args(self):

        "Return a dictionary containing start and/or end navigation details."

        view_period = self.get_view_period()
        view_start = view_period.get_start()
        view_end = view_period.get_end()
        link_args = {}
        if view_start:
            link_args["start"] = format_datetime(view_start)
        if view_end:
            link_args["end"] = format_datetime(view_end)
        return link_args

    def get_view_period(self):

        "Return the current view period."

        view_start, view_end = self.get_time_navigation()

        # Without any explicit limits, impose a reasonable view period.

        if not (view_start or view_end):
            view_start = get_date()
            view_end = get_date(timedelta(7))

        return Period(view_start, view_end, self.get_tzid())

    def show_view_period(self, view_period):

        "Show a description of the 'view_period'."

        _ = self.get_translator()

        page = self.page

        view_start = view_period.get_start()
        view_end = view_period.get_end()

        if not (view_start or view_end):
            return

        page.p(class_="view-period")

        if view_start and view_end:
            page.add(_("Showing events from %(start)s until %(end)s") % {
                "start" : self.format_date(view_start, "full"),
                "end" : self.format_date(view_end, "full")})
        elif view_start:
            page.add(_("Showing events from %s") % self.format_date(view_start, "full"))
        elif view_end:
            page.add(_("Showing events until %s") % self.format_date(view_end, "full"))

        page.p.close()

    def get_period_group_details(self, freebusy, participants, view_period):

        """
        Return details of periods in the given 'freebusy' collection and for the
        collections of the given 'participants'.
        """

        _ = self.get_translator()

        # Obtain the user's timezone.

        tzid = self.get_tzid()

        # Requests are listed and linked to their tentative positions in the
        # calendar. Other participants are also shown.

        request_summary = self._get_request_summary()

        period_groups = [request_summary, freebusy]
        period_group_types = ["request", "freebusy"]
        period_group_sources = [_("Pending requests"), _("Your schedule")]

        for i, participant in enumerate(participants):
            period_groups.append(self.store.get_freebusy_for_other(self.user, get_uri(participant)))
            period_group_types.append("freebusy-part%d" % i)
            period_group_sources.append(participant)

        groups = []
        group_columns = []
        group_types = period_group_types
        group_sources = period_group_sources
        all_points = set()

        # Obtain time point information for each group of periods.

        for periods in period_groups:

            # Filter periods outside the given view.

            if view_period:
                periods = periods.get_overlapping([view_period])

            # Get the time scale with start and end points.

            scale = get_scale(periods, tzid, view_period)

            # Get the time slots for the periods.
            # Time slots are collections of Point objects with lists of active
            # periods.

            slots = get_slots(scale)

            # Add start of day time points for multi-day periods.

            add_day_start_points(slots, tzid)

            # Remove the slot at the end of a view.

            if view_period:
                remove_end_slot(slots, view_period)

            # Record the slots and all time points employed.

            groups.append(slots)
            all_points.update([point for point, active in slots])

        # Partition the groups into days.

        days = {}
        partitioned_groups = []
        partitioned_group_types = []
        partitioned_group_sources = []

        for slots, group_type, group_source in zip(groups, group_types, group_sources):

            # Propagate time points to all groups of time slots.

            add_slots(slots, all_points)

            # Count the number of columns employed by the group.

            columns = 0

            # Partition the time slots by day.

            partitioned = {}

            for day, day_slots in partition_by_day(slots).items():

                # Construct a list of time intervals within the day.

                intervals = []

                # Convert each partition to a mapping from points to active
                # periods.

                partitioned[day] = day_points = {}

                last = None

                for point, active in day_slots:
                    columns = max(columns, len(active))
                    day_points[point] = active

                    if last:
                        intervals.append((last, point))

                    last = point

                if last:
                    intervals.append((last, None))

                if not days.has_key(day):
                    days[day] = set()

                # Record the divisions or intervals within each day.

                days[day].update(intervals)

            # Only include the requests column if it provides objects.

            if group_type != "request" or columns:
                if group_type != "request":
                    columns += 1
                group_columns.append(columns)
                partitioned_groups.append(partitioned)
                partitioned_group_types.append(group_type)
                partitioned_group_sources.append(group_source)

        return days, partitioned_groups, partitioned_group_types, partitioned_group_sources, group_columns

    # Full page output methods.

    def show(self):

        "Show the calendar for the current user."

        _ = self.get_translator()

        self.new_page(title=_("Calendar"))
        page = self.page

        if self.handle_newevent():
            return

        freebusy = self.store.get_freebusy(self.user)
        participants = self.update_participants()

        # Form controls are used in various places on the calendar page.

        page.form(method="POST")
        self.validator()
        self.show_user_navigation()
        self.show_requests_on_page()
        self.show_participants_on_page(participants)

        # Get the view period and details of events within it and outside it.

        view_period = self.get_view_period()

        # Day view: start at the earliest known day and produce days until the
        # latest known day, with expandable sections of empty days.

        (days, partitioned_groups, partitioned_group_types, partitioned_group_sources, group_columns) = \
            self.get_period_group_details(freebusy, participants, view_period)

        # Add empty days.

        add_empty_days(days, self.get_tzid(), view_period.get_start(), view_period.get_end())

        # Show controls to change the calendar appearance.

        self.show_view_period(view_period)
        self.show_calendar_controls()
        self.show_time_navigation(freebusy, view_period)

        # Show the calendar itself.

        self.show_calendar_days(days, partitioned_groups, partitioned_group_types, partitioned_group_sources, group_columns)

        # End the form region.

        page.form.close()

    # More page fragment methods.

    def show_calendar_day_controls(self, day):

        "Show controls for the given 'day' in the calendar."

        page = self.page
        daystr, dayid = self._day_value_and_identifier(day)

        # Generate a dynamic stylesheet to allow day selections to colour
        # specific days.
        # NOTE: The style details need to be coordinated with the static
        # NOTE: stylesheet.

        page.style(type="text/css")

        page.add("""\
input.newevent.selector#%s:checked ~ table#region-%s label.day,
input.newevent.selector#%s:checked ~ table#region-%s label.timepoint {
    background-color: #5f4;
    text-decoration: underline;
}
""" % (dayid, dayid, dayid, dayid))

        page.style.close()

        # Generate controls to select days.

        slots = self.env.get_args().get("slot", [])
        value, identifier = self._day_value_and_identifier(day)
        self._slot_selector(value, identifier, slots)

    def show_calendar_interval_controls(self, day, intervals):

        "Show controls for the intervals provided by 'day' and 'intervals'."

        page = self.page
        daystr, dayid = self._day_value_and_identifier(day)

        # Generate a dynamic stylesheet to allow day selections to colour
        # specific days.
        # NOTE: The style details need to be coordinated with the static
        # NOTE: stylesheet.

        l = []

        for point, endpoint in intervals:
            timestr, timeid = self._slot_value_and_identifier(point, endpoint)
            l.append("""\
input.newevent.selector#%s:checked ~ table#region-%s th#region-%s""" % (timeid, dayid, timeid))

        page.style(type="text/css")

        page.add(",\n".join(l))
        page.add(""" {
    background-color: #5f4;
    text-decoration: underline;
}
""")

        page.style.close()

        # Generate controls to select time periods.

        slots = self.env.get_args().get("slot", [])
        last = None

        # Produce controls for the intervals/slots. Where instants in time are
        # encountered, they are merged with the following slots, permitting the
        # selection of contiguous time periods. However, the identifiers
        # employed by controls corresponding to merged periods will encode the
        # instant so that labels may reference them conveniently.

        intervals = list(intervals)
        intervals.sort()

        for point, endpoint in intervals:

            # Merge any previous slot with this one, producing a control.

            if last:
                _value, identifier = self._slot_value_and_identifier(last, last)
                value, _identifier = self._slot_value_and_identifier(last, endpoint)
                self._slot_selector(value, identifier, slots)

            # If representing an instant, hold the slot for merging.

            if endpoint and point.point == endpoint.point:
                last = point

            # If not representing an instant, produce a control.

            else:
                value, identifier = self._slot_value_and_identifier(point, endpoint)
                self._slot_selector(value, identifier, slots)
                last = None

        # Produce a control for any unmerged slot.

        if last:
            _value, identifier = self._slot_value_and_identifier(last, last)
            value, _identifier = self._slot_value_and_identifier(last, endpoint)
            self._slot_selector(value, identifier, slots)

    def show_calendar_participant_headings(self, group_types, group_sources, group_columns):

        """
        Show headings for the participants and other scheduling contributors,
        defined by 'group_types', 'group_sources' and 'group_columns'.
        """

        page = self.page

        page.colgroup(span=1, id="columns-timeslot")

        # Make column groups at least two cells wide.

        for group_type, columns in zip(group_types, group_columns):
            page.colgroup(span=max(columns, 2), id="columns-%s" % group_type)

        page.thead()
        page.tr()
        page.th("", class_="emptyheading")

        for group_type, source, columns in zip(group_types, group_sources, group_columns):
            page.th(source,
                class_=(group_type == "request" and "requestheading" or "participantheading"),
                colspan=max(columns, 2))

        page.tr.close()
        page.thead.close()

    def show_calendar_days(self, days, partitioned_groups, partitioned_group_types,
        partitioned_group_sources, group_columns):

        """
        Show calendar days, defined by a collection of 'days', the contributing
        period information as 'partitioned_groups' (partitioned by day), the
        'partitioned_group_types' indicating the kind of contribution involved,
        the 'partitioned_group_sources' indicating the origin of each group, and
        the 'group_columns' defining the number of columns in each group.
        """

        _ = self.get_translator()

        page = self.page

        # Determine the number of columns required. Where participants provide
        # no columns for events, one still needs to be provided for the
        # participant itself.

        all_columns = sum([max(columns, 1) for columns in group_columns])

        # Determine the days providing time slots.

        all_days = days.items()
        all_days.sort()

        # Produce a heading and time points for each day.

        i = 0

        for day, intervals in all_days:
            groups_for_day = [partitioned.get(day) for partitioned in partitioned_groups]
            is_empty = True

            for slots in groups_for_day:
                if not slots:
                    continue

                for active in slots.values():
                    if active:
                        is_empty = False
                        break

            daystr, dayid = self._day_value_and_identifier(day)

            # Put calendar tables within elements for quicker CSS selection.

            page.div(class_="calendar")

            # Show the controls permitting day selection as well as the controls
            # configuring the new event display.

            self.show_calendar_day_controls(day)
            self.show_calendar_interval_controls(day, intervals)

            # Show an actual table containing the day information.

            page.table(cellspacing=5, cellpadding=5, class_="calendar %s" % (is_empty and " empty" or ""), id="region-%s" % dayid)

            page.caption(class_="dayheading container separator")
            self._day_heading(day)
            page.caption.close()

            self.show_calendar_participant_headings(partitioned_group_types, partitioned_group_sources, group_columns)

            page.tbody(class_="points")
            self.show_calendar_points(intervals, groups_for_day, partitioned_group_types, group_columns)
            page.tbody.close()

            page.table.close()

            # Show a button for scheduling a new event.

            page.p(class_="newevent-with-periods")
            page.label(_("Summary:"))
            page.input(name="summary-%d" % i, type="text")
            page.input(name="newevent-%d" % i, type="submit", value=_("New event"), accesskey="N")
            page.p.close()

            page.p(class_="newevent-with-periods")
            page.label(_("Clear selections"), for_="reset", class_="reset")
            page.p.close()

            page.div.close()

            i += 1

    def show_calendar_points(self, intervals, groups, group_types, group_columns):

        """
        Show the time 'intervals' along with period information from the given
        'groups', having the indicated 'group_types', each with the number of
        columns given by 'group_columns'.
        """

        _ = self.get_translator()

        page = self.page

        # Obtain the user's timezone.

        tzid = self.get_tzid()

        # Get view information for links.

        link_args = self.get_time_navigation_args()

        # Produce a row for each interval.

        intervals = list(intervals)
        intervals.sort()

        for point, endpoint in intervals:
            continuation = point.point == get_start_of_day(point.point, tzid)

            # Some rows contain no period details and are marked as such.

            have_active = False
            have_active_request = False

            for slots, group_type in zip(groups, group_types):
                if slots and slots.get(point):
                    if group_type == "request":
                        have_active_request = True
                    else:
                        have_active = True

            # Emit properties of the time interval, where post-instant intervals
            # are also treated as busy.

            css = " ".join([
                "slot",
                (have_active or point.indicator == Point.REPEATED) and "busy" or \
                    have_active_request and "suggested" or "empty",
                continuation and "daystart" or ""
                ])

            page.tr(class_=css)

            # Produce a time interval heading, spanning two rows if this point
            # represents an instant.

            if point.indicator == Point.PRINCIPAL:
                timestr, timeid = self._slot_value_and_identifier(point, endpoint)
                page.th(class_="timeslot", id="region-%s" % timeid,
                    rowspan=(endpoint and point.point == endpoint.point and 2 or 1))
                self._time_point(point, endpoint)
                page.th.close()

            # Obtain slots for the time point from each group.

            for columns, slots, group_type in zip(group_columns, groups, group_types):

                # Make column groups at least two cells wide.

                columns = max(columns, 2)
                active = slots and slots.get(point)

                # Where no periods exist for the given time interval, generate
                # an empty cell. Where a participant provides no periods at all,
                # one column is provided; otherwise, one more column than the
                # number required is provided.

                if not active:
                    self._empty_slot(point, endpoint, max(columns, 2))
                    continue

                slots = slots.items()
                slots.sort()
                spans = get_spans(slots)

                empty = 0

                # Show a column for each active period.

                for p in active:

                    # The period can be None, meaning an empty column.

                    if p:

                        # Flush empty slots preceding this one.

                        if empty:
                            self._empty_slot(point, endpoint, empty)
                            empty = 0

                        key = p.get_key()
                        span = spans[key]

                        # Produce a table cell only at the start of the period
                        # or when continued at the start of a day.
                        # Points defining the ends of instant events should
                        # never define the start of new events.

                        if point.indicator == Point.PRINCIPAL and (point.point == p.get_start() or continuation):

                            has_continued = continuation and point.point != p.get_start()
                            will_continue = not ends_on_same_day(point.point, p.get_end(), tzid)
                            is_organiser = p.organiser == self.user

                            css = " ".join([
                                "event",
                                has_continued and "continued" or "",
                                will_continue and "continues" or "",
                                p.transp == "ORG" and "only-organising" or is_organiser and "organising" or "attending",
                                self._have_request(p.uid, p.recurrenceid, "COUNTER", True) and "counter" or "",
                                ])

                            # Only anchor the first cell of events.
                            # Need to only anchor the first period for a recurring
                            # event.

                            html_id = "%s-%s-%s" % (group_type, p.uid, p.recurrenceid or "")

                            if point.point == p.get_start() and html_id not in self.html_ids:
                                page.td(class_=css, rowspan=span, id=html_id)
                                self.html_ids.add(html_id)
                            else:
                                page.td(class_=css, rowspan=span)

                            # Only link to events if they are not being updated
                            # by requests.

                            if not p.summary or \
                                group_type != "request" and self._have_request(p.uid, p.recurrenceid, None, True):

                                page.span(p.summary or _("(Participant is busy)"))

                            # Link to requests and events (including ones for
                            # which counter-proposals exist).

                            elif group_type == "request" and self._have_request(p.uid, p.recurrenceid, "COUNTER", True):
                                d = {"counter" : self._period_identifier(p)}
                                d.update(link_args)
                                page.a(p.summary, href=self.link_to(p.uid, p.recurrenceid, d))

                            else:
                                page.a(p.summary, href=self.link_to(p.uid, p.recurrenceid, link_args))

                            page.td.close()
                    else:
                        empty += 1

                # Pad with empty columns.

                empty = columns - len(active)

                if empty:
                    self._empty_slot(point, endpoint, empty, True)

            page.tr.close()

    def _day_heading(self, day):

        """
        Generate a heading for 'day' of the following form:

        <label class="day" for="day-20150203">Tuesday, 3 February 2015</label>
        """

        page = self.page
        value, identifier = self._day_value_and_identifier(day)
        page.label(self.format_date(day, "full"), class_="day", for_=identifier)

    def _time_point(self, point, endpoint):

        """
        Generate headings for the 'point' to 'endpoint' period of the following
        form:

        <label class="timepoint" for="slot-20150203T090000-20150203T100000">09:00:00 CET</label>
        <span class="endpoint">10:00:00 CET</span>
        """

        page = self.page
        tzid = self.get_tzid()
        value, identifier = self._slot_value_and_identifier(point, endpoint)
        page.label(self.format_time(point.point, "long"), class_="timepoint", for_=identifier)
        page.span(self.format_time(endpoint and endpoint.point or get_end_of_day(point.point, tzid), "long"), class_="endpoint")

    def _slot_selector(self, value, identifier, slots):

        """
        Provide a timeslot control having the given 'value', employing the
        indicated HTML 'identifier', and using the given 'slots' collection
        to select any control whose 'value' is in this collection, unless the
        "reset" request parameter has been asserted.
        """

        reset = self.env.get_args().has_key("reset")
        page = self.page
        if not reset and value in slots:
            page.input(name="slot", type="checkbox", value=value, id=identifier, class_="newevent selector", checked="checked")
        else:
            page.input(name="slot", type="checkbox", value=value, id=identifier, class_="newevent selector")

    def _empty_slot(self, point, endpoint, colspan, at_end=False):

        """
        Show an empty slot cell for the given 'point' and 'endpoint', with the
        given 'colspan' configuring the cell's appearance.
        """

        _ = self.get_translator()

        page = self.page
        page.td(class_="empty%s%s" % (point.indicator == Point.PRINCIPAL and " container" or "", at_end and " padding" or ""), colspan=colspan)
        if point.indicator == Point.PRINCIPAL:
            value, identifier = self._slot_value_and_identifier(point, endpoint)
            page.label(_("Select/deselect period"), class_="newevent popup", for_=identifier)
        page.td.close()

    def _day_value_and_identifier(self, day):

        "Return a day value and HTML identifier for the given 'day'."

        value = format_datetime(day)
        identifier = "day-%s" % value
        return value, identifier

    def _slot_value_and_identifier(self, point, endpoint):

        """
        Return a slot value and HTML identifier for the given 'point' and
        'endpoint'.
        """

        value = "%s-%s" % (format_datetime(point.point), endpoint and format_datetime(endpoint.point) or "")
        identifier = "slot-%s" % value
        return value, identifier

    def _period_identifier(self, period):
        return "%s-%s" % (format_datetime(period.get_start()), format_datetime(period.get_end()))

    def get_date_arg(self, args, name):
        values = args.get(name)
        if not values:
            return None
        return get_datetime(values[0], {"VALUE-TYPE" : "DATE"})

# vim: tabstop=4 expandtab shiftwidth=4
