#!/usr/bin/env python

"""
Common handler functionality for different entities.

Copyright (C) 2014, 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.data import get_address, get_uri, make_freebusy, to_part, \
                           uri_dict
from imiptools.dates import format_datetime
from imiptools.period import FreeBusyPeriod, Period

class CommonFreebusy:

    "Common free/busy mix-in."

    def _record_freebusy(self, from_organiser=True):

        """
        Record free/busy information for a message originating from an organiser
        if 'from_organiser' is set to a true value.
        """

        if from_organiser:
            organiser_item = self.require_organiser(from_organiser)
            if not organiser_item:
                return

            senders = [organiser_item]
        else:
            oa = self.require_organiser_and_attendees(from_organiser)
            if not oa:
                return

            organiser_item, attendees = oa
            senders = attendees.items()

            if not senders:
                return

        freebusy = [FreeBusyPeriod(p.get_start_point(), p.get_end_point()) for p in self.obj.get_period_values("FREEBUSY")]
        dtstart = self.obj.get_datetime("DTSTART")
        dtend = self.obj.get_datetime("DTEND")
        period = Period(dtstart, dtend, self.get_tzid())

        for sender, sender_attr in senders:
            stored_freebusy = self.store.get_freebusy_for_other_for_update(self.user, sender)
            stored_freebusy.replace_overlapping(period, freebusy)
            self.store.set_freebusy_for_other(self.user, stored_freebusy, sender)

    def request(self):

        """
        Respond to a request by preparing a reply containing free/busy
        information for each indicated attendee.
        """

        oa = self.require_organiser_and_attendees()
        if not oa:
            return

        (organiser, organiser_attr), attendees = oa

        # Get the details for each attendee.

        responses = []
        rwrite = responses.append

        # For replies, the organiser and attendee are preserved.

        for attendee, attendee_attr in attendees.items():
            freebusy = self.store.get_freebusy(attendee)

            # Indicate the actual sender of the reply.

            self.update_sender(attendee_attr)

            dtstart = self.obj.get_datetime("DTSTART")
            dtend = self.obj.get_datetime("DTEND")
            period = dtstart and dtend and Period(dtstart, dtend, self.get_tzid()) or None

            rwrite(make_freebusy(freebusy, self.uid, organiser, organiser_attr, attendee, attendee_attr, period))

        # Return the reply.

        self.add_result("REPLY", [get_address(organiser)], to_part("REPLY", responses))

class CommonEvent:

    "Common outgoing message handling functionality mix-in."

    def is_usable(self, method=None):

        "Return whether the current object is usable with the given 'method'."

        return self.obj and (
            method in ("CANCEL", "REFRESH") or
            self.obj.get_datetime("DTSTART") and
                (self.obj.get_datetime("DTEND") or self.obj.get_duration("DURATION")))

    def will_refresh(self):

        """
        Indicate whether a REFRESH message should be used to respond to an ADD
        message.
        """

        return not self.get_stored_object_version() or self.get_add_method_response() == "refresh"

    def make_refresh(self):

        "Make a REFRESH message."

        organiser = get_uri(self.obj.get_value("ORGANIZER"))
        attendees = uri_dict(self.obj.get_value_map("ATTENDEE"))

        # Indicate the actual sender of the message.

        attendee_attr = attendees[self.user]
        self.update_sender(attendee_attr)

        # Make a new object with a minimal property selection.

        obj = self.obj.copy()
        obj.preserve(("ORGANIZER", "DTSTAMP", "UID", "RECURRENCE-ID"))
        obj["ATTENDEE"] = [(self.user, attendee_attr)]

        # Send a REFRESH message in response.

        self.add_result("REFRESH", [get_address(organiser)], obj.to_part("REFRESH"))

    def is_newly_separated_occurrence(self):

        "Return whether the current object is a newly-separated occurrence."

        # Obtain any stored object.

        obj = self.get_stored_object_version()

        # Handle any newly-separated, valid occurrence.

        return not obj and self.is_recurrence()

    def make_separate_occurrence(self, for_organiser=False):

        """
        Set the current object as a separate occurrence and redefine free/busy
        records in terms of this new occurrence for other participants.
        """

        parent = self.get_parent_object()
        if not parent:
            return False

        # Transfer attendance information from the parent.

        parent_attendees = uri_dict(parent.get_value_map("ATTENDEE"))
        attendee_map = uri_dict(self.obj.get_value_map("ATTENDEE"))

        for attendee, attendee_attr in parent_attendees.items():
            if not attendee_map.has_key(attendee):
                attendee_map[attendee] = attendee_attr

        self.obj["ATTENDEE"] = attendee_map.items()
        self.obj.remove_all(["RDATE", "RRULE"])

        # Create or revive the occurrence.

        self.store.remove_cancellation(self.user, self.uid, self.recurrenceid)
        self.store.set_event(self.user, self.uid, self.recurrenceid, self.obj.to_node())

        # Update free/busy details for the current object for all attendees.

        self.update_freebusy_from_attendees(attendee_map.keys())

        return True

# vim: tabstop=4 expandtab shiftwidth=4
