/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor, Jon Tai
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Web;
using System.Threading;
using System.Text.RegularExpressions;

/* A big thanks to Jorn Baayen of Muine fame */
namespace Cowbell.Base
{
	[MetadataProxyPriority (20)]
	public class AmazonMetadataProxy : IMetadataProxy
	{
		/* private delegate */
		private delegate void AmazonImportHandler (string artist, string album);
		private delegate void DatabaseUpdateHandler (int i, Song s);

		/* public events */
		public event StringHandler Heartbeat;
		public event StringHandler ConnectionFailed;
		public event StringHandler ConnectionRejected;
		public event VoidHandler Completed;

		/* public fields */
		public bool Block;

		/* public methods */
		public void Import ()
		{
			if (!Block) {
				Runtime.Dispatcher.BackgroundDispatch (new AmazonImportHandler (ThreadedImport),
				                                       Runtime.Database.GlobalData.Artist,
				                                       Runtime.Database.GlobalData.Album);
			} else {
				ThreadedImport (Runtime.Database.GlobalData.Artist,
				                Runtime.Database.GlobalData.Album);
			}
		}

		/* private fields */
		static string DevTag = "1G1CFFX6R1ZWFXDMVJG2";

		private string year;

		/* private methods */
		private void Pulse (string text)
		{
			if (Heartbeat != null) {
				Runtime.Dispatcher.GuiDispatch (new StringHandler (Heartbeat),
			                                        text);
			}
		}

		private void ThreadedImport (string artist, string album)
		{
			Track[] tracks = new Track[0];
			Song s; int best_score, score = 0;
			int pos;

			Pulse (Catalog.GetString ("Retrieving results from server..."));

			// retry 3 times
			bool finished = false;
			int tries = 0;
			do {
				try {
					Pulse (Catalog.GetString ("Retrieving results from server..."));
					tracks = GetMetadataFromAmazon (artist, album);
					// Album really found or got bogus answers?
					if (tracks != null) {
						finished = true;
					}
					tries++;
				} catch (System.Net.WebException e) {
					if (ConnectionFailed != null) {
						Runtime.Dispatcher.GuiDispatch (new StringHandler (ConnectionFailed),
						                                Catalog.GetString ("The connection to http://soap.amazon.com "
						                                                   + "failed.  Please check your internet "
						                                                   + "connection and try again later.")); 

					}
					return;
				} catch (Exception e) {
					Runtime.Debug ("Got exception " + e.Message + ", " + e.GetType ());
					Thread.Sleep (500);
					tries++;
				}
			} while (!finished && tries < 3);
			
			if (!finished) {
				if (ConnectionRejected != null) {
					Runtime.Dispatcher.GuiDispatch (new StringHandler (ConnectionRejected),
					                                Catalog.GetString ("Amazon couldn't find the album "
					                                                   + "that you specified.  Please verify the Album "
				                                                           + "and Artist names and try again."));
 
				}
				return;
			}

			Pulse (Catalog.GetString ("Processing Results..."));
			for (int i = 0; i < Runtime.Database.Count; i++)
			{
				s = (Song)Runtime.Database[i];
				string[] db_tokens = SanitizeString (s.Title).Split (' ');

				best_score = 0;
				pos = -1;

				Pulse (String.Format (Catalog.GetString ("Matching {0}..."), ((Song)Runtime.Database[i]).Title));

				// Match these up with the tracks in the database
				for (int j = 0; j < tracks.Length; j++)
				{
					if (tracks[j] == null) {
						continue;
					}

					string[] amz_tokens = SanitizeString (tracks[j].TrackName).Split (' ');
					score = CalculateStringMatchScore ((string[])db_tokens.Clone (), amz_tokens);

					if (score >= 0) {
						Runtime.Debug ("Matching \"{0}\" <=> \"{1}\" ; score {2}", SanitizeString (s.Title), SanitizeString (tracks[j].TrackName), score);
					}

					if (score > best_score) {
						pos = j;
						best_score = score;
					}
				}

				if (best_score < 0) {
					// None of the words matched, so lets move on
					Pulse (String.Format (Catalog.GetString ("Found no match for {0}"), ((Song)Runtime.Database[i]).Title));
					continue;
				}
				
				if (pos > -1) {
					// we must have hit the best match now
					Song temp = (Song)Runtime.Database[i]; 

					// dump the amazon supplied information into the song
					temp.Artist = artist;
					if ((artist == String.Empty || artist == "Various Artists")
					    && tracks[pos].ByArtist != null) {
						temp.Artist = tracks[pos].ByArtist;
					} 

					temp.TrackNumber = Convert.ToUInt32 (pos + 1);
					temp.Title = tracks[pos].TrackName; 
					temp.Album = album;
					temp.Year = ParseAmazonDate (year);

					Runtime.Dispatcher.GuiDispatch (new DatabaseUpdateHandler (UpdateDatabase),
					                                i, temp);

					tracks[pos] = null;
				}
			}

			if (Completed != null) {
				Runtime.Dispatcher.GuiDispatch (new VoidHandler (Completed));
			}

			// update global data
			Runtime.Dispatcher.GuiDispatch (new VoidHandler (UpdateGlobalData));
		}

		/**
		 * Based on provided album and artist, look up data in Amazon
		 * and get a listing of tracks in the album.
		 */
		private Track[] GetMetadataFromAmazon (string artist, string album)
		{
			AmazonSearchService service = new AmazonSearchService ();
			Track[] result = null;
			int total, current, max;
			
			// FIXME: Make this change based on locale
			service.Url = "http://soap.amazon.com/onca/soap3";

			// Get rid of [disc 1], etc
			album = AmazonQuerySanitizeString (album);
			album = Regex.Replace (album, @"[,:]?\s* (cd|dis[ck])\s* (\d+|one|two|three|four|five|six|seven|eight|nine|ten)\s*$", "");
		
			artist = AmazonQuerySanitizeString (artist);	

			// handle multi-page results
			total = 1;
			current = 1;
			max = 2;
		
			// initialize an artist request
			ArtistRequest request = new ArtistRequest ();
			request.devtag = DevTag;

			if (!Runtime.Database.MultipleArtists) {
				request.artist = artist;
			} else {
				request.artist = "Various Artists";
			}

			request.keywords = album;
			request.type = "heavy";
			request.mode = "music";
			request.tag = "webservices-20";

			int best_distance = Int32.MaxValue;

			while (current <= total && current <= max)
			{
				ProductInfo pi;

				request.page = current.ToString ();

				// Amazon API makes us wait a tick
				Thread.Sleep (1000);
				service.Timeout = 3000;

				pi = service.ArtistSearchRequest (request);

				int num_results = pi.Details.Length;
				total = Convert.ToInt32 (pi.TotalPages);

				if (num_results < 1)
					return null;

				for (int i = 0; i < num_results; i++)
				{
					int distance;

					// Ignore bracketed text on the result from Amazon
					distance = Utils.LevenshteinDistance (SanitizeString (pi.Details[i].ProductName),
					                                      SanitizeString (album));

					// good lower bound for a "match"
					if (distance / (double)pi.Details[i].ProductName.Length > 0.6)	// TODO: test this heuristic
						continue;
					
					Track[] tracks = pi.Details[i].Tracks;

					if (tracks == null || tracks.Length < 1) {
						Runtime.Debug ("Got a null or empty track list from Amazon.  Rejecting result.");
						continue;
					}

					// If we have more tracks than Amazon
					// gave us within a fuzz (+/- 2), we didn't
					// find the right album.
					if (Runtime.Database.Count > (tracks.Length + 2)) {
						Runtime.Debug ("We have 3 more tracks than Amazon's results; we didn't find the right album.");
						continue;
					}
					
					if (distance > best_distance)
						continue;

					// we've found the best match for right now
					result = tracks;	
					best_distance = distance;
					
					artist = String.Join (",", pi.Details[i].Artists);
					album = pi.Details[i].ProductName;
					year = pi.Details[i].ReleaseDate;
				}
				current++;
			}

			return result;
		}

		private int CalculateStringMatchScore (string[] a, string[] b)
		{
			int score = 0;
			for (int i = 0; i < a.Length; i++)
			{
				int best_distance = 3;
				int best_pos = -1;
				for (int j = 0; j < b.Length; j++)
				{
					if (a[i] != String.Empty && b[j] != String.Empty) {
						int distance = Utils.LevenshteinDistance (a[i], b[j]);
						if (distance < best_distance) {
							best_distance = distance;
							best_pos = j;
						}
					}
				}

				if (best_pos > -1) {
					b[best_pos] = String.Empty;
					a[i] = String.Empty;

					score += 30 - (best_distance * 10);
				}
			}

			foreach (string token in a)
			{
				if (token != String.Empty) {
					score -= 2;
				}
			}

			foreach (string token in b)
			{
				if (token != String.Empty) {
					score -= 2;
				}
			}

			return score;
		}
		
		private string SanitizeString (string s)
		{
			s = s.ToLower ();
			s = s.Replace ("-", " ");
			s = s.Replace ("_", " ");
			s = Regex.Replace (s, " +", " ");

			return s;
		}

		private string AmazonQuerySanitizeString (string s)
		{
			s = s.ToLower ();
			s = Regex.Replace (s, "\\(.*\\)", "");
			s = Regex.Replace (s, "\\[.*\\]", "");
			s = s.Replace ("-", " ");
			s = s.Replace ("_", " ");
			s = Regex.Replace (s, " +", " ");

			return s;
		}

		/**
		 * Parses Amazon's date style (dd MMMM, YYYY) and returns the
		 * year part
		 */
		private uint ParseAmazonDate (string d)
		{
			uint ret = 0;
			try {
				ret = Convert.ToUInt32 (d.Substring (d.Length - 4));
			} catch { }
			
			return ret;
		}

		private void UpdateGlobalData ()
		{
			if (Runtime.Database.Count > 0) {
				Runtime.Database.GlobalData = (Song)Runtime.Database[0];
			}
		}

		private void UpdateDatabase (int i, Song s)
		{
			Runtime.Database[i] = s;
		}
	}
}
