/* soundfile.c - low-level sound file I/O implementation
 *
 * Based on libsndfile (http://www.mega-nerd.com/libsndfile/).
 *
 * Copyright 2010, 2016 Petteri Hintsanen <petterih@iki.fi>
 *
 * This file is part of abx.
 *
 * abx is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * abx is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with abx.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "soundfile.h"
#include <assert.h>
#include <glib.h>
#include <sndfile.h>

/*
 * Sound file handle type.  This struct should not be accessed
 * directly from user code.  See open_sound_file.
 */
struct Sound_file {
    SNDFILE* sffile;            /* Initialized sndfile handle. */
    Metadata metadata;
};

/*
 * Open sound file for reading.  Return handle to the opened sound
 * file, or NULL on error.
 */
Sound_file *
open_sound_file(const char *filename)
{
    Sound_file* sndfile;
    SNDFILE *sffile;
    SF_INFO sfinfo;

    if (!filename) return NULL;

    sfinfo.format = 0;
    sffile = sf_open(filename, SFM_READ, &sfinfo);
    if (!sffile) return NULL;

    sndfile = g_malloc(sizeof(Sound_file));
    sndfile->sffile = sffile;

    if (sfinfo.format & SF_FORMAT_PCM_S8) sndfile->metadata.bits = 8;
    else if (sfinfo.format & SF_FORMAT_PCM_16) sndfile->metadata.bits = 16;
    else if (sfinfo.format & SF_FORMAT_PCM_24) sndfile->metadata.bits = 24;
    else if (sfinfo.format & SF_FORMAT_PCM_32) sndfile->metadata.bits = 32;
    else {
        g_error("unsupported bits per sample "
                "(must be 8, 16, 24, or 32)");
    }

    sndfile->metadata.filename = g_strdup(filename);
    sndfile->metadata.rate = sfinfo.samplerate;
    sndfile->metadata.channels = sfinfo.channels;
    sndfile->metadata.frames = sfinfo.frames;
    sndfile->metadata.duration = (1.0 * sndfile->metadata.frames 
                                  / sndfile->metadata.rate);
    sndfile->metadata.minutes = 
	(int) sndfile->metadata.duration / 60;
    sndfile->metadata.seconds = 
	(int) sndfile->metadata.duration % 60;

    /* g_debug("opened '%s':\n" */
    /*         "  frames         : %i\n" */
    /*         "  sampling rate  : %i\n" */
    /*         "  bits per sample: %i\n" */
    /*         "  channels       : %i\n" */
    /*         "  duration       : %f (%d:%d)",  */
    /*         filename, sndfile->metadata.frames,  */
    /*         sndfile->metadata.rate, sndfile->metadata.bits,  */
    /*         sndfile->metadata.channels, sndfile->metadata.duration,  */
    /*         sndfile->metadata.minutes, sndfile->metadata.seconds); */

    return sndfile;
}

/*
 * Close sound file.  Return 0 on success, or non-zero on error.
 */
int
close_sound_file(Sound_file *sndfile)
{
    int rval;
    assert(sndfile);
    rval = sf_close(sndfile->sffile);
    g_free(sndfile->metadata.filename);
    g_free(sndfile);
    return rval;
}

/*
 * Return sound file metadata.
 */
Metadata
get_metadata(Sound_file *sndfile)
{
    assert(sndfile);
    return sndfile->metadata;
}

/*
 * Seek sound file.  Offset (in seconds) and whence are as in fseek.
 * Return the new location, or -1 on error.
 */
double
seek_sound_file(Sound_file *sndfile, double offset, int whence)
{
    sf_count_t frames, loc;
    assert(sndfile);
    /* convert offset to frame count */
    frames = sndfile->metadata.rate * offset;
    loc = sf_seek(sndfile->sffile, frames, whence);
    if (loc != -1) loc = 1.0 * loc / sndfile->metadata.rate;
    return loc;
}

/*
 * Read nframes frames of normalized (32-bit floating point) PCM data
 * from sound file to buf.  Return the number of frames read.
 */
unsigned int
read_pcm_data(Sound_file *sndfile, float *buf, unsigned int nframes)
{
    assert(sndfile && buf);
    return sf_readf_float(sndfile->sffile, buf, nframes);
}
